<?php

    /**
     * Make a subscription update button.
     *
     * @param  ID_TEXT        The purchase ID.
     * @param    ID_TEXT        Serialized data which stores transaction keys
     * @return tempcode        The button
     */
    public function make_update_button($purchase_id, $auto_found_key = null)
    {
        $url = build_url(array('page' => 'subscriptions', 'type' => 'update', 'id' => $purchase_id), get_module_zone('subscriptions'));
        $url = $url->evaluate();

        return do_template('ECOM_UPDATE_BUTTON_VIA_AUTHORIZE', array('URL' => $url));
    }

    /**
     * Get form fields.
     *
     * @param ID Subscription Id
     * @return tempcode The form fields
     */
    public function get_update_form_fields($subscription_id)
    {
        require_code('ecommerce');

        list($card_det, $hidden) = get_general_update_form_fields($subscription_id, $this);

        /*	General transaction fields returned from ecommerce api, attaching the details to the form.
			The fields can add/remove here, depending on the customised payment gateway features.
		*/
        $fields = new ocp_tempcode();
        foreach ($card_det as $field) {
            $fields->attach($field);
        }

        $verified_account_logo = $this->get_confidence_logos();
        return array($fields, $hidden, $verified_account_logo);
    }

    /**
     * Update subscription
     *
     * @param  ID           Subscription Id
     * @param  SHORT_TEXT    Cardholder first name
     * @param  SHORT_TEXT    Cardholder last name
     * @param  SHORT_TEXT    Card number
     * @param  SHORT_TEXT    Card Type
     * @param  SHORT_TEXT    Card Expiry date
     * @param  SHORT_TEXT    Card CV2 number (security number)
     * @param  SHORT_TEXT    Address for card
     * @param  SHORT_TEXT    City for card
     * @param  SHORT_TEXT    State for card
     * @param  SHORT_TEXT    Zip/post-code for card
     * @param  SHORT_TEXT    Country for card
     * @param  ID Member Id (NULL: Admin)
     * @return ARRAY
     */
    public function update_subscription($subscription_id, $first_name, $last_name, $card_number, $card_type, $expiry_date, $cv2, $address1, $city, $state, $zip, $country)
    {
        $temp = $GLOBALS['SITE_DB']->query_select_value('subscriptions', 's_auto_fund_key', array('id' => $subscription_id));
        $data = unserialize($temp);
        $authorize_subscription_id = $data['id'];

        $expiry_month = substr($expiry_date, 5, 2);
        $expiry_year = substr($expiry_date, 0, 4);
        $card_number = str_replace('-', '', str_replace(' ', '', $card_number));

        $this->_set_update_api_parameters($authorize_subscription_id, $first_name, $last_name, $card_number, $card_type, $expiry_month, $expiry_year, $cv2, $address1, $city, $state, $zip, $country);

        $response = http_download_file($this->url, null, true, false, 'Composr', array($this->api_parameters), null, null, null, null, null, null, null, 12.0, true);

        if ($response !== null) {
            $response_result = $this->parse_return($response);
            $success = ($response_result[0] == 'OK');
            $result = array($success, $response_result[1], $response_result[2], $response_result[2]);
        } else {
            $result = array(false, null, null, null);
        }

        return $result;
    }


    /**
     * This function defines the parameters needed to make an API call.
     *
     * @param  ID           Subscription Id
     * @param  SHORT_TEXT    Cardholder first name
     * @param  SHORT_TEXT    Cardholder last name
     * @param  SHORT_TEXT    Card number
     * @param  SHORT_TEXT    Card Type
     * @param  SHORT_TEXT    Card Expiry date
     * @param  SHORT_TEXT    Card CV2 number (security number)
     * @param  SHORT_TEXT    Address for card
     * @param  SHORT_TEXT    City for card
     * @param  SHORT_TEXT    State for card
     * @param  SHORT_TEXT    Zip/post-code for card
     * @param  SHORT_TEXT    Country for card
     * @param  ID Member Id (NULL: Admin)
     */
    protected function _set_update_api_parameters($authorize_subscription_id, $first_name, $last_name, $card_number, $card_type, $expiry_month, $expiry_year, $cv2, $address1, $city, $state, $zip, $country)
    {
        // NOTE: The first name and last name (and possibly others) need to be
        // given (usually via a custom profile field) otherwise Authorize.net
        // won't work.
        list($loginid, $transaction_key) = $this->_get_access_details();

        if (ecommerce_test_mode()) {
            //URL for test account
            $this->url = "https://apitest.authorize.net/xml/v1/request.api";
        } else {
            //URL for live account
            $this->url = "https://api.authorize.net/xml/v1/request.api";
        }

        $this->api_parameters =
            "<ARBUpdateSubscriptionRequest xmlns=\"AnetApi/xml/v1/schema/AnetApiSchema.xsd\">" .
            "<merchantAuthentication>" .
            "<name>" . $loginid . "</name>" .
            "<transactionKey>" . $transaction_key . "</transactionKey>" .
            "</merchantAuthentication>" .
            "<subscriptionId>" . $authorize_subscription_id . "</subscriptionId>" .
            "<subscription>" .
            "<payment>" .
            "<creditCard>" .
            "<cardNumber>" . $card_number . "</cardNumber>" .
            "<expirationDate>" . $expiry_year . '-' . $expiry_month . "</expirationDate>" .
            "</creditCard>" .
            "</payment>" .
            "<billTo>";

        if ($first_name) {
            $this->api_parameters .= "<firstName>" . $first_name . "</firstName>";
        }
        if ($last_name) {
            $this->api_parameters .= "<lastName>" . $last_name . "</lastName>";
        }
        if ($address1) {
            $this->api_parameters .= "<address>" . $address1 . "</address>";
        }
        if ($city) {
            $this->api_parameters .= "<city>" . $city . "</city>";
        }
        if ($state) {
            $this->api_parameters .= "<state>" . $state . "</state>";
        }
        if ($zip) {
            $this->api_parameters .= "<zip>" . $zip . "</zip>";
        }
        if ($country) {
            $this->api_parameters .= "<country>" . $country . "</country>";
        }

        $this->api_parameters .=
            "</billTo>" .
            "</subscription>" .
            "</ARBUpdateSubscriptionRequest>";
    }


    /**
     * Function to parse CIM response
     *
     * @param SHORT_TEXT Response
     * @param SHORT_TEXT API type
     * @return ARRAY
     */
    protected function _parse_cim_return($response, $api_type)
    {
        $result_code = $this->_substring_between($response, '<resultCode>', '</resultCode>');
        $code = $this->_substring_between($response, '<code>', '</code>');
        $text = $this->_substring_between($response, '<text>', '</text>');

        switch ($api_type) {
            case 'profile':
                $profile_id = $this->_substring_between($response, '<customerProfileId>', '</customerProfileId>');
                $response = array(strtoupper($result_code), $code, $text, $profile_id);
                break;

            case 'payment_profile':
                $payment_profile_id = $this->_substring_between($response, '<customerPaymentProfileId>', '</customerPaymentProfileId>');
                $response = array(strtoupper($result_code), $code, $text, $payment_profile_id);
                break;

            case 'shipping':
                $shipping_address_id = $this->_substring_between($response, '<customerAddressId>', '</customerAddressId>');
                $response = array(strtoupper($result_code), $code, $text, $shipping_address_id);
                break;

            case 'transaction':
                $direct_response = $this->_substring_between($response, '<directResponse>', '</directResponse>');
                $direct_response_fields = explode(',', $direct_response);

                $response_code = $direct_response_fields[0]; // 1 = Approved 2 = Declined 3 = Error
                $response_reason_code = $direct_response_fields[2];
                $response_reason_text = $direct_response_fields[3];
                $approval_code = $direct_response_fields[4];
                $gateway_trans_id = $direct_response_fields[6];
                $amount = $direct_response_fields[9];

                $response = array(strtoupper($result_code), $code, $text, $response_code, $response_reason_code, $response_reason_text, $approval_code, $gateway_trans_id, $amount);
                break;

            default:
                $response = array(strtoupper($result_code), $code, $text, '');
        }

        return $response;
    }

    /**
     * Get CIM duplicate profile id
     *
     * @param SHORT_TEXT Response
     * @return ID_TEXT
     */
    protected function _get_cim_duplicate_profile_id($response)
    {
        $profile_id = $this->_substring_between($response, 'A duplicate record with id ', ' already exists');

        return $profile_id;
    }

    /**
     * Get CIM error
     *
     * @param SHORT_TEXT Error code
     * @return SHORT_TEXT Error
     */
    protected function _get_cim_error($code)
    {
        require_lang('ecommerce');

        require_lang('ecommerce_gateway_authorize');

        switch ($code) {
            case '100003':
                $error_msg = do_lang_tempcode('CIM_RECORD_DELETED');
                break;

            case 'E00001':
                $error_msg = do_lang_tempcode('CIM_UNEXPECTED_ERROR');
                break;

            case 'E00002':
                $error_msg = do_lang_tempcode('CIM_CONTENT_TYPE_ERROR');
                break;

            case 'E00003':
                $error_msg = do_lang_tempcode('CIM_XML_PARSE_ERROR');
                break;

            case 'E00004':
                $error_msg = do_lang_tempcode('CIM_API_METHOD_INVALID');
                break;

            case 'E00005':
                $error_msg = do_lang_tempcode('CIM_TRANSACTION_KEY_ERROR');
                break;

            case 'E00006':
                $error_msg = do_lang_tempcode('CIM_LOGIN_ID_ERROR');
                break;

            case 'E00007':
                $error_msg = do_lang_tempcode('CIM_AUTHENTICATION_ERROR');
                break;

            case 'E00008':
                $error_msg = do_lang_tempcode('CIM_ACCOUNT_INACTIVE');
                break;

            case 'E00009':
                $error_msg = do_lang_tempcode('CIM_ACCOUNT_TEST_MODE');
                break;

            case 'E00010':
                $error_msg = do_lang_tempcode('CIM_PERMISSION_DENIED');
                break;

            case 'E00011':
                $error_msg = do_lang_tempcode('CIM_PERMISSION_DENIED');
                break;

            case 'E00013':
                $error_msg = do_lang_tempcode('CIM_FIELD_INVALID');
                break;

            case 'E00014':
                $error_msg = do_lang_tempcode('CIM_REQUIRED_FIELD_MISSING');
                break;

            case 'E00015':
                $error_msg = do_lang_tempcode('CIM_FIELD_LENGTH_INVALID');
                break;

            case 'E00016':
                $error_msg = do_lang_tempcode('CIM_FIELD_TYPE_INVALID');
                break;

            case 'E00019':
                $error_msg = do_lang_tempcode('CIM_REQUIRED_TAX_LICENSE');
                break;

            case 'E00027':
                $error_msg = do_lang_tempcode('CIM_TRANSACTION_UNSUCCESSFUL');
                break;

            case 'E00029':
                $error_msg = do_lang_tempcode('CIM_PAYMENT_INFO_REQUIRED');
                break;

            case 'E00039':
                $error_msg = do_lang_tempcode('CIM_DUPLICATE');
                break;

            case 'E00040':
                $error_msg = do_lang_tempcode('CIM_NO_RECORED');
                break;

            case 'E00041':
                $error_msg = do_lang_tempcode('CIM_NO_FIELD_VALUE');
                break;

            case 'E00042':
                $error_msg = do_lang_tempcode('CIM_MAXIMUM_PAYMENT_PROFILE');
                break;

            case 'E00043':
                $error_msg = do_lang_tempcode('CIM_MAXIMUM_SHIPPING_ADDRESS');
                break;

            case 'E00044':
                $error_msg = do_lang_tempcode('CIM_NOT_ENABLED');
                break;

            case 'E00051':
                $error_msg = do_lang_tempcode('CIM_NO_ORIGINAL_TRANSACTION');
                break;

            default:
                $error_msg = do_lang_tempcode('CIM_GENERAL_ERROR');
        }

        return $error_msg;
    }

    /**
     * This function defines the parameters needed to make an API call for profile creation.
     *
     * @param SHORT_TEXT Email Id
     * @param SHORT_TEXT Description
     */
    protected function _set_cim_profile_parameters($email, $description = null)
    {
        $access_detail = $this->_get_access_details();

        if (ecommerce_test_mode()) {
            $this->url = 'https://apitest.authorize.net/xml/v1/request.api';
        } else {
            $this->url = 'https://api.authorize.net/xml/v1/request.api';
        }

        $this->api_parameters =
            '<createCustomerProfileRequest xmlns="AnetApi/xml/v1/schema/AnetApiSchema.xsd">' .
            '<merchantAuthentication>' .
            '<name>' . $access_detail[0] . '</name>' .
            '<transactionKey>' . $access_detail[1] . '</transactionKey>' .
            '</merchantAuthentication>' .
            '<profile>' .
            '<description>' . $description . '</description>' .
            '<email>' . $email . '</email>' .
            '</profile>' .
            '</createCustomerProfileRequest>';
    }

    /**
     * Create CIM profile.
     *
     * @param SHORT_TEXT Email Id
     * @param SHORT_TEXT Description
     * @param ARRAY
     */
    protected function _create_cim_profile($email, $description = null)
    {
        $this->_set_cim_profile_parameters($email, $description);

        $response = http_download_file($this->url, null, true, false, 'Composr', array($this->api_parameters), null, null, null, null, null, null, null, 12.0, true);

        if ($response) {
            $response_result = $this->_parse_cim_return($response, 'profile');
            $success = ($response_result[0] == 'OK');
            $result = array($success, $response_result[1], $response_result[2], $response_result[3]);
        } else {
            $result = array(false, null, null, null);
        }

        return $result;
    }

    /**
     * This function defines the parameters needed to make an API call for payment profile creation.
     *
     * @param SHORT_TEXT Profile Id
     * @param SHORT_TEXT First name
     * @param SHORT_TEXT Last name
     * @param SHORT_TEXT Card number
     * @param SHORT_TEXT Card expiry year
     * @param SHORT_TEXT Card expiry month
     * @param SHORT_TEXT Address
     * @param SHORT_TEXT City
     * @param SHORT_TEXT A valid two-character state code
     * @param SHORT_TEXT Zip
     * @param SHORT_TEXT Country
     * @param SHORT_TEXT Last name
     */
    protected function _set_cim_payment_profile_parameters($profile_id, $first_name, $last_name, $card_number, $expiry_year, $expiry_month, $address = null, $city = null, $state = null, $zip = null, $country = null)
    {
        $access_detail = $this->_get_access_details();

        if (ecommerce_test_mode()) {
            $this->url = 'https://apitest.authorize.net/xml/v1/request.api';
            $mode = 'testMode';
        } else {
            $this->url = 'https://api.authorize.net/xml/v1/request.api';
            $mode = 'liveMode';
        }

        $this->api_parameters =
            '<createCustomerPaymentProfileRequest xmlns="AnetApi/xml/v1/schema/AnetApiSchema.xsd">' .
            '<merchantAuthentication>' .
            '<name>' . $access_detail[0] . '</name>' .
            '<transactionKey>' . $access_detail[1] . '</transactionKey>' .
            '</merchantAuthentication>' .
            '<customerProfileId>' . $profile_id . '</customerProfileId>' .
            '<paymentProfile>' .
            '<billTo>' .
            '<firstName>' . $first_name . '</firstName>' .
            '<lastName>' . $last_name . '</lastName>';
        if ($address) {
            $this->api_parameters .= '<address>' . $address . '</address>';
        }
        if ($city) {
            $this->api_parameters .= '<city>' . $city . '</city>';
        }
        if ($state) {
            $this->api_parameters .= '<state>' . $state . '</state>';
        }
        if ($zip) {
            $this->api_parameters .= '<zip>' . $zip . '</zip>';
        }
        if ($country) {
            $this->api_parameters .= '<country>' . $country . '</country>';
        }
        $this->api_parameters .= '</billTo>' .
                                 '<payment>' .
                                 '<creditCard>' .
                                 '<cardNumber>' . $card_number . '</cardNumber>' .
                                 '<expirationDate>' . $expiry_year . '-' . $expiry_month . '</expirationDate>' .
                                 '</creditCard>' .
                                 '</payment>' .
                                 '</paymentProfile>' .
                                 '<validationMode>' . $mode . '</validationMode>' .
                                 '</createCustomerPaymentProfileRequest>';
    }

    /**
     * Create CIM payment profile.
     *
     * @param SHORT_TEXT Profile Id
     * @param SHORT_TEXT First name
     * @param SHORT_TEXT Last name
     * @param SHORT_TEXT Card number
     * @param SHORT_TEXT Card expiry year
     * @param SHORT_TEXT Card expiry month
     * @param SHORT_TEXT Address
     * @param SHORT_TEXT City
     * @param SHORT_TEXT A valid two-character state code
     * @param SHORT_TEXT Zip
     * @param SHORT_TEXT Country
     * @param ARRAY
     */
    protected function _create_cim_payment_profile($profile_id, $first_name, $last_name, $card_number, $expiry_year, $expiry_month, $address = null, $city = null, $state = null, $zip = null, $country = null)
    {
        $card_number = str_replace('-', '', str_replace(' ', '', $card_number));

        $this->_set_cim_payment_profile_parameters($profile_id, $first_name, $last_name, $card_number, $expiry_year, $expiry_month, $address, $city, $state, $zip, $country);

        $response = http_download_file($this->url, null, true, false, 'Composr', array($this->api_parameters), null, null, null, null, null, null, null, 12.0, true);

        if ($response) {
            $response_result = $this->_parse_cim_return($response, 'payment_profile');
            $success = ($response_result[0] == 'OK');
            $result = array($success, $response_result[1], $response_result[2], $response_result[3]);
        } else {
            $result = array(false, null, null, null);
        }

        return $result;
    }

    /**
     * This function defines the parameters needed to make an API call for shipping address id.
     *
     * @param SHORT_TEXT Profile Id
     * @param SHORT_TEXT First name
     * @param SHORT_TEXT Last name
     * @param SHORT_TEXT Address
     * @param SHORT_TEXT City
     * @param SHORT_TEXT A valid two-character state code
     * @param SHORT_TEXT Zip
     * @param SHORT_TEXT Country
     */
    protected function _set_cim_shipping_address_parameters($profile_id, $first_name, $last_name, $address = null, $city = null, $state = null, $zip = null, $country = null)
    {
        $access_detail = $this->_get_access_details();

        if (ecommerce_test_mode()) {
            $this->url = 'https://apitest.authorize.net/xml/v1/request.api';
        } else {
            $this->url = 'https://api.authorize.net/xml/v1/request.api';
        }

        $this->api_parameters =
            '<createCustomerShippingAddressRequest xmlns="AnetApi/xml/v1/schema/AnetApiSchema.xsd">' .
            '<merchantAuthentication>' .
            '<name>' . $access_detail[0] . '</name>' .
            '<transactionKey>' . $access_detail[1] . '</transactionKey>' .
            '</merchantAuthentication>' .
            '<customerProfileId>' . $profile_id . '</customerProfileId>' .
            '<address>' .
            '<firstName>' . $first_name . '</firstName>' .
            '<lastName>' . $last_name . '</lastName>';
        if ($address) {
            $this->api_parameters .= '<address>' . $address . '</address>';
        }
        if ($city) {
            $this->api_parameters .= '<city>' . $city . '</city>';
        }
        if ($state) {
            $this->api_parameters .= '<state>' . $state . '</state>';
        }
        if ($zip) {
            $this->api_parameters .= '<zip>' . $zip . '</zip>';
        }
        if ($country) {
            $this->api_parameters .= '<country>' . $country . '</country>';
        }
        $this->api_parameters .= '</address>' .
                                 '</createCustomerShippingAddressRequest>';
    }

    /**
     * Create CIM shipping address id.
     *
     * @param SHORT_TEXT Profile Id
     * @param SHORT_TEXT First name
     * @param SHORT_TEXT Last name
     * @param SHORT_TEXT Address
     * @param SHORT_TEXT City
     * @param SHORT_TEXT A valid two-character state code
     * @param SHORT_TEXT Zip
     * @param SHORT_TEXT Country
     * @param ARRAY
     */
    protected function _create_cim_shipping_address_id($profile_id, $first_name, $last_name, $address = null, $city = null, $state = null, $zip = null, $country = null)
    {
        $this->_set_cim_shipping_address_parameters($profile_id, $first_name, $last_name, $address, $city, $state, $zip, $country);

        $response = http_download_file($this->url, null, true, false, 'Composr', array($this->api_parameters), null, null, null, null, null, null, null, 12.0, true);

        if ($response) {
            $response_result = $this->_parse_cim_return($response, 'shipping');
            $success = ($response_result[0] == 'OK');
            $result = array($success, $response_result[1], $response_result[2], $response_result[3]);
        } else {
            $result = array(false, null, null, null);
        }

        return $result;
    }

    /**
     * This function defines the parameters needed to make an API call for deleting a profile.
     *
     * @param SHORT_TEXT Profile Id
     */
    protected function _set_cim_delete_profile_parameters($profile_id)
    {
        $access_detail = $this->_get_access_details();

        if (ecommerce_test_mode()) {
            $this->url = 'https://apitest.authorize.net/xml/v1/request.api';
        } else {
            $this->url = 'https://api.authorize.net/xml/v1/request.api';
        }

        $this->api_parameters =
            '<deleteCustomerProfileRequest xmlns="AnetApi/xml/v1/schema/AnetApiSchema.xsd">' .
            '<merchantAuthentication>' .
            '<name>' . $access_detail[0] . '</name>' .
            '<transactionKey>' . $access_detail[1] . '</transactionKey>' .
            '</merchantAuthentication>' .
            '<customerProfileId>' . $profile_id . '</customerProfileId>' .
            '</deleteCustomerProfileRequest>';
    }

    /**
     * Delete CIM profile.
     *
     * @param SHORT_TEXT Profile Id
     * @param ARRAY
     */
    protected function _delete_cim_profile($profile_id)
    {
        $this->_set_cim_delete_profile_parameters($profile_id);

        $response = http_download_file($this->url, null, true, false, 'Composr', array($this->api_parameters), null, null, null, null, null, null, null, 12.0, true);

        if ($response) {
            $response_result = $this->_parse_cim_return($response, 'delete');
            $success = ($response_result[0] == 'OK');
            $result = array($success, $response_result[1], $response_result[2], $response_result[3]);
        } else {
            $result = array(false, null, null, null);
        }

        return $result;
    }

    /**
     * This function defines the parameters needed to make an API call for transaction.
     *
     * @param SHORT_TEXT Item Profile Id
     * @param SHORT_TEXT Item Payment profile Id
     * @param SHORT_TEXT Item Shipping address Id
     * @param SHORT_TEXT Transaction amount (should include tax, shipping, and everything.)
     * @param SHORT_TEXT Shipping amount
     * @param SHORT_TEXT Shipping name
     * @param SHORT_TEXT Shipping description
     * @param SHORT_TEXT Tax amount
     * @param SHORT_TEXT Tax name
     * @param SHORT_TEXT Tax description
     * @param SHORT_TEXT Item name
     * @param SHORT_TEXT Item description
     * @param SHORT_TEXT Item quantity
     * @param SHORT_TEXT Item unit price
     */
    protected function _set_cim_transaction_parameters($profile_id, $payment_profile_id, $shipping_address_id, $amount, $shipping_amount = '0.00', $shipping_name = 'Free Shipping', $shipping_description = 'Free', $tax_amount = '0.00', $tax_name = 'Tax Free', $tax_description = 'Tax Free', $item_id = null, $item_name = null, $item_description = null, $item_quantity = null, $item_unit_price = null)
    {
        $access_detail = $this->_get_access_details();

        if (ecommerce_test_mode()) {
            $this->url = 'https://apitest.authorize.net/xml/v1/request.api';
        } else {
            $this->url = 'https://api.authorize.net/xml/v1/request.api';
        }

        $this->api_parameters =
            '<createCustomerProfileTransactionRequest xmlns="AnetApi/xml/v1/schema/AnetApiSchema.xsd">' .
            '<merchantAuthentication>' .
            '<name>' . $access_detail[0] . '</name>' .
            '<transactionKey>' . $access_detail[1] . '</transactionKey>' .
            '</merchantAuthentication>' .
            '<transaction>' .
            '<profileTransAuthOnly>' .
            '<amount>' . $amount . '</amount>' .
            '<tax>' .
            '<amount>' . $tax_amount . '</amount>' .
            '<name>' . $tax_name . '</name>' .
            '<description>' . $tax_description . '</description>' .
            '</tax>' .
            '<shipping>' .
            '<amount>' . $shipping_amount . '</amount>' .
            '<name>' . $shipping_name . '</name>' .
            '<description>' . $shipping_description . '</description>' .
            '</shipping>';

        if ($item_id && $item_name && $item_description && $item_quantity && $item_unit_price) {
            $this->api_parameters .= '<lineItems>' .
                                     '<itemId>' . $item_id . '</itemId>' .
                                     '<name>' . $item_name . '</name>' .
                                     '<description>' . $item_description . '</description>' .
                                     '<quantity>' . $item_quantity . '</quantity>' .
                                     '<unitPrice>' . $item_unit_price . '</unitPrice>';
            '</lineItems>';
        }

        $this->api_parameters .= '<customerProfileId>' . $profile_id . '</customerProfileId>' .
                                 '<customerPaymentProfileId>' . $payment_profile_id . '</customerPaymentProfileId>' .
                                 '<customerShippingAddressId>' . $shipping_address_id . '</customerShippingAddressId>' .
                                 '</profileTransAuthOnly>' .
                                 '</transaction>' .
                                 '</createCustomerProfileTransactionRequest>';
    }

    /**
     * Make a transaction using CIM
     *
     * @param SHORT_TEXT Item Profile Id
     * @param SHORT_TEXT Item Payment profile Id
     * @param SHORT_TEXT Item Shipping address Id
     * @param SHORT_TEXT Transaction amount (should include tax, shipping, and everything.)
     * @param SHORT_TEXT Shipping amount
     * @param SHORT_TEXT Shipping name
     * @param SHORT_TEXT Shipping description
     * @param SHORT_TEXT Tax amount
     * @param SHORT_TEXT Tax name
     * @param SHORT_TEXT Tax description
     * @param SHORT_TEXT Item name
     * @param SHORT_TEXT Item description
     * @param SHORT_TEXT Item quantity
     * @param SHORT_TEXT Item unit price
     * @param ARRAY
     */
    protected function _do_cim_transaction($profile_id, $payment_profile_id, $shipping_address_id, $amount, $shipping_amount = '0.00', $shipping_name = 'Free Shipping', $shipping_description = 'Free', $tax_amount = '0.00', $tax_name = 'Tax Free', $tax_description = 'Tax Free', $item_id = null, $item_name = null, $item_description = null, $item_quantity = null, $item_unit_price = null)
    {
        $this->_set_cim_transaction_parameters($profile_id, $payment_profile_id, $shipping_address_id, $amount, $shipping_amount, $shipping_name, $shipping_description, $tax_amount, $tax_name, $tax_description, $item_id, $item_name, $item_description, $item_quantity, $item_unit_price);

        $response = http_download_file($this->url, null, true, false, 'Composr', array($this->api_parameters), null, null, null, null, null, null, null, 12.0, true);

        if ($response) {
            $response_result = $this->_parse_cim_return($response, 'transaction');
            $success = ($response_result[0] == 'OK');
            $result = array($success, $response_result[1], $response_result[2], $response_result[3], $response_result[4], $response_result[5], $response_result[6], $response_result[7], $response_result[8]);
        } else {
            $result = array(false, null, null, null, null, null, null, null, null);
        }

        return $result;
    }
