<?php /*

 ocPortal
 Copyright (c) ocProducts, 2004-2009

 See text/en/licence.txt for full licencing information.

*/

/**
* @license		http://opensource.org/licenses/cpal_1.0 Common Public Attribution License
* @copyright	ocProducts Ltd
* @package		shopping
*/

class Hook_catalogue_items
{
	var $product_id;
	var $shipping_amounts;
	var $shipping_equation_amts;
	/**
	 *	Get the products handled by this eCommerce hook.
    *
	 * IMPORTANT NOTE TO PROGRAMMERS: This function may depend only on the database, and not on get_member() or any GET/POST values.
    *  Such dependencies will break IPN, which works via a Guest and no dependable environment variables. It would also break manual transactions from the Admin Zone.
	 *
	 * @param  boolean	Whether to make sure the language for item_name is the site default language (crucial for when we read/go to third-party sales systems and use the item_name as a key).
	 * @param  boolean 	Weather item name should be the key of return array
	 * @return array		A map of product name to list of product details.
	 */
	function get_products($site_lang=false)
	{
		require_code('catalogues');

		$products		=	array();

		$price			=	array();

		$product_title	=	array();

		$qry			=	"SELECT t1.id,t1.c_name from ".get_table_prefix()."catalogue_entries t1,  ".get_table_prefix()."catalogues t2 WHERE t1.c_name=t2.c_name and t2.c_ecommerce=1";

		$ecomm_catalogues	=	$GLOBALS['SITE_DB']->query($qry);

		foreach($ecomm_catalogues as $ecomm_catalogue)
		{
			$map		=	get_catalogue_entry_field_values($ecomm_catalogue['c_name'],$ecomm_catalogue['id']);

			$item_price	=	'0.0';
			$tax			=	0.0;

			$product_weight	=	0.0;

			if(array_key_exists(2,$map))
				$item_price	=	$map[2]['effective_value'];

			if(array_key_exists(0,$map) && array_key_exists('effective_value_pure',$map[0]))
			{
				$product_title[$ecomm_catalogue['id']]	=	$map[0]['effective_value_pure'];
			}

			if(array_key_exists(6,$map))
				$tax		=	(is_object($map[6]['effective_value'])?$map[6]['effective_value']->evaluate():$map[6]['effective_value']);

			if(array_key_exists(8,$map))
				$product_weight	=	floatval(is_object($map[8]['effective_value'])?$map[8]['effective_value']->evaluate():$map[8]['effective_value']);

			list($tax_type,$tax_rate)			=	$this->get_product_tax($tax);

			$price[$ecomm_catalogue['id']]	=	float_to_raw_string($this->calculate_product_price(floatval($item_price),$tax_rate,$product_weight));

			$product		=	do_lang('PRODUCT_CATALOGUE_ITEMS',strval($ecomm_catalogue['id']));

			$products[$product]	=	array(PRODUCT_CATALOGUE,$price[$ecomm_catalogue['id']],'handle_product_orders_items',array('id'=>$ecomm_catalogue['id'],'tax'=>$tax),$product_title[$ecomm_catalogue['id']]);
		}

		return $products;
	}

	/**
	 * Check whether the product has shipping feature
	 *
	 * @return boolean	Whether it is.
	 */
	function has_shipping_feature()
	{
		return true;
	}

	/**
	 * Check whether the product code is available for purchase by the member.
	 *
	 * @param  ID_TEXT	The product.
	 * @param  ?MEMBER	The member we are checking against (NULL: current meber).
	 * @param  integer	The number required.
	 * @return boolean	Whether it is.
	 */
	function is_available($product,$member=NULL,$req_quantity=1)
	{
		require_code('catalogues');

		$res			=	$GLOBALS['SITE_DB']->query_select('catalogue_entries',array('*'),array('id'=>$product));

		if(!array_key_exists(0,$res)) return AVAILABLE_NO;

		$product_det	=	$res[0];

		$fields	=	get_catalogue_entry_field_values($product_det['c_name'],$product_det['id']);

		$str	=	NULL;

		if(!array_key_exists(3,$fields)) return AVAILABLE_YES;

		if((is_null($fields[4]['effective_value'])) || (intval($fields[4]['effective_value'])==0)) return AVAILABLE_YES;

		if($fields[3]['effective_value']!='')
		{
			$available_stock	=	intval($fields[3]['effective_value']);

			$rows	=	$GLOBALS['SITE_DB']->query("SELECT id FROM ".get_table_prefix()."shopping_order WHERE order_status='ORDER_STATUS_awaiting_payment' AND add_date< ".strval(time()-60*60*1)." AND session_id=".strval(get_session_id()));

			if(array_key_exists(0,$rows))
			{
				foreach($rows as $row)
				{
					$GLOBALS['SITE_DB']->query_delete('shopping_order',array('id'=>$row['id']));
					$GLOBALS['SITE_DB']->query_delete('shopping_order_details',array('order_id'=>$row['id']));
				}
			}

			$res	=	$GLOBALS['SITE_DB']->query('SELECT sum(t2.p_quantity) as qty FROM '.get_table_prefix().'shopping_order t1,'.get_table_prefix().'shopping_order_details t2 WHERE t1.id=t2.order_id AND t1.order_status=\'ORDER_STATUS_awaiting_payment\' AND t2.p_id='.strval(intval($product)));

			if(array_key_exists(0,$res))
				$item_count	=	intval($res[0]['qty']);
			else
				$item_count	=	0;

			if(($available_stock-$item_count)<$req_quantity) return AVAILABLE_NO; else return AVAILABLE_YES;
		}

		return true;
	}

	/**
	 * Get currently available quantity of selected product
	 *
	 * @param  ID_TEXT	The product.
	 * @return ?integer	Quantity (NULL: no limit).
	 */
	function get_available_quantity($product)
	{
		require_code('catalogues');

		$res		=	$GLOBALS['SITE_DB']->query_select('catalogue_entries',array('*'),array('id'=>$product));

		if(!array_key_exists(0,$res)) return 0;

		$product_det	=	$res[0];

		$fields	=	get_catalogue_entry_field_values($product_det['c_name'],$product_det['id']);

		$str	=	NULL;


		if((is_null($fields[4]['effective_value'])) || (intval($fields[4]['effective_value'])==0)) return NULL;

		if($fields[3]['effective_value']!='')
		{
			$available_stock	=	intval($fields[3]['effective_value']);

			//Locked order check
			$res	=	$GLOBALS['SITE_DB']->query('SELECT sum(t2.p_quantity) as qty FROM '.get_table_prefix().'shopping_order t1,'.get_table_prefix().'shopping_order_details t2 WHERE t1.id=t2.order_id AND '.db_string_equal_to('t1.order_status','ORDER_STATUS_awaiting_payment').' AND t2.p_id='.strval(intval($product)).' AND t1.session_id<>'.strval(get_session_id()));

			if(array_key_exists(0,$res))
				$item_count	=	intval($res[0]['qty']);
			else
				$item_count	=	0;

			return ($available_stock-$item_count);
		}

		return NULL;
	}

	/**
	 * Get the message for use in the purchase wizard
	 *
	 * @param  string		The product in question.
	 * @return tempcode	The message.
	 */
	function get_message($product_id)
	{
		require_code('catalogues');

		$catalogue_name	=	$GLOBALS['SITE_DB']->query_value('catalogue_entries','c_name',array('id'=>$product_id));

		$catalogues	=	$GLOBALS['SITE_DB']->query_select('catalogues',array('*'),array('c_name'=>$catalogue_name),'',1);

		if (!array_key_exists(0,$catalogues)) warn_exit(do_lang_tempcode('CATALOGUE_NOT_FOUND',$catalogue_name));

		$catalogue	=	$catalogues[0];

		$entries	=	$GLOBALS['SITE_DB']->query_select('catalogue_entries',array('*'),array('id'=>$product_id),'',1);

		if (!array_key_exists(0,$entries)) return warn_screen(get_page_title('CATALOGUES'),do_lang_tempcode('MISSING_RESOURCE'));

		$entry		=	$entries[0];

		$map		=	get_catalogue_entry_map($entry,$catalogue,'PAGE',$catalogue_name,$product_id,NULL,NULL,true,true);

		return do_template('ECOMMERCE_ITEM_DETAILS',$map,NULL,false,'ECOMMERCE_ITEM_DETAILS');
	}

	/**
	 *	Get the products details
	 *
	 *	@param	?AUTO_LINK	Product ID (NULL: read from environment, product_id)
	 *	@return 	array			A map of product name to list of product details.
	 */
	function get_product_details($pid=NULL)
	{
		require_code('catalogues');

		$product_det	=	array();

		if(is_null($pid))
		{
			$pid	=	either_param_integer('product_id');
		}

		$qty		=	post_param_integer('quantity',1);

		$catalogue_name	=	$GLOBALS['SITE_DB']->query_value('catalogue_entries','c_name',array('id'=>$pid));

		$product_det	=	get_catalogue_entry_field_values($catalogue_name,$pid);

		foreach($product_det as $key=>$value)
		{
			$product_det[$key]	=	array_key_exists('effective_value_pure',$value)?$value['effective_value_pure']:$value['effective_value'];
		}

		list($tax_type,$tax_rate)	=	$this->get_product_tax($product_det[6]);

		$product	=	array
						(
							'product_id'	=>	$pid,
							'product_name'	=>	$product_det[0],
							'product_code'	=>	$product_det[1],
							'price'			=>	$product_det[2],
							'tax'				=>	$tax_rate,
							'tax_type'		=> $tax_type,
							'description'	=>	$product_det[9],
							'quantity'		=>	$qty,
							'product_type'	=>	'catalogue_items',
							'product_weight'=> $product_det[8]
						);

		return $product;
	}

	/**
	 *	Find tax from the tax string of catalogue
	 *
	 *	@param	?ID_TEXT		Tax rate setting (eg: Nevada=6.85%)
	 *	@return 	array			Array of tax type and Float value of tax setting
	 */
	function get_product_tax($tax)
	{
		$tax_bits=explode('=',$tax,2);
		if (!array_key_exists(1,$tax_bits)) $tax_bits[1]=$tax_bits[0];
		list($tax_type,$tax_rate)=$tax_bits;
		if ($tax_rate=='nil') $tax_rate='0.0';
		$tax_rate=preg_replace('#[^\d\.]#','',$tax_rate);
		return	array($tax_type,$tax_rate);
	}

	/**
	 *	Add an order
	 *
	 *	@param  array		Array of product details.
	 * @return AUTO_LINK	Order id of newly added order.
	 */
	function add_order($product_det)
	{
		if (method_exists($this,'is_available'))
		{
			if($this->is_available($product_det['product_id'],get_member())==AVAILABLE_NO)
				warn_exit(do_lang_tempcode('PRODUCT_UNAVAILABLE_NAMED',escape_html($product_det['product_name'])));
			elseif($this->is_available($product_det['product_id'],get_member())==AVAILABLE_IF_WAS_LOGGED_IN && is_guest())
				warn_exit(do_lang_tempcode('NOT_AS_GUEST'));
		}

		$qty	=	$GLOBALS['SITE_DB']->query_value_null_ok('shopping_cart','quantity',array('product_code'=>$product_det['product_code'],'session_id'=>get_session_id(),'is_deleted'=>0));

		if($qty==0)
		{
			$id=$GLOBALS['SITE_DB']->query_insert('shopping_cart',
					array(
						'session_id'		=>	get_session_id(),
						'ordered_by'		=>	get_member(),
						'product_id'		=>	$product_det['product_id'],
						'product_name'		=>	$product_det['product_name'],
						'product_code'		=>	$product_det['product_code'],
						'quantity'			=>	$product_det['quantity'],
						'price'				=>	round(floatval($product_det['price']),2),
						'tax'	            =>	floatval($product_det['tax']),
						'tax_type'			=> $product_det['tax_type'],
						'product_description'	=>	$product_det['description'],
						'product_type'		=>	$product_det['product_type'],
						'product_weight'	=>	floatval($product_det['product_weight']),
						'is_deleted' => 0,
					)
			);
		}
		else
		{
			$id=$GLOBALS['SITE_DB']->query_update('shopping_cart',
					array(
						'quantity'			=>	($qty+$product_det['quantity']),
						'price'				=>	round(floatval($product_det['price']),2),
						'tax'	=>	floatval($product_det['tax']),
					),
					array('product_name'=>$product_det['product_name'],'product_code'=>$product_det['product_code'],'session_id'=>get_session_id())
			);
		}

		return $id;
	}

	/**
	 * Add order - (order coming from purchase module)
	 *
	 * @param AUTO_LINK	Product id
	 * @param array		Product details
	 * @return AUTO_LINK	order id
	 */
	function set_needed_fields($product_id,$product_det)
	{
		require_lang('shopping');
		$unique_product_name	=	do_lang('PRODUCT_CATALOGUE_ITEMS',strval($product_id));

		if(get_option('allow_opting_out_of_tax')=='1' && post_param_integer('tax_opted_out',0)==1)
			$tax_opted_out	=	1;
		else
			$tax_opted_out	=	0;

		if (method_exists($this,'calculate_tax') && $tax_opted_out==0)
		{
			$tax_percentage	=	array_key_exists(0,$product_det[3])? $product_det[3][0] : 0;
			$tax	=	round($this->calculate_tax($product_det[1],$tax_percentage),2);
		}
		else
			$tax	=	0.0;

		$seller		=	$this->get_seller($product_id);
		$ship_opt	=	$GLOBALS['SITE_DB']->query_value_null_ok('shipping_equations','id',array('is_default'=>1));

		$order_id	=	$GLOBALS['SITE_DB']->query_insert('shopping_order',
								array(
									'c_member'			=>	get_member(),
									'session_id'		=>	get_session_id(),
									'add_date'			=>	time(),
									'tot_price'			=>	floatval($product_det[1]+$tax),
									'order_status'		=>	'ORDER_STATUS_awaiting_payment',
									'notes'				=>	'',
									'product_name'		=>	$unique_product_name,
									'transaction_id'	=> '',
									'tax_opted_out'	=>	$tax_opted_out,
									'to_member_id'		=>	$seller,
									'shipping_option'	=>	$ship_opt,
								),true);

		$GLOBALS['SITE_DB']->query_insert('shopping_order_details',
												array(
														'p_id'		=>	intval($product_id),
														'p_name'		=>	$product_det[4],
														'p_code'		=>	strval($product_det[0]),
														'p_type'		=>	'catalogue_items',
														'p_quantity'=>	1,
														'p_price'	=>	floatval($product_det[1]),
														'order_id'	=>	$order_id,
														'dispatch_status' => '',
														'included_tax'=>	$tax
												),true);
		return $order_id;
	}

	/**
	 * Show shopping cart entries
	 *
	 *	@param  tempcode	Tempcode object of shopping cart result table.
	 * @param  array		Details of new entry to the shopping cart.
	 * @return tempcode	Tempcode object of shopping cart result table.
	 */
	function show_cart_entry(&$shopping_cart,$entry)
	{
		$equation_amts = array();

		$tpl_set		=	'cart';

		require_code('images');

		$edit_qnty	=	do_template('SHOPPING_ITEM_QUANTITY_FIELD',array(
						'PRODUCT_ID'	=>	strval($entry['product_id']),
						'QUANTITY'	=>	strval($entry['quantity'])
						)
					);

		$tax				=	$this->calculate_tax($entry['price'],$entry['tax']);

		$shipping_cost	=	$this->calculate_shipping_cost($entry['product_weight'],true);

		if ($shipping_cost===false) attach_message(do_lang_tempcode('SHIPPING_EQUATION_ERROR'));

		$del_item	=	do_template('SHOPPING_ITEM_REMOVE_FIELD',array(
							'PRODUCT_ID'	=>	strval($entry['product_id']),
							)
					);

		$catalogue_name=	$GLOBALS['SITE_DB']->query_value('catalogue_entries','c_name',array('id'=>$entry['product_id']));

		$image			=	$this->get_product_image($catalogue_name,$entry['product_id']);

		$product_image	=	get_item_image_thumb($entry['product_name'],$image);

		$currency		=	ecommerce_get_currency_symbol();

		$price			=	number_format(round((round($entry['price']+$tax+$shipping_cost,2))*$entry['quantity'],2),2);

		$product_url	=	build_url(array('page'=>'catalogues','type'=>'entry','id'=>$entry['product_id']),'_SELF');

		$product_link	=	hyperlink($product_url,$entry['product_name'],false,false,do_lang('INDEX'));

		$shipping_cost	=	$shipping_cost*$entry['quantity'];

		$shopping_cart->attach(results_entry(
						array(
							$product_image,
							$product_link,
							$currency.float_format(round($entry['price'],2)),
							$edit_qnty,
							$currency.float_format($tax),
							$currency.float_format($shipping_cost,2),
							$currency.(string)$price,
							$del_item
						),false,$tpl_set
					)
				);

		$shopping_cart->catalogue	=	$catalogue_name;	//Setting shopping cart catalogue name to the object

		if(is_array($this->shipping_amounts))
			foreach($this->shipping_amounts as $eqid => $amt)
			{
				$equation_amts[$eqid]	=	$amt*$entry['quantity'];
				$shopping_cart->shipping_equation_amts	=	$equation_amts;
			}

		return $shopping_cart;
	}

	/**
	 * Calculate tax of catalogue product.
	 *
	 * @param  float		Gross cost of product.
	 * @param  float		Tax in percentage
	 * @return float		Calculated tax for the product.
	 */
	function calculate_tax($gross_cost,$tax_percentage)
	{
		if(addon_installed('shopping'))
		{
			require_code('shopping');
			if(get_order_tax_opt_out_status()==1) return 0.0;
		}

		$tax	=	($gross_cost*$tax_percentage)/100.0;
		return $tax;
	}

	/**
	 * Calculate shipping cost of product.
	 *
	 * @param  float		Weight of product
	 *	@param	BINARY	Whether the calculation of shipping is for selected shipping method or evaluate all shipping method amount
	 * @return float		Calculated shipping cost for the product
	 */
	function calculate_shipping_cost($weight,$evaluate_all=false)
	{
		$equation				=	NULL;
		$order_shipping_amt	=	NULL;
		$shipping_cost       =  NULL;

		$row			=	$GLOBALS['SITE_DB']->query('SELECT shipping_option FROM '.get_table_prefix().'shopping_order WHERE c_member='.strval(get_member()).' AND session_id='.strval(get_session_id()).' ORDER BY add_date DESC');

		$option_id	=	array_key_exists(0,$row)?$row[0]['shipping_option']:$GLOBALS['SITE_DB']->query_value_null_ok('shipping_equations','id',array('is_default'=>1));

		$cond			=	(!is_null($option_id) && !$evaluate_all)?	" WHERE id=".strval($option_id):NULL;

		$row			=	$GLOBALS['SITE_DB']->query('SELECT * FROM '.get_table_prefix().'shipping_equations '.db_escape_string($cond).' ORDER BY is_default DESC, add_date DESC');

		foreach($row as $equation_entry)
		{
			$equation	=	$equation_entry['equation'];
			$equation	=	'$shipping_cost= '.$equation.';';
			$status		=	@eval($equation);
			if(!is_null($status))	return false;
			if($equation_entry['id']==$option_id)	$order_shipping_amt	=	$shipping_cost;	//Current order's shipping cost.
			$this->shipping_amounts[$equation_entry['id']]=$shipping_cost;	//Array of shipping costs for this item.
		}
		if(count($row)==0)	$order_shipping_amt	=	$weight*floatval(get_option('shipping_cost_factor'))/100.0;
		if(is_null($order_shipping_amt))	$order_shipping_amt	=	$shipping_cost;
		return $order_shipping_amt;
	}

	/**
	 * Calculate product price
	 *
	 * @param  float		Weight of product
	 * @param  float		Tax in percentage
	 * @param  integer	Weight of item
	 * @return float		Calculated shipping cost for the product
	 */
	function calculate_product_price($item_price,$tax,$item_weight)
	{
		$item_price	=	round($item_price,2);

		$shipping_cost	=	$this->calculate_shipping_cost($item_weight);

		if(get_option('allow_opting_out_of_tax')=='1' && post_param_integer('tax_opted_out',0)==1)
			$tax		=	0.0;
		else
			$tax		=	$this->calculate_tax($item_price,$tax);

		$product_price	=	round(($item_price+$tax+$shipping_cost),2);

		return $product_price;
	}

	/**
	 * Find product image for a specific catalogue product
	 *
	 * @param  ID_TEXT		Catalogue name
	 * @param  AUTO_LINK		Catalogue entry id
	 * @return ?SHORT_TEXT	Image name (NULL: no image)
	 */
	function get_product_image($catalogue_name,$entry_id)
	{
		require_code('catalogues');
		$map		=	get_catalogue_entry_field_values($catalogue_name,$entry_id);		
		if(array_key_exists(7,$map))
			return $map[7]['effective_value'];
		else
			return NULL;
	}

	/**
	 * Calculate product price
	 *
	 * @param  AUTO_LINK	Catalogue entry id
	 * @param  integer	Quantity
	 */
	function update_stock($entry_id,$quantity)
	{
		require_code('catalogues');

		$stock_level_warn_threshold	=	0;

		$current_stock			=	0;

		$stock_maintained		=	false;

		$res	=	$GLOBALS['SITE_DB']->query_select('catalogue_entries',array('c_name','cc_id'),array('id'=>$entry_id));

		if(!array_key_exists(0,$res)) return;

		$row		=	$res[0];

		$catalogue_name	=	$row['c_name'];

		$fields		=	get_catalogue_entry_field_values($catalogue_name,$entry_id);

		if(array_key_exists(3,$fields))	//Stock level
		{
			if($fields[3]['effective_value']=='')	return;

			$stock_field			=	$fields[3]['id'];
			$current_stock			=	intval($fields[3]['effective_value']);
		}

		if(array_key_exists(4,$fields))	//Stock maintained
		{
			if(is_null($fields[4]['effective_value']))	return;

			$stock_maintained		=	intval($fields[4]['effective_value'])==1;
		}

		if(array_key_exists(5,$fields))	//Stock level warn threshold
		{
			if(is_null($fields[5]['effective_value']))	return;

			$stock_level_warn_threshold	=	intval($fields[5]['effective_value']);
		}

		$product_name	=	get_translated_text($row['cc_id']);

		if($current_stock<$quantity && $stock_maintained)
		{
			require_code('site');
			attach_message(do_lang_tempcode('LOW_STOCK_DESPATCH_FAILED',$product_name));
		}

		$stock_after_dispatch	=	$current_stock-$quantity;

		if($stock_after_dispatch<$stock_level_warn_threshold)
		{
			stock_maintain_warn_mail($product_name,$entry_id);
		}

		$GLOBALS['SITE_DB']->query_update('catalogue_efv_short',array('cv_value'=>$stock_after_dispatch),array('cf_id'=>$stock_field,'ce_id'=>$entry_id));
	}

	/**
	 * Function to return dispatch type of product.
	 *
	 * @return  ID_TEXT	Dispatch type (manual/automatic)
	 */
	function get_product_dispatch_type()
	{
		return "manual";
	}

	/**
	 * Function to return manual transaction status. Currently manual transaction for catalogue products can only do with uncompleted product orders, that is listed in order numbers (that is getting from cart_orders hook)
	 *
	 * @return  BINARY	Enable/Disable
	 */
	function enable_manual_transaction()
	{
		return false;
	}

	/**
	* Return product info details
	*
	* @param  AUTO_LINK	Product id
	* @return tempcode	Product information
	*/
	function product_info($id)
	{
		return render_catalogue_entry_screen($id,true);
	}

	/**
	 * Get custom fields for ecommerce product
	 *
	 *	@param	AUTO_LINK	Product entry Id
	 *	@param	array			Map where product details are placed
	 */
	function get_custom_product_map_fields($id,&$map)
	{
		require_code('feedback');

		require_code('ecommerce');

		require_code('images');

		$shopping_cart_url			=	build_url(array('page'=>'shopping','type'=>'view_cart'),'_SELF');

		$product_title					=	NULL;

		if(array_key_exists('FIELD_0',$map))
		{
			$product_title	=	$map['FIELD_0_PLAIN'];
			if (is_object($product_title)) $product_title=html_entity_decode(strip_tags($product_title->evaluate()),ENT_QUOTES);
		}

		$product	=	do_lang('PRODUCT_CATALOGUE_ITEMS',strval($id));

		require_lang('shopping');

		$licence	=	method_exists($this,'get_agreement')?$this->get_agreement($id):'';
		$fields	=	method_exists($this,'get_needed_fields')?$this->get_needed_fields($id):NULL;

		$purchase_mod_url=build_url(array('page'=>'purchase','type'=>($licence=='')?(is_null($fields)?'pay':'details'):'licence','product'=>$product),get_page_zone('purchase'));

		$cart_url		=	build_url(array('page'=>'shopping','type'=>'add_item','hook'=>'catalogue_items'),'_SELF');

		if(array_key_exists('FIELD_3',$map))
		{
			$out_of_stock=$map['FIELD_3']=='0';
		} else
		{
			$out_of_stock=false;
		}

		$product_det	=	$this->get_product_details($id);
		$shipping_cost	=	$this->calculate_shipping_cost($product_det['product_weight'],true);
		$tax				=	round($this->calculate_tax($product_det['price'],$product_det['tax']),2);
		$tot_price		=	$this->calculate_product_price(floatval($product_det['price']),$product_det['tax'],$product_det['product_weight']);

		$map['CART_BUTTONS']	=	do_template('CATALOGUE_ENTRY_ADD_TO_CART',array('TAX_TYPE'=>$product_det['tax_type'],'TAX_RATE'=>float_to_raw_string($tax),'OUT_OF_STOCK'=>$out_of_stock,'ACTION_URL'=>$cart_url,'PRODUCT_ID'=>strval($id),'CART_URL'=>$shopping_cart_url,'QUANTITY'=>do_lang_tempcode('QUANTITY'),'ADD_TO_CART'=>do_lang_tempcode('ADD_TO_CART'),'PURCHASE_ONE'=>do_lang_tempcode('PURCHASE_ONE'),'VIEW_CART'=>do_lang_tempcode('VIEW_CART'),'PURCHASE_ACTION_URL'=>$purchase_mod_url,'ALLOW_OPTOUT_TAX'=>get_option('allow_opting_out_of_tax'),'SHIPPING_COST'=>float_to_raw_string($shipping_cost),'PRICE'=>float_to_raw_string($product_det['price']),'TOTAL_PRICE'=>float_to_raw_string($tot_price)),NULL,false);

		$map['CART_LINK']=show_cart_image();
	}

	/**
	 * Returns	seller of this product type
	 *
	 * @param	AUTO_LINK	Product Id
 	 * @return	SHORT_TEXT	Seller	(User/NULL)
 	 */
	function get_seller($product)
	{
		if (get_option('enable_member_payments')=='1')
		{
			return $GLOBALS['SITE_DB']->query_value_null_ok('catalogue_entries','ce_submitter',array('id'=>$product));
		}
		else
		{
			return NULL;
		}
	}

	/**
	 * Find the actual purchase amount of
	 *
	 * @param	AUTO_LINK	Order id
 	 * @return	REAL			Purchase amount
 	 */
	function	find_purchased_price($order_id)
	{
		return	$GLOBALS['SITE_DB']->query_value_null_ok('shopping_order','tot_price',array('id'=>intval($order_id)));
	}

	/**
	 * Cancel an order
	 *
	 * @param ID Order id
 	 */
	function cancel_order($id)
	{
		$order_detail = $GLOBALS['SITE_DB']->query_select('shopping_order',array('*'),array('id'=>$id));
		if ($order_detail && $order_detail[0]['order_status'] != 'ORDER_STATUS_dispatched')
		{
			$GLOBALS['SITE_DB']->query_update('shopping_order',array('order_status'=>'ORDER_STATUS_cancelled'),array('id'=>$id),'',1);
			$GLOBALS['SITE_DB']->query_update('shopping_order_details',array('dispatch_status'=>'ORDER_STATUS_cancelled'),array('order_id'=>$id),'',1);
		}
	}

	/**
	 * Standard function to get the member id of the purchase
	 *
	 * @param  SHORT_TEXT	item id
	 * @return ?USER 			The id of the member/NULL
	 */
	function member_for($id)
	{
		$member=$GLOBALS['SITE_DB']->query_value_null_ok('shopping_order','c_member',array('id'=>$id));
		return $member;
	}

	/**
	 * Standard function to return all sales figures of this product type
	 *
	 * @return array	 		An array of sales details
	 */
	function get_product_sales_data()
	{
		$query	=	"SELECT t1.add_date,t3.id,t3.ce_submitter,t3.c_name	FROM ".get_table_prefix()."shopping_order t1	JOIN ".get_table_prefix()."shopping_order_details t2 ON t1.id = t2.order_id	JOIN ".get_table_prefix()."catalogue_entries t3 ON t2.p_id = t3.id WHERE t2.dispatch_status='ORDER_STATUS_payment_received' OR t2.dispatch_status='ORDER_STATUS_dispatched'";

		$rows				=	$GLOBALS['SITE_DB']->query($query);
		$sales_figures	=	array();
		foreach($rows as $perm_sale)
		{
			$sales_figures[$perm_sale['id']][]	=	array('pro_obj'=>$this,'title'=>'','purchase_date'=>$perm_sale['add_date'],'product_image'=>$this->get_product_image($perm_sale['c_name'],$perm_sale['id']));
		}

		return	$sales_figures;
	}

	/**
	 * Function to return product url
	 *
	 * @return object	 	Hyperlink for product
	 */
	function get_product_link($url_only=false)
	{
		$entry_id		=	$this->product_id[3]['id'];		
		$product_url	=	build_url(array('page'=>'catalogues','type'=>'entry','id'=>$entry_id),get_module_zone('catalogues'));
		if($url_only)	return	$product_url;
		return	hyperlink($product_url,$this->product_id[4],false,false,do_lang('INDEX'));
	}
}
/**
 * Update order status,transaction id after transaction
 *
 * @param  AUTO_LINK	Purchase/Order id.
 * @param  array		Details of product.
 */
function handle_product_orders_items($purchase_id,$details)
{
	if(get_option('enable_member_payments')=='0')	//If direct member payment is disabled, assign credit points to owner.
	{
		require_code('ocf_members');
		$store_owner = $GLOBALS['SITE_DB']->query_value('shopping_order','to_member_id',array('id'=>$purchase_id));
		if(!is_null($store_owner))
		{
			$cost = $GLOBALS['SITE_DB']->query_value('shopping_order','tot_price',array('id'=>$purchase_id));
			if(has_pro_account_access($store_owner))
			{
				$GLOBALS['SITE_DB']->query_insert('ecommerce_credits',array('src_code'=>'store_sale','note'=>'','member_id'=>$store_owner,'number_credits'=>$cost,'timestamp_given'=>time(),'transaction_id'=>strval($purchase_id),'credit_type'=>'store_sale'));
			}
		}
	}

	$status	=	$details['ORDER_STATUS'];

	$GLOBALS['SITE_DB']->query_update('shopping_order_details',array('dispatch_status'=>$status),array('order_id'=>$purchase_id));

	$GLOBALS['SITE_DB']->query_update('shopping_order',array('order_status'=>$status,'transaction_id'=>$details['txn_id']),array('id'=>$purchase_id));
}