<?php /*

 ocPortal
 Copyright (c) ocProducts, 2004-2009

 See text/en/licence.txt for full licencing information.

*/

/**
 * @license		http://opensource.org/licenses/cpal_1.0 Common Public Attribution License
 * @copyright	ocProducts Ltd
 * @package		ecommerce
 */

class Hook_secpay
{
	/**
	* Get some title/description about payment method
	*
	* @return SHORT_TEXT		Title/Description about paypal
	*/
	function info()
	{
		return do_lang('SECPAY_DESCRIPTION');
	}

	/**
	* Function to install paypal config options
	*
	*/
	function install_config()
	{
		add_config_option('SECPAY_ENABLE','secpay_enable','special_group__tick_control__secpay__tick','return\'0\';','ECOMMERCE','PAYMENT_GATEWAY');
		add_config_option('SECPAY_USERNAME','secpay_username','special_group__tick_control__secpay__line','return \'\';','ECOMMERCE','PAYMENT_GATEWAY');
		add_config_option('SECPAY_VPN_PASSWORD','secpay_vpn_password','special_group__tick_control__secpay__line','return \'\';','ECOMMERCE','PAYMENT_GATEWAY');
		add_config_option('SECPAY_REMOTE_PASSWORD','secpay_remote_password','special_group__tick_control__secpay__line','return \'\';','ECOMMERCE','PAYMENT_GATEWAY');
		add_config_option('SECPAY_DIGEST_KEY','secpay_digest_key','special_group__tick_control__secpay__line','return \'\';','ECOMMERCE','PAYMENT_GATEWAY');
		add_config_option('LOCAL_SUBSCRIPTION_METHODS','secpay_local_subscription_methods','special_group__tick_control__secpay__list','return \'automatic\';','ECOMMERCE','PAYMENT_GATEWAY',0,'automatic|local_controlled');
	}

	/**
	* Function to unistall paypal config options
	*
	*/
	function uninstall_config()
	{
		delete_config_option('secpay_enable');
		delete_config_option('secpay_username');
		delete_config_option('secpay_vpn_password');
		delete_config_option('secpay_remote_password');
		delete_config_option('secpay_digest_key');
		delete_config_option('secpay_local_subscription_methods');
	}

	/**
	* Find the payment gateway enabled or not
	*
	* @return BINARY		Enabled/Disabled
	*/
	function is_enabled()
	{
		return get_option('secpay_enable');
	}

	/**
	* Get gateway CPFs
	*
	* @return array
	*/
	function get_gateway_cpfs()
	{
		return array('secpay_username','secpay_vpn_password','secpay_remote_password','secpay_digest_key');
	}

	/**
	* Whether the cpf is of this gateway
	*
	* @param SHORT_TEXT The cpf
	* @return bool
	*/
	function is_gateway_cpf($cpf)
	{
		return in_array($cpf,$this->get_gateway_cpfs());
	}

	/**
	* Get logos/verified seal of payment method
	*
	* @return tempcode	Logo/verified sign
	*/
	function get_confidence_logos()
	{
		require_css('ecommerce');
		$logo	=	'<a href="http://www.paypoint.net/"><img border="0" alt="secpay logo" src="http://www.paypoint.net/assets/images/paypoint_com_logo.gif"/></a>';

		$read_text_link	=	build_url(array('page'=>'purchase','type'=>'payment_note','screen'=>'secure_payment_paypal','wide_high'=>1),get_module_zone('purchase'));

		$read_text_link	=	$read_text_link->evaluate();

		$js	=	"function secure_shopping_notes() {
						window.open('$read_text_link','ruleswindow','menubar=1,resizable=1,scrollbars=1,width=970,height=650');
						return false;
					}";
		return do_template('VERIFIED_MERCHANT_LOGO_SCREEN',array('LOGO'=>$logo,'READ_TEXT_LINK'=>$read_text_link,'JAVASCRIPT'=>$js,'PAYMENT_PROCESSING_LINK'=>'','PAYMENT_TEXT_URL'=>'site:secure_payment_authorize'));
	}

	/**
	* Get the gateway username.
	*
	* @return string			The answer.
	*/
	function _get_username()
	{
		return ecommerce_test_mode()?get_option('ipn_test'):get_option('ipn');
	}

	/**
	* Show option buton
	*
	*	@return tempcode	Form for secpay option
	*/
	function show_option_button()
	{
		return do_template('SECPAY_OPTION_BUTTON',array('URL'=>get_self_url()));
	}

	/**
	* Get form fields for local payment.
	*
	* @param  ?ID_TEXT		The transaction ID (NULL: auto-generate)
	* @param  ID_TEXT		The purchase ID
	* @param  SHORT_TEXT	The item name
	* @param  SHORT_TEXT	The amount
	* @param  ?integer		The length (NULL: not a subscription)
	* @param  ID_TEXT		The length units
	* @return tempcode		The form fields
	*/
	function get_transaction_form_fields($trans_id,$purchase_id,$item_name,$amount,$length,$length_units)
	{
		require_code('ecommerce');
		if (is_null($trans_id))	$trans_id=$this->generate_trans_id();

		list($card_det,$hidden)	=	get_general_transaction_form_fields($trans_id,$purchase_id,$item_name,$amount,$length,$length_units,$this);

		/*	General transaction fields returned from ecommerce api, attaching the details to the form.
			The fields can add/remove here, depending on the customised payment gateway features.
		*/
		$fields	=	new ocp_tempcode();
		foreach($card_det as $field)
		{
			$fields->attach($field);
		}

		$hidden->attach(form_input_hidden('payment_gateway','secpay'));

		$verified_account_logo	=	$this->get_confidence_logos();
		return array($fields,$hidden,$verified_account_logo);
	}

	/**
	 * Whether the member could enter the subscription start date in the payment form
	 *
	 * @return boolean
	 */
	function is_allow_subscription_start_date()
	{
		return true;
	}

	/**
	 * Whether to allow shipping detail in the local payment form
	 *
	 * @return boolean
	 */
	function is_allow_shipping()
	{
		return true;
	}

	/**
	* Get the IPN URL.
	*
	* @return URLPATH		The IPN url.
	*/
	function get_ipn_url()
	{
		return 'https://www.secpay.com/java-bin/ValCard';
	}

	/**
	* Generate a transaction ID.
	*
	* @return string			A transaction ID.
	*/
	function generate_trans_id()
	{
		return md5(uniqid(strval((mt_rand(0,32000))),true));
	}

	/**
	* Make a transaction (payment) button.
	*
	* @param  ID_TEXT		The product codename.
	* @param  SHORT_TEXT	The human-readable product title.
	* @param  ID_TEXT		The purchase ID.
	* @param  float			A transaction amount.
	* @param  ID_TEXT		The currency to use.
	* @param  ID_TEXT      The seller
	* @return tempcode		The button
	*/
	function make_transaction_button($product,$item_name,$purchase_id,$amount,$currency,$seller=NULL)
	{
		$access_detail = $this->get_access_detail($seller);
		$username      = $access_detail[0];
		$ipn_url       = $this->get_ipn_url();
		$trans_id      = $this->generate_trans_id();
		$digest        = md5('INSTANT_'.$trans_id.float_to_raw_string($amount).$access_detail[2]);
		$module			= get_param('page','purchase');
		$callback_url  = build_url(array('page'=>$module,'type'=>'finish','from'=>'secpay'),get_module_zone($module),NULL,false,false,true);
		$_callback_url = $callback_url->evaluate();
		$is_ssl        = (substr($_callback_url,0,5) == 'https') ? true : false;

		$GLOBALS['SITE_DB']->query_insert('trans_expecting',array(
			'id'=>$trans_id,
			'e_purchase_id'=>strval($purchase_id),
			'e_item_name'=>$item_name,
			'e_member_id'=>get_member(),
			'e_amount'=>float_to_raw_string($amount),
			'e_ip_address'=>get_ip_address(),
			'e_session_id'=>get_session_id(),
			'e_time'=>time(),
			'e_length'=>NULL,
			'e_length_units'=>'',
		));

		return do_template('ECOM_BUTTON_VIA_SECPAY',array('_GUID'=>'e68e80cb637f8448ef62cd7d73927722','IS_SSL'=>$is_ssl,'CALLBACK_URL'=>$callback_url,'PRODUCT'=>$product,'DIGEST'=>$digest,'TEST'=>ecommerce_test_mode(),'TRANS_ID'=>$trans_id,'ITEM_NAME'=>$item_name,'PURCHASE_ID'=>strval($purchase_id),'AMOUNT'=>float_to_raw_string($amount),'CURRENCY'=>$currency,'USERNAME'=>$username,'IPN_URL'=>$ipn_url));
	}

	/**
	* Find details for a subscription in secpay format.
	*
	* @param  integer	The subscription length in the units.
	* @param  ID_TEXT	The length units.
	* @param  DATE      Subscription start date
	* @set    d w m y
	* @return array		A tuple: the period in secpay units, the date of the first repeat
	*/
	function _translate_subscription_details($length,$length_units,$start_date=NULL)
	{
		$start_day   = NULL;
		$start_month = NULL;
		$start_year  = NULL;

		switch ($length_units)
		{
			case 'd':
				$length_units_2='daily';
				$single_length=60*60*24;
				break;
			case 'w':
				$length_units_2='weekly';
				$single_length=60*60*24*7;
				break;
			case 'm':
				$length_units_2='monthly';
				$single_length=60*60*24*31;
				break;
			case 'y':
				$length_units_2='yearly';
				$single_length=60*60*24*365;
				break;
		}
		if (($length_units=='m') && ($length==3))
		{
			$length_units_2='quarterly';
			$single_length=60*60*24*92;
		}

		if (is_null($start_date))
		{
			$first_repeat = date('Ymd');
		}
		else
		{
			$start_date_explode = explode('-',$start_date);
			$first_repeat = $start_date_explode[0].$start_date_explode[1].$start_date_explode[2];
		}

		return array($length_units_2,$first_repeat);
	}

	/**
	* Make a subscription (payment) button.
	*
	* @param  ID_TEXT		The product codename.
	* @param  SHORT_TEXT	The human-readable product title.
	* @param  ID_TEXT		The purchase ID.
	* @param  float			A transaction amount.
	* @param  integer		The subscription length in the units.
	* @param  ID_TEXT		The length units.
	* @set    d w m y
	* @param  ID_TEXT		The currency to use.
	* @param  ID_TEXT      The seller
	* @return tempcode		The button
	*/
	function make_subscription_button($product,$item_name,$purchase_id,$amount,$length,$length_units,$currency,$seller=NULL)
	{
		$access_detail = $this->get_access_detail($seller);
		$username      = $access_detail[0];
		$ipn_url       = $this->get_ipn_url();
		$trans_id      = $this->generate_trans_id();
		$digest        = md5('SUBSCR_'.$trans_id.float_to_raw_string($amount).$access_detail[2]);
		$module			= get_param('page','purchase');
		$callback_url  = build_url(array('page'=>$module,'type'=>'finish','from'=>'secpay'),get_module_zone($module),NULL,false,false,true);
		$_callback_url = $callback_url->evaluate();
		$is_ssl        = (substr($_callback_url,0,5) == 'https') ? true : false;

		list($length_units_2,$first_repeat)=$this->_translate_subscription_details($length,$length_units);

		$start_day   = date('d');
		$start_month = date('m');
		$start_year  = date('Y');

		$start_days =array();
		for($i=1;$i<=31;$i++) {
			$start_days[] = array('DAY'=>str_pad($i,2,'0',STR_PAD_LEFT));
		}

		require_lang('dates');
		require_lang('ocf');
		$start_months = array();
		for($i=1;$i<=12;$i++) {
			$month = str_pad($i,2,'0',STR_PAD_LEFT);

			switch ($i)
			{
				case 1:
					$month_text=do_lang_tempcode('JANUARY');
					break;
				case 2:
					$month_text=do_lang_tempcode('FEBRUARY');
					break;
				case 3:
					$month_text=do_lang_tempcode('MARCH');
					break;
				case 4:
					$month_text=do_lang_tempcode('APRIL');
					break;
				case 5:
					$month_text=do_lang_tempcode('MAY');
					break;
				case 6:
					$month_text=do_lang_tempcode('JUNE');
					break;
				case 7:
					$month_text=do_lang_tempcode('JULY');
					break;
				case 8:
					$month_text=do_lang_tempcode('AUGUST');
					break;
				case 9:
					$month_text=do_lang_tempcode('SEPTEMBER');
					break;
				case 10:
					$month_text=do_lang_tempcode('OCTOBER');
					break;
				case 11:
					$month_text=do_lang_tempcode('NOVEMBER');
					break;
				case 12:
					$month_text=do_lang_tempcode('DECEMBER');
					break;
			}

			$start_months[] = array('MONTH'=>$month,'MONTH_TEXT'=>$month_text);
		}

		$start_years = array();
		for($i=date('Y');$i<=date('Y')+2;$i++) {
			$start_years[] = array('YEAR'=>strval($i));
		}

		$GLOBALS['SITE_DB']->query_insert('trans_expecting',array(
			'id'=>$trans_id,
			'e_purchase_id'=>$purchase_id,
			'e_item_name'=>$item_name,
			'e_member_id'=>get_member(),
			'e_amount'=>float_to_raw_string($amount),
			'e_ip_address'=>get_ip_address(),
			'e_session_id'=>get_session_id(),
			'e_time'=>time(),
			'e_length'=>$length,
			'e_length_units'=>$length_units,
		));

		require_javascript('javascript_ajax');
		$ajax_url = find_script('subscription_start_date');

		return do_template('ECOM_SUBSCRIPTION_BUTTON_VIA_SECPAY',array('_GUID'=>'e5e6d6835ee6da1a6cf02ff8c2476aa6','IS_SSL'=>$is_ssl,'CALLBACK_URL'=>$callback_url,'PRODUCT'=>$product,'DIGEST'=>$digest,'TEST'=>ecommerce_test_mode(),'TRANS_ID'=>$trans_id,'FIRST_REPEAT'=>$first_repeat,'LENGTH'=>strval($length),'LENGTH_UNITS_2'=>$length_units_2,'ITEM_NAME'=>$item_name,'PURCHASE_ID'=>strval($purchase_id),'AMOUNT'=>float_to_raw_string($amount),'CURRENCY'=>$currency,'USERNAME'=>$username,'IPN_URL'=>$ipn_url,'START_DAY'=>$start_day,'START_DAYS'=>$start_days,'START_MONTH'=>$start_month,'START_MONTHS'=>$start_months,'START_YEAR'=>$start_year,'START_YEARS'=>$start_years,'AJAX_URL'=>$ajax_url));
	}

   /**
	 * Find whether the hook auto-cancels (if it does, auto cancel the given trans-id).
	 *
	 * @param ID Subscription Id
	 * @param ID Member Id (NULL: Admin)
	 * @return ?boolean	True: yes. False: no. (NULL: cancels via a user-URL-directioning)
	 */
	function auto_cancel($subscription_id,$member_id=NULL)
	{
		require_lang('ecommerce');
		require_code('xmlrpc');

		$temp	= $GLOBALS['SITE_DB']->query_value('subscriptions','s_auto_fund_key',array('id'=>$subscription_id));
		$data	= unserialize($temp);
		$secpay_subscription_id	= $data['id'];

		$this->set_cancel_api_parameters($secpay_subscription_id,$member_id);

		$response = xml_rpc('https://www.secpay.com:443/secxmlrpc/make_call','SECVPN.repeatCardFullAddr',$this->api_parameters,true);

		if (!is_null($response))
		{
			$pos_1 = strpos($response,'<value>');

			if ($pos_1)
			{
				$pos_2 = strpos($response,'</value>');
				$value = @html_entity_decode(trim(substr($response,$pos_1+7,$pos_2-$pos_1-7)),ENT_QUOTES,get_charset());
				if (substr($value,0,1)=='?') $value=substr($value,1);

				$_map = explode('&',$value);
				$map  = array();
				foreach ($_map as $x)
				{
					$explode = explode('=',$x);
					if (count($explode)==2)
						$map[$explode[0]]=$explode[1];
				}

				$code    = (array_key_exists('code',$map)) ? $map['code'] : NULL;
				$success = ($code == 'A') ?  true : false;
			}
			else
			{
				$success = false;
			}
		}
		else
		{
			$success = false;
		}

		return $success;
	}

	/**
	* Find a transaction fee from a transaction amount. Regular fees aren't taken into account.
	*
	* @param  float	A transaction amount.
	* @return float	The fee
	*/
	function get_transaction_fee($amount)
	{
		//return 0.39; // the fee for <60 transactions per month. If it's more, I'd hope ocPortal's simplistic accountancy wasn't being relied on!
		return round(floatval(get_option('secpay_transaction_flat_cost')),2);
	}

	/**
	* Get a list of card types.
	*
	* @param  ?string	The card type to select by default (NULL: don't care)
	* @return tempcode	The list
	*/
	function nice_get_card_types($it=NULL)
	{
		$list=new ocp_tempcode();
		$array=array('Visa','Master Card','Switch','UK Maestro','Maestro','Solo','Delta','American Express','Diners Card','JCB');
		foreach ($array as $x)
		{
			$list->attach(form_input_list_entry($x,$it==$x));
		}
		return $list;
	}

	/**
	* Perform a transaction.
	*
	* @param  ?ID_TEXT		The transaction ID (NULL: generate one)
	* @param  SHORT_TEXT	Cardholder first name
	* @param  SHORT_TEXT	Cardholder last name
	* @param  SHORT_TEXT	Card number
	* @param  SHORT_TEXT	Transaction amount
	* @param  SHORT_TEXT	Card Expiry date
	* @param  integer		Card Issue number
	* @param  SHORT_TEXT	Card Start date
	* @param  SHORT_TEXT	Card Type
	* @set    "Visa" "Master Card" "Switch" "UK Maestro" "Maestro" "Solo" "Delta" "American Express" "Diners Card" "JCB"
	* @param  SHORT_TEXT	Card CV2 number (security number)
	* @param  ?integer		The subscription length in the units. (NULL: not a subscription)
	* @param  ?ID_TEXT		The length units. (NULL: not a subscription)
	* @set    d w m y
	* @param  SHORT_TEXT	Description for the transaction
	* @param  ?MEMBER		Member making the transaction (NULL: current member)
	* @param  SHORT_TEXT	Address for card
	* @param  SHORT_TEXT	City for card
	* @param  SHORT_TEXT	State for card
	* @param  SHORT_TEXT	Zip/post-code for card
	* @param  SHORT_TEXT	Country for card
	* @param  SHORT_TEXT	Shipping Address for card
	* @param  SHORT_TEXT	Shipping City for card
	* @param  SHORT_TEXT	Shipping State for card
	* @param  SHORT_TEXT	Shipping Zip/post-code for card
	* @param  SHORT_TEXT	Shipping Country for card
	* @param  integer		Total occurence for subscription
	* @param  integer		Total trial occurence for subscription
	* @param  integer		Trial amount for subscription
	* @return array			A tuple: success (boolean), trans-id (string), message (string), raw message (string)
	*/
	function do_transaction($trans_id,$first_name,$last_name,$card_number,$amount,$expiry_date,$issue_number,$start_date,$card_type,$cv2,$length=NULL,$length_units=NULL,$description='',$member_id=NULL,$address='',$city='',$state='',$zip='',$country='',$shipping_first_name='',$shipping_last_name='',$shipping_address='',$shipping_city='',$shipping_state='',$shipping_zip='',$shipping_country='',$total_occurrences=9999,$trial_occurrences=0,$trial_amount=0)
	{
		require_lang('ecommerce');
		require_code('xmlrpc');

		if(is_null($trans_id))	$trans_id =	$this->generate_trans_id();

		$purchase_id =	post_param_integer('customfld1','-1');
		$card_number = str_replace(' ','',$card_number);

		$expiry_month = str_pad(substr($expiry_date,5,2),2,'0');
		$expiry_year  = substr($expiry_date,2,2);

		$username=$this->_get_username();
		$password_2=get_option('vpn_password');
		$digest=md5($trans_id.strval($amount).get_option('ipn_password'));

		if (!is_null($length))
		{
			list($length_units_2,$first_repeat)=$this->_translate_subscription_details($length,$length_units,$start_date);

			$this->set_recurring_api_parameters($purchase_id,$card_type,$card_number,$cv2,$expiry_month,$expiry_year,$length_units_2,$first_repeat,$description,$amount,$first_name,$last_name,$address,$city,$state,$zip,$country,$shipping_first_name,$shipping_last_name,$shipping_address,$shipping_city,$shipping_state,$shipping_zip,$shipping_country,$member_id);

			$response = xml_rpc('https://www.secpay.com:443/secxmlrpc/make_call','SECVPN.validateCardFull',$this->api_parameters);
		}
		else
		{
			$this->set_api_parameters($purchase_id,$card_type,$card_number,$cv2,$expiry_month,$expiry_year,$description,$amount,$first_name,$last_name,$address,$city,$state,$zip,$country,$shipping_first_name,$shipping_last_name,$shipping_address,$shipping_city,$shipping_state,$shipping_zip,$shipping_country,$member_id);

			$response = xml_rpc('https://www.secpay.com:443/secxmlrpc/make_call','SECVPN.validateCardFull',$this->api_parameters);
		}

		if (!is_null($response))
		{
			$pos_1 = strpos($response,'<value>');

			if ($pos_1)
			{
				$pos_2 = strpos($response,'</value>');
				$value = @html_entity_decode(trim(substr($response,$pos_1+7,$pos_2-$pos_1-7)),ENT_QUOTES,get_charset());
				if (substr($value,0,1)=='?') $value=substr($value,1);

				$_map = explode('&',$value);
				$map  = array();
				foreach ($_map as $x)
				{
					$explode = explode('=',$x);
					if (count($explode)==2)
						$map[$explode[0]]=$explode[1];
				}

				$code          = (array_key_exists('code',$map)) ? $map['code'] : NULL;
				$success       = ($code == 'A') ?  true : false;
				$message_raw   =  array_key_exists('message',$map) ? $map['message'] : '';
				$response_code = ($success) ? $map['trans_id'] : $code;

				if(addon_installed('shopping'))
				{
					$this->store_shipping_address($purchase_id);
				}

				$result = array($success,$response_code,$code,$message_raw);
			}
			else
			{
				$result = array(false,NULL,NULL,NULL);
			}
		}
		else
		{
			$result = array(false,NULL,NULL,NULL);
		}

		return $result;
	}

	/**
	* This function defines the parameters needed to make an API call.
	*
	* @param  AUTO_LINK	Purchase-id
	* @param  ?ID_TEXT	The transaction ID (NULL: generate one)
	* @param  SHORT_TEXT	Card Type
	* @param  SHORT_TEXT	Card number
	* @param  SHORT_TEXT	Card CV2 number (security number)
	* @param  SHORT_TEXT	Card Expiry Month
	* @param  SHORT_TEXT	Card Expiry Yeary
	* @param  SHORT_TEXT	Subscription frequency
	* @param  SHORT_TEXT	Subscription start date
	* @param  SHORT_TEXT	Description
	* @param  SHORT_TEXT	Transaction amount
	* @param  SHORT_TEXT	Address for card
	* @param  SHORT_TEXT	City for card
	* @param  SHORT_TEXT	State for card
	* @param  SHORT_TEXT	Zip/post-code for card
	* @param  SHORT_TEXT	Country for card
	* @param  SHORT_TEXT	Shipping Address for card
	* @param  SHORT_TEXT	Shipping City for card
	* @param  SHORT_TEXT	Shipping State for card
	* @param  SHORT_TEXT	Shipping Zip/post-code for card
	* @param  SHORT_TEXT	Shipping Country for card
	* @param  ?MEMBER		Member making the transaction (NULL: admin)
	*/
	function set_recurring_api_parameters($purchase_id,$card_type,$card_number,$cv2,$expiry_month,$expiry_year,$length_units_2,$first_repeat,$description,$amount,$first_name=NULL,$last_name=NULL,$address=NULL,$city=NULL,$state=NULL,$zip=NULL,$country=NULL,$shipping_first_name=NULL,$shipping_last_name=NULL,$shipping_address=NULL,$shipping_city=NULL,$shipping_state=NULL,$shipping_zip=NULL,$shipping_country=NULL,$member_id=NULL)
	{
		$access_detail = $this->get_access_detail($member_id);
		$item_name     = find_item_name_of_order($purchase_id);
		$module			= get_param('page','purchase');
		$callback_url  = build_url(array('page'=>$module,'type'=>'finish','from'=>'secpay'),get_module_zone($module),NULL,false,false,true);
		$_callback_url = $callback_url->evaluate();
		$is_ssl        = (substr($_callback_url,0,5) == 'https') ? true : false;

		$user_name    = $access_detail[0];
		$vpn_password = $access_detail[1];
		$trans_id     = 'SUBSCR_'.md5(uniqid(strval((mt_rand(0,32000))),true)).strval($purchase_id);
		$ip           = get_ip_address();
		$name         = $first_name;
		$card_number  = $card_number;
		$amount       = strval($amount);
		$expiry_date  = $expiry_month.'/'.$expiry_year;
		$issue_number = '';
		$start_date   = '';
		$order        = "prod=$item_name,item_amount=$amount";
		$shipping     = 'name='.$shipping_first_name.' '.$shipping_last_name.',addr_1='.$shipping_address.',city='.$shipping_city.',state='.$shipping_state.',post_code='.$shipping_zip;
		$billing      = 'name='.$first_name.' '.$last_name.',addr_1='.$address.',city='.$city.',state='.$state.',post_code='.$zip;
		$options      = (ecommerce_test_mode()) ? 'test_status=true' : 'test_status=false';
		$options     .= ',currency='.get_option('currency').',card_type='.str_replace(',','',$card_type).',cv2='.strval(intval($cv2));
		$options     .= ',repeat='.$first_repeat.'/'.$length_units_2.'/0/'.$amount.',callback='.urlencode($_callback_url).',repeat_callback='.urlencode($_callback_url);
		$options     .= ',product='.$item_name.',cb_flds=product:currency';
		$options     .= ($is_ssl) ? ',ssl_cb=true' : '';

		$this->api_parameters = array($user_name,$vpn_password,$trans_id,$ip,$name,$card_number,$amount,$expiry_date,$issue_number,$start_date,$order,$shipping,$billing,$options);
	}

	/**
	* This function defines the parameters needed to make an API call.
	*
	* @param  NUMBER       Purchase-id
	* @param  ?ID_TEXT		The transaction ID (NULL: generate one)
	* @param  SHORT_TEXT	Card Type
	* @param  SHORT_TEXT	Card number
	* @param  SHORT_TEXT	Card CV2 number (security number)
	* @param  SHORT_TEXT	Card Expiry Month
	* @param  SHORT_TEXT	Card Expiry Yeary
	* @param  SHORT_TEXT	Description
	* @param  SHORT_TEXT	Transaction amount
	* @param  SHORT_TEXT	Address for card
	* @param  SHORT_TEXT	City for card
	* @param  SHORT_TEXT	State for card
	* @param  SHORT_TEXT	Zip/post-code for card
	* @param  SHORT_TEXT	Country for card
	* @param  SHORT_TEXT	Shipping Address for card
	* @param  SHORT_TEXT	Shipping City for card
	* @param  SHORT_TEXT	Shipping State for card
	* @param  SHORT_TEXT	Shipping Zip/post-code for card
	* @param  SHORT_TEXT	Shipping Country for card
	* @param  ?MEMBER		Member making the transaction (NULL: admin)
	*/
	function set_api_parameters($purchase_id,$card_type,$card_number,$cv2,$expiry_month,$expiry_year,$description,$amount,$first_name=NULL,$last_name=NULL,$address=NULL,$city=NULL,$state=NULL,$zip=NULL,$country=NULL,$shipping_first_name=NULL,$shipping_last_name=NULL,$shipping_address=NULL,$shipping_city=NULL,$shipping_state=NULL,$shipping_zip=NULL,$shipping_country=NULL,$member_id=NULL)
	{
		$access_detail = $this->get_access_detail($member_id);
		$item_name     = find_item_name_of_order($purchase_id);
		$module			= get_param('page','purchase');
		$callback_url  = build_url(array('page'=>$module,'type'=>'finish','from'=>'secpay'),get_module_zone($module),NULL,false,false,true);
		$_callback_url = $callback_url->evaluate();
		$is_ssl        = (substr($_callback_url,0,5) == 'https') ? true : false;

		$user_name    = $access_detail[0];
		$vpn_password = $access_detail[1];
		$trans_id     = 'INSTANT_'.md5(uniqid(strval((mt_rand(0,32000))),true)).strval($purchase_id);
		$ip           = '192.168.0.1';
		$name         = $first_name;
		$card_number  = $card_number;
		$amount       = strval($amount);
		$expiry_date  = $expiry_month.'/'.$expiry_year;
		$issue_number = '';
		$start_date   = '';
		$order        = "prod=$item_name,item_amount=$amount";
		$shipping     = 'name='.$shipping_first_name.' '.$shipping_last_name.',addr_1='.$shipping_address.',city='.$shipping_city.',state='.$shipping_state.',post_code='.$shipping_zip;
		$billing      = 'name='.$first_name.' '.$last_name.',addr_1='.$address.',city='.$city.',state='.$state.',post_code='.$zip;
		$options      = (ecommerce_test_mode()) ? 'test_status=true' : 'test_status=false';
		$options     .= ',currency='.get_option('currency').',card_type='.str_replace(',','',$card_type).',cv2='.strval(intval($cv2));
		$options     .= ',product='.$item_name.',cb_flds=product:currency'.',cb_post=true,callback='.urlencode($_callback_url);
		$options     .= ($is_ssl) ? ',ssl_cb=true' : '';

		$this->api_parameters = array($user_name,$vpn_password,$trans_id,$ip,$name,$card_number,$amount,$expiry_date,$issue_number,$start_date,$order,$shipping,$billing,$options);
	}

	/**
	* This function defines the parameters needed to make an API call.
	*
	* @param  SHORT_TEXT  Paymentgateway transaction id
	* @param  ?MEMBER		Member making the transaction (NULL: admin)
	*/
	function set_cancel_api_parameters($trans_id,$member_id=NULL)
	{
		$access_detail = $this->get_access_detail($member_id);

		$user_name       = $access_detail[0];
		$vpn_password    = $access_detail[1];
		$trans_id        = $trans_id;
		$amount          = '-1';
		$remote_password = $access_detail[2];
		$new_trans_id    = '';
		$expiry_date     = '';
		$order           = '';
		$shipping        = '';
		$billing         = '';
		$options         = 'repeat_change=true,repeat=false';

		$this->api_parameters = array($user_name,$vpn_password,$trans_id,$amount,$remote_password,$new_trans_id,$expiry_date,$order,$shipping,$billing,$options);
	}

	/**
	* Get secpay access detail
	*
	* @param ID Member Id
	* @return array
	*/
	function get_access_detail($member_id=NULL)
	{
		$username        = NULL;
		$vpn_password    = NULL;
		$remote_password = NULL;
		$digest_key      = NULL;

		if(!is_null($member_id) && get_option('enable_member_payments'))
		{
			$username        = get_ocp_cpf('secpay_username',$member_id);
			$vpn_password    = get_ocp_cpf('secpay_vpn_password',$member_id);
			$remote_password = get_ocp_cpf('secpay_remote_password',$member_id);
			$digest_key      = get_ocp_cpf('secpay_digest_key',$member_id);
		}

		if (is_null($username) || $username=='' || is_null($vpn_password) || $vpn_password=='' || is_null($remote_password) || $remote_password=='' || is_null($digest_key) || $digest_key=='')
		{
			$username        = get_option('secpay_username');
			$vpn_password    = get_option('secpay_vpn_password');
			$remote_password = get_option('secpay_remote_password');
			$digest_key      = get_option('secpay_digest_key');
		}

		return array($username,$vpn_password,$remote_password,$digest_key);
	}

	/**
    * Get subscription error
    *
    * @param  SHORT_TEXT   The error code
    * @param  SHORT_TEXT   The error
    * @return SHORT_TEXT   Error message
    */
   function get_subscription_error($code,$error)
   {
		require_lang('ecommerce');

		switch ($code)
		{
			case 'N':
				$error_msg = do_lang_tempcode('SECPAY_NOT_AUTHORISED');
				break;

			case 'C':
				$error_msg = do_lang_tempcode('SECPAY_COMMUNICATION_PROBLEM');
				break;

			case 'F':
				$error_msg = do_lang_tempcode('SECPAY_FRAUD_TRANSACTION');
				break;

			case 'P:A':
				$error_msg = do_lang_tempcode('SECPAY_AMOUNT_INVALID');
				break;

			case 'P:X':
				$error_msg = do_lang_tempcode('SECPAY_NOT_ALL_FIELDS');
				break;

			case 'P:P':
				$error_msg = do_lang_tempcode('SECPAY_SAME_PAYMENT');
				break;

			case 'P:S':
				$error_msg = do_lang_tempcode('SECPAY_START_DATE_INVALID');
				break;

			case 'P:E':
				$error_msg = do_lang_tempcode('SECPAY_EXPIRY_DATE_INVALID');
				break;

			case 'P:I':
				$error_msg = do_lang_tempcode('SECPAY_ISSUE_NUMBER_INVALID');
				break;

			case 'P:C':
				$error_msg = do_lang_tempcode('SECPAY_CARD_NUMBER_INVALID');
				break;

			case 'P:T':
				$error_msg = do_lang_tempcode('SECPAY_CARD_TYPE_INVALID');
				break;

			case 'P:N':
				$error_msg = do_lang_tempcode('SECPAY__NO_CUSTOMER_NAME');
				break;

			case 'P:M':
				$error_msg = do_lang_tempcode('SECPAY_NO_MERCHANT');
				break;

			case 'P:B':
				$error_msg = do_lang_tempcode('SECPAY_NO_CARD_TYPE_MERCHANT');
				break;

			case 'P:D':
				$error_msg = do_lang_tempcode('SECPAY_NO_CURRENCY_MERCHANT');
				break;

			case 'P:V':
				$error_msg = do_lang_tempcode('SECPAY_NO_CV2');
				break;

			case 'P:R':
				$error_msg = do_lang_tempcode('SECPAY_TIMED_OUT');
				break;

			case 'P:#':
				$error_msg = do_lang_tempcode('SECPAY_NO_HASH');
				break;

			default:
				$error_msg = do_lang('CARD_GENERAL_ERROR');
		}

		return $error_msg;
	}


	/**
   * Get transaction error
   *
   * @param  SHORT_TEXT   The error code
   * @param  SHORT_TEXT   The error
   * @return SHORT_TEXT   Error message
   */
   function get_transaction_error($code,$error)
	{
		require_lang('ecommerce');

		switch ($code)
		{
			case 'N':
				$error_msg = do_lang_tempcode('SECPAY_NOT_AUTHORISED');
				break;

			case 'C':
				$error_msg = do_lang_tempcode('SECPAY_COMMUNICATION_PROBLEM');
				break;

			case 'F':
				$error_msg = do_lang_tempcode('SECPAY_FRAUD_TRANSACTION');
				break;

			case 'P:A':
				$error_msg = do_lang_tempcode('SECPAY_AMOUNT_INVALID');
				break;

			case 'P:X':
				$error_msg = do_lang_tempcode('SECPAY_NOT_ALL_FIELDS');
				break;

			case 'P:P':
				$error_msg = do_lang_tempcode('SECPAY_SAME_PAYMENT');
				break;

			case 'P:S':
				$error_msg = do_lang_tempcode('SECPAY_START_DATE_INVALID');
				break;

			case 'P:E':
				$error_msg = do_lang_tempcode('SECPAY_EXPIRY_DATE_INVALID');
				break;

			case 'P:I':
				$error_msg = do_lang_tempcode('SECPAY_ISSUE_NUMBER_INVALID');
				break;

			case 'P:C':
				$error_msg = do_lang_tempcode('SECPAY_CARD_NUMBER_INVALID');
				break;

			case 'P:T':
				$error_msg = do_lang_tempcode('SECPAY_CARD_TYPE_INVALID');
				break;

			case 'P:N':
				$error_msg = do_lang_tempcode('SECPAY__NO_CUSTOMER_NAME');
				break;

			case 'P:M':
				$error_msg = do_lang_tempcode('SECPAY_NO_MERCHANT');
				break;

			case 'P:B':
				$error_msg = do_lang_tempcode('SECPAY_NO_CARD_TYPE_MERCHANT');
				break;

			case 'P:D':
				$error_msg = do_lang_tempcode('SECPAY_NO_CURRENCY_MERCHANT');
				break;

			case 'P:V':
				$error_msg = do_lang_tempcode('SECPAY_NO_CV2');
				break;

			case 'P:R':
				$error_msg = do_lang_tempcode('SECPAY_TIMED_OUT');
				break;

			case 'P:#':
				$error_msg = do_lang_tempcode('SECPAY_NO_HASH');
				break;

			default:
				$error_msg = do_lang('CARD_GENERAL_ERROR');
		}

		return $error_msg;
	}

	/**
	* Handle IPN's. The function may produce output, which would be returned to the Payment Gateway. The function may do transaction verification.
	*
	* @return array	A long tuple of collected data.
	*/
	function handle_transaction()
	{
		$reason_code = post_param('code');
		$hash        = post_param('hash');
		$txn_id      = post_param('trans_id');
		$product     = post_param('product');
		$mc_gross    = post_param('amount');
		$mc_currency = post_param('currency');

		$memo           = '';
		$parent_txn_id  = $txn_id;
		$pending_reason = '';

		$success = ($reason_code == 'A') ? true : false;

		$object	    =	find_product($product);
		$product_det =	$object->product_id;
		$product_id	 =	$product_det[3]['id'];
		$seller	    =	(method_exists($object,'get_seller'))?$object->get_seller($product_id):NULL;

		$access_detail = $this->get_access_detail($seller);

		$my_hash = md5('trans_id='.$txn_id.'&amount='.float_to_raw_string($mc_gross).'&'.$access_detail[2]);

		if ($hash!=$my_hash) my_exit(do_lang('IPN_UNVERIFIED'));

		if (substr($txn_id,0,7)=='SUBSCR_')
		{
			$subscription = true;
			$txn_id = substr($txn_id,7);
		}
		else
		{
			$subscription = false;
			$txn_id = substr($txn_id,8);
		}

		$transaction_rows = $GLOBALS['SITE_DB']->query_select('trans_expecting',array('*'),array('id'=>$txn_id),'',1);
		if (!array_key_exists(0,$transaction_rows)) warn_exit(do_lang_tempcode('MISSING_RESOURCE'));
		$transaction_row = $transaction_rows[0];

		$member_id   = $transaction_row['e_member_id'];
		$purchase_id = $transaction_row['e_purchase_id'];

		$payment_status = ($success) ? 'Completed' : 'Failed';
		if ($subscription && !$success) $payment_status = 'SCancelled';

		if(addon_installed('shopping'))
		{
			$this->store_shipping_address($purchase_id);
		}

		return array($purchase_id,$product,$payment_status,$reason_code,$pending_reason,$memo,$mc_gross,$mc_currency,$txn_id,$parent_txn_id);
	}

	/**
	* Store shipping address for orders
	*
	* @param  AUTO_LINK		Order id
	* @return ?mixed			Address id (NULL: No address record found)
	*/
	function store_shipping_address($order_id)
	{
		$shipping_address	=	array();

		if(is_null($GLOBALS['SITE_DB']->query_value_null_ok('shopping_order_addresses','id',array('order_id'=>$order_id))))
		{
			$shipping_address['order_id']			=	$order_id;
			$shipping_address['address_name']	=	post_param('ship_first_name','')." ".post_param('ship_last_name','');
			$shipping_address['address_street']	=	post_param('ship_addr_1','');
			$shipping_address['address_city']	=	post_param('ship_city','');
			$shipping_address['address_state']	=	post_param('ship_state','');
			$shipping_address['address_zip']		=	post_param('ship_post_code','');
			$shipping_address['address_country']=	post_param('shipping_country','');
			$shipping_address['receiver_email']	=	'';//post_param('receiver_email','');

			$cpfs_to_save = array
										(
											'ocp_shipping_firstname' => post_param('ship_first_name',''),
											'ocp_shipping_lastname'	 => post_param('ship_last_name',''),
											'ocp_shipping_street'	 => post_param('ship_addr_1',''),
											'ocp_shipping_country'   => post_param('shipping_country',''),
											'ocp_shipping_state'		 => post_param('ship_state',''),
											'ocp_shipping_city'	    => post_param('ship_city',''),
											'ocp_shipping_zip'		 => post_param('ship_post_code','')
										);
			set_ocp_cpfs($cpfs_to_save);

			return $GLOBALS['SITE_DB']->query_insert('shopping_order_addresses',$shipping_address,true);
		}
	}

	/** Function to return local payment method for subscriptions
	 *
	 * @return ID_TEXT	automatic|local_controlled
	 */
	function get_subscription_local_payment_method()
	{
		return get_option('secpay_local_subscription_methods');
	}

	/**
	 * Function to return profile ids and types in serialized array format
	 *
	 *	@param	AUTO_LINK	Transaction id
	 * @param	AUTO_LINK 	Parent transaction id (Subscription profile id of a particular subscription)
	 * @return	?SHORT_TEXT	Type, profile id data in serialized string
	 */
	function	map_payment_ids($txn_id,$parent_txn_id)
	{
		$data = array('id'=>$parent_txn_id,'type'=>'subscr');
		return serialize($data);
	}
}