<?php /*

 ocPortal
 Copyright (c) ocProducts, 2004-2009

 See text/en/licence.txt for full licencing information.

*/

/**
 * @license		http://opensource.org/licenses/cpal_1.0 Common Public Attribution License
 * @copyright	ocProducts Ltd
 * @package		catalogues
 */

/**
 * Create an eCommerce catalogue.
 *
 * @param  ID_TEXT			The codename of the catalogue
 * @param  AUTO_LINK			Language code reference to the title of the catalogue
 * @param  ?MEMBER			The owner of the catalogue (NULL: nobody)
 * @return AUTO_LINK			The ID of the first new catalogues root category
 */
function create_ecommerce_catalogue($catalogue_name,$title,$owner=NULL)
{
	require_lang('catalogues');

	$first_cat=actual_add_catalogue($catalogue_name,$title,'',1,1,'',0,1,'never',$owner);

	$default_tax_list="10%|20%|30%";
	if ($catalogue_name!='products')
	{
		$_default_tax_list=$GLOBALS['SITE_DB']->query_value('catalogue_fields','cf_default',array('c_name'=>'products','cf_order'=>6,'cf_type'=>'list'),'ORDER BY cf_order');
		if (!is_null($_default_tax_list)) $default_tax_list=$_default_tax_list;
	}

	$fields=array(
		//		Name							 Description			Type			  Defines order  Required  Visible  Searchable
		array('ECOM_CAT_product_title','DESCRIPTION_TITLE','short_trans',1,1,1,1),
		array('ECOM_CAT_item_code','ECOM_CATD_item_code','random',0,1,1,1),
		array('ECOM_CAT_price_pre_tax','ECOM_CATD_price_pre_tax','float',0,1,1,1),
		array('ECOM_CAT_stock_level','ECOM_CATD_stock_level','integer',0,0,1,0),
		array('ECOM_CAT_stock_level_warn_at','ECOM_CATD_stock_level_warn_at','integer',0,0,0,0),
		array('ECOM_CAT_stock_level_maintain','ECOM_CATD_stock_level_maintain','tick',0,1,0,0),
		array('ECOM_CAT_tax_type','ECOM_CATD_tax_type','list',0,1,0,0,$default_tax_list,0),
		array('ECOM_CAT_image','ECOM_CATD_image','picture',0,0,1,1),
		array('ECOM_CAT_weight','ECOM_CATD_weight','float',0,1,0,0),
		array('ECOM_CAT_description','DESCRIPTION_DESCRIPTION','long_trans',0,1,1,1)
	);

	foreach ($fields as $i=>$field)
	{
		actual_add_catalogue_field($catalogue_name, // $c_name
											lang_code_to_default_content($field[0],false,3), // $name
											lang_code_to_default_content($field[1],true,3), // $description
											($field[2]=='tick')?'list':$field[2], // $type
											$i, // $order
											$field[3], // $defines_order
											$field[5], // $visible
											$field[6], // $searchable
											($field[2]=='tick')?(do_lang('NO').'|'.do_lang('YES')):(array_key_exists(7,$field)?$field[7]:''), // $default
											$field[4], // $required
											array_key_exists(5,$field)?$field[5]:0, // $put_in_category
											array_key_exists(5,$field)?$field[5]:0 // $put_in_search
										);
	}

	//Add permission to catalogues
	$groups=$GLOBALS['FORUM_DRIVER']->get_usergroup_list();
	foreach (array_keys($groups) as $group_id)
	{
		$GLOBALS['SITE_DB']->query_insert('group_category_access',array('module_the_name'=>'catalogues_catalogue','category_name'=>$catalogue_name,'group_id'=>$group_id));
		$GLOBALS['SITE_DB']->query_insert('group_category_access',array('module_the_name'=>'catalogues_category','category_name'=>strval($first_cat),'group_id'=>$group_id));
	}

	if (!is_null($owner))
	{
		foreach (array_keys($groups) as $group_id)
		{
			$perms=array('submit_cat_midrange_content','edit_cat_midrange_content','delete_cat_midrange_content','submit_midrange_content','edit_midrange_content','delete_midrange_content');
			foreach ($perms as $perm)
			{
				$GLOBALS['SITE_DB']->query_insert('gsp',array('specific_permission'=>$perm,'group_id'=>$group_id,'the_page'=>'','the_value'=>1,'module_the_name'=>'catalogues_catalogue','category_name'=>$catalogue_name));
			}
		}
	}

	return $first_cat;
}

/**
 * Add a catalogue using all the specified values.
 *
 * @param  ID_TEXT			The codename of the catalogue
 * @param  mixed				The title of the catalogue (either language code or string)
 * @param  mixed				A description (either language code or string)
 * @param  SHORT_INTEGER	The display type
 * @param  BINARY				Whether the catalogue uses a tree system (as opposed to mere categories in an index)
 * @param  LONG_TEXT			Hidden notes pertaining to this catalogue
 * @param  integer			How many points a member gets by submitting to this catalogue
 * @param  BINARY				Whether the catalogue is an eCommerce catalogue
 * @param  ID_TEXT			How to send view reports
 * @set    never daily weekly monthly quarterly
 * @param  ?MEMBER			The owner of the catalogue (NULL: nobody)
 * @return ?AUTO_LINK		The ID of the first new catalogues root category (NULL: no root, as it's not a tree catalogue)
 */
function actual_add_catalogue($name,$title,$description,$display_type,$is_tree,$notes,$submit_points,$ecommerce=0,$send_view_reports='never',$c_owner=NULL)
{
	require_code('type_validation');
	if (!is_alphanumeric($name)) warn_exit(do_lang_tempcode('BAD_CODENAME'));

	// Check doesn't already exist
	$test=$GLOBALS['SITE_DB']->query_value_null_ok('catalogues','c_name',array('c_name'=>$name));
	if (!is_null($test)) warn_exit(do_lang_tempcode('ALREADY_EXISTS',escape_html($name)));

	// Create
	if (!is_integer($description)) $description=insert_lang_comcode($description,2);
	if (!is_integer($title)) $title=insert_lang($title,1);
	$GLOBALS['SITE_DB']->query_insert('catalogues',array('c_name'=>$name,'c_title'=>$title,'c_send_view_reports'=>$send_view_reports,'c_ecommerce'=>$ecommerce,'c_description'=>$description,'c_display_type'=>$display_type,'c_is_tree'=>$is_tree,'c_notes'=>$notes,'c_add_date'=>time(),'c_submit_points'=>$submit_points,'c_owner'=>$c_owner));

	if ($is_tree==1)
	{
		// Create root node
		$root_title=get_translated_text($title);
		$category=$GLOBALS['SITE_DB']->query_insert('catalogue_categories',array('cc_move_days_lower'=>30,'cc_move_days_higher'=>60,'cc_move_target'=>NULL,'rep_image'=>'','c_name'=>$name,'cc_title'=>insert_lang($root_title,1),'cc_description'=>insert_lang_comcode('',3),'cc_notes'=>'','cc_add_date'=>time(),'cc_parent_id'=>NULL),true);
	} else $category=NULL;

	log_it('ADD_CATALOGUE',$name);

	return $category;
}

/**
 * Add a field to the specified catalogue, without disturbing any other data in that catalogue.
 *
 * @param  ID_TEXT		The codename of the catalogue the field is for
 * @param  mixed			The name of the field (either language code or string)
 * @param  mixed			A description (either language code or string)
 * @param  ID_TEXT		The type of the field
 * @param  integer		The field order (the field order determines what order the fields are displayed within an entry)
 * @param  BINARY			Whether this field defines the catalogue order
 * @param  BINARY			Whether this is a visible field
 * @param  BINARY			Whether the field is usable as a search key
 * @param  LONG_TEXT		The default value for the field
 * @param  BINARY			Whether this field is required
 * @param  BINARY			Whether the field is to be shown in category views (not applicable for the list display type)
 * @param  BINARY			Whether the field is to be shown in search views (not applicable for the list display type)
 */
function actual_add_catalogue_field($c_name,$name,$description,$type,$order,$defines_order,$visible,$searchable,$default,$required,$put_in_category=1,$put_in_search=1)
{
	if (!is_integer($description)) $description=insert_lang($description,2);
	if (!is_integer($name)) $name=insert_lang($name,2);
	$cf_id=$GLOBALS['SITE_DB']->query_insert('catalogue_fields',array('c_name'=>$c_name,'cf_name'=>$name,'cf_description'=>$description,'cf_type'=>$type,'cf_order'=>$order,'cf_defines_order'=>$defines_order,'cf_visible'=>$visible,'cf_searchable'=>$searchable,'cf_default'=>$default,'cf_required'=>$required,'cf_put_in_category'=>$put_in_category,'cf_put_in_search'=>$put_in_search),true);

	require_code('hooks/modules/catalogue_fields/'.filter_naughty($type));
	$ob=object_factory('Hook_catalogue_field_'.filter_naughty($type));

	// Now add field values for all pre-existing entries (in the ideal world, there would be none yet) - we find our entries via our categories
	$categories=collapse_1d_complexity('id',$GLOBALS['SITE_DB']->query_select('catalogue_categories',array('id'),array('c_name'=>$c_name)));
	foreach ($categories as $category)
	{
		$entries=collapse_1d_complexity('id',$GLOBALS['SITE_DB']->query_select('catalogue_entries',array('id'),array('cc_id'=>$category)));
		foreach ($entries as $entry)
		{
			list($raw_type,$default,$_type)=$ob->get_field_value_row_bits($cf_id,$required==1,$default);

			$map=array('cf_id'=>$cf_id,'ce_id'=>$entry,'cv_value'=>$default);

			$GLOBALS['SITE_DB']->query_insert('catalogue_efv_'.$_type,$map);
		}
	}
}

/**
 * Get a fresh value for an auto_increment valued field.
 *
 * @param  AUTO_LINK		The field ID
 * @param  string			The field default
 * @return string			The value
 */
function get_field_auto_increment($field_id,$default='')
{
	$value=$GLOBALS['SITE_DB']->query_value_null_ok('catalogue_efv_short','cv_value',array('cf_id'=>$field_id),'ORDER BY ce_id DESC');
	if (is_null($value))
	{
		$value=strval(intval($default)-1);
	}

	$test=NULL;
	do
	{
		$value=strval(intval($value)+1);

		$test=$GLOBALS['SITE_DB']->query_value_null_ok('catalogue_efv_short','ce_id',array('cv_value'=>$value,'cf_id'=>$field_id));
	}
	while (!is_null($test));

	return $value;
}

/**
 * Get a fresh value for a random valued field.
 *
 * @param  AUTO_LINK		The field ID
 * @param  string			The field default
 * @return string			The value
 */
function get_field_random($field_id,$default='')
{
	$rand_array='1234567890abcdefghijklmnopqrstuvwxyz';
	$c=strlen($rand_array)-1;
	$length=intval($default);
	if ($length==0) $length=10;
	$test=NULL;
	do
	{
		$value='';
		for ($i=0;$i<$length;$i++)
		{
			$value.=$rand_array[mt_rand(0,$c)];
		}

		$test=$GLOBALS['SITE_DB']->query_value_null_ok('catalogue_efv_short','ce_id',array('cv_value'=>$value,'cf_id'=>$field_id));
	}
	while (!is_null($test));

	return $value;
}

/**
 * Edit a catalogue.
 *
 * @param  ID_TEXT			The current name of the catalogue
 * @param  ID_TEXT			The new name of the catalogue
 * @param  SHORT_TEXT		The human readable name/title of the catalogue
 * @param  LONG_TEXT			The description
 * @param  SHORT_INTEGER	The display type
 * @param  LONG_TEXT			Admin notes
 * @param  integer			How many points are given to a member that submits to the catalogue
 * @param  BINARY				Whether the catalogue is an eCommerce catalogue
 * @param  ID_TEXT			How to send view reports
 * @set    never daily weekly monthly quarterly
 */
function actual_edit_catalogue($old_name,$name,$title,$description,$display_type,$notes,$submit_points,$ecommerce,$send_view_reports,$c_owner=NULL)
{
	if ($old_name!=$name)
	{
		// Check doesn't already exist
		$test=$GLOBALS['SITE_DB']->query_value_null_ok('catalogues','c_name',array('c_name'=>$name));
		if (!is_null($test)) warn_exit(do_lang_tempcode('ALREADY_EXISTS',escape_html($name)));

		require_code('type_validation');
		if (!is_alphanumeric($name)) warn_exit(do_lang_tempcode('BAD_CODENAME'));
	}

	$rows=$GLOBALS['SITE_DB']->query_select('catalogues',array('c_description','c_title'),array('c_name'=>$old_name));
	if (!array_key_exists(0,$rows))
	{
		warn_exit(do_lang_tempcode('MISSING_RESOURCE'));
	}
	$myrow=$rows[0];
	$_title=$myrow['c_title'];
	$_description=$myrow['c_description'];

	// Edit
	$GLOBALS['SITE_DB']->query_update('catalogues',array('c_send_view_reports'=>$send_view_reports,'c_display_type'=>$display_type,'c_ecommerce'=>$ecommerce,'c_name'=>$name,'c_title'=>lang_remap($_title,$title),'c_description'=>lang_remap_comcode($_description,$description),'c_notes'=>$notes,'c_add_date'=>time(),'c_submit_points'=>$submit_points,'c_owner'=>$c_owner),array('c_name'=>$old_name),'',1);

	// If we're renaming, then we better change a load of references
	if ($name!=$old_name)
	{
		$GLOBALS['SITE_DB']->query_update('catalogue_categories',array('c_name'=>$name),array('c_name'=>$old_name));
		$GLOBALS['SITE_DB']->query_update('catalogue_fields',array('c_name'=>$name),array('c_name'=>$old_name));
		$GLOBALS['SITE_DB']->query_update('catalogue_entries',array('c_name'=>$name),array('c_name'=>$old_name));

		$types=$GLOBALS['SITE_DB']->query_select('award_types',array('id'),array('a_content_type'=>'catalogue'));
		foreach ($types as $type)
		{
			$GLOBALS['SITE_DB']->query_update('award_archive',array('content_id'=>$name),array('content_id'=>$old_name,'a_type_id'=>$type['id']));
		}
	}

	log_it('EDIT_CATALOGUE',$name);
}

/**
 * Whether the catalogue exists or not.
 *
 * @param  ID_TEXT		The name of the catalogue
 * @return bool
 */
function is_catalogue($name)
{
	$value_check = $GLOBALS['FORUM_DB']->query_value_null_ok('catalogues','c_title',array('c_name'=>$name));
	return ($value_check) ? true : false;
}

/**
 * Delete a catalogue.
 *
 * @param  ID_TEXT		The name of the catalogue
 */
function actual_delete_catalogue($name)
{
	// Delete lang
	$rows=$GLOBALS['SITE_DB']->query_select('catalogues',array('c_description','c_title'),array('c_name'=>$name));
	if (!array_key_exists(0,$rows))
	{
		warn_exit(do_lang_tempcode('MISSING_RESOURCE'));
	}
	$myrow=$rows[0];
	delete_lang($myrow['c_title']);
	delete_lang($myrow['c_description']);

	// Delete anything involved (ha ha destruction!)
	$entries=collapse_1d_complexity('id',$GLOBALS['SITE_DB']->query_select('catalogue_entries',array('id'),array('c_name'=>$name)));
	foreach ($entries as $entry)
	{
		actual_delete_catalogue_entry($entry);
	}
	$GLOBALS['SITE_DB']->query_delete('catalogues',array('c_name'=>$name),'',1);
	$categories=collapse_1d_complexity('id',$GLOBALS['SITE_DB']->query_select('catalogue_categories',array('id'),array('c_name'=>$name)));
	foreach ($categories as $category)
	{
		actual_delete_catalogue_category($category,true);
	}
	$fields=collapse_1d_complexity('id',$GLOBALS['SITE_DB']->query_select('catalogue_fields',array('id'),array('c_name'=>$name)));
	foreach ($fields as $field)
	{
		actual_delete_catalogue_field($field);
	}
	$GLOBALS['SITE_DB']->query_delete('group_category_access',array('module_the_name'=>'catalogues_catalogue','category_name'=>$name));
	$GLOBALS['SITE_DB']->query_delete('gsp',array('module_the_name'=>'catalogues_catalogue','category_name'=>$name));

	log_it('DELETE_CATALOGUE',$name);
}

/**
 * Edit a catalogue field.
 *
 * @param  AUTO_LINK		The ID of the field
 * @param  ID_TEXT		The name of the catalogue
 * @param  SHORT_TEXT	The name of the field
 * @param  LONG_TEXT		Description for the field
 * @param  integer		The field order (the field order determines what order the fields are displayed within an entry)
 * @param  BINARY			Whether the field defines entry ordering
 * @param  BINARY			Whether the field is visible when an entry is viewed
 * @param  BINARY			Whether the field is usable as a search key
 * @param  LONG_TEXT		The default value for the field
 * @param  BINARY			Whether the field is required
 * @param  BINARY			Whether the field is to be shown in category views (not applicable for the list display type)
 * @param  BINARY			Whether the field is to be shown in search views (not applicable for the list display type)
 * @param  ?ID_TEXT		The field type (NULL: do not change)
 */
function actual_edit_catalogue_field($id,$c_name,$name,$description,$order,$defines_order,$visible,$searchable,$default,$required,$put_in_category=1,$put_in_search=1,$type=NULL) // You cannot edit a field type
{
	$rows=$GLOBALS['SITE_DB']->query_select('catalogue_fields',array('cf_description','cf_name'),array('id'=>$id));
	if (!array_key_exists(0,$rows))
	{
		warn_exit(do_lang_tempcode('MISSING_RESOURCE'));
	}
	$myrow=$rows[0];
	$_name=$myrow['cf_name'];
	$_description=$myrow['cf_description'];

	$map=array('c_name'=>$c_name,'cf_name'=>lang_remap($_name,$name),'cf_description'=>lang_remap($_description,$description),'cf_order'=>$order,'cf_defines_order'=>$defines_order,'cf_visible'=>$visible,'cf_searchable'=>$searchable,'cf_default'=>$default,'cf_required'=>$required,'cf_put_in_category'=>$put_in_category,'cf_put_in_search'=>$put_in_search);
	if (!is_null($type)) $map['cf_type']=$type;

	$GLOBALS['SITE_DB']->query_update('catalogue_fields',$map,array('id'=>$id),'',1);
}

/**
 * Delete a catalogue field.
 *
 * @param  AUTO_LINK		The ID of the field
 */
function actual_delete_catalogue_field($id)
{
	$rows=$GLOBALS['SITE_DB']->query_select('catalogue_fields',array('cf_name','cf_description','cf_type'),array('id'=>$id));
	if (!array_key_exists(0,$rows))
	{
		warn_exit(do_lang_tempcode('MISSING_RESOURCE'));
	}
	$myrow=$rows[0];
	delete_lang($myrow['cf_name']);
	delete_lang($myrow['cf_description']);

	$GLOBALS['SITE_DB']->query_delete('catalogue_fields',array('id'=>$id),'',1);
}

/**
 * Add a catalogue category
 *
 * @param  ID_TEXT		The codename of the catalogue the category is in
 * @param  mixed			The title of this category (either language code or string)
 * @param  mixed			A description (either language code or string)
 * @param  LONG_TEXT		Hidden notes pertaining to this category
 * @param  ?AUTO_LINK	The ID of this categories parent (NULL: a root category, or not a tree catalogue)
 * @param  URLPATH		The representative image for the category (blank: none)
 * @param  integer		The number of days before expiry (lower limit)
 * @param  integer		The number of days before expiry (higher limit)
 * @param  ?AUTO_LINK	The expiry category (NULL: do not expire)
 * @param  ?TIME			The add time (NULL: now)
 * @param  ?AUTO_LINK	Force an ID (NULL: don't force an ID)
 * @return AUTO_LINK		The ID of the new category
 */
function actual_add_catalogue_category($catalogue_name,$title,$description,$notes,$parent_id,$rep_image,$move_days_lower=30,$move_days_higher=60,$move_target=NULL,$add_date=NULL,$id=NULL)
{
	if (is_null($add_date)) $add_date=time();
	if (!is_integer($description)) $description=insert_lang_comcode($description,3);
	if (!is_integer($title)) $title=insert_lang($title,2);
	$map=array('cc_move_days_lower'=>$move_days_lower,'cc_move_days_higher'=>$move_days_higher,'cc_move_target'=>$move_target,'rep_image'=>$rep_image,'cc_add_date'=>$add_date,'c_name'=>$catalogue_name,'cc_title'=>$title,'cc_description'=>$description,'cc_notes'=>$notes,'cc_parent_id'=>$parent_id);
	if (!is_null($id)) $map['id']=$id;
	$id=$GLOBALS['SITE_DB']->query_insert('catalogue_categories',$map,true);

	log_it('ADD_CATALOGUE_CATEGORY',strval($id));

	require_code('seo2');
	if (!is_numeric($title)) seo_meta_set_for_implicit('catalogue_category',strval($id),array($title,$description),$title);

	return $id;
}

/**
 * Edit a catalogue category.
 *
 * @param  AUTO_LINK		The ID of the category
 * @param  SHORT_TEXT	The title of the category
 * @param  LONG_TEXT		Description for the category
 * @param  LONG_TEXT		Admin notes
 * @param  ?AUTO_LINK	The ID of the parent category (NULL: no parent)
 * @param  SHORT_TEXT	Meta keywords for the category
 * @param  LONG_TEXT		Meta description for the category
 * @param  URLPATH		The representative image for the category (blank: none)
 * @param  integer		The number of days before expiry (lower limit)
 * @param  integer		The number of days before expiry (higher limit)
 * @param  ?AUTO_LINK	The expiry category (NULL: do not expire)
 */
function actual_edit_catalogue_category($id,$title,$description,$notes,$parent_id,$meta_keywords,$meta_description,$rep_image,$move_days_lower,$move_days_higher,$move_target)
{
	$under_category_id=$parent_id;
	while ((!is_null($under_category_id)) && ($under_category_id!=INTEGER_MAGIC_NULL))
	{
		if ($id==$under_category_id) warn_exit(do_lang_tempcode('OWN_PARENT_ERROR'));
		$under_category_id=$GLOBALS['SITE_DB']->query_value('catalogue_categories','cc_parent_id',array('id'=>$under_category_id));
	}

	$rows=$GLOBALS['SITE_DB']->query_select('catalogue_categories',array('cc_description','cc_title'),array('id'=>$id));
	if (!array_key_exists(0,$rows))
	{
		warn_exit(do_lang_tempcode('MISSING_RESOURCE'));
	}
	$myrow=$rows[0];
	$_title=$myrow['cc_title'];
	$_description=$myrow['cc_description'];

	$map=array('cc_move_days_lower'=>$move_days_lower,'cc_move_days_higher'=>$move_days_higher,'cc_move_target'=>$move_target,'cc_title'=>lang_remap($_title,$title),'cc_description'=>lang_remap_comcode($_description,$description),'cc_notes'=>$notes,'cc_parent_id'=>$parent_id);

	if (!is_null($rep_image))
	{
		$map['rep_image']=$rep_image;
		require_code('files2');
		delete_upload('uploads/grepimages','catalogue_categories','rep_image','id',$id,$rep_image);
	}

	$GLOBALS['SITE_DB']->query_update('catalogue_categories',$map,array('id'=>$id),'',1);

	require_code('urls2');
	suggest_new_idmoniker_for('catalogues','category',strval($id),$title);

	require_code('seo2');
	seo_meta_set_for_explicit('catalogue_category',strval($id),$meta_keywords,$meta_description);

	log_it('EDIT_CATALOGUE_CATEGORY',$title);
}

/**
 * Delete a catalogue category.
 *
 * @param  AUTO_LINK		The ID of the category
 * @param  boolean		Whether we're deleting everything under the category; if FALSE we will actively reassign child categories to be directly under the root
 */
function actual_delete_catalogue_category($id,$deleting_all=false)
{
	// Info about our category
	$rows=$GLOBALS['SITE_DB']->query_select('catalogue_categories c LEFT JOIN '.$GLOBALS['SITE_DB']->get_table_prefix().'catalogues x ON c.c_name=x.c_name',array('c_is_tree','c.c_name','cc_description','cc_title','cc_parent_id'),array('id'=>$id));
	if (!array_key_exists(0,$rows))
	{
		warn_exit(do_lang_tempcode('MISSING_RESOURCE'));
	}
	$myrow=$rows[0];

	// If we aren't deleting the entire catalogue, make sure we don't delete the root category
	if ((!$deleting_all) && ($myrow['c_is_tree']==1))
	{
		$root_category=$GLOBALS['SITE_DB']->query_value('catalogue_categories','MIN(id)',array('c_name'=>$myrow['c_name']));
		if ($id==$root_category) warn_exit(do_lang_tempcode('CATALOGUE_NO_DELETE_ROOT'));
	}

	require_code('files2');
	delete_upload('uploads/grepimages','catalogue_categories','rep_image','id',$id);

	if (!$deleting_all)
	{
		// If we're in a tree
		if ($myrow['c_is_tree']==1)
		{
			$GLOBALS['SITE_DB']->query_update('catalogue_categories',array('cc_parent_id'=>$myrow['cc_parent_id']),array('cc_parent_id'=>$id));
			$GLOBALS['SITE_DB']->query_update('catalogue_entries',array('cc_id'=>$myrow['cc_parent_id']),array('cc_id'=>$id));
		} else
		{
			$GLOBALS['SITE_DB']->query_delete('catalogue_categories',array('cc_parent_id'=>$id)); // Does nothing, in theory
			$entries=$GLOBALS['SITE_DB']->query_select('catalogue_entries',array('id'),array('cc_id'=>$id));
			foreach ($entries as $entry)
			{
				actual_delete_catalogue_entry($entry['id']);
			}
		}

		$GLOBALS['SITE_DB']->query_update('catalogue_categories',array('cc_move_target'=>NULL),array('cc_move_target'=>$id));
	}

	require_code('seo2');
	seo_meta_erase_storage('catalogue_category',strval($id));

	log_it('DELETE_CATALOGUE_CATEGORY',strval($id),get_translated_text($myrow['cc_title']));

	// Delete lang
	delete_lang($myrow['cc_title']);
	delete_lang($myrow['cc_description']);

	/*$entries=collapse_1d_complexity('id',$GLOBALS['SITE_DB']->query_select('catalogue_entries',array('id'),array('cc_id'=>$id)));
	foreach ($entries as $entry)
	{
		actual_delete_catalogue_entry($entry);
	}*/

	$GLOBALS['SITE_DB']->query_delete('catalogue_categories',array('id'=>$id),'',1);
	$GLOBALS['SITE_DB']->query_delete('group_category_access',array('module_the_name'=>'catalogues_category','category_name'=>strval($id)));
	$GLOBALS['SITE_DB']->query_delete('gsp',array('module_the_name'=>'catalogues_category','category_name'=>strval($id)));
}

/**
 * Adds an entry to the specified catalogue.
 *
 * @param  AUTO_LINK			The ID of the category that the entry is in
 * @param  BINARY				Whether the entry has been validated
 * @param  LONG_TEXT			Hidden notes pertaining to the entry
 * @param  BINARY				Whether the entry may be rated
 * @param  SHORT_INTEGER	Whether comments are allowed (0=no, 1=yes, 2=review style)
 * @param  BINARY				Whether the entry may be trackbacked
 * @param  array				A map of field IDs, to values, that defines the entries settings
 * @param  ?TIME				The time the entry was added (NULL: now)
 * @param  ?MEMBER			The entries submitter (NULL: current user)
 * @param  ?TIME				The edit time (NULL: never)
 * @param  integer			The number of views
 * @param  ?AUTO_LINK		Force an ID (NULL: don't force an ID)
 * @return AUTO_LINK			The ID of the newly added entry
 */
function actual_add_catalogue_entry($category_id,$validated,$notes,$allow_rating,$allow_comments,$allow_trackbacks,$map,$time=NULL,$submitter=NULL,$edit_date=NULL,$views=0,$id=NULL)
{
	if (is_null($time)) $time=time();
	if (is_null($submitter)) $submitter=get_member();

	$catalogue_name=$GLOBALS['SITE_DB']->query_value('catalogue_categories','c_name',array('id'=>$category_id));
	$fields=collapse_2d_complexity('id','cf_type',$GLOBALS['SITE_DB']->query_select('catalogue_fields',array('id','cf_type'),array('c_name'=>$catalogue_name)));

	require_code('comcode_check');

	$imap=array('c_name'=>$catalogue_name,'ce_edit_date'=>$edit_date,'cc_id'=>$category_id,'ce_last_moved'=>time(),'ce_submitter'=>$submitter,'ce_add_date'=>$time,'ce_views'=>$views,'ce_views_prior'=>$views,'ce_validated'=>$validated,'notes'=>$notes,'allow_rating'=>$allow_rating,'allow_comments'=>$allow_comments,'allow_trackbacks'=>$allow_trackbacks);
	if (!is_null($id)) $imap['id']=$id;
	$val=mixed();
	foreach ($map as $field_id=>$val)
	{
		$type=$fields[$field_id];

		if (($type=='short_trans') || ($type=='long_trans'))
			check_comcode($val);
	}
	$id=$GLOBALS['SITE_DB']->query_insert('catalogue_entries',$imap,true);

	$catalog_entry_title	=	'';

	foreach ($map as $field_id=>$val)
	{
		if ($val==STRING_MAGIC_NULL) $val='';

		$type=$fields[$field_id];
		if((($type=='short_text') || ($type=='short_trans')) && ($catalog_entry_title==''))	$catalog_entry_title=post_param('field_'.strval($field_id),'');
		require_code('hooks/modules/catalogue_fields/'.filter_naughty($type));
		$ob=object_factory('Hook_catalogue_field_'.filter_naughty($type));
		list(,,$sup_table_name)=$ob->get_field_value_row_bits($field_id);

		if (($type=='short_trans') || ($type=='long_trans'))
			$val=insert_lang_comcode($val,3);

		$GLOBALS['SITE_DB']->query_insert('catalogue_efv_'.$sup_table_name,array('cf_id'=>$field_id,'ce_id'=>$id,'cv_value'=>$val));
	}

	decache('main_cc_embed');
	decache('main_recent_cc_entries');

	require_code('seo2');
	seo_meta_set_for_implicit('catalogue_entry',strval($id),$map,'');

	log_it('ADD_CATALOGUE_ENTRY',strval($id));

	$is_product=(get_param('catalogue_name','')=='products')?true:false;

	$is_ecommerce_product=$GLOBALS['SITE_DB']->query_value('catalogues','c_ecommerce',array('c_name'=>$catalogue_name));

	if (has_category_access($GLOBALS['FORUM_DRIVER']->get_guest_id(),'catalogues_category',strval($category_id)))
	{
		$sbmitter	=	$GLOBALS['FORUM_DRIVER']->get_username(get_member());
		say_activity(($is_ecommerce_product)?'catalogues:NEW_PRODUCT_ADDED':'catalogues:CATALOGUES_ENTRY_ADDED',$sbmitter,$catalog_entry_title,$catalogue_name,'_SEARCH:members:view:'.strval(get_member()),'_SEARCH:catalogues:entry:'.strval($id),'','','catalogues');
	}

	return $id;
}

/**
 * Edit the specified catalogue entry
 *
 * @param  AUTO_LINK			The ID of the entry being edited
 * @param  AUTO_LINK			The ID of the category that the entry is in
 * @param  BINARY				Whether the entry has been validated
 * @param  LONG_TEXT			Hidden notes pertaining to the entry
 * @param  BINARY				Whether the entry may be rated
 * @param  SHORT_INTEGER	Whether comments are allowed (0=no, 1=yes, 2=review style)
 * @param  BINARY				Whether the entry may be trackbacked
 * @param  array				A map of field IDs, to values, that defines the entries settings
 * @param  ?SHORT_TEXT		Meta keywords for this resource (NULL: do not edit)
 * @param  ?LONG_TEXT		Meta description for this resource (NULL: do not edit)
 */
function actual_edit_catalogue_entry($id,$category_id,$validated,$notes,$allow_rating,$allow_comments,$allow_trackbacks,$map,$meta_keywords,$meta_description)
{
	$catalogue_name=$GLOBALS['SITE_DB']->query_value('catalogue_categories','c_name',array('id'=>$category_id));
	$fields=collapse_2d_complexity('id','cf_type',$GLOBALS['SITE_DB']->query_select('catalogue_fields',array('id','cf_type'),array('c_name'=>$catalogue_name)));

	require_code('files2');
	foreach ($fields as $cf_id=>$cf_type)
	{
		if ((($cf_type=='upload') || ($cf_type=='picture')) && (array_key_exists($cf_id,$map)) && ($map[$cf_id]!=STRING_MAGIC_NULL))
		{
			delete_upload('uploads/catalogues','catalogue_efv_short','cv_value',array('ce_id'=>$id,'cf_id'=>$cf_id),$id,$map[$cf_id]);
		}
	}

	$GLOBALS['SITE_DB']->query_update('catalogue_entries',array('ce_edit_date'=>time(),'cc_id'=>$category_id,'ce_validated'=>$validated,'notes'=>$notes,'allow_rating'=>$allow_rating,'allow_comments'=>$allow_comments,'allow_trackbacks'=>$allow_trackbacks),array('id'=>$id),'',1);
	foreach ($map as $field_id=>$val)
	{
		$type=$fields[$field_id];

		require_code('hooks/modules/catalogue_fields/'.filter_naughty($type));
		$ob=object_factory('Hook_catalogue_field_'.filter_naughty($type));
		list(,,$sup_table_name)=$ob->get_field_value_row_bits($field_id);

		if (substr($sup_table_name,-6)=='_trans')
		{
			$_val=$GLOBALS['SITE_DB']->query_value_null_ok('catalogue_efv_'.$sup_table_name,'cv_value',array('cf_id'=>$field_id,'ce_id'=>$id));
			if (is_null($_val))
			{
				$val=insert_lang_comcode($val,3);
			} else
			{
				$val=lang_remap_comcode($_val,$val);
			}
		}

		$GLOBALS['SITE_DB']->query_update('catalogue_efv_'.$sup_table_name,array('cv_value'=>$val),array('cf_id'=>$field_id,'ce_id'=>$id),'',1);
	}

	require_code('urls2');
	suggest_new_idmoniker_for('catalogues','entry',strval($id),strip_comcode(array_shift($map)));

	require_code('seo2');
	seo_meta_set_for_explicit('catalogue_entry',strval($id),$meta_keywords,$meta_description);

	decache('main_cc_embed');
	decache('main_recent_cc_entries');

	log_it('EDIT_CATALOGUE_ENTRY',strval($id));
}

/**
 * Delete a catalogue entry.
 *
 * @param  AUTO_LINK		The ID of the entry to delete
 */
function actual_delete_catalogue_entry($id)
{
	$catalogue_name=$GLOBALS['SITE_DB']->query_value('catalogue_entries','c_name',array('id'=>$id));
	$url_fields=$GLOBALS['SITE_DB']->query('SELECT id FROM '.$GLOBALS['SITE_DB']->get_table_prefix().'catalogue_fields WHERE '.db_string_equal_to('c_name',$catalogue_name).' AND ('.db_string_equal_to('cf_type','upload').' OR '.db_string_equal_to('cf_type','picture').')');
	require_code('files2');
	foreach ($url_fields as $field)
		delete_upload('uploads/catalogues','catalogue_efv_short','cv_value',array('ce_id'=>$id,'cf_id'=>$field['id']),$id);

	$lang1=$GLOBALS['SITE_DB']->query_select('catalogue_efv_long_trans',array('cv_value'),array('ce_id'=>$id));
	$lang2=$GLOBALS['SITE_DB']->query_select('catalogue_efv_short_trans',array('cv_value'),array('ce_id'=>$id));
	$lang=array_merge($lang1,$lang2);
	foreach ($lang as $lang_to_delete)
	{
		delete_lang($lang_to_delete['cv_value']);
	}

	$GLOBALS['SITE_DB']->query_delete('catalogue_efv_long_trans',array('ce_id'=>$id));
	$GLOBALS['SITE_DB']->query_delete('catalogue_efv_short_trans',array('ce_id'=>$id));
	$GLOBALS['SITE_DB']->query_delete('catalogue_efv_long',array('ce_id'=>$id));
	$GLOBALS['SITE_DB']->query_delete('catalogue_efv_short',array('ce_id'=>$id));

	$GLOBALS['SITE_DB']->query_delete('catalogue_entries',array('id'=>$id),'',1);
	$GLOBALS['SITE_DB']->query_delete('trackbacks',array('trackback_for_type'=>'catalogues','trackback_for_id'=>$id));
	$GLOBALS['SITE_DB']->query_delete('rating',array('rating_for_type'=>'catalogues','rating_for_id'=>$id));

	require_code('seo2');
	seo_meta_erase_storage('catalogue_entry',strval($id));

	decache('main_recent_cc_entries');
	decache('main_cc_embed');

	log_it('DELETE_CATALOGUE_ENTRY',strval($id));
}


