<?php /*

 ocPortal
 Copyright (c) ocProducts, 2004-2008

 See text/en/licence.txt for full licencing information.

*/

/**
 * @license		http://opensource.org/licenses/cpal_1.0 Common Public Attribution License
 * @copyright	ocProducts Ltd
 * @package		catalogues
 */

/**
 * Standard code module initialisation function.
 */
function init__catalogues()
{
	global $PT_PAIR_CACHE;
	$PT_PAIR_CACHE=array();

	// We do not actually necessarily use these constants in the code (they're based on an extensive of an old BINARY field): but they're here for reference so as to understand the codes
	if (!defined('C_DT_MAPS'))
	{
		define('C_DT_MAPS',0);
		define('C_DT_LIST',1);
		define('C_DT_MATRIX',2);
	}
}

/**
 * Find the URL or catalogue entry count of a member's store catalogue.
 *
 * @param  MEMBER				The owner of the catalogue
 * @return tempcode			Link to the root catalogue category, or blank if there is no catalogue for this member yet
 */
function get_catalogue_store_for($member_id,$isCount=false)
{
	// Find a store, if they have one
        $count = 0;
	$store_url = new ocp_tempcode();
	$catalogue_name=$GLOBALS['SITE_DB']->query_value_null_ok('catalogues','c_name',array('c_owner'=>$member_id));
	if (!is_null($catalogue_name))
	{
		$root_category=$GLOBALS['SITE_DB']->query_value_null_ok('catalogue_categories','id',array('c_name'=>$catalogue_name));
		if (!is_null($root_category))
		{
                        if ($isCount) {
                            $entries = $GLOBALS['SITE_DB']->query_select('catalogue_entries',array('id'),array('cc_id'=>"$root_category",'ce_validated'=>'1'));
                            $count = count($entries);
                        } else {
                            $store_url=build_url(array('page'=>'catalogues','type'=>'category','id'=>$root_category),get_module_zone('catalogues'));
                        }
		}
	}

	return ($isCount) ? $count : $store_url;
}

/**
 * Get tempcode for a catalogue category 'feature box' for the given row
 *
 * @param  array			The database field row of it
 * @param  ID_TEXT		The zone to use
 * @param  boolean		Whether to put it in a box
 * @return tempcode		A box for it, linking to the full page
 */
function get_catalogue_category_html($row,$zone='_SEARCH',$put_in_box=true)
{
	$content=paragraph(get_translated_tempcode($row['cc_description']),'yghjgfjftgerr');
	$url=build_url(array('page'=>'catalogues','type'=>'category','id'=>$row['id']),$zone);
	$content->attach(paragraph(hyperlink($url,escape_html(get_translated_text($row['cc_title']))),'gdfhrthfgdgdf'));
	if (!$put_in_box) return $content;

	$tree=catalogue_category_breadcrumbs($row['id']);
	if (!$tree->is_empty()) $content->attach(paragraph(do_lang_tempcode('LOCATED_IN',$tree)));

	return put_in_standard_box($content,do_lang_tempcode('CATALOGUE_CATEGORY'));
}

/**
 * Grant all usergroups access to the specified catalogue category.
 *
 * @param  AUTO_LINK		The ID of the category that access is being given to
 */
function grant_catalogue_full_access($category_id)
{
	$groups=$GLOBALS['FORUM_DRIVER']->get_usergroup_list();
	foreach (array_keys($groups) as $group_id)
	{
		$GLOBALS['SITE_DB']->query_insert('group_category_access',array('module_the_name'=>'catalogues_category','category_name'=>strval($category_id),'group_id'=>$group_id));
	}
}

/**
 * Count the entries and subcategories underneath the specified category, recursively.
 *
 * @param  AUTO_LINK		The ID of the category for which count details are collected
 * @return array			The number of entries is returned in $output['num_entries'], and the number of subcategories is returned in $output['num_children'], and the (possibly recursive) number of entries is returned in $output['num_entries_children'].
 */
function count_catalogue_category_children($category_id)
{
	$out=array();
	$out['num_children']=$GLOBALS['SITE_DB']->query_value('catalogue_categories','COUNT(*)',array('cc_parent_id'=>$category_id));
	$out['num_entries']=$GLOBALS['SITE_DB']->query_value('catalogue_entries','COUNT(*)',array('cc_id'=>$category_id,'ce_validated'=>1));

	$out['num_entries_children']=$out['num_entries'];
	if (true) // Disable if you want to improve efficiency at cost of showing tree counts
	{
		$rows=$GLOBALS['SITE_DB']->query_select('catalogue_categories',array('id'),array('cc_parent_id'=>$category_id));
		foreach ($rows as $child)
		{
			$temp=count_catalogue_category_children($child['id']);
			$out['num_entries_children']+=$temp['num_entries_children'];
		}
	}

	return $out;
}

/**
 * Get an ordered array of all the entries in the specified catalogue.
 *
 * @param  ?AUTO_LINK		The ID of the category for which the entries are being collected (NULL: entries are [and must be] passed instead)
 * @param  ID_TEXT			The name of the catalogue
 * @param  array				A database row of the catalogue we are working with
 * @param  ID_TEXT			The view type we're doing
 * @set    PAGE SEARCH CATEGORY
 * @param  ID_TEXT			The template set we are rendering this category using
 * @param  ?integer			The maximum number of entries to show on a single page of this this category (ignored if $select is not NULL) (NULL: all)
 * @param  ?integer			The entry number to start at (ignored if $select is not NULL) (NULL: all)
 * @param  ?array				The entries to show (NULL: use $start and $max)
 * @param  AUTO_LINK			The virtual root for display of this category (-1 means "whatever")
 * @param  ?SHORT_INTEGER The display type to use (NULL: lookup from $catalogue)
 * @param  boolean			Whether to perform sorting
 * @param  ?array				A list of entry rows (NULL: select them normally)
 * @return array				An array containing our built up entries (renderable tempcode), our sort options (a dropdown list of sort options), and our entries (entry records from database, with an additional 'map' field)
 */
function get_catalogue_category_entry_buildup($id,$catalogue_name,$catalogue,$view_type,$tpl_set,$max,$start,$select,$root,$display_type=NULL,$do_sorting=true,$entries=NULL,$filter_fields=NULL,$isCategoryListing=false)
{
	require_code('ecommerce');

	$is_ecomm=	is_ecommerce_catalogue($catalogue_name);
	
	if ((is_null($id)) && (is_null($entries))) fatal_exit(do_lang_tempcode('INTERNAL_ERROR'));

	if (is_null($display_type))
	{
		$display_type=get_param_integer('keep_cat_display_type',$catalogue['c_display_type']);
	}

	// Find order field
	$fields=$GLOBALS['SITE_DB']->query_select('catalogue_fields',array('*'),array('c_name'=>$catalogue_name),'ORDER BY cf_order');
	if ($do_sorting)
	{
		$_order_by=get_param('order','');
		if ($_order_by=='')
		{	
			$order_by=0;
			$direction='ASC';
			foreach ($fields as $i=>$field)
			{
				if ($field['cf_defines_order']!=0)
				{
					$order_by=$i;
					$direction=($field['cf_defines_order']==1)?'ASC':'DESC';
					$_order_by=strval($field['id']).' '.$direction;
					break;
				}
			}			
		} else
		{	
			list($order_by,$direction)=explode(' ',$_order_by);
			foreach ($fields as $i=>$field)
			{
				if ($order_by==$field['id'])
				{
					$order_by=$i;
					break;
				}
			}
		}
	} else $order_by=NULL;

	// Get entries in this category
	$map=array('cc_id'=>$id);

	if (!has_specific_permission(get_member(),'see_unvalidated')) $map['ce_validated']=1;
	if (is_null($entries)) $entries=$GLOBALS['SITE_DB']->query_select('catalogue_entries',array('*'),$map);
	foreach ($entries as $i=>$entry)
	{
		if(!is_null($filter_fields))
		{
			$field_values	=	get_catalogue_entry_field_values($catalogue_name,$entry,NULL,$fields);
			if(!found_match_values($field_values,$filter_fields))	
			{
				unset($entries[$i]);
				continue;
			}
		}

		$entries[$i]['map']=get_catalogue_entry_map($entry,$catalogue,$view_type,$tpl_set,$root,$fields,(($display_type==1) && (!$is_ecomm) && (!is_null($order_by)))?array(0,$order_by):NULL,false,false,$order_by,$isCategoryListing);

	}

	$_entries	=	array();
	foreach($entries as $entry)
		$_entries[]	=	$entry;
	
	$entries	=$_entries;

	$num_entries=count($entries);

	if ($do_sorting)
	{
		// Sort entries
		$sort_options=new ocp_tempcode();
		foreach ($fields as $field)
		{
			if ($field['cf_searchable']==1)
			{
				$potential_sorter_name=get_translated_text($field['cf_name']);
				$asc_sel=($_order_by==strval($field['id']).' ASC');
				$desc_sel=($_order_by==strval($field['id']).' DESC');
				$_potential_sorter_name=new ocp_tempcode();
				$_potential_sorter_name->attach(escape_html($potential_sorter_name));
				$_potential_sorter_name->attach(do_lang_tempcode('_ASCENDING'));
				$sort_options->attach(form_input_list_entry(strval($field['id']).' ASC',$asc_sel,protect_from_escaping($_potential_sorter_name)));
				$_potential_sorter_name=new ocp_tempcode();
				$_potential_sorter_name->attach(escape_html($potential_sorter_name));
				$_potential_sorter_name->attach(do_lang_tempcode('_DESCENDING'));
				$sort_options->attach(form_input_list_entry(strval($field['id']).' DESC',$desc_sel,protect_from_escaping($_potential_sorter_name)));
			}
		}
		for ($i=0;$i<$num_entries;$i++)
		{
			for ($j=$i+1;$j<$num_entries;$j++)
			{
				$a=@$entries[$j]['map']['FIELD_'.strval($order_by)];
				$b=@$entries[$i]['map']['FIELD_'.strval($order_by)];
				if (is_object($a)) $a=$a->evaluate();
				if (is_object($b)) $b=$b->evaluate();
	
				if (((strcasecmp($a,$b)<0) && ($direction=='ASC')) || ((strcasecmp($a,$b)>0) && ($direction=='DESC')))
				{
					$temp=$entries[$i];
					$entries[$i]=$entries[$j];
					$entries[$j]=$temp;
				}
			}
		}
	} else $sort_options=new ocp_tempcode();

	// Build up entries
	$entry_buildup=new ocp_tempcode();

	$extra_map=array();
	if($is_ecomm)
	{	
		require_lang('shopping');
		$i=0;
		for ($i=0;$i<$num_entries;$i++)
		{
			$entry=$entries[$i];
			$extra_map[$i]['ADD_TO_CART']=build_url(array('page'=>'shopping','type'=>'add_item','product_id'=>$entry['id'],'hook'=>'catalogue_items'),'_SELF');
		}
	}

	
	if ($display_type==2)
	{	
		for ($i=0;$i<$num_entries;$i++)
		{
			$entry=$entries[$i];			
			if (((is_null($select)) && ((is_null($start)) || ($i>=$start) && ($i<$start+$max))) || ((!is_null($select)) && (in_array($entry['id'],$select))))
			{
				$tab_entry_map=$entry['map']+(array_key_exists($i,$extra_map)?$extra_map[$i]:array());
				if (($entry['allow_comments']>=1) || ($entry['allow_rating']==1) || ($entry['allow_trackbacks']==1))
				{
					$tab_entry_map['VIEW_URL']=build_url(array('page'=>'catalogues','type'=>'entry','id'=>$entry['id'],'root'=>($root==-1)?NULL:$root),get_module_zone('catalogues'));
				} else
				{
					$tab_entry_map['VIEW_URL']='';
				}

				$entry_buildup->attach(do_template('CATALOGUE_'.$tpl_set.'_TAB_ENTRY',$tab_entry_map,NULL,false,'CATALOGUE_DEFAULT_TAB_ENTRY'));
			}
		}
		if (!$entry_buildup->is_empty())
		{
			$head=new ocp_tempcode();
			$field_count=0;
			foreach ($fields as $i=>$field)
			{
				if (((($field['cf_put_in_category']==1) && ($view_type=='CATEGORY')) || (($field['cf_put_in_search']==1) && ($view_type=='SEARCH'))) && ($field['cf_visible']==1))
				{
					$sort_url_asc=get_self_url(false,false,array('order'=>strval($field['id']).' ASC'),true);
					$sort_url_desc=get_self_url(false,false,array('order'=>strval($field['id']).' DESC'),true);
					$head->attach(do_template('CATALOGUE_'.$tpl_set.'_TAB_FIELD_HEAD',array('SORT_URL_ASC'=>$sort_url_asc,'SORT_URL_DESC'=>$sort_url_desc,'CATALOGUE'=>$catalogue_name,'FIELDID'=>strval($i),'_FIELDID'=>strval($field['id']),'FIELD'=>get_translated_text($field['cf_name']),'FIELDTYPE'=>$field['cf_type']),NULL,false,'CATALOGUE_DEFAULT_TAB_FIELD_HEAD'));
					$field_count++;
				}
			}
			$entry_buildup=do_template('CATALOGUE_'.$tpl_set.'_TAB_WRAP',array('CATALOGUE'=>$catalogue_name,'HEAD'=>$head,'CONTENT'=>$entry_buildup,'FIELD_COUNT'=>strval($field_count)),NULL,false,'CATALOGUE_DEFAULT_TAB_WRAP');
		}
	}
	elseif ($display_type==0)
	{
		require_code('tracking');

		for ($i=0;$i<$num_entries;$i++)
		{
			$entry=$entries[$i];

			if(is_tracked('catalogues',$id))
			{
				$track_url=build_url(array('page'=>'tracking','track_type'=>'catalogues','id'=>$entry['id'],'type'=>'untrack','redirect'=>get_self_url(true)),get_module_zone('tracking'));
				$img="untrack_resource";
			}
			else
			{
				$track_url=build_url(array('page'=>'tracking','track_type'=>'catalogues','id'=>$entry['id'],'type'=>'misc','redirect'=>get_self_url(true)),get_module_zone('tracking'));
				$img="track_resource";
			}

			// Admin functions
			if ((has_actual_page_access(NULL,'cms_catalogues',NULL,NULL)) && (has_catalogue_edit_permission($catalogue['c_owner'],get_member(),$entry['ce_submitter'],'cms_catalogues',array('catalogues_catalogue',$catalogue_name,'catalogues_category',$entry['cc_id']))))
			{
				$entry['map']['EDIT_URL']=build_url(array('page'=>'cms_catalogues','type'=>'_edit_entry','catalogue_name'=>$catalogue_name,'id'=>$entry['id']),get_module_zone('cms_catalogues'));
			} else 
			{
				$entry['map']['EDIT_URL']=new ocp_tempcode();				
			}

			if (has_specific_permission(get_member(),'use_alert_system'))
				$send_alert=build_url(array('page'=>'cms_catalogues','type'=>'send_alerts','catalogue_name'=>$catalogue_name,'id'=>$entry['id']),get_module_zone('cms_catalogues'));
			else
				$send_alert='';

         $private_message_url= build_url(array('page'=>'topics','type'=>'new_pt','id'=>$entry['ce_submitter']),get_module_zone('topics'));

			if (((is_null($select)) && ($i>=$start) && ($i<$start+$max)) || ((!is_null($select)) && (in_array($entry['id'],$select)))){
            $entry_buildup->attach(do_template('CATALOGUE_'.$tpl_set.'_ENTRY_EMBED',$entry['map']+(array_key_exists($i,$extra_map)?$extra_map[$i]:array())+array('TRACK_URL'=>$track_url,'TRACK_IMG'=>$img,'SEND_ALERT'=>$send_alert,'PRIVATE_MESSAGE_URL'=>$private_message_url),NULL,false,'CATALOGUE_DEFAULT_ENTRY_EMBED'));
         }
		}
	}	
	else
	{
		$display_first_count	=	0;
		for ($i=0;$i<$num_entries;$i++)
		{
			$entry=$entries[$i];
			$display_first_count++;

			$entry['map']['DIV_ID']='div_first_'.$display_first_count;
         $entry['map']['TABLE_ID']='table_first_'.$display_first_count;
         $entry['map']['IMAGE_TD_ID']='image_first_td_'.$display_first_count;
         $entry['map']['IMAGE_ID']='image_first_'.$display_first_count;

			$entry['map']['BREADCRUMBS']	=	new ocp_tempcode();//catalogue_category_breadcrumbs($entry['id'],NULL);

			if (((is_null($select)) && ($i>=$start) && ($i<$start+$max)) || ((!is_null($select)) && (in_array($entry['id'],$select))))
				$entry_buildup->attach(do_template('CATALOGUE_'.$tpl_set.'_LINE',$entry['map']+(array_key_exists($i,$extra_map)?$extra_map[$i]:array()),NULL,false,'CATALOGUE_DEFAULT_LINE'));
		}
		if (!$entry_buildup->is_empty()) $entry_buildup=do_template('CATALOGUE_'.$tpl_set.'_LINE_WRAP',array('CATALOGUE'=>$catalogue_name,'CONTENT'=>$entry_buildup),NULL,false,'CATALOGUE_DEFAULT_LINE_WRAP');
	}

	return array($entry_buildup,$sort_options,$entries);
}

function found_match_values($field_values,$filter_fields)
{
	foreach($field_values as $field_val)
	{
		$field_title	=	get_translated_text($field_val['cf_name']);
		if(array_key_exists($field_title,$filter_fields))
		{
			$filter_fld_val	=	post_param($filter_fields[$field_title],'');
			if($filter_fld_val!='')
			{
				$effective_val	=	$field_val['effective_value'];
				if($field_title=='Project location')	//Special case for location, because it contains state,city,zip values
				{
					$locations	=	explode(",",$effective_val);
					if(!in_array($filter_fld_val,$locations))	return false;
				}
				else
					if($field_val['effective_value']!=$filter_fld_val)	return false;
			}
		}
	}	
	return true;
}

/**
 * Get a map of the fields for the given entry.
 *
 * @param  array			A database row of the entry we are working with
 * @param  array			A database row of the catalogue we are working with
 * @param  ID_TEXT		The view type we're doing
 * @set    PAGE SEARCH CATEGORY
 * @param  ID_TEXT		The template set we are rendering this category using
 * @param  AUTO_LINK		The virtual root for display of this category
 * @param  ?array			The database rows for the fields for this catalogue (NULL: find them)
 * @param  ?array			A list of fields that we are limiting ourselves to (NULL: get ALL fields)
 * @param  boolean		Whether to grab the feedback details
 * @param  boolean		Whether to grab the tree details
 * @param  ?integer		Field index to order by (NULL: none)
 * @return array			A map of information relating to the entry. The map contains 'FIELDS' (tempcode for all accumulated fields), 'FIELD_x' (for each field x applying to the entry), STAFF_DETAILS, COMMENT_DETAILS, RATING_DETAILS, VIEW_URL, TREE (tempcode category tree to this entry)
 */
function get_catalogue_entry_map($entry,$catalogue,$view_type,$tpl_set,$root,$fields=NULL,$only_fields=NULL,$feedback_details=false,$tree_details=false,$order_by=NULL,$isCategoryListing=false)
{
	$id=$entry['id'];

	// Views
	if (get_db_type()!='xml')
	{
		$entry['ce_views']++;
	}

	$catalogue_name=$catalogue['c_name'];
	$fields=get_catalogue_entry_field_values($catalogue_name,$entry,$only_fields,$fields);
	$map=array();
	$map['FIELDS']=new ocp_tempcode();
	$map['FIELDS_TABLE']=new ocp_tempcode();
	$map['fields']=$fields;
	$fields_1d=array();
	$fields_2d=array();
	$all_visible=true;
   $isFilmFestival=false;
	foreach ($fields as $i=>$field)
	{
		if (!array_key_exists('effective_value',$field))
		{
			$all_visible=false;
			continue;
		}

      if($catalogue_name == 'events')
      {
         if($i==0 && is_string($field['effective_value']) && strpos($field['effective_value'],'Film Festival') !== FALSE)
         {
            $isFilmFestival=true;
         }

         if($isCategoryListing && !$isFilmFestival && ($i==12 || $i==13))
         {
            continue;
         }

         if($isCategoryListing && $i==14)
         {
            continue;
         }
      }

		$ev=$field['effective_value'];
		$dereference_ev=is_object($field['effective_value'])?$field['effective_value']->evaluate():$field['effective_value'];
		$use_ev=$dereference_ev;
		if (($field['cf_visible']==1) || ($i==0) || ($order_by===$i)) // If it's visible, or if it is the first (name) field, or if it is our order field
		{
			require_code('hooks/modules/catalogue_fields/'.filter_naughty($field['cf_type']));
			$ob=object_factory('Hook_catalogue_field_'.filter_naughty($field['cf_type']));
			$use_ev=$ob->render_field_value($ev,$catalogue_name,$i,$only_fields);
			if (($i==0) && ($catalogue['c_display_type']==1))
			{
				//$use_ev=hyperlink(build_url(array('page'=>'catalogues','type'=>'entry','id'=>$id,'root'=>($root==-1)?$root),get_module_zone('catalogues')),$ev,false,!is_object($ev));
				$use_ev=$ev;
			}

			//Set image thumbnail in to the map array
 			if($field['cf_type']=='picture')
 			{
				if((!is_null($ev)) && ($ev!='') && ((!url_is_local(is_object($ev)?$ev->evaluate():$ev)) || (file_exists(get_custom_file_base().'/'.rawurldecode(is_object($ev)?$ev->evaluate():$ev)))))
				{
					require_code('ecommerce');
					$currency	=	ecommerce_get_currency_symbol();
					$tool_tip	=	($catalogue['c_ecommerce'])?do_lang_tempcode('PRODUCT_TOOL_TIP',$map['FIELD_0'],$currency.$map['FIELD_2']):$map['FIELD_0'];

					$tool_tip	=	($catalogue['c_ecommerce'])?do_lang_tempcode('PRODUCT_TOOL_TIP',$map['FIELD_0'],$currency.$map['FIELD_2']):$map['FIELD_0'];
					
					//$tool_tip=$temp->evaluate();
					
					$map['FIELD_'.strval($i).'_THUMB']=do_image_thumb($ev,$tool_tip,array_key_exists('FIELD_0',$map)?$map['FIELD_0']:'',false,100,100);
				} else
				{
					$map['FIELD_'.strval($i).'_THUMB']=new ocp_tempcode();
				}
				$map['_FIELD_'.strval($field['id']).'_THUMB']=$map['FIELD_'.strval($i).'_THUMB'];
 			}

			$map['FIELD_'.strval($i)]=$use_ev;
			$map['_FIELD_'.strval($field['id'])]=$use_ev;
			$map['FIELD_'.strval($i).'_PLAIN']=$ev;
			$map['_FIELD_'.strval($field['id']).'_PLAIN']=$ev;
			$field_name=get_translated_text($field['cf_name']);
			$map['FIELDNAME_'.strval($i)]=$field_name;
			$fields_2d[]=array('NAME'=>$field_name,'VALUE'=>$use_ev);
			$field_type=$field['cf_type'];
			$map['FIELDTYPE_'.strval($i)]=$field_type;

			if (($view_type=='PAGE') || (($field['cf_put_in_category']==1) && ($view_type=='CATEGORY')) || (($field['cf_put_in_search']==1) && ($view_type=='SEARCH')))
			{
				$use_ev_enhanced=$use_ev;
				if (($view_type!='PAGE') && ($i==0) && (($field['cf_type']=='short_trans') || ($field['cf_type']=='short_text')))
				{
					foreach ($fields as $field_temp)
					{
						if ((array_key_exists('effective_value',$field_temp)) && ($field_temp['cf_type']=='url') && (!((($field_temp['cf_put_in_category']==1) && ($view_type=='CATEGORY')) || (($field_temp['cf_put_in_search']==1) && ($view_type=='SEARCH')))) && ($field_temp['cf_visible']==1))
						{
							$use_ev_enhanced=hyperlink($field_temp['effective_value'],$use_ev,true);
							break;
						}
					}
				}

				if (($field['cf_visible']==1) || ($i==0))
				{
					if($field_name == 'Quotes')
					{
						$qval = $use_ev_enhanced;
						$qval = str_replace(' ~~~ ','<br/><br/>',$qval);
						$qval = str_replace(' ||| ','<br/>&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;-',$qval);
					// 	exit($qval);
						$use_ev_enhanced = $qval;

						$map['FIELD_'.strval($i).'_ENHANCED']=$use_ev_enhanced;
					}
					else if ($field_name == 'Upload Picture')
					{
						$qval = $use_ev_enhanced;
						$pics = explode(' ||| ',$qval);
						$final = array();
						foreach($pics as $pic)
						{
							$final[] = "<a href='".get_base_url()."/$pic'><img src='".get_base_url()."/$pic' /></a>";
						}
						$use_ev_enhanced = implode("<br />",$final);

						$map['FIELD_'.strval($i).'_ENHANCED']=$use_ev_enhanced;
					}

					$_field=do_template('CATALOGUE_'.$tpl_set.'_ENTRY_FIELD',array('ENTRYID'=>strval($id),'CATALOGUE'=>$catalogue_name,'TYPE'=>$field['cf_type'],'FIELD'=>$field_name,'FIELDID'=>strval($i),'_FIELDID'=>strval($field['id']),'FIELDTYPE'=>$field_type,'VALUE_PLAIN'=>$ev,'VALUE'=>$use_ev_enhanced),NULL,false,'CATALOGUE_DEFAULT_ENTRY_FIELD');
					if ((!is_string($ev)) || ($ev!='')) $map['FIELDS']->attach($_field);
					$_field=do_template('CATALOGUE_'.$tpl_set.'_TAB_FIELD',array('ENTRYID'=>strval($id),'CATALOGUE'=>$catalogue_name,'TYPE'=>$field['cf_type'],'FIELD'=>$field_name,'FIELDID'=>strval($i),'_FIELDID'=>strval($field['id']),'FIELDTYPE'=>$field_type,'VALUE_PLAIN'=>$ev,'VALUE'=>$use_ev_enhanced),NULL,false,'CATALOGUE_DEFAULT_TAB_FIELD');
					$map['FIELDS_TABLE']->attach($_field);
				}
			}

			$fields_1d[]=$field;
		} else $all_visible=false;
	}

	$map['FIELDS_1D']=$fields_1d;
	$map['FIELDS_2D']=$fields_2d;

	// Admin functions
	if ((has_actual_page_access(NULL,'cms_catalogues',NULL,NULL)) && (has_catalogue_edit_permission($catalogue['c_owner'],get_member(),$entry['ce_submitter'],'cms_catalogues',array('catalogues_catalogue',$catalogue_name,'catalogues_category',$entry['cc_id']))))
	{
		$map['EDIT_URL']=build_url(array('page'=>'cms_catalogues','type'=>'_edit_entry','catalogue_name'=>$catalogue_name,'id'=>$id),get_module_zone('cms_catalogues'));
	} else $map['EDIT_URL']='';

	if($tpl_set == 'events')
	{
		if ((has_actual_page_access(NULL,'cms_catalogues',NULL,NULL)) && get_member()==$entry['ce_submitter'])
		{
			$map['EDIT_URL']=build_url(array('page'=>'cms_catalogues','type'=>'_edit_entry','catalogue_name'=>$catalogue_name,'id'=>$id),get_module_zone('cms_catalogues'));
		} else $map['EDIT_URL']=NULL;
	}

	$map['SUBMITTER']=strval($entry['ce_submitter']);
	$map['VIEWS']=strval($entry['ce_views']);
	$map['ADD_DATE_RAW']=strval($entry['ce_add_date']);
	$map['EDIT_DATE_RAW']=is_null($entry['ce_edit_date'])?'':strval($entry['ce_edit_date']);
	$map['ADD_DATE']=get_timezoned_date($entry['ce_add_date']);
	$map['EDIT_DATE']=get_timezoned_date($entry['ce_edit_date']);
	$map['ID']=strval($id);
	$submitterid = $entry['ce_submitter'];
	if ((get_option('is_on_comments')=='1') && (!has_no_forum()) && ($entry['allow_comments']>=1)) $map['COMMENT_COUNT']='1';

	// Feedback
	//Set basic rating display of item to the map array
	require_code('feedback');
	$map['RATING']=display_rating('catalogues',strval($id));
	if ($feedback_details)
	{
		require_code('feedback');
		do_rating($entry['allow_rating']==1,'catalogues',strval($id));
		$c_value=array_key_exists('FIELD_0_PLAIN_PURE',$map)?$map['FIELD_0_PLAIN_PURE']:$map['FIELD_0_PLAIN'];
		if (is_object($c_value)) $c_value=$c_value->evaluate();
		if ((!is_null(post_param('title',NULL))) || ($entry['ce_validated']==1))
			do_comments($entry['allow_comments']>=1,'catalogues',strval($id),build_url(array('page'=>'catalogues','type'=>'entry','id'=>$id),get_module_zone('catalogues')),$submitterid,$c_value,get_value('comment_forum__catalogues__'.$catalogue_name));
		//do_trackback($myrow['allow_trackbacks'],'catalogues',$id);
		$map['COMMENT_DETAILS']=get_comment_details('catalogues',$entry['allow_comments']==1,strval($id),false,get_value('comment_forum__catalogues__'.$catalogue_name),NULL,NULL,false,false,$entry['ce_submitter'],$entry['allow_comments']==2);
		$map['RATING_DETAILS']=get_rating_text('catalogues',strval($id),$entry['allow_rating']==1);
		$map['TRACKBACK_DETAILS']=get_trackback_details('catalogues',strval($id),$entry['allow_trackbacks']==1);
	}

	if (($entry['allow_comments']>=1) || ($entry['allow_rating']==1) || ($entry['allow_trackbacks']==1) || (!$all_visible))
	{
		$map['VIEW_URL']=($tpl_set=='WML')?make_string_tempcode('wap.php?page=catalogues&type=entry&id='.strval($id)):build_url(array('page'=>'catalogues','type'=>'entry','id'=>$id,'root'=>(is_null(get_param('root',NULL)))?NULL:$root),get_module_zone('catalogues'));
	} else
	{
		$map['VIEW_URL']='';
	}
	if ($tree_details)
	{
		$map['TREE']='';
		if (($catalogue['c_is_tree']==1) && (is_null($only_fields)))
		{
			$tree=catalogue_category_breadcrumbs($entry['cc_id'],$root,false);
			$map['TREE']=$tree;
		}
	}
	$map['CATALOGUE']=$catalogue_name;

	return $map;
}

/**
 * Get a nice, formatted, XHTML list of all the catalogues.
 *
 * @param  ?ID_TEXT		The name of the currently selected catalogue (NULL: none selected)
 * @param  boolean		Whether special permission will be required on the catalogue
 * @return tempcode		Catalogue selection list
 */
function nice_get_catalogues($it=NULL,$need_permission=false)
{
	$rows=$GLOBALS['SITE_DB']->query_select('catalogues',array('c_owner','c_name','c_title'));
	$out=new ocp_tempcode();
	foreach ($rows as $row)
	{
		if (!has_category_access(get_member(),'catalogues_catalogue',$row['c_name'])) continue;

		if (($need_permission) && (!is_null($row['c_owner'])) && (!has_specific_permission(get_member(),'can_submit_to_others_categories','cms_catalogues',array('catalogue',$row['c_name']))))
		{
			if ($row['c_owner']!=get_member()) continue;
		}

		$selected=($row['c_name']==$it);
		$out->attach(form_input_list_entry($row['c_name'],$selected,get_translated_text($row['c_title'])));
	}

	return $out;
}

/**
 * Get the values for the specified fields, for the stated catalogue entry.
 *
 * @param  ?ID_TEXT		The catalogue name we are getting an entry in (NULL: lookup)
 * @param  mixed			The ID of the entry we are getting OR the row
 * @param  ?array			A list of fields that we are limiting ourselves to (NULL: get ALL fields)
 * @param  ?array			The database rows for the fields for this catalogue (NULL: find them)
 * @return array			A list of maps (each field for the entry gets a map), where each map contains 'effective_value' (the value for the field). Some maps get additional fields (effective_value_nontrans, effective_value_pure), depending on the field type
 */
function get_catalogue_entry_field_values($catalogue_name,$entry_id,$only_fields=NULL,$fields=NULL)
{
	if (is_null($fields))
	{
		if (is_null($catalogue_name)) $catalogue_name=$GLOBALS['SITE_DB']->query_value('catalogue_entries','c_name',array('id'=>$entry_id));
		$fields=$GLOBALS['SITE_DB']->query_select('catalogue_fields',array('*'),array('c_name'=>$catalogue_name),'ORDER BY cf_order');
	}

	foreach ($fields as $i=>$field)
	{
		$field_id=$field['id'];

		if ((!is_null($only_fields)) && (!in_array($i,$only_fields))) continue;

		require_code('hooks/modules/catalogue_fields/'.filter_naughty($field['cf_type']));
		$ob=object_factory('Hook_catalogue_field_'.filter_naughty($field['cf_type']));
		list($raw_type,,)=$ob->get_field_value_row_bits($field['id']);

		if (is_null($raw_type)) $raw_type=$field['cf_type'];

		switch ($raw_type)
		{
			case 'short_trans':
				$fields[$i]['effective_value_nontrans']=_get_catalogue_entry_field($field_id,$entry_id,'short_trans');
				if (is_null($fields[$i]['effective_value_nontrans']))
				{
					$fields[$i]['effective_value']=do_lang_tempcode('INTERNAL_ERROR');
					$fields[$i]['effective_value_pure']=do_lang('INTERNAL_ERROR');
					break;
				}
				$fields[$i]['effective_value']=get_translated_tempcode($fields[$i]['effective_value_nontrans']);
				$fields[$i]['effective_value_pure']=get_translated_text($fields[$i]['effective_value_nontrans']);
				break;
			case 'long_trans':
				$fields[$i]['effective_value_nontrans']=_get_catalogue_entry_field($field_id,$entry_id,'long_trans');
				if (is_null($fields[$i]['effective_value_nontrans']))
				{
					$fields[$i]['effective_value']=do_lang_tempcode('INTERNAL_ERROR');
					$fields[$i]['effective_value_pure']=do_lang('INTERNAL_ERROR');
					break;
				}
				$fields[$i]['effective_value']=get_translated_tempcode($fields[$i]['effective_value_nontrans']);
				$fields[$i]['effective_value_pure']=get_translated_text($fields[$i]['effective_value_nontrans']);
				break;
			case 'long_text':
				$fields[$i]['effective_value_pure']=_get_catalogue_entry_field($field_id,$entry_id,'long');
				$fields[$i]['effective_value']=$fields[$i]['effective_value_pure'];
				if (is_null($fields[$i]['effective_value']))
				{
					$fields[$i]['effective_value']=do_lang_tempcode('INTERNAL_ERROR');
					$fields[$i]['effective_value_pure']=do_lang('INTERNAL_ERROR');
					break;
				}
				break;
			case 'short_text':
				$fields[$i]['effective_value_pure']=_get_catalogue_entry_field($field_id,$entry_id);
				$fields[$i]['effective_value']=$fields[$i]['effective_value_pure'];
				if (is_null($fields[$i]['effective_value']))
				{
					$fields[$i]['effective_value']=do_lang_tempcode('INTERNAL_ERROR');
					$fields[$i]['effective_value_pure']=do_lang('INTERNAL_ERROR');
					break;
				}
				break;
			case 'long_unescaped':
				$fields[$i]['effective_value']=_get_catalogue_entry_field($field_id,$entry_id,'long');
				if (is_null($fields[$i]['effective_value']))
				{
					$fields[$i]['effective_value']=do_lang_tempcode('INTERNAL_ERROR');
					$fields[$i]['effective_value_pure']=do_lang('INTERNAL_ERROR');
					break;
				}
				break;
			case 'short_unescaped':
				$fields[$i]['effective_value']=_get_catalogue_entry_field($field_id,$entry_id);
				if (is_null($fields[$i]['effective_value']))
				{
					$fields[$i]['effective_value']=do_lang_tempcode('INTERNAL_ERROR');
					$fields[$i]['effective_value_pure']=do_lang('INTERNAL_ERROR');
					break;
				}
				break;
			default:
				warn_exit(do_lang_tempcode('INTERNAL_ERROR'));
		}
	}

	return $fields;
}

/**
 * Get the value for the specified field, for the stated catalogue entry.
 *
 * @param  AUTO_LINK		The ID of the field we are getting
 * @param  mixed			The ID of the entry we are getting for OR the row
 * @param  ID_TEXT		The type of field
 * @set    short long
 * @return string			The value
 */
function _get_catalogue_entry_field($field_id,$entry_id,$type='short')
{
	if (is_array($entry_id)) $entry_id=$entry_id['id'];
	$value=$GLOBALS['SITE_DB']->query_value_null_ok('catalogue_efv_'.$type,'cv_value',array('cf_id'=>$field_id,'ce_id'=>$entry_id));
	return $value;
}

/**
 * Get a nice, formatted XHTML list of entries, in catalogue category tree structure
 *
 * @param  ID_TEXT		The catalogue name
 * @param  ?AUTO_LINK	The currently selected entry (NULL: none selected)
 * @param  ?AUTO_LINK	Only show entries submitted by this member (NULL: no filter)
 * @param  boolean		Whether to only show for what may be edited by the current member
 * @return tempcode		The list of entries
 */
function nice_get_catalogue_entries_tree($catalogue_name,$it=NULL,$submitter=NULL,$editable_filter=false)
{
	$tree=get_catalogue_entries_tree($catalogue_name,$submitter,NULL,NULL,NULL,NULL,$editable_filter);

	$out=''; // HACKHACK
	foreach ($tree as $category)
	{
		foreach ($category['entries'] as $eid=>$etitle)
		{
			$selected=($eid==$it);
			$line=do_template('CATALOGUE_ENTRIES_LIST_LINE',array('_GUID'=>'0ccffeff5b80b1840188b83aaee8d9f2','TREE'=>$category['tree'],'NAME'=>$etitle));
			$out.='<option value="'.strval($eid).'"'.($selected?'selected="selected"':'').'>'.$line->evaluate().'</option>';
		}
	}

	if ($GLOBALS['XSS_DETECT']) ocp_mark_as_escaped($out);

	return make_string_tempcode($out);
}

/**
 * Get a list of maps containing all the catalogue entries, and path information, under the specified category - and those beneath it, recursively.
 *
 * @param  ID_TEXT		The catalogue name
 * @param  ?AUTO_LINK	Only show entries submitted by this member (NULL: no filter)
 * @param  ?AUTO_LINK	The category being at the root of our recursion (NULL: true root)
 * @param  ?string		The tree up to this point in the recursion (NULL: blank, as we are starting the recursion)
 * @param  ?ID_TEXT		The name of the $category_id we are currently going through (NULL: look it up). This is here for efficiency reasons, as finding children IDs to recurse to also reveals the childs title
 * @param  ?integer		The number of recursive levels to search (NULL: all)
 * @param  boolean		Whether to only show for what may be edited by the current member
 * @param  ?array			Row of catalogue details (NULL: look up)
 * @return array			A list of maps for all categories. Each map entry containins the fields 'id' (category ID) and 'tree' (tree path to the category, including the categories own title), and more.
 */
function get_catalogue_entries_tree($catalogue_name,$submitter=NULL,$category_id=NULL,$tree=NULL,$title=NULL,$levels=NULL,$editable_filter=false,$catalogue_row=NULL)
{
	if (is_null($catalogue_row))
	{
		$catalogue_rows=$GLOBALS['SITE_DB']->query_select('catalogues',array('*'),array('c_name'=>$catalogue_name),'',1);
		if (!array_key_exists(0,$catalogue_rows)) return array();
		$catalogue_row=$catalogue_rows[0];
	}

	if (is_null($category_id))
	{
		$is_tree=$catalogue_row['c_is_tree'];
		if (is_null($is_tree)) return array();
		if ($is_tree==0)
		{
			$temp_rows=$GLOBALS['SITE_DB']->query_select('catalogue_categories',array('id','cc_title'),array('c_name'=>$catalogue_name,'cc_parent_id'=>NULL));
			$children=array();
			foreach ($temp_rows as $row)
			{
				$children=array_merge(get_catalogue_entries_tree($catalogue_name,$submitter,$row['id'],NULL,get_translated_text($row['cc_title']),1,$editable_filter,$catalogue_row),$children);
			}
			return $children;
		}

		$temp_rows=$GLOBALS['SITE_DB']->query_select('catalogue_categories',array('id','cc_title'),array('c_name'=>$catalogue_name),'ORDER BY id',1);
		if (!array_key_exists(0,$temp_rows)) return array();
		$category_id=$temp_rows[0]['id'];
		$title=get_translated_text($temp_rows[0]['cc_title']);
	}
	if (is_null($tree)) $tree='';

	if (!has_category_access(get_member(),'catalogues_catalogue',$catalogue_name)) return array();
	if (!has_category_access(get_member(),'catalogues_category',strval($category_id))) return array();

	// Put our title onto our tree
	if (is_null($title)) $title=get_translated_text($GLOBALS['SITE_DB']->query_value('catalogue_categories','cc_title',array('id'=>$category_id)));
	$tree.=$title;

	// We'll be putting all children in this entire tree into a single list
	$children=array();
	$children[0]=array();
	$children[0]['id']=$category_id;
	$children[0]['title']=$title;
	$children[0]['tree']=$tree;

	// Children of this category
	$rows=$GLOBALS['SITE_DB']->query_select('catalogue_categories',array('id','cc_title'),array('cc_parent_id'=>$category_id));
	$where=array('cc_id'=>$category_id);
	if (!is_null($submitter)) $where['ce_submitter']=$submitter;
	$erows=$GLOBALS['SITE_DB']->query_select('catalogue_entries',array('id','ce_submitter'),$where);
	$children[0]['entries']=array();
	foreach ($erows as $row)
	{
		if (($editable_filter) && (!has_catalogue_edit_permission($catalogue_row['c_owner'],get_member(),$row['ce_submitter'],'cms_catalogues',array('catalogues_catalogue',$catalogue_name,'catalogues_category',$category_id)))) continue;

		$entry_fields=get_catalogue_entry_field_values($catalogue_name,$row['id'],array(0));
		$name=$entry_fields[0]['effective_value']; // 'Name' is value of first field

		$children[0]['entries'][$row['id']]=$name;
	}
	$children[0]['child_entry_count']=count($children[0]['entries']);
	if ($levels===0) // We throw them away now because they're not on the desired level
	{
		$children[0]['entries']=array();
	}
	$children[0]['child_count']=count($rows);
	$tree.=' > ';
	if ($levels!==0)
	{
		foreach ($rows as $child)
		{
			$child_id=$child['id'];
			$child_title=get_translated_text($child['cc_title']);
			$child_tree=$tree;

			$child_children=get_catalogue_entries_tree($catalogue_name,$submitter,$child_id,$child_tree,$child_title,is_null($levels)?NULL:max(0,$levels-1),$editable_filter,$catalogue_row);

			$children=array_merge($children,$child_children);
		}
	}

	return $children;
}

/**
 * Get a nice, formatted XHTML list extending from the root, and showing all subcategories, and their subcategories (ad infinitum). The tree bit is because each entry in the list is shown to include the path through the tree that gets to it
 *
 * @param  ID_TEXT		The catalogue name
 * @param  ?AUTO_LINK	The currently selected entry (NULL: none)
 * @param  boolean		Whether to only show for what may be added to by the current member
 * @param  boolean		Whether to make the list elements store comma separated child lists instead of IDs
 * @return tempcode		The list of categories
 */
function nice_get_catalogue_category_tree($catalogue_name,$it=NULL,$addable_filter=false,$use_compound_list=false)
{
	$tree=array();
	$temp_rows=$GLOBALS['SITE_DB']->query('SELECT id,cc_title FROM '.get_table_prefix().'catalogue_categories WHERE c_name=\''.db_escape_string($catalogue_name).'\' AND cc_parent_id IS NULL');
	foreach ($temp_rows as $row)
	{
		$category_id=$row['id'];
		$title=get_translated_text($row['cc_title']);
		$subtree=get_catalogue_category_tree($catalogue_name,$category_id,NULL,$title,NULL,$addable_filter,$use_compound_list);
		if (($use_compound_list) && (array_key_exists(0,$subtree))) $subtree=$subtree[0];
		$tree=array_merge($tree,$subtree);
	}

	$out=new ocp_tempcode();
	foreach ($tree as $category)
	{
		if (($addable_filter) && (!$category['addable'])) continue;

		$selected=($category['id']==$it);
		$line=do_template('CATALOGUE_CATEGORIES_LIST_LINE',array('_GUID'=>'9f6bfc4f28c154c8f5d8887ce0d47c1c','TREE'=>$category['tree'],'COUNT'=>number_format($category['count'])));
		$out->attach(form_input_list_entry(!$use_compound_list?strval($category['id']):$category['compound_list'],$selected,$line->evaluate()));
	}

	return $out;
}

/**
 * Get a list of maps containing all the subcategories, and path information, of the specified category - and those beneath it, recursively.
 *
 * @param  ID_TEXT		The catalogue name
 * @param  ?AUTO_LINK	The category being at the root of our recursion (NULL: true root category)
 * @param  ?tempcode		The tree up to this point in the recursion (NULL: blank, as we are starting the recursion)
 * @param  ?string		The category name of the $category_id we are currently going through (NULL: look it up). This is here for efficiency reasons, as finding children IDs to recurse to also reveals the childs title
 * @param  ?integer		The number of recursive levels to search (NULL: all)
 * @param  boolean		Whether to only show for what may be added to by the current member
 * @param  boolean		Whether to make the list elements store comma separated child lists instead of IDs
 * @return array			A list of maps for all subcategories. Each map entry containins the fields 'id' (category ID) and 'tree' (tree path to the category, including the categories own title), and 'count' (the number of entries in the category).
 */
function get_catalogue_category_tree($catalogue_name,$category_id,$tree=NULL,$title=NULL,$levels=NULL,$addable_filter=false,$use_compound_list=false)
{
	if ($levels==-1) return array();

	if (!has_category_access(get_member(),'catalogues_catalogue',$catalogue_name)) return array();
	if ((!is_null($category_id)) && (!has_category_access(get_member(),'catalogues_category',strval($category_id)))) return array();

	if (is_null($tree)) $tree=new ocp_tempcode();

	// Put our title onto our tree
	if (is_null($title))
	{
		if (is_null($category_id))
		{
			$_title=$GLOBALS['SITE_DB']->query_value_null_ok('catalogue_categories','cc_title',array('id'=>$category_id));
		} else
		{
			$_title=$GLOBALS['SITE_DB']->query_value('catalogue_categories','cc_title',array('id'=>$category_id));
		}
		$title=is_null($_title)?do_lang('HOME'):get_translated_text($_title);
	}
	$tree->attach($title);

	// We'll be putting all children in this entire tree into a single list
	$children=array();
	$catalogue_rows=$GLOBALS['SITE_DB']->query_select('catalogues',array('*'),array('c_name'=>$catalogue_name),'',1);
	if (!array_key_exists(0,$catalogue_rows)) warn_exit(do_lang_tempcode('_MISSING_RESOURCE','catalogue:'.escape_html($catalogue_name)));
	$catalogue_row=$catalogue_rows[0];
	$is_tree=$catalogue_row['c_is_tree'];
	if ((!is_null($category_id))/* || ($is_tree==1)*/)
	{
		$children[0]['id']=$category_id;
		$children[0]['title']=$title;
		$children[0]['tree']=$tree;
		$children[0]['compound_list']=strval($category_id).',';
		$children[0]['count']=$GLOBALS['SITE_DB']->query_value('catalogue_entries','COUNT(*)',array('cc_id'=>$category_id));
		if ($addable_filter) $children[0]['addable']=has_catalogue_submit_permission($catalogue_row['c_owner'],get_member(),get_ip_address(),'cms_catalogues',array('catalogues_catalogue',$catalogue_name,'catalogues_category',$category_id));
	}

	// Children of this category
	$tree2=new ocp_tempcode();
	$tree2->attach($tree);
	$tree2->attach(do_template('BREADCRUMB'));
	$rows=$GLOBALS['SITE_DB']->query_select('catalogue_categories c LEFT JOIN '.$GLOBALS['SITE_DB']->get_table_prefix().'translate t ON '.db_string_equal_to('language',user_lang()).' AND t.id=c.cc_title',array('c.id','cc_title','text_original'),array('c_name'=>$catalogue_name,'cc_parent_id'=>$category_id),' ORDER BY text_original');
	$no_root=!array_key_exists(0,$children);
	if (!$no_root) $children[0]['child_count']=count($rows);
	if ($levels!==0)
	{
		foreach ($rows as $child)
		{
			if (is_null($child['text_original'])) $child['text_original']=get_translated_text($child['cc_title']);

			$child_id=$child['id'];
			$child_title=$child['text_original'];
			$child_tree=new ocp_tempcode();
			$child_tree->attach($tree2);
			//MANU:shell_exec("echo Catalogue_name=".$catalogue_name." >> /home/manu/debug.txt");
			$child_children=get_catalogue_category_tree($catalogue_name,$child_id,$child_tree,$child_title,is_null($levels)?NULL:max(0,$levels-1),$addable_filter,$use_compound_list);

			if ($use_compound_list)
			{
				list($child_children,$_compound_list)=$child_children;
				if (!$no_root) $children[0]['compound_list'].=$_compound_list;
			}

			$children=array_merge($children,$child_children);
		}
	}

	return $use_compound_list?array($children,$no_root?'':$children[0]['compound_list']):$children;
}

/**
 * Get a formatted XHTML string of the route back to the specified root, from the specified category.
 *
 * @param  AUTO_LINK		The category we are finding for
 * @param  ?AUTO_LINK	The root of the tree (NULL: the true root)
 * @param  boolean		Whether to include category links at this level (the recursed levels will always contain links - the top level is optional, hence this parameter)
 * @return tempcode		The tree route
 */
function catalogue_category_breadcrumbs($category_id,$root=NULL,$no_link_for_me_sir=true,$js_onclick=NULL)
{
	$map=array('page'=>'catalogues','type'=>'category','id'=>$category_id);
	if (!is_null($root)) $map['root']=$root;
   $url=build_url($map,get_module_zone('catalogues'));

	$onclick	=	(!is_null($js_onclick))?"return $js_onclick($category_id)":NULL;

   if (is_null($category_id)) return new ocp_tempcode();

   if (($category_id!=$root) || (!$no_link_for_me_sir))
   {
	   global $PT_PAIR_CACHE;
	   if (!array_key_exists($category_id,$PT_PAIR_CACHE))
	   {
	      $category_rows=$GLOBALS['SITE_DB']->query_select('catalogue_categories',array('cc_parent_id','cc_title'),array('id'=>$category_id),'',1);
	      if (!array_key_exists(0,$category_rows)) fatal_exit(do_lang_tempcode('CAT_NOT_FOUND',escape_html(strval($category_id))));
	      $PT_PAIR_CACHE[$category_id]=$category_rows[0];
	   }

	   if ($PT_PAIR_CACHE[$category_id]['cc_parent_id']==$category_id) fatal_exit(do_lang_tempcode('RECURSIVE_TREE_CHAIN',escape_html(strval($category_id))));
	}

   if ($category_id==$root)
   {
		$below=new ocp_tempcode();
	} else
	{
	   $below=catalogue_category_breadcrumbs($PT_PAIR_CACHE[$category_id]['cc_parent_id'],$root,false,$js_onclick);
	}

   if (!$no_link_for_me_sir)
   {
	   $title=get_translated_text($PT_PAIR_CACHE[$category_id]['cc_title']);
      if (!$below->is_empty()) $tpl_url=do_template('BREADCRUMB_ESCAPED'); else $tpl_url=new ocp_tempcode();
      $tpl_url->attach(hyperlink($url,escape_html($title),false,false,do_lang_tempcode('GO_BACKWARDS_TO'),NULL,NULL,'up',$onclick));
   } else $tpl_url=new ocp_tempcode();

   $below->attach($tpl_url);
   return $below;
}

/**
 * Check the current catalogue is an ecommerce catalogue
 * @param  SHORT_TEXT		Catalogue name
 * @return BINARY				Status of ecommerce catalogue check
*/
function is_ecommerce_catalogue($catalogue_name)
{
	if(is_null($GLOBALS['SITE_DB']->query_value_null_ok('catalogues','c_name',array('c_name'=>$catalogue_name,'c_ecommerce'=>1))))
		return false;
	else
		return true;
}
/**
 * Check selected entry is an ecommerce catalogue entry
 *
 * @param  AUTO_LINK			Entry ID
 * @return BINARY				Status of entry type check
*/
function is_ecommerce_catalogue_entry($entry_id)
{
	$catalogue_name	=	$GLOBALS['SITE_DB']->query_value('catalogue_entries','c_name',array('id'=>$entry_id));

	return is_ecommerce_catalogue($catalogue_name);
}

/**
 * Display a catalogue entry
 *
 *	@param  AUTO_LINK		Entry Id
 * @return tempcode		Tempcode interface to display an entry
 */
function render_catalogue_entry_screen($entry_id,$isCatalogueListing=false)
{
	require_code('feedback');
	require_code('ecommerce');
	require_code('images');
	require_code('shopping');
	require_css('catalogues');
	require_lang('catalogues');
	require_lang('ecommerce');

	log_cart_actions('View item');

	$entries=$GLOBALS['SITE_DB']->query_select('catalogue_entries',array('*'),array('id'=>$entry_id),'',1);
	if (!array_key_exists(0,$entries))
	{
		return warn_screen(get_page_title('CATALOGUES'),do_lang_tempcode('MISSING_RESOURCE'));
	}
	$entry=$entries[0];

	$categories=$GLOBALS['SITE_DB']->query_select('catalogue_categories',array('*'),array('id'=>$entry['cc_id']),'',1);
	if (!array_key_exists(0,$categories)) warn_exit(do_lang_tempcode('CAT_NOT_FOUND',strval($entry['cc_id'])));
	$category=$categories[0];
	$GLOBALS['FEED_URL']=find_script('backend').'?mode=catalogues&filter='.strval($entry['cc_id']);

	$catalogue_name=$category['c_name'];
	$catalogues=$GLOBALS['SITE_DB']->query_select('catalogues',array('*'),array('c_name'=>$catalogue_name),'',1);
	if (!array_key_exists(0,$catalogues)) warn_exit(do_lang_tempcode('CATALOGUE_NOT_FOUND',$catalogue_name));
	$catalogue=$catalogues[0];

	// Permission for here?
	if (!has_category_access(get_member(),'catalogues_catalogue',$catalogue_name))
	{
		access_denied('CATALOGUE_ACCESS');
	}
	if (!has_category_access(get_member(),'catalogues_category',strval($entry['cc_id'])))
	{
		access_denied('CATEGORY_ACCESS');
	}

	$tpl_set		=	(is_ecommerce_catalogue($catalogue_name))?'products':$catalogue_name;

	$root=get_param_integer('root',$GLOBALS['SITE_DB']->query_value('catalogue_categories','MIN(id)',array('c_name'=>$catalogue_name)),true);

	$map=get_catalogue_entry_map($entry,$catalogue,'PAGE',$tpl_set,$root,NULL,NULL,true,true,NULL,$isCatalogueListing);
	if (get_db_type()!='xml')
	{
		$entry['ce_views']++;
		$GLOBALS['SITE_DB']->query_update('catalogue_entries',array('ce_views'=>$entry['ce_views']),array('id'=>$entry_id),'',1);
	}

	// Validation
	if ($entry['ce_validated']==0)
	{
		if (!has_specific_permission(get_member(),'jump_to_unvalidated'))
			access_denied('SPECIFIC_PERMISSION','jump_to_unvalidated');

		$map['WARNINGS']=do_template('WARNING_TABLE',array('_GUID'=>'bf604859a572ca53e969bec3d91f9cfb','WARNING'=>do_lang_tempcode('UNVALIDATED_TEXT')));
	} else $map['WARNINGS']='';

	$product	=	do_lang('PRODUCT_CATALOGUE_ITEMS',$entry_id);
	$object	=	find_product($product);
	if (is_object($object) && method_exists($object,'get_custom_product_map_fields'))
	{
		$object->get_custom_product_map_fields($entry_id,$map);
	}

   if($catalogue_name == 'events')
   {
      $event_quote=array();
      $quotes=explode('~~~',$map['FIELD_14']);
      $quote_num=1;
      $quote_tpl = new ocp_tempcode();
      foreach($quotes as $quote)
      {
         if($quote)
         {
            $quote_detail=explode('|||',$quote);
            $quote_tpl->attach(do_template('CATALOGUE_'.$tpl_set.'_QUOTE',array('IS_FIRST_QUOTE'=>($quote_num==1)?true:NULL,'QUOTE_NUM'=>strval($quote_num),'QUOTE'=>$quote_detail[0],'ATTRIBUTION'=>$quote_detail[1])));
         }
         $quote_num++;
      }
   } else {
      $quote_tpl=NULL;
   }

   $map['QUOTE_DISPLAY']=$quote_tpl;
   $map['SHOW_FILM_FIELDS']=($catalogue_name == 'events' && strpos($map['FIELD_0'],'Film Festival') !== FALSE) ? true : false;
	$map['ENTRY']=do_template('CATALOGUE_'.$tpl_set.'_ENTRY',$map,NULL,false,'CATALOGUE_DEFAULT_ENTRY');
	$map['ADD_DATE']=get_timezoned_date($entry['ce_add_date']);
	$map['ADD_DATE_RAW']=strval($entry['ce_add_date']);
	$map['EDIT_DATE']=is_null($entry['ce_edit_date'])?'':get_timezoned_date($entry['ce_edit_date']);
	$map['EDIT_DATE_RAW']=is_null($entry['ce_edit_date'])?'':strval($entry['ce_edit_date']);
	$map['VIEWS']=number_format($entry['ce_views']);
	$title_to_use=do_lang_tempcode($catalogue_name.'__CATALOGUE_ENTRY',$map['FIELD_0']);
	$title_to_use_2=do_lang($catalogue_name.'__CATALOGUE_ENTRY',$map['FIELD_0_PLAIN'],NULL,NULL,NULL,false);
	if (is_null($title_to_use_2))
	{
		$title_to_use=do_lang_tempcode('DEFAULT__CATALOGUE_ENTRY',$map['FIELD_0']);
		$title_to_use_2=do_lang('DEFAULT__CATALOGUE_ENTRY',$map['FIELD_0_PLAIN']);
	}
	$map['TITLE']=get_page_title($title_to_use,false);
	$map['SUBMITTER']=strval($entry['ce_submitter']);

	require_code('seo2');
	if (is_object($title_to_use_2)) $title_to_use_2=$title_to_use_2->evaluate();
	seo_meta_load_for('catalogue_entry',strval($entry_id),strip_tags($title_to_use_2));

	if ($map['TREE']==='')
	{
		$map['TREE']=new ocp_tempcode();
		$url=build_url(array('page'=>'_SELF','type'=>'index','id'=>$catalogue_name),'_SELF');
		$map['TREE']->attach(hyperlink($url,escape_html(get_translated_text($catalogue['c_title'])),false,false,do_lang('INDEX')));
		$map['TREE']->attach(do_template('BREADCRUMB_ESCAPED'));
		$url=build_url(array('page'=>'_SELF','type'=>'category','id'=>$category['id']),'_SELF');
		$map['TREE']->attach(hyperlink($url,escape_html(get_translated_text($category['cc_title'])),false,false,do_lang('GO_BACKWARDS_TO',get_translated_text($category['cc_title'])),NULL,NULL,'up'));
	}

	$map['TAGS']=get_loaded_tags('catalogue_entries');

	breadcrumb_add_segment($map['TREE'],$title_to_use);
	if (is_null(get_param('root',NULL))) breadcrumb_set_parents(array(array('_SELF:_SELF:misc'.(is_ecommerce_catalogue($catalogue_name)?':ecommerce=1':''),do_lang('CATALOGUES'))));

	$GLOBALS['META_DATA']+=array(
		'created'=>date('Y-m-d',$entry['ce_add_date']),
		'creator'=>$GLOBALS['FORUM_DRIVER']->get_username($entry['ce_submitter']),
		'publisher'=>'', // blank means same as creator
		'modified'=>is_null($entry['ce_edit_date'])?'':date('Y-m-d',$entry['ce_edit_date']),
		'type'=>get_translated_text($catalogue['c_title']).' entry',
		'title'=>$title_to_use_2,
		'identifier'=>'_SEARCH:catalogues:entry:'.strval($entry_id),
		'description'=>'',
	);

	return do_template('CATALOGUE_'.$tpl_set.'_ENTRY_SCREEN',$map,NULL,false,'CATALOGUE_DEFAULT_ENTRY_SCREEN');
}

/**
 * See if a member has permissions to operate in a specific catalogue, based on an ownership check
 *
 * @param  mixed			The name of the catalogue for operating in OR the owner ID
 * @param  ?MEMBER		The member being checked whether to have the access (NULL: current member)
 * @return boolean		Whether the member can
 */
function _catalogue_ownership_check($catalogue_name,$member)
{
	if (is_null($catalogue_name)) return true;

	if (is_null($member)) $member=get_member();

	if (is_integer($catalogue_name))
	{
		$owner=$catalogue_name;
		$catalogue_name='store_'.strval($owner);
	} else
	{
		$catalogue_rows=$GLOBALS['SITE_DB']->query_select('catalogues',array('*'),array('c_name'=>$catalogue_name),'',1);
		if (!array_key_exists(0,$catalogue_rows)) warn_exit(do_lang_tempcode('_MISSING_RESOURCE','catalogue:'.escape_html($catalogue_name)));
		$catalogue_row=$catalogue_rows[0];
		$owner=$catalogue_row['c_owner'];
	}
	if ((!is_null($owner)) && ($owner!=$member))
	{
		if (!has_specific_permission(get_member(),'can_submit_to_others_categories','cms_catalogues',array('catalogue',$catalogue_name)))
			return false;
	}
	return true;
}

/**
 * Find if a member's has access to a specified page. Zone permissions are taken into account for wherever the page is found at. Also support for category access and specific permissions. No support for entry-point checks, which are only carried out as an extension of page permissions when actually at a page.
 *
 * @param  mixed			The name of the catalogue for operating in OR the owner ID
 * @param  ?MEMBER		The member being checked whether to have the access (NULL: current member)
 * @param  ?ID_TEXT		The ID code for the page being checked (NULL: current page)
 * @param  ?ID_TEXT		The ID code for the zone being checked (NULL: search)
 * @param  ?array			A list of cat details to require access to (c-type-1,c-id-1,c-type-2,c-d-2,...) (NULL: N/A)
 * @param  ?mixed			Either the ID code of a specific-permission, an array of alternatives that are acceptable (NULL: none required)
 * @return boolean		Whether the member has zone and page access
 */
function has_actual_catalogue_page_access($catalogue_name,$member=NULL,$page=NULL,$zone=NULL,$cats=NULL,$sp=NULL)
{
	if (!_catalogue_ownership_check($catalogue_name,$member)) return false;

	return has_actual_page_access($member,$page,$zone,$cats,$sp);
}

/**
 * Find if a member's has permission to submit, into a specific catalogue
 *
 * @param  mixed			The name of the catalogue for operating in OR the owner ID
 * @param  MEMBER			The member being checked whether to have the access
 * @param  IP				The member's IP address
 * @param  ?ID_TEXT		The ID code for the page being checked (NULL: current page)
 * @param  ?mixed			Either the ID code of a specific-permission, an array of alternatives that are acceptable (NULL: none required)
 * @return boolean		Whether the member can submit in this range
 */
function has_catalogue_submit_permission($catalogue_name,$member,$ip,$page,$cats=NULL)
{
	$range='mid';

	if (!_catalogue_ownership_check($catalogue_name,$member)) return false;

	return has_submit_permission($range,$member,$ip,$page,$cats);
}

/**
 * Find if a member's has permission to edit, into a specific catalogue
 *
 * @param  mixed			The name of the catalogue for operating in OR the owner ID
 * @param  MEMBER			The member being checked for access
 * @param  ?MEMBER		The member that owns this resource (NULL: no-one)
 * @param  ?ID_TEXT		The ID code for the page being checked (NULL: current page)
 * @param  ?mixed			Either the ID code of a specific-permission, an array of alternatives that are acceptable (NULL: none required)
 * @return boolean		Whether the member may edit the resource
 */
function has_catalogue_edit_permission($catalogue_name,$member,$resource_owner,$page,$cats=NULL)
{
	$range='mid';

	if (!_catalogue_ownership_check($catalogue_name,$member)) return false;

	return has_edit_permission($range,$member,$resource_owner,$page,$cats);
}

/**
* Functions to return a thumbnail image of catalogue item
*
*	@param	ID_TEXT		Title
* 	@return  ID_TEXT		Image path
*/
function get_item_image_thumb($title,$image_path=NULL)
{
	require_code('images');
	if(is_null($image_path) || $image_path=='')
		return do_image_thumb('themes/default/images/no_image.png',do_lang('NO_IMAGE'),do_lang('NO_IMAGE'),false,50,50);
	else
		return do_image_thumb(get_custom_base_url().'/'.$image_path,$title,$title,false,50,50);
}

/**
* Function to return all entries of a catalogue
*
*	@param	SHORT_TEXT	Catalogue name
* 	@return  ARRAY
*/
function get_catalogue_entries($catalogue_name)
{
	$entries_rows = $GLOBALS['SITE_DB']->query_select('catalogue_entries',array('id'),array('c_name'=>$catalogue_name));

	return $entries_rows;
}

/**
 * Render catalogue category screen
 *
 *	@param	ID_TEXT		Category id
 *	@param	BINARY		Title needs or not
 *	@param	BINARY		View in profile screen or not
 *	@result	tempcode		Tempcode of rendered catalogue
 */
function render_catalogue_category_screen($category_id=NULL,$no_title=false,$view_in_profile=false)
{
	require_code('feedback');
	require_code('images');
	require_code('tracking');
	require_lang('tracking');
	require_css('catalogues');
	require_code('banners');

	$id=is_null($category_id)?get_param_integer('id',$category_id):$category_id;
	if ($id==-1)
	{
		$id=$GLOBALS['SITE_DB']->query_value('catalogue_categories','MIN(id)',array('c_name'=>get_param('catalogue_name')));
	}
	$GLOBALS['FEED_URL']=find_script('backend').'?mode=catalogues&filter='.strval($id);

	$categories=$GLOBALS['SITE_DB']->query_select('catalogue_categories',array('*'),array('id'=>$id));
	if (!array_key_exists(0,$categories))
	{
		return warn_screen(get_page_title('CATALOGUES'),do_lang_tempcode('MISSING_RESOURCE'));
	}
	$category=$categories[0];

	// Permission for here?
	if (!has_category_access(get_member(),'catalogues_catalogue',$category['c_name']))
	{
		access_denied('CATALOGUE_ACCESS');
	}
	if (!has_category_access(get_member(),'catalogues_category',strval($id)))
	{
		access_denied('CATEGORY_ACCESS');
	}

	$catalogue_name=$category['c_name'];
	$min=$GLOBALS['SITE_DB']->query_value('catalogue_categories','MIN(id)',array('c_name'=>$catalogue_name));
	$root=get_param_integer('root',$min,true);

	$_title=get_translated_text($category['cc_title']);
	$catalogues=$GLOBALS['SITE_DB']->query_select('catalogues',array('*'),array('c_name'=>$catalogue_name));
	if (!array_key_exists(0,$catalogues)) warn_exit(do_lang_tempcode('CATALOGUE_NOT_FOUND',$catalogue_name));
	
	$catalogue=$catalogues[0];

	if(is_ecommerce_catalogue($catalogue_name))
	{
		$is_ecommerce=true;
		$tpl_set='products';
	}
	else
	{
		$tpl_set=$catalogue_name;
		$is_ecommerce=false;
	}


	$title_to_use=do_lang_tempcode($catalogue_name.'__CATALOGUE_CATEGORY',escape_html($_title));
	$title_to_use_2=do_lang($catalogue_name.'__CATALOGUE_CATEGORY',$_title,NULL,NULL,NULL,false);
	if (is_null($title_to_use_2))
	{
		$title_to_use=do_lang_tempcode('DEFAULT__CATALOGUE_CATEGORY',escape_html($_title));
		$title_to_use_2=do_lang('DEFAULT__CATALOGUE_CATEGORY',$_title);
	}
	$title=($no_title)?new ocp_tempcode():get_page_title($title_to_use,false);
	
	require_code('seo2');
	if (is_object($title_to_use_2) && !$no_title)
	{
		$title_to_use_2=$title_to_use_2->evaluate();
		seo_meta_load_for('catalogue_category',strval($id),$title_to_use_2);
	}
	
	$max=get_param_integer('max',30);
	$start=get_param_integer('start',0);

	list($entry_buildup,$sort_options,$entries)=get_catalogue_category_entry_buildup($id,$catalogue_name,$catalogue,'CATEGORY',$tpl_set,$max,$start,NULL,$root);

	// Build up subcategories
	$rows_subcategories=$GLOBALS['SITE_DB']->query_select('catalogue_categories',array('*'),array('cc_parent_id'=>$id));
	foreach ($rows_subcategories as $i=>$subcategory) // Dereference language
	{
		$rows_subcategories[$i]['cc_title']=get_translated_text($subcategory['cc_title']);
	}
	for ($i=0;$i<count($rows_subcategories);$i++) // Sort
	{
		for ($j=$i+1;$j<count($rows_subcategories);$j++)
		{
			if ($rows_subcategories[$i]['cc_title']>$rows_subcategories[$j]['cc_title'])
			{
				$temp=$rows_subcategories[$j];
				$rows_subcategories[$j]=$rows_subcategories[$i];
				$rows_subcategories[$i]=$temp;
			}
		}
	}
	$subcategories=new ocp_tempcode();
	$uses_rep_image=false;
	foreach ($rows_subcategories as $sc)
	{
		if ($sc['rep_image']!='') $uses_rep_image=true;
	}
	foreach ($rows_subcategories as $sc)
	{
		$child_counts=count_catalogue_category_children($sc['id']);
		$num_children=$child_counts['num_children'];
		$num_entries=$child_counts['num_entries_children'];
		$map=array('page'=>'_SELF','id'=>$sc['id'],'type'=>'category');
		if (!is_null(get_param('root',NULL))) $map['root']=$root;
		$url=build_url($map,'_SELF');
		if ($uses_rep_image || $is_ecommerce)
		{
			if ($num_entries!=0)
			{
				if ($num_children>0)
				{
					$display_string=do_lang_tempcode('CATEGORY_SUBORDINATE',number_format($num_entries),number_format($num_children));
				} else
				{
					$display_string=do_lang_tempcode('CATEGORY_SUBORDINATE_2',number_format($num_entries));
				}
			} else
			{
				if ($num_children>0)
				{
					$display_string=do_lang_tempcode('CATEGORY_SUBORDINATE_NO_ENTRIES',number_format($num_children));
				} else
				{
					$display_string=do_lang_tempcode('CATEGORY_SUBORDINATE_NO_ENTRIES_OR_SUBCATEGORIES');
				}
			}
			if($sc['rep_image'])
				$rep_image=do_image_thumb($sc['rep_image'],$sc['cc_title'],$sc['cc_title'],false);
			else
				$rep_image=new ocp_tempcode();

			$description=$sc['cc_description'];
			$description=get_translated_tempcode($description);
			$ajax_edit_url='_SEARCH:cms_catalogues:type=__edit_category:id='.strval($sc['id']);
			$subcategories->attach(do_template('CATEGORY_'.$tpl_set.'_ENTRY',array('_GUID'=>'871f5f5195fcb426fac716e35569d6a3','ID'=>strval($sc['id']),'NAME_FIELD'=>'title','AJAX_EDIT_URL'=>$ajax_edit_url,'URL'=>$url,'REP_IMAGE'=>$rep_image,'CHILDREN'=>$display_string,'NAME'=>$sc['cc_title'],'NAME_PLAIN'=>$sc['cc_title'],'DESCRIPTION'=>$description,'CATEGORY_ID'=>strval($sc['id']),'VIEW_IN_PROFILE'=>$view_in_profile),NULL,false,'CATEGORY_ENTRY'));
		} else
		{	
			$subcategories->attach(do_template('CATALOGUE_'.$tpl_set.'_SUBCATEGORY',array('ID'=>strval($sc['id']),'CATALOGUE'=>$catalogue_name,'URL'=>$url,'NUM_CHILDREN'=>number_format($num_children),'NUM_ENTRIES'=>number_format($num_entries),'NAME'=>$sc['cc_title']),NULL,false,'CATALOGUE_DEFAULT_SUBCATEGORY'));
		}
	}
	if (!$subcategories->is_empty())
	{
		if ($uses_rep_image)
		{
			$subcategories=do_template('CATEGORY_LIST',array('_GUID'=>'3041620a953e52229f752b1bccfbeee3','CONTENT'=>$subcategories));
		} else
		{
			$subcategories=do_template('CATALOGUE_'.$tpl_set.'_SUBCATEGORY_WRAP',array('CATALOGUE'=>$catalogue_name,'CONTENT'=>$subcategories),NULL,false,'CATALOGUE_DEFAULT_SUBCATEGORY_WRAP');
		}
	}

	// Links to add to catalogue category etc
	if (has_actual_catalogue_page_access($catalogue_name,NULL,'cms_catalogues',NULL,array('catalogues_catalogue',$catalogue_name,'catalogues_category',$id),'submit_midrange_content'))
	{
		$add_link=build_url(array('page'=>'cms_catalogues','type'=>'add_entry','catalogue_name'=>$catalogue_name,'category_id'=>$id),get_module_zone('cms_catalogues'));
	} else $add_link=new ocp_tempcode();
	if (has_actual_catalogue_page_access($catalogue_name,NULL,'cms_catalogues',NULL,array('catalogues_catalogue',$catalogue_name,'catalogues_category',$id),'submit_cat_midrange_content'))
	{
		$add_cat_url=build_url(array('page'=>'cms_catalogues','type'=>'add_category','catalogue_name'=>$catalogue_name,'parent_id'=>$id),get_module_zone('cms_catalogues'));
	} else $add_cat_url=new ocp_tempcode();
	if (has_actual_catalogue_page_access($catalogue_name,NULL,'cms_catalogues',NULL,array('catalogues_catalogue',$catalogue_name,'catalogues_category',$id),'edit_cat_midrange_content'))
	{
		$edit_cat_url=build_url(array('page'=>'cms_catalogues','type'=>'_edit_category','catalogue_name'=>$catalogue_name,'id'=>$id),get_module_zone('cms_catalogues'));
	} else $edit_cat_url=new ocp_tempcode();
	if (has_actual_catalogue_page_access($catalogue_name,NULL,'cms_catalogues',NULL,array('catalogues_catalogue',$catalogue_name),'edit_cat_highrange_content'))
	{
		$edit_catalogue_url=build_url(array('page'=>'cms_catalogues','type'=>'_edit_catalogue','id'=>$catalogue_name),get_module_zone('cms_catalogues'));
	} else $edit_catalogue_url=new ocp_tempcode();

	$max_rows=count($entries);
	require_code('templates_results_browser');
	$browser=results_browser(do_lang_tempcode('ENTRIES'),$id,$start,'start',$max,'max',$max_rows,$root,'category',$entries);

	$catalogue_title=get_translated_text($catalogue['c_title']);

	if ($catalogue['c_is_tree']==1)
	{
		$tree=catalogue_category_breadcrumbs($id,is_null(get_param('root',NULL))?NULL:$root);
		if (!$tree->is_empty()) $tree->attach(do_template('BREADCRUMB_ESCAPED'));
		if (has_specific_permission(get_member(),'open_virtual_roots'))
		{
			$url=get_self_url(false,false,(is_null(get_param('root',NULL)))?NULL:array('root'=>$id));
			$tree->attach(hyperlink($url,escape_html($_title),false,false,do_lang_tempcode('VIRTUAL_ROOT')));
		} else $tree->attach(escape_html($_title));
	} else
	{
		$tree=new ocp_tempcode();
		$url=build_url(array('page'=>'_SELF','type'=>'index','id'=>$catalogue_name),'_SELF');
		$tree->attach(hyperlink($url,escape_html($catalogue_title),false,false,do_lang_tempcode('GO_BACKWARDS_TO',escape_html($catalogue_name))));
		$tree->attach(do_template('BREADCRUMB_ESCAPED'));
		$tree->attach(escape_html($_title));
	}

	if(!$no_title)
	{
		breadcrumb_add_segment($tree);
		if (is_null(get_param('root',NULL))) breadcrumb_set_parents(array(array('_SELF:_SELF:misc'.($is_ecommerce?':ecommerce=1':''),do_lang('CATALOGUES'))));
	}

	$GLOBALS['META_DATA']+=array(
		'created'=>date('Y-m-d',$category['cc_add_date']),
		'creator'=>'',
		'publisher'=>'', // blank means same as creator
		'modified'=>'',
		'type'=>get_translated_text($catalogue['c_title']).' category',
		'title'=>$_title,
		'identifier'=>'_SEARCH:catalogues:category:'.strval($id),
		'description'=>get_translated_text($category['cc_description']),
	);

	$cart_link =	new ocp_tempcode();
	$breadcrumb	=	new ocp_tempcode();
	$banner		=	new ocp_tempcode();

	if($is_ecommerce) 
	{
		if (get_forum_type()!='ocf') warn_exit(do_lang_tempcode('NO_OCF'));

		require_code('shopping');
		require_lang('shopping');

		$cart_link=show_cart_image();
		$breadcrumb	=	catalogue_category_breadcrumbs($id,is_null(get_param('root',NULL))?NULL:$root,false,'get_store_content');		
		$banner_name	=	$GLOBALS['SITE_DB']->query_value_null_ok('banners','name',array('b_type'=>'Store','submitter'=>$catalogue['c_owner']));
		
		if(!is_null($banner_name))
		{
			$banner		=	banners_script(true,'',$banner_name,'Store',true);
		}
	}

	if(is_tracked('catalogues',$id))
	{
		$track_url=build_url(array('page'=>'tracking','track_type'=>'catalogues','id'=>$id,'type'=>'untrack','redirect'=>get_self_url(true)),get_module_zone('tracking'));
		$img="untrack_resource";
	}
	else
	{
		$track_url=build_url(array('page'=>'tracking','track_type'=>'catalogues','id'=>$id,'type'=>'misc','redirect'=>get_self_url(true)),get_module_zone('tracking'));
		$img="track_resource";
	}
	// If the its a guest then unset the tracking image
	if(is_guest())
		$img=NULL;
	
	return do_template('CATALOGUE_'.$tpl_set.'_CATEGORY_SCREEN',array('TITLE'=>$title,'ID'=>strval($id),'_TITLE'=>$_title,'TAGS'=>get_loaded_tags('catalogue_categories'),'CATALOGUE'=>$catalogue_name,'CATALOGUE_TITLE'=>$catalogue_title,'BROWSER'=>$browser,'SORT_OPTIONS'=>$sort_options,'ADD_LINK'=>$add_link,'ADD_CAT_URL'=>$add_cat_url,'EDIT_CAT_URL'=>$edit_cat_url,'EDIT_CATALOGUE_URL'=>$edit_catalogue_url,'ENTRIES'=>$entry_buildup,'SUBCATEGORIES'=>$subcategories,'DESCRIPTION'=>get_translated_tempcode($category['cc_description']),'CART_LINK'=>$cart_link,'TRACK_URL'=>$track_url,'TRACK_IMG'=>$img,'BREAD_CRUMB'=>$breadcrumb,'BANNER'=>$banner,'VIEW_IN_PROFILE'=>$view_in_profile),NULL,false,'CATALOGUE_DEFAULT_CATEGORY_SCREEN');
}

/**
 * Function to check specifi permissions for adding entries to event & project catalogue
 *
 *	@param	ID_TEXT	Catalogue name
 * @return	tempcode	
*/
function check_entry_add_permission_for_catalogue($catalogue_name)
{
	if(!(has_specific_permission(get_member(),'can_add_'.$catalogue_name,'cms_catalogues',($catalogue_name=='events')?NULL:array('catalogues_catalogue',$catalogue_name)) || (has_specific_permission(get_member(),'submit_midrange_content','cms_catalogues',($catalogue_name=='')?NULL:array('catalogues_catalogue',$catalogue_name)) && $catalogue_name!='events' && $catalogue_name!='projects'))) access_denied('SPECIFIC_PERMISSION','can_add_'.$catalogue_name);
}
