<?php /*

 ocPortal
 Copyright (c) ocProducts, 2004-2009

 See text/en/licence.txt for full licencing information.

*/

/**
 * @license		http://opensource.org/licenses/cpal_1.0 Common Public Attribution License
 * @copyright	ocProducts Ltd
 * @package		catalogues
 */

require_code('aed_module');

class Module_cms_catalogues extends standard_aed_module
{
	var $lang_type='CATALOGUE_ENTRY';
	var $select_name='ENTRY';
	var $permissions_require='mid';
	var $permissions_cat_require='catalogues_catalogue';
	var $permissions_cat_name='catalogue_name';
	var $permissions_cat_require_b='catalogues_category';
	var $permissions_cat_name_b='category_id';
	var $user_facing=true;
	var $seo_type='catalogue_entry';
	var $catalogue=true;
	var $award_type='catalogue_entry';
	var $possibly_some_kind_of_upload=true;
	var $do_preview=NULL;
	var $menu_label='CATALOGUES';

	var $donext_category_id;
	var $donext_catalogue_name;
	var $saleable_entries=true;

	/**
	 * Standard modular entry-point finder function.
	 *
	 * @return ?array	A map of entry points (type-code=>language-code) (NULL: disabled).
	 */
	function get_entry_points()
	{
		return array_merge(array('misc'=>'MANAGE_CATALOGUES','create_my_store'=>'CREATE_MY_STORE'),parent::get_entry_points());
	}

	/**
	 * Standard modular specific-permission-overide finder function.
	 *
	 * @return array	A map of specific-permissions that are overridable; sp to 0 or 1. 0 means "not category overridable". 1 means "category overridable".
	 */
	function get_sp_overrides()
	{
		require_lang('catalogues');
		return array('can_submit_to_others_categories'=>1,'high_catalogue_entry_timeout'=>1,'submit_cat_highrange_content'=>0,'edit_cat_highrange_content'=>0,'delete_cat_highrange_content'=>0,'submit_cat_midrange_content'=>0,'edit_cat_midrange_content'=>0,'delete_cat_midrange_content'=>0,'submit_midrange_content'=>1,'bypass_validation_midrange_content'=>1,'edit_own_midrange_content'=>1,'edit_midrange_content'=>1,'delete_own_midrange_content'=>1,'delete_midrange_content'=>1);
	}

	/**
	 * Standard aed_module run_start.
	 *
	 * @param  ID_TEXT		The type of module execution
	 * @return tempcode		The output of the run
	 */
	function run_start($type)
	{
		$this->cat_aed_module=new Module_cms_catalogues_cat();
		$this->alt_aed_module=new Module_cms_catalogues_alt();
		$GLOBALS['MODULE_CMS_CATALOGUES']=$this;

		$GLOBALS['HELPER_PANEL_TUTORIAL']='tut_catalogues';
		$GLOBALS['HELPER_PANEL_PIC']='pagepics/catalogues';

		require_lang('catalogues');
		require_lang('fields');
		require_code('catalogues');
		require_code('catalogues2');
		require_code('submit');
		require_code('tracking');
		require_code('mail');

		// Decide what to do
		if ($type=='misc') return $this->misc();
		if ($type=='import') return $this->import_catalogue();
		if ($type=='_import') return $this->_import_catalogue();
		if ($type=='create_my_store') return $this->create_my_store();
		if ($type=='_create_my_store') return $this->_create_my_store();

		return new ocp_tempcode();
	}

	/**
	 * The do-next manager for before content management.
	 *
	 * @return tempcode		The UI
	 */
	function misc()
	{
		require_code('templates_donext');
		$catalogue_name=get_param('catalogue_name','');
		$orders_do_next=NULL;
		if ($catalogue_name=='')
		{
			$extra_map=array();
			$extra_map_2=array();
		} else
		{
			require_lang('do_next');
			$extra_map=array('catalogue_name'=>$catalogue_name);
			$extra_map_2=array('id'=>$catalogue_name);
			$cat_rows=$GLOBALS['SITE_DB']->query_select('catalogues',array('c_title','c_description','c_ecommerce','c_owner'),array('c_name'=>$catalogue_name),'',1);
			if (!array_key_exists(0,$cat_rows)) warn_exit(do_lang_tempcode('MISSING_RESOURCE'));
			$cat_title=$cat_rows[0]['c_title'];
			$cat_description=$cat_rows[0]['c_description'];
			$orders_do_next=($cat_rows[0]['c_ecommerce'] && $cat_rows[0]['c_owner']==get_member())?array('orders',array('cms_orders',array('type'=>'misc','catalogue'=>$catalogue_name),get_module_zone('cms_orders')),do_lang('ECOMMERCE_ORDERS')):NULL;
		}

		if ((!is_null($catalogue_name)) && ($catalogue_name!=''))
		{
			$cat_count=$GLOBALS['SITE_DB']->query_value('catalogue_categories','COUNT(*)',array('c_name'=>$catalogue_name));
			$has_categories=($cat_count!=0);
		} else $has_categories=true;

		return do_next_manager(($catalogue_name!='')?get_page_title(escape_html(get_translated_text($cat_title)),false):get_page_title('MANAGE_CATALOGUES'),($catalogue_name!='')?get_translated_tempcode($cat_description):comcode_to_tempcode(do_lang('DOC_CATALOGUES'),NULL,true),
					array(
						/*	 type							  page	 params													 zone	  */
						(has_specific_permission(get_member(),'submit_cat_highrange_content','cms_catalogues') && ($catalogue_name==''))?array('add_one_catalogue',array('_SELF',array_merge($extra_map,array('type'=>'add_catalogue')),'_SELF'),do_lang('ADD_CATALOGUE')):NULL,
						has_specific_permission(get_member(),'edit_cat_highrange_content','cms_catalogues')?array('edit_one_catalogue',array('_SELF',array_merge($extra_map_2,array('type'=>($catalogue_name=='')?'edit_catalogue':'_edit_catalogue')),'_SELF'),do_lang('EDIT_CATALOGUE')):NULL,
						has_specific_permission(get_member(),'submit_cat_midrange_content','cms_catalogues',($catalogue_name=='')?NULL:array('catalogues_catalogue',$catalogue_name))?array('add_one_category',array('_SELF',array_merge($extra_map,array('type'=>'add_category')),'_SELF'),($catalogue_name!='')?do_lang('NEXT_ITEM_add_one_category'):do_lang('ADD_CATALOGUE_CATEGORY')):NULL,
						has_specific_permission(get_member(),'edit_cat_midrange_content','cms_catalogues',($catalogue_name=='')?NULL:array('catalogues_catalogue',$catalogue_name))?array('edit_one_category',array('_SELF',array_merge($extra_map,array('type'=>'edit_category')),'_SELF'),($catalogue_name!='')?do_lang('NEXT_ITEM_edit_one_category'):do_lang('EDIT_CATALOGUE_CATEGORY')):NULL,
						(!$has_categories)?NULL:(has_specific_permission(get_member(),'submit_midrange_content','cms_catalogues',($catalogue_name=='')?NULL:array('catalogues_catalogue',$catalogue_name))?array('add_one',array('_SELF',array_merge($extra_map,array('type'=>'add_entry')),'_SELF'),($catalogue_name!='')?do_lang('NEXT_ITEM_add_one'):do_lang('ADD_CATALOGUE_ENTRY')):NULL),
						(!$has_categories)?NULL:(has_specific_permission(get_member(),'edit_midrange_content','cms_catalogues',($catalogue_name=='')?NULL:array('catalogues_catalogue',$catalogue_name))?array('edit_one',array('_SELF',array_merge($extra_map,array('type'=>'edit_entry')),'_SELF'),($catalogue_name!='')?do_lang('NEXT_ITEM_edit_one'):do_lang('EDIT_CATALOGUE_ENTRY')):NULL),
						(!$has_categories)?NULL:(has_specific_permission(get_member(),'import_own_highrange_content','cms_catalogues',($catalogue_name=='')?NULL:array('catalogues_catalogue',$catalogue_name))?array('import',array('_SELF',array_merge($extra_map,array('type'=>'import')),'_SELF'),do_lang('IMPORT_CATALOGUE_ENTRIES')):NULL),
						($catalogue_name=='' && $GLOBALS['FORUM_DRIVER']->is_staff(get_member()))?array('orders',array('cms_orders',array('type'=>'misc'),get_module_zone('cms_orders')),do_lang('ECOMMERCE_ORDERS')):$orders_do_next,
					),
					do_lang('MANAGE_CATALOGUES')
		);
	}

	/**
	 * Standard aed_module list function.
	 *
	 * @return tempcode		The tree field
	 */
	function create_my_store()
	{
		$title=get_page_title('CREATE_MY_STORE');

		$test=get_catalogue_store_for(get_member());
		if (!$test->is_empty())
		{
			$url=get_catalogue_store_for(get_member());
			return redirect_screen($title,$url,do_lang_tempcode('STORE_ALREADY_EXISTS'));
		}

		$preview=do_lang('CREATE_MY_STORE_CONFIRM',get_site_name());

		$next_url=build_url(array('page'=>'_SELF','type'=>'_create_my_store'),'_SELF');

		return do_template('CONFIRM_SCREEN',array('TITLE'=>$title,'PREVIEW'=>$preview,'URL'=>$next_url,'HIDDEN'=>'','FIELDS'=>''));
	}

	/**
	 * Standard aed_module list function.
	 *
	 * @return tempcode		The tree field
	 */
	function _create_my_store()
	{
		$title=get_page_title('CREATE_MY_STORE');

		$test=get_catalogue_store_for(get_member());
		if (!$test->is_empty()) warn_exit(do_lang_tempcode('STORE_ALREADY_EXISTS'));

		$username=$GLOBALS['FORUM_DRIVER']->get_username(get_member());

		create_ecommerce_catalogue('store_'.strval(get_member()),insert_lang(do_lang('DEFAULT_CATALOGUE_PRODUCTS_TITLE_MEMBER',$username),2),get_member());

		$url=get_catalogue_store_for(get_member());
		require_code('templates_redirect_screen');
		return redirect_screen($title,$url,do_lang_tempcode('SUCCESS'));
	}

	/**
	 * Standard aed_module list function.
	 *
	 * @return tempcode		The tree field
	 */
	function nice_get_ajax_tree()
	{
		$catalogue_name=get_param('catalogue_name');
		$only_owned=has_specific_permission(get_member(),'edit_midrange_content','cms_catalogues')?NULL:get_member();
		return form_input_tree_list(do_lang_tempcode('ENTRY'),'','id',NULL,'choose_catalogue_entry',array('catalogue_name'=>$catalogue_name,'only_owned'=>$only_owned,'editable_filter'=>true),true);
	}

	/**
	 * Get tempcode for a catalogue entry adding/editing form.
	 *
	 * @param  ?ID_TEXT			The catalogue for the entry (NULL: detect)
	 * @param  ?AUTO_LINK		The category for the entry (NULL: first)
	 * @param  BINARY				Whether the entry is validated
	 * @param  LONG_TEXT			Staff notes
	 * @param  BINARY				Whether rating is allowed
	 * @param  SHORT_INTEGER	Whether comments are allowed (0=no, 1=yes, 2=review style)
	 * @param  BINARY				Whether trackbacks are allowed
	 * @param  ?AUTO_LINK		The ID of the entry (NULL: not yet added)
	 * @return array				A pair: the tempcode for the visible fields, and the tempcode for the hidden fields
	 */
	function get_form_fields($catalogue_name=NULL,$category_id=NULL,$validated=1,$notes='',$allow_rating=1,$allow_comments=1,$allow_trackbacks=1,$id=NULL)
	{
		if (is_null($catalogue_name)) $catalogue_name=get_param('catalogue_name');

		if (!_catalogue_ownership_check($catalogue_name,get_member())) access_denied('SPECIFIC_PERMISSION','can_submit_to_others_categories');

		require_code('feedback');
		require_code('form_templates');

		$fields=new ocp_tempcode();

		$hidden=form_input_hidden('catalogue_name',$catalogue_name);

		if ((is_null($id)) && (is_null($category_id)))
		{
			$category_id=get_param_integer('category_id',-1);
			if ($category_id==-1) $category_id=NULL;
		}

		// Standard fields
		// ===============

		// Category
		if ((is_null($id)) && (is_null($category_id)))
		{
			$category_id=$GLOBALS['SITE_DB']->query_value_null_ok('catalogue_categories','MIN(id)',array('c_name'=>$catalogue_name));
		}

		$fields->attach(form_input_tree_list(do_lang_tempcode('CATEGORY'),do_lang_tempcode('DESCRIPTION_CATEGORY_TREE'),'category_id',NULL,'choose_catalogue_category',array('catalogue_name'=>$catalogue_name,'addable_filter'=>true),true,is_null($category_id)?'':strval($category_id)));

		// Special fields
		// ==============

		if (!is_null($id))
		{
			$special_fields=get_catalogue_entry_field_values($catalogue_name,$id);
		} else
		{
			$special_fields=$GLOBALS['SITE_DB']->query_select('catalogue_fields',array('*'),array('c_name'=>$catalogue_name),'ORDER BY cf_order');
		}

		foreach ($special_fields as $field)
		{
			require_code('hooks/modules/catalogue_fields/'.filter_naughty($field['cf_type']));
			$ob=object_factory('Hook_catalogue_field_'.filter_naughty($field['cf_type']));

			$current=NULL;
			if (array_key_exists('effective_value_pure',$field)) $current=$field['effective_value_pure'];
			elseif (array_key_exists('effective_value',$field)) $current=$field['effective_value'];
			$_cf_name=get_translated_text($field['cf_name']);
			$_cf_description=escape_html(get_translated_text($field['cf_description']));
			$GLOBALS['NO_DEBUG_MODE_FULLSTOP_CHECK']=true;
			$result=$ob->get_field_inputter($_cf_name,$_cf_description,$field,$current,is_null($id));
			$GLOBALS['NO_DEBUG_MODE_FULLSTOP_CHECK']=false;
			if (is_null($result)) continue;
			$fields->attach($result);

			if (strpos($field['cf_type'],'_trans')!==false) $this->do_preview=true;

			unset($result);
			unset($ob);
		}

		$fields->attach(form_input_tick(do_lang_tempcode('VALIDATED'),do_lang_tempcode('DESCRIPTION_VALIDATED'),'validated',$validated==1));

		$fields->attach(feedback_fields($allow_rating==1,$allow_comments==1,$allow_trackbacks==1,false,$notes,$allow_comments==2));

		return array($fields,$hidden);
	}

	/**
	 * Standard aed_module submitter getter.
	 *
	 * @param  ID_TEXT		The entry for which the submitter is sought
	 * @return array			The submitter, and the time of submission (null submission time implies no known submission time)
	 */
	function get_submitter($id)
	{
		$rows=$GLOBALS['SITE_DB']->query_select('catalogue_entries',array('ce_submitter','ce_add_date'),array('id'=>intval($id)),'',1);
		if (!array_key_exists(0,$rows)) return array(NULL,NULL);
		return array($rows[0]['ce_submitter'],$rows[0]['ce_add_date']);
	}

	/**
	 * Standard aed_module cat getter.
	 *
	 * @param  AUTO_LINK		The entry for which the cat is sought
	 * @return string			The cat
	 */
	function get_cat_b($id)
	{
		$temp=$GLOBALS['SITE_DB']->query_value_null_ok('catalogue_entries','cc_id',array('id'=>$id));
		if (is_null($temp)) warn_exit(do_lang_tempcode('MISSING_RESOURCE'));
		return strval($temp);
	}

	/**
	 * Standard aed_module cat getter.
	 *
	 * @param  AUTO_LINK		The entry for which the cat is sought
	 * @return string			The cat
	 */
	function get_cat($id)
	{
		$cat=$this->get_cat_b($id);
		return $GLOBALS['SITE_DB']->query_value('catalogue_categories','c_name',array('id'=>intval($cat)));
	}

	/**
	 * Standard aed_module edit form filler.
	 *
	 * @param  ID_TEXT		The entry being edited
	 * @return array			A tuple of lots of info
	 */
	function fill_in_edit_form($_id)
	{
		$id=intval($_id);

		$myrows=$GLOBALS['SITE_DB']->query_select('catalogue_entries',array('*'),array('id'=>$id));
		if (!array_key_exists(0,$myrows))
		{
			warn_exit(do_lang_tempcode('MISSING_RESOURCE'));
		}
		$myrow=$myrows[0];

		$catalogue_name=$GLOBALS['SITE_DB']->query_value_null_ok('catalogue_entries','c_name',array('id'=>$id));
		if (is_null($catalogue_name)) warn_exit(do_lang_tempcode('MISSING_RESOURCE'));
		$this->edit_text=paragraph(do_lang_tempcode('FOR_CATALOGUE',escape_html($catalogue_name)));

		return $this->get_form_fields($catalogue_name,$myrow['cc_id'],$myrow['ce_validated'],$myrow['notes'],$myrow['allow_rating'],$myrow['allow_comments'],$myrow['allow_trackbacks'],$id);
	}

	/**
	 * Get a entry-id=>value map of what a submitted catalogue entry form has set
	 *
	 * @param  ID_TEXT		The name of the catalogue that was used
	 * @param  boolean		Whether we were editing (because on edit, files might need deleting)
	 * @return array			The map
	 */
	function get_set_field_map($catalogue_name,$editing=false)
	{
		// Get field values
		$fields=$GLOBALS['SITE_DB']->query_select('catalogue_fields',array('*'),array('c_name'=>$catalogue_name));
		$map=array();
		foreach ($fields as $field)
		{
			require_code('hooks/modules/catalogue_fields/'.filter_naughty($field['cf_type']));
			$ob=object_factory('Hook_catalogue_field_'.filter_naughty($field['cf_type']));
			$value=$ob->inputted_to_field_value($editing,$field['id'],$field['cf_default']);

			$map[$field['id']]=$value;
		}

		return $map;
	}

	/**
	 * Standard aed_module add actualiser.
	 *
	 * @return AUTO_LINK		The ID of the entry added
	 */
	function add_actualisation()
	{
		$category_id=post_param_integer('category_id');
		$validated=post_param_integer('validated',0);
		$notes=post_param('notes');
		$allow_rating=post_param_integer('allow_rating',0);
		$allow_comments=post_param_integer('allow_comments',0);
		$allow_trackbacks=post_param_integer('allow_trackbacks',0);

		$catalogue_name=$GLOBALS['SITE_DB']->query_value_null_ok('catalogue_categories','c_name',array('id'=>$category_id));
		if (is_null($catalogue_name)) warn_exit(do_lang_tempcode('MISSING_RESOURCE'));

		$map=$this->get_set_field_map($catalogue_name);

		if ((get_member()!=$GLOBALS['FORUM_DRIVER']->get_guest_id()) && (addon_installed('points')))
		{
			$points=$GLOBALS['SITE_DB']->query_value('catalogues','c_submit_points',array('c_name'=>$catalogue_name));
			require_code('points2');
			system_gift_transfer(do_lang('ADD_CATALOGUE_ENTRY'),intval($points),get_member());
		}

		$owner=$GLOBALS['SITE_DB']->query_value_null_ok('catalogues','c_owner',array('c_name'=>$catalogue_name));
		if ((!is_null($owner)) && ($owner!=get_member())) check_specific_permission('can_submit_to_others_categories',array('catalogue',$catalogue_name));

		$id=actual_add_catalogue_entry($category_id,$validated,$notes,$allow_rating,$allow_comments,$allow_trackbacks,$map);

		// If this category is under tracking
		// Getting the users list
		$user_list=tracking_users('catalogues',$category_id);

		if(count($user_list)>0)
		{
			send_alert('catalogues',$user_list,build_url(array('page'=>'catalogues','type'=>'entry','id'=>$id),get_module_zone('catalogues'),NULL,false,false,true));
		}

		$this->donext_category_id=$category_id;
		$this->donext_catalogue_name=$catalogue_name;

		return $id;
	}

	/**
	 * Standard aed_module edit actualiser.
	 *
	 * @param  ID_TEXT		The entry being edited
	 */
	function edit_actualisation($_id)
	{
		$id=intval($_id);

		$category_id=post_param_integer('category_id',INTEGER_MAGIC_NULL);
		$validated=post_param_integer('validated',fractional_edit()?INTEGER_MAGIC_NULL:0);
		$notes=post_param('notes',STRING_MAGIC_NULL);
		$allow_rating=post_param_integer('allow_rating',fractional_edit()?INTEGER_MAGIC_NULL:0);
		$allow_comments=post_param_integer('allow_comments',fractional_edit()?INTEGER_MAGIC_NULL:0);
		$allow_trackbacks=post_param_integer('allow_trackbacks',fractional_edit()?INTEGER_MAGIC_NULL:0);

		$catalogue_name=$GLOBALS['SITE_DB']->query_value_null_ok('catalogue_categories','c_name',array('id'=>$category_id));
		if (is_null($catalogue_name)) warn_exit(do_lang_tempcode('MISSING_RESOURCE'));
		$map=$this->get_set_field_map($catalogue_name,true);

		$owner=$GLOBALS['SITE_DB']->query_value_null_ok('catalogues','c_owner',array('c_name'=>$catalogue_name));
		if ((!is_null($owner)) && ($owner!=get_member())) check_edit_permission('can_submit_to_others_categories',array('catalogue',$catalogue_name));

		actual_edit_catalogue_entry($id,$category_id,$validated,$notes,$allow_rating,$allow_comments,$allow_trackbacks,$map,post_param('meta_keywords',STRING_MAGIC_NULL),post_param('meta_description',STRING_MAGIC_NULL));

		$this->donext_category_id=$category_id;
		$this->donext_catalogue_name=$catalogue_name;
	}

	/**
	 * Standard aed_module delete actualiser.
	 *
	 * @param  ID_TEXT		The entry being deleted
	 */
	function delete_actualisation($_id)
	{
		$id=intval($_id);

		$category_id=$GLOBALS['SITE_DB']->query_value_null_ok('catalogue_entries','cc_id',array('id'=>$id));
		if (is_null($category_id)) warn_exit(do_lang_tempcode('MISSING_RESOURCE'));

		$catalogue_name=$GLOBALS['SITE_DB']->query_value_null_ok('catalogue_categories','c_name',array('id'=>$category_id));
		$owner=$GLOBALS['SITE_DB']->query_value_null_ok('catalogues','c_owner',array('c_name'=>$catalogue_name));
		if ((!is_null($owner)) && ($owner!=get_member())) check_edit_permission('can_submit_to_others_categories',array('catalogue',$catalogue_name));

		actual_delete_catalogue_entry($id);

		$catalogue_name=$GLOBALS['SITE_DB']->query_value('catalogue_categories','c_name',array('id'=>$category_id));
		$this->donext_category_id=$category_id;
		$this->donext_catalogue_name=$catalogue_name;
	}

	/**
	 * Standard aed_module delete possibility checker.
	 *
	 * @param  ID_TEXT		The entry being potentially deleted
	 * @return boolean		Whether it may be deleted
	 */
	function may_delete_this($id)
	{
		if(!is_ecommerce_catalogue_entry(intval($id))) return true;
		return is_null($GLOBALS['SITE_DB']->query_value_null_ok('shopping_order_details','id',array('p_id'=>intval($id),'p_type'=>'catalogue_items')));
	}

	/**
	 * The do-next manager for after content management.
	 *
	 * @param  tempcode		The title (output of get_page_title)
	 * @param  tempcode		Some description to show, saying what happened
	 * @param  ?AUTO_LINK	The ID of whatever was just handled (NULL: N/A)
	 * @return tempcode		The UI
	 */
	function do_next_manager($title,$description,$id)
	{
		breadcrumb_set_self(do_lang_tempcode('DONE'));

		$c_name=$this->donext_catalogue_name;
		$category_id=$this->donext_category_id;

		require_code('templates_donext');
		return do_next_manager($title,$description,
					NULL,
					NULL,
					/*		TYPED-ORDERED LIST OF 'LINKS'		*/
					/*	 page	 params				  zone	  */
					array('_SELF',array('type'=>'add_entry','catalogue_name'=>$c_name,'category_id'=>$category_id),'_SELF'),								  // Add one
					(is_null($id) || (!has_specific_permission(get_member(),'edit_own_midrange_content','cms_catalogues',array('catalogues_category',$category_id))))?NULL:array('_SELF',array('type'=>'_edit_entry','id'=>$id,'catalogue_name'=>$c_name),'_SELF'),				 // Edit this
					has_specific_permission(get_member(),'edit_own_midrange_content','cms_catalogues')?array('_SELF',array('type'=>'edit_entry','catalogue_name'=>$c_name),'_SELF'):NULL, // Edit one
					is_null($id)?NULL:array('catalogues',array('type'=>'entry','id'=>$id),get_module_zone('catalogues')),						  // View this
					NULL,																						// View archive
					NULL,																						// Add to category
					has_specific_permission(get_member(),'submit_cat_midrange_content','cms_catalogues')?array('_SELF',array('type'=>'add_category','catalogue_name'=>$c_name),'_SELF'):NULL,			  // Add one category
					has_specific_permission(get_member(),'edit_own_cat_midrange_content','cms_catalogues')?array('_SELF',array('type'=>'edit_category','catalogue_name'=>$c_name),'_SELF'):NULL,			  // Edit one category

					NULL,																						// Edit this category
					NULL,																						// View this category
					/*	  SPECIALLY TYPED 'LINKS'				  */
					array(),
					array(),
					array(
						/*	 type							  page	 params													 zone	  */
						has_specific_permission(get_member(),'submit_cat_highrange_content','cms_catalogues')?array('add_one_catalogue',array('_SELF',array('type'=>'add_catalogue'),'_SELF')):NULL,
						has_specific_permission(get_member(),'edit_own_cat_highrange_content','cms_catalogues')?array('edit_this_catalogue',array('_SELF',array('type'=>'_edit_catalogue','id'=>$c_name),'_SELF')):NULL,
						has_specific_permission(get_member(),'edit_own_cat_highrange_content','cms_catalogues')?array('edit_one_catalogue',array('_SELF',array('type'=>'edit_catalogue'),'_SELF')):NULL,
						array('view_this',array('catalogues',array('type'=>'index','id'=>$c_name),get_module_zone('catalogues')),do_lang('VIEW_CATALOGUE'))
					),
					do_lang('MANAGE_CATALOGUES')
		);
	}

	/**
	 * The UI to choose a catalogue to import catalogue entries
	 *
	 * @return tempcode		The UI
	 */
	function import_catalogue()
	{
		$title=get_page_title('CATALOGUE_IMPORT');

		$catalogue_select=$this->choose_catalogue($title);

		if (!is_null($catalogue_select)) return $catalogue_select;

		$catalogue_name=get_param('catalogue_name');

		$post_url=build_url(array('page'=>'_SELF','type'=>'_import','old_type'=>get_param('type',''),'catalogue_name'=>$catalogue_name),'_SELF');

		$submit_name=do_lang_tempcode('PROCEED');

		// Build up form
		$fields=new ocp_tempcode();

		require_code('form_templates');

		$fields->attach(form_input_upload(do_lang_tempcode('UPLOAD'),do_lang_tempcode('UPLOAD_DESC'),'file_novalidate',true));

		return do_template('FORM_SCREEN',array('TITLE'=>$title,'TEXT'=>do_lang_tempcode('CATALOGUE_IMPORT_TEXT'),'HIDDEN'=>'','FIELDS'=>$fields,'SUBMIT_NAME'=>$submit_name,'URL'=>$post_url));
	}

	/**
	 * Standard actualiser to import catalogue entries
	 *
	 * @return tempcode		The UI
	 */
	function _import_catalogue()
	{
		$catalog_root	=	NULL;

		$title		=	get_page_title('CATALOGUE_IMPORT');

		$catalogue_name =	get_param('catalogue_name');

		$categories	=	array();

		$fields=$GLOBALS['SITE_DB']->query_select('catalogue_fields',array('*'),array('c_name'=>$catalogue_name));

		$catsrow=$GLOBALS['SITE_DB']->query("SELECT t1.id,t2.text_original, t1.cc_parent_id FROM ".$GLOBALS['SITE_DB']->get_table_prefix()."catalogue_categories t1,".$GLOBALS['SITE_DB']->get_table_prefix()."translate t2 WHERE t1.cc_title=t2.id AND t1.c_name='".db_escape_string($catalogue_name)."'");

		foreach($catsrow as $values)
		{
			$categories[$values['text_original']]	=	$values['id'];
		}

		require_code('uploads');
		is_swf_upload(true);
		if (array_key_exists('file_novalidate',$_FILES))
		{
			if (is_uploaded_file($_FILES['file_novalidate']['tmp_name']))
			{
				$csv_name=$_FILES['file_novalidate']['tmp_name'];
			}
		}

		$handle 		= 	fopen($csv_name, "r");

		$csv_field_titles	=	fgetcsv($handle, 1000, ",");

		$csv_field_titles	=	array_flip($csv_field_titles);

		if(!array_key_exists('CATEGORY',$csv_field_titles)) //Checking the fields title contains 'CATEGORY'.
			warn_exit(do_lang_tempcode('FIELDS_UNMATCH'));

		while (($data = fgetcsv($handle, 1000, ",")) !== FALSE)
		{
			$this->import_csv_lines($catalogue_name,$data,NULL,$fields,$categories,$csv_field_titles,$catalog_root);
		}

		$description=is_null($this->do_next_description)?do_lang_tempcode('SUCCESS'):$this->do_next_description;

		return $this->do_next_manager($title,$description,NULL);
	}

	/**
	 * Create an entry-id=>value map of uploaded csv data and it's importing
	 *
	 * @param  ID_TEXT		The name of the catalogue that was used
	 * @param  array		Data array of CSV imported file's lines
	 * @param  integer		catalogue root id
	 * @param  array		Array of catalogue fields
 	 * @param  array		Array of categories
 	 * @param  array		Array of csv field titles
	 */
	function import_csv_lines($catalogue_name,$csv_data,$catalog_root,$fields,&$categories,$csv_field_titles)
	{
		$map=array();
		$match_flag=false;

		$curr_cat=$csv_data[$csv_field_titles['CATEGORY']];

		if($curr_cat=='')
		{
			//Checks the general category exists or not
			if(array_key_exists($catalogue_name,$categories))
				$catid=$categories[$catalogue_name];
			else	//If category field is null, record adds to a general category named catalogue name.
			{
				$catalog_title=$GLOBALS['FORUM_DB']->query_value_null_ok('catalogues','c_title',array('c_name'=>$catalogue_name));

				$catid=actual_add_catalogue_category($catalogue_name,$catalog_title,$catalog_title,$catalog_title,NULL,'');

				$categories	=	array_merge(array($catalogue_name=>$catid),$categories);
			}
		}
		elseif(array_key_exists($curr_cat,$categories))
		{
			$catid=$categories[$curr_cat];
		}
		else
		{
			$catid	=				actual_add_catalogue_category($catalogue_name,$curr_cat,$curr_cat,$curr_cat,$catalog_root,'');

			$categories	=	array_merge(array($curr_cat=>$catid),$categories);

			$this->set_permissions($catid);
		}

		foreach ($fields as $field)
		{
			$field_name	=	get_translated_text($field['cf_name']);

			if(array_key_exists($field_name,$csv_field_titles))
			{
				$map[$field['id']]	=	str_replace('"','',$csv_data[$csv_field_titles[$field_name]]);
				$match_flag=true; // to check matching of csv and db fields
			}
			else
			{
				$map[$field['id']]	=	$field['cf_default'];
			}
		}

		if ($match_flag)
			$id=actual_add_catalogue_entry($catid,0,'',1,1,1,$map);
		else
			warn_exit(do_lang_tempcode('FIELDS_UNMATCH'));
	}
}

class Module_cms_catalogues_cat extends standard_aed_module
{
	var $lang_type='CATALOGUE_CATEGORY';
	var $select_name='NAME';
	var $permissions_require='cat_mid';
	var $permission_module='catalogues_category';
	var $permissions_cat_require='catalogues_catalogue';
	var $permissions_cat_name='catalogue_name';
	var $seo_type='catalogue_category';
	var $catalogue=true;
	var $award_type='catalogue_category';
	var $upload='image';
	var $javascript='var mt=document.getElementById(\'move_target\'); var form=mt.form; var crf=function() { var s=mt.selectedIndex==0; form.elements[\'move_days_lower\'].disabled=s; form.elements[\'move_days_higher\'].disabled=s; }; crf(); mt.onclick=crf;';
	var $menu_label='CATALOGUES';

	var $donext_catalogue_name;
	var $saleable_entries=true;

	/**
	 * Standard aed_module list function.
	 *
	 * @return tempcode		The tree field
	 */
	function nice_get_ajax_tree()
	{
		$catalogue_name=get_param('catalogue_name');
		return form_input_tree_list(do_lang_tempcode('CODENAME'),'','id',NULL,'choose_catalogue_category',array('catalogue_name'=>$catalogue_name),true);
	}

	/**
	 * Standard aed_module cat getter.
	 *
	 * @param  AUTO_LINK		The entry for which the cat is sought
	 * @return string			The cat
	 */
	function get_cat($id)
	{
		return $GLOBALS['SITE_DB']->query_value('catalogue_categories','c_name',array('id'=>$id));
	}

	/**
	 * Get tempcode for a catalogue category adding/editing form.
	 *
	 * @param  ?ID_TEXT		The name of the catalogue the category is in (NULL: detect)
	 * @param  SHORT_TEXT	The title of the category
	 * @param  LONG_TEXT		Description for the category
	 * @param  LONG_TEXT		Admin notes
	 * @param  ?AUTO_LINK	The ID of the parent category (NULL: no parent) (-1: arbitrary default)
	 * @param  ?AUTO_LINK	The ID of this category (NULL: we're adding, not editing)
	 * @param  URLPATH		The rep-image for the catalogue category
	 * @param  integer		The number of days before expiry (lower limit)
	 * @param  integer		The number of days before expiry (higher limit)
	 * @param  ?AUTO_LINK	The expiry category (NULL: do not expire)
	 * @return array			A pair: the tempcode for the visible fields, and the tempcode for the hidden fields
	 */
	function get_form_fields($catalogue_name=NULL,$title='',$description='',$notes='',$parent_id=-1,$id=NULL,$rep_image='',$move_days_lower=30,$move_days_higher=60,$move_target=NULL) // Not the fields in a category (no such thing: fields are in catalogues) - the HTML form fields to input the details for a category
	{
		if (is_null($catalogue_name)) $catalogue_name=get_param('catalogue_name');

		if (!_catalogue_ownership_check($catalogue_name,get_member())) access_denied('SPECIFIC_PERMISSION','can_submit_to_others_categories');

		if ($parent_id==-1) $parent_id=get_param_integer('parent_id',-1);

		$fields=new ocp_tempcode();
		require_code('form_templates');
		$hidden=new ocp_tempcode();
		$hidden->attach(form_input_hidden('catalogue_name',$catalogue_name));
		$fields->attach(form_input_line(do_lang_tempcode('TITLE'),do_lang_tempcode('DESCRIPTION_TITLE'),'title',$title,true));
		$fields->attach(form_input_text_comcode(do_lang_tempcode('DESCRIPTION'),do_lang_tempcode('DESCRIPTION_DESCRIPTION'),'description',$description,false));
		$fields->attach(form_input_text(do_lang_tempcode('NOTES'),do_lang_tempcode('DESCRIPTION_NOTES'),'notes',$notes,false));
		$fields->attach(form_input_upload(do_lang_tempcode('REPRESENTATIVE_IMAGE'),do_lang_tempcode('DESCRIPTION_REPRESENTATIVE_IMAGE'),'rep_image',false,$rep_image));
		$list=new ocp_tempcode();
		$list->attach(form_input_list_entry(strval(-1),false,do_lang_tempcode('NA_EM')));
		$list->attach(nice_get_catalogue_category_tree($catalogue_name,$move_target));

		// Is the catalogue a tree?
		$is_tree=$GLOBALS['SITE_DB']->query_value_null_ok('catalogues','c_is_tree',array('c_name'=>$catalogue_name));
		if (is_null($is_tree)) warn_exit(do_lang_tempcode('MISSING_RESOURCE'));
		if (($is_tree==1) && (!is_null($parent_id)))
		{
			$fields->attach(form_input_tree_list(do_lang_tempcode('PARENT'),do_lang_tempcode('DESCRIPTION_PARENT'),'parent_id',NULL,'choose_catalogue_category',array('catalogue_name'=>$catalogue_name),true,is_null($parent_id)?'':strval($parent_id)));
		}

		$fields->attach(do_template('FORM_SCREEN_FIELD_SPACER',array('SECTION_HIDDEN'=>is_null($move_target),'TITLE'=>do_lang_tempcode('CLASSIFIED_ADS'))));
		$fields->attach(form_input_list(do_lang_tempcode('EXPIRY_MOVE_TARGET'),do_lang_tempcode('DESCRIPTION_EXPIRY_MOVE_TARGET'),'move_target',$list));
		$fields->attach(form_input_integer(do_lang_tempcode('EXPIRY_MOVE_DAYS_LOWER'),do_lang_tempcode('DESCRIPTION_EXPIRY_MOVE_DAYS_LOWER'),'move_days_lower',$move_days_lower,true));
		$fields->attach(form_input_integer(do_lang_tempcode('EXPIRY_MOVE_DAYS_HIGHER'),do_lang_tempcode('DESCRIPTION_EXPIRY_MOVE_DAYS_HIGHER'),'move_days_higher',$move_days_higher,true));

		// Permissions
		$fields->attach($this->get_permission_fields(is_null($id)?'':strval($id),NULL,is_null($id)));

		return array($fields,$hidden);
	}

	/**
	 * Standard aed_module edit form filler.
	 *
	 * @param  ID_TEXT		The entry being edited
	 * @return array			A tuple of lots of info
	 */
	function fill_in_edit_form($_id)
	{
		$category_id=intval($_id);

		$catalogue_name=get_param('catalogue_name');
		$this->edit_text=paragraph(do_lang_tempcode('FOR_CATALOGUE',escape_html($catalogue_name)));

		$rows=$GLOBALS['SITE_DB']->query_select('catalogue_categories',array('*'),array('id'=>$category_id));
		if (!array_key_exists(0,$rows))
		{
			warn_exit(do_lang_tempcode('MISSING_RESOURCE'));
		}
		$myrow=$rows[0];

		return $this->get_form_fields($catalogue_name,get_translated_text($myrow['cc_title']),get_translated_text($myrow['cc_description']),$myrow['cc_notes'],$myrow['cc_parent_id'],$category_id,$myrow['rep_image'],$myrow['cc_move_days_lower'],$myrow['cc_move_days_higher'],$myrow['cc_move_target']);
	}

	/**
	 * Standard aed_module add actualiser.
	 *
	 * @return AUTO_LINK		The ID of what was added
	 */
	function add_actualisation()
	{
		$catalogue_name=post_param('catalogue_name');

		$title=post_param('title');
		$description=post_param('description');
		$notes=post_param('notes');
		$parent_id=post_param_integer('parent_id',-1);
		if ($parent_id==-1) $parent_id=NULL;
		$urls=get_url('image_url','rep_image','uploads/grepimages',0,OCP_UPLOAD_IMAGE);
		$rep_image=$urls[0];
		$move_days_lower=post_param_integer('move_days_lower',30);
		$move_days_higher=post_param_integer('move_days_higher',60);
		$move_target=post_param_integer('move_target');
		if ($move_target==-1) $move_target=NULL;

		$owner=$GLOBALS['SITE_DB']->query_value_null_ok('catalogues','c_owner',array('c_name'=>$catalogue_name));
		if ((!is_null($owner)) && ($owner!=get_member())) check_specific_permission('can_submit_to_others_categories',array('catalogue',$catalogue_name));

		$category_id=actual_add_catalogue_category($catalogue_name,$title,$description,$notes,$parent_id,$rep_image,$move_days_lower,$move_days_higher,$move_target);
		$this->set_permissions($category_id);

		$this->donext_category_id=$category_id;
		$this->donext_catalogue_name=$catalogue_name;

		return $category_id;
	}

	/**
	 * Standard aed_module edit actualiser.
	 *
	 * @param  ID_TEXT		The entry being edited
	 */
	function edit_actualisation($_id)
	{
		$category_id=intval($_id);

		$catalogue_name=$GLOBALS['SITE_DB']->query_value('catalogue_categories','c_name',array('id'=>$category_id));

		$title=post_param('title');
		$description=post_param('description',STRING_MAGIC_NULL);
		$notes=post_param('notes',STRING_MAGIC_NULL);
		$parent_id=post_param_integer('parent_id',INTEGER_MAGIC_NULL);
		$move_days_lower=post_param_integer('move_days_lower',INTEGER_MAGIC_NULL);
		$move_days_higher=post_param_integer('move_days_higher',INTEGER_MAGIC_NULL);
		$move_target=post_param_integer('move_target',INTEGER_MAGIC_NULL);
		if ($move_target==-1) $move_target=NULL;
		if (!fractional_edit())
		{
			$urls=get_url('image_url','rep_image','uploads/grepimages',0,OCP_UPLOAD_IMAGE);
			$rep_image=$urls[0];
			if (($rep_image=='') && (post_param_integer('rep_image_unlink',0)!=1)) $rep_image=NULL;
		} else $rep_image=STRING_MAGIC_NULL;

		$owner=$GLOBALS['SITE_DB']->query_value_null_ok('catalogues','c_owner',array('c_name'=>$catalogue_name));
		if ((!is_null($owner)) && ($owner!=get_member())) check_edit_permission('can_submit_to_others_categories',array('catalogue',$catalogue_name));

		actual_edit_catalogue_category($category_id,$title,$description,$notes,$parent_id,post_param('meta_keywords',STRING_MAGIC_NULL),post_param('meta_description',STRING_MAGIC_NULL),$rep_image,$move_days_lower,$move_days_higher,$move_target);
		if (!fractional_edit())
		{
			$this->set_permissions($category_id);
		}

		$this->donext_category_id=$category_id;
		$this->donext_catalogue_name=$catalogue_name;
	}

	/**
	 * Standard aed_module delete actualiser.
	 *
	 * @param  ID_TEXT		The entry being deleted
	 */
	function delete_actualisation($id)
	{
		$category_id=intval($id);

		$catalogue_name=$GLOBALS['SITE_DB']->query_value_null_ok('catalogue_categories','c_name',array('id'=>$category_id));
		$owner=$GLOBALS['SITE_DB']->query_value_null_ok('catalogues','c_owner',array('c_name'=>$catalogue_name));
		if ((!is_null($owner)) && ($owner!=get_member())) check_edit_permission('can_submit_to_others_categories',array('catalogue',$catalogue_name));

		actual_delete_catalogue_category($category_id);

		$this->donext_catalogue_name=post_param('catalogue_name');
	}

	/**
	 * The do-next manager for after catalogue content management.
	 *
	 * @param  tempcode		The title (output of get_page_title)
	 * @param  tempcode		Some description to show, saying what happened
	 * @param  ?AUTO_LINK	The ID of whatever catalogue category was just handled (NULL: deleted)
	 * @return tempcode		The UI
	 */
	function do_next_manager($title,$description,$id)
	{
		$catalogue_name=$this->donext_catalogue_name;

		breadcrumb_set_self(do_lang_tempcode('DONE'));

		require_code('templates_donext');
		return do_next_manager($title,$description,
					NULL,
					NULL,
					/*		TYPED-ORDERED LIST OF 'LINKS'		*/
					/*	 page	 params				  zone	  */
					(!is_null($id))?NULL:array('_SELF',array('type'=>'add_entry','catalogue_name'=>$catalogue_name),'_SELF'),// Add one
					NULL,																						// Edit this
					has_specific_permission(get_member(),'edit_own_midrange_content','cms_catalogues')?array('_SELF',array('type'=>'edit_entry','catalogue_name'=>$catalogue_name),'_SELF'):NULL,  // Edit one
					NULL,																						// View this
					NULL,																						// View archive
					is_null($id)?NULL:array('_SELF',array('type'=>'add_entry','category_id'=>$id,'catalogue_name'=>$catalogue_name),'_SELF'),// Add to category
					has_specific_permission(get_member(),'submit_cat_midrange_content','cms_catalogues')?array('_SELF',array('type'=>'add_category','catalogue_name'=>$catalogue_name),'_SELF'):NULL, // Add one category
					has_specific_permission(get_member(),'edit_own_cat_midrange_content','cms_catalogues')?array('_SELF',array('type'=>'edit_category','catalogue_name'=>$catalogue_name),'_SELF'):NULL, // Edit one category
					(is_null($id) || (!has_specific_permission(get_member(),'edit_own_cat_midrange_content','cms_catalogues')))?NULL:array('_SELF',array('type'=>'_edit_category','id'=>$id,'catalogue_name'=>$catalogue_name),'_SELF'), // Edit this category
					is_null($id)?NULL:array('catalogues',array('type'=>'category','id'=>$id),get_module_zone('catalogues')),// View this category

					/*	  SPECIALLY TYPED 'LINKS'				  */
					array(),array(),
					array(
						/*	 type							  page	 params													 zone	  */
						has_specific_permission(get_member(),'submit_cat_highrange_content','cms_catalogues')?array('add_one_catalogue',array('_SELF',array('type'=>'add_catalogue'),'_SELF')):NULL,
						has_specific_permission(get_member(),'edit_own_cat_highrange_content','cms_catalogues')?array('edit_this_catalogue',array('_SELF',array('type'=>'_edit_catalogue','id'=>$catalogue_name),'_SELF')):NULL,
						has_specific_permission(get_member(),'edit_own_cat_highrange_content','cms_catalogues')?array('edit_one_catalogue',array('_SELF',array('type'=>'edit_catalogue'),'_SELF')):NULL,
						array('view_this',array('catalogues',array('type'=>'index','id'=>$catalogue_name),get_module_zone('catalogues')),do_lang('INDEX'))
					),
					do_lang('MANAGE_CATALOGUES')
		);
	}

}

class Module_cms_catalogues_alt extends standard_aed_module
{
	var $lang_type='CATALOGUE';
	var $select_name='CATALOGUE';
	var $select_name_description='DESCRIPTION_CATALOGUE';
	var $permissions_require='cat_high';
	var $permission_module='catalogues_catalogue';
	var $non_integer_id=true;
	var $award_type='catalogue';
	var $is_tree_catalogue=false; // Set for usage by do-next-manager
	var $menu_label='CATALOGUES';
	var $saleable_entries=true;
	var $javascript="var fn=document.getElementById('title'); if (fn) { var form=fn.form; fn.onchange=function() { if ((form.elements['name']) && (form.elements['name'].value=='')) form.elements['name'].value=fn.value.toLowerCase().replace(/[^\w\d\.\-]/g,'_').replace(/\_+\$/,''); }; }";

	/**
	 * Standard aed_module list function.
	 *
	 * @return tempcode		The selection list
	 */
	function nice_get_entries()
	{
		return nice_get_catalogues();
	}

	/**
	 * Get tempcode for a catalogue adding/editing form.
	 *
	 * @param  ID_TEXT			The name of the catalogue
	 * @param  SHORT_TEXT		The human readable name/title of the catalogue
	 * @param  LONG_TEXT			The description
	 * @param  SHORT_INTEGER	The display type
	 * @param  BINARY				Whether the catalogue uses a hierarchy
	 * @param  LONG_TEXT			Admin notes
	 * @param  integer			How many points are given to a member that submits to the catalogue
	 * @param  BINARY				Whether the catalogue is an eCommerce catalogue
	 * @param  ID_TEXT			How to send view reports
	 * @set    never daily weekly monthly quarterly
	 *	@param	?USER				Owner of catalogue
	 * @return array				A tuple: the tempcode for the visible fields, and the tempcode for the hidden fields, ..., and action fields
	 */
	function get_form_fields($name='',$title='',$description='',$display_type=0,$is_tree=0,$notes='',$submit_points=0,$ecommerce=0,$send_view_reports='never',$c_owner=NULL)
	{
		$fields=new ocp_tempcode();
		$hidden=new ocp_tempcode();
		require_code('form_templates');

		$fields->attach(form_input_line(do_lang_tempcode('TITLE'),do_lang_tempcode('DESCRIPTION_TITLE'),'title',$title,true));
		$fields->attach(form_input_codename(do_lang_tempcode('CODENAME'),do_lang_tempcode('DESCRIPTION_CODENAME'),'name',$name,true));
		$fields->attach(form_input_text_comcode(do_lang_tempcode('DESCRIPTION'),do_lang_tempcode('DESCRIPTION_CATALOGUE_DESCRIPTION'),'description',$description,false));

		$display_types=new ocp_tempcode();
		$display_types->attach(form_input_list_entry(strval(0),$display_type==0,do_lang_tempcode('DT_MAPS')));
		$display_types->attach(form_input_list_entry(strval(1),$display_type==1,do_lang_tempcode('DT_LIST')));
		$display_types->attach(form_input_list_entry(strval(2),$display_type==2,do_lang_tempcode('DT_MATRIX')));
		$fields->attach(form_input_list(do_lang_tempcode('DISPLAY_TYPE'),do_lang_tempcode('DESCRIPTION_DISPLAY_TYPE'),'display_type',$display_types));

		if (addon_installed('shopping'))
		{
			if ($ecommerce==1)
			{
				if (get_forum_type()!='ocf') warn_exit(do_lang_tempcode('NO_OCF'));
			}

			$fields->attach(form_input_tick(do_lang_tempcode('CAT_ECOMMERCE'),do_lang_tempcode('DESCRIPTION_CAT_ECOMMERCE'),'ecommerce',$ecommerce==1));
		}
		if ($name=='') // For new catalogues only
			$fields->attach(form_input_tick(do_lang_tempcode('IS_TREE'),do_lang_tempcode('DESCRIPTION_IS_TREE'),'is_tree',$is_tree==1));
		if ($name=='') $fields->attach(form_input_line(do_lang_tempcode('AUTO_FILL'),do_lang_tempcode('DESCRIPTION_AUTO_FILL'),'auto_fill','',false,NULL,10000));

		$fields->attach(form_input_username(do_lang_tempcode('CATALOGUE_OWNER'),do_lang_tempcode('DESCRIPTION_CATALOGUE_OWNER'),'catalogue_owner',$c_owner,false));

		$fields->attach(do_template('FORM_SCREEN_FIELD_SPACER',array('SECTION_HIDDEN'=>$notes=='' && $submit_points==0 && $send_view_reports=='never','TITLE'=>do_lang_tempcode('ADVANCED'))));

		$fields->attach(form_input_text(do_lang_tempcode('NOTES'),do_lang_tempcode('DESCRIPTION_NOTES'),'notes',$notes,false));
		if (addon_installed('points'))
		{
			$fields->attach(form_input_integer(do_lang_tempcode('SUBMIT_POINTS'),do_lang_tempcode('DESCRIPTION_SUBMIT_POINTS'),'submit_points',$submit_points,false));
		}

		$view_report_types=new ocp_tempcode();
		$view_report_types->attach(form_input_list_entry('never',$send_view_reports=='never',do_lang_tempcode('VR_NEVER')));
		$view_report_types->attach(form_input_list_entry('daily',$send_view_reports=='daily',do_lang_tempcode('VR_DAILY')));
		$view_report_types->attach(form_input_list_entry('weekly',$send_view_reports=='weekly',do_lang_tempcode('VR_WEEKLY')));
		$view_report_types->attach(form_input_list_entry('monthly',$send_view_reports=='monthly',do_lang_tempcode('VR_MONTHLY')));
		$view_report_types->attach(form_input_list_entry('quarterly',$send_view_reports=='quarterly',do_lang_tempcode('VR_QUARTERLY')));
		$fields->attach(form_input_list(do_lang_tempcode('VIEW_REPORTS'),do_lang_tempcode('DESCRIPTION_VIEW_REPORTS'),'send_view_reports',$view_report_types));

		// Permissions
		$fields->attach($this->get_permission_fields($name,NULL,($name=='')));

		$actions=new ocp_tempcode();
		if ($name!='')
			$actions->attach(form_input_tick(do_lang_tempcode('RESET_CATEGORY_PERMISSIONS'),do_lang_tempcode('DESCRIPTION_RESET_CATEGORY_PERMISSIONS'),'reset_category_permissions',false));

		return array($fields,$hidden,NULL,NULL,false,NULL,$actions);
	}

	/**
	 * Get tempcode for a catalogue field adding/editing form (many of these are put together to add/edit a single catalogue!).
	 *
	 * @param  boolean		Whether this is the first field of the entry fields
	 * @param  integer		The number of fields that will be on the screen
	 * @param  string			The prefix the field input fields are given (e.g. new1_)
	 * @param  integer		The order of the field relative to the other fields
	 * @param  SHORT_TEXT	The name of the field
	 * @param  LONG_TEXT		Description for the field
	 * @param  ID_TEXT		The field type
	 * @set    short_text long_text short_trans long_trans integer float picture upload url email user
	 * @param  BINARY			Whether the field defines entry ordering
	 * @param  BINARY			Whether the field is searchable
	 * @param  BINARY			Whether the field is visible when an entry is viewed
	 * @param  SHORT_TEXT	Default value for the field
	 * @param  BINARY			Whether the field is required
	 * @param  BINARY			Whether the field is to be shown in category views (not applicable for the list display type)
	 * @param  BINARY			Whether the field is to be shown in search views (not applicable for the list display type)
	 * @return array			A pair: the tempcode for the visible fields, and the tempcode for the hidden fields
	 */
	function get_field_fields($first_field,$num_fields_to_show,$prefix,$order,$name='',$description='',$type='short_text',$defines_order=0,$visible=1,$searchable=1,$default='',$required=1,$put_in_category=1,$put_in_search=1)
	{
		$fields=new ocp_tempcode();
		$hidden=new ocp_tempcode();

		require_code('form_templates');
		$fields->attach(form_input_line(do_lang_tempcode('NAME'),do_lang_tempcode('DESCRIPTION_FIELD_NAME'),$prefix.'name',$name,($name!='') || $first_field)); // If this is gonna be a new field that might not be filled in, don't make them fill it in
		$fields->attach(form_input_line(do_lang_tempcode('DESCRIPTION'),do_lang_tempcode('DESCRIPTION_FIELD_DESCRIPTION'),$prefix.'description',$description,false));
		$fields->attach(form_input_line(do_lang_tempcode('DEFAULT_VALUE'),do_lang_tempcode('DESCRIPTION_FIELD_DEFAULT'),$prefix.'default',$default,false,NULL,10000));

		$all_types=find_all_hooks('modules','catalogue_fields');
		if ($name!='')
		{
			require_code('hooks/modules/catalogue_fields/'.$type);
			$ob=object_factory('Hook_catalogue_field_'.$type);
			$types=array();
			list(,,$db_type)=$ob->get_field_value_row_bits(db_get_first_id());
			foreach ($all_types as $this_type=>$hook_type)
			{
				require_code('hooks/modules/catalogue_fields/'.$this_type);
				$ob=object_factory('Hook_catalogue_field_'.$this_type);
				list(,,$this_db_type)=$ob->get_field_value_row_bits(db_get_first_id());

				if ($this_db_type==$db_type)
					$types[$this_type]=$hook_type;
			}
		} else $types=$all_types;
		$orderings=array(
			do_lang_tempcode('FIELD_TYPES__TEXT'),'short_trans','long_trans','short_text','long_text',
			do_lang_tempcode('FIELD_TYPES__NUMBERS'),'integer','float',
			do_lang_tempcode('FIELD_TYPES__CHOICES'),'list','radiolist','tick',
			do_lang_tempcode('FIELD_TYPES__UPLOADSANDURLS'),'upload','picture','url',
			do_lang_tempcode('FIELD_TYPES__MAGIC'),'auto_increment','random',
			do_lang_tempcode('FIELD_TYPES__REFERENCES'),'isbn','reference');
//			do_lang_tempcode('FIELD_TYPES__OTHER'),'user','email','date',			Will go under OTHER automatically
		$_types=array();
		$done_one_in_section=true;
		foreach ($orderings as $o)
		{
			if (is_object($o))
			{
				if (!$done_one_in_section) array_pop($_types);
				$_types[]=$o;
				$done_one_in_section=false;
			} else
			{
				if (array_key_exists($o,$types))
				{
					$_types[]=$o;
					unset($types[$o]);
					$done_one_in_section=true;
				}
			}
		}
		if (!$done_one_in_section) array_pop($_types);
		if (count($types)!=0)
		{
			$types=array_merge($_types,array(do_lang_tempcode('FIELD_TYPES__OTHER')),array_keys($types));
		} else $types=$_types;
		$type_list=new ocp_tempcode();
		foreach ($types as $_type)
		{
			if (is_object($_type))
			{
				if (!$type_list->is_empty()) $type_list->attach(form_input_list_entry('',false,escape_html(''),false,true));
				$type_list->attach(form_input_list_entry('',false,$_type,false,true));
			} else
			{
				$type_list->attach(form_input_list_entry($_type,($_type==$type),do_lang_tempcode('FIELD_TYPE_'.$_type)));
			}
		}

		$fields->attach(form_input_list(do_lang_tempcode('TYPE'),do_lang_tempcode(($name=='')?'DESCRIPTION_FIELD_TYPE_FIRST_TIME':'DESCRIPTION_FIELD_TYPE'),$prefix.'type',$type_list));
		$order_list=new ocp_tempcode();
		for ($i=0;$i<$num_fields_to_show;$i++)
		{
			$order_list->attach(form_input_list_entry(strval($i),$i==$order,number_format($i+1).(($i==0)?do_lang('NEW_FIELD_TITLE'):'')));
		}
		$fields->attach(form_input_list(do_lang_tempcode('ORDER'),do_lang_tempcode('DESCRIPTION_FIELD_ORDER_CLEVER'),$prefix.'order',$order_list));

		// Defines order?
		$radios=form_input_radio_entry($prefix.'defines_order','0',$defines_order==0,do_lang_tempcode('NO'));
		$radios->attach(form_input_radio_entry($prefix.'defines_order','1',$defines_order==1,do_lang_tempcode('ASCENDING')));
		$radios->attach(form_input_radio_entry($prefix.'defines_order','2',$defines_order==2,do_lang_tempcode('DESCENDING')));
		$fields->attach(form_input_radio(do_lang_tempcode('DEFINES_ORDER'),do_lang_tempcode('DESCRIPTION_DEFINES_ORDER'),$radios));

		if ($first_field)
		{
			$hidden->attach(form_input_hidden($prefix.'visible','1'));
			$hidden->attach(form_input_hidden($prefix.'required','1'));
		} else
		{
			$fields->attach(form_input_tick(do_lang_tempcode('VISIBLE'),do_lang_tempcode('DESCRIPTION_VISIBLE'),$prefix.'visible',$visible==1));
			$fields->attach(form_input_tick(do_lang_tempcode('REQUIRED'),do_lang_tempcode('DESCRIPTION_REQUIRED'),$prefix.'required',$required==1));
		}
		$fields->attach(form_input_tick(do_lang_tempcode('SEARCHABLE'),do_lang_tempcode('DESCRIPTION_SEARCHABLE'),$prefix.'searchable',$searchable==1));
		$fields->attach(form_input_tick(do_lang_tempcode('PUT_IN_CATEGORY'),do_lang_tempcode('DESCRIPTION_PUT_IN_CATEGORY'),$prefix.'put_in_category',$put_in_category==1));
		$fields->attach(form_input_tick(do_lang_tempcode('PUT_IN_SEARCH'),do_lang_tempcode('DESCRIPTION_PUT_IN_SEARCH'),$prefix.'put_in_search',$put_in_search==1));

		return array($fields,$hidden);
	}

	/**
	 * Standard aed_module edit form filler.
	 *
	 * @param  ID_TEXT		The entry being edited
	 * @return tempcode		The edit form
	 */
	function fill_in_edit_form($catalogue_name)
	{
		$rows=$GLOBALS['SITE_DB']->query_select('catalogues',array('*'),array('c_name'=>$catalogue_name));
		if (!array_key_exists(0,$rows))
		{
			warn_exit(do_lang_tempcode('MISSING_RESOURCE'));
		}
		$myrow=$rows[0];

		$title=get_translated_text($myrow['c_title']);
		$description=get_translated_text($myrow['c_description']);

		$c_owner	=	$GLOBALS['FORUM_DRIVER']->get_username($myrow['c_owner']);

		return $this->get_form_fields($catalogue_name,$title,$description,$myrow['c_display_type'],$myrow['c_is_tree'],$myrow['c_notes'],$myrow['c_submit_points'],$myrow['c_ecommerce'],$myrow['c_send_view_reports'],$c_owner);
	}

	/**
	 * Standard aed_module add actualiser.
	 *
	 * @return ID_TEXT		The entry added
	 */
	function add_actualisation()
	{
		$name=post_param('name');
		$title=post_param('title');
		$description=post_param('description');
		$display_type=post_param_integer('display_type');
		$is_tree=post_param_integer('is_tree',0);
		$this->is_tree_catalogue=($is_tree==1);
		$notes=post_param('notes');
		$submit_points=post_param_integer('submit_points',0);
		$cat_tab=post_param_integer('cat_tab',0);
		$ecommerce=post_param_integer('ecommerce',0);
		$send_view_reports=post_param('send_view_reports');
		$catalogue_owner_name=post_param('catalogue_owner',NULL);
		$c_owner=$GLOBALS['FORUM_DB']->query_value_null_ok('f_members','id',array('m_username'=>$catalogue_owner_name));

		// What fields do we have?
		$new=array();
		foreach ($_POST as $key=>$val)
		{
			if (!is_string($val)) continue;

			if (get_magic_quotes_gpc()) $val=stripslashes($val);

			$matches=array();
			if (preg_match('#new_field_(\d*)_(.*)#A',$key,$matches)!=0)
			{
				$new[$matches[1]][$matches[2]]=$val;
			}
		}
		$num_fields=0;
		foreach ($new as $field)
		{
			if ($field['name']!='') $num_fields++;
		}
		if ($num_fields==0) warn_exit(do_lang_tempcode('NO_FIELDS'));

		$category_id=actual_add_catalogue($name,$title,$description,$display_type,$is_tree,$notes,$submit_points,$ecommerce,$send_view_reports,$c_owner);
		$this->set_permissions($name);
		if (!is_null($category_id)) $GLOBALS['MODULE_CMS_CATALOGUES']->cat_aed_module->set_permissions($category_id);

		// Now onto the fields
		foreach ($new as $field)
		{
			if (!array_key_exists('default',$field)) warn_exit(do_lang_tempcode('IMPROPERLY_FILLED_IN'));
			if (!array_key_exists('description',$field)) warn_exit(do_lang_tempcode('IMPROPERLY_FILLED_IN'));
			if (!array_key_exists('name',$field)) warn_exit(do_lang_tempcode('IMPROPERLY_FILLED_IN'));
			if (!array_key_exists('order',$field)) warn_exit(do_lang_tempcode('IMPROPERLY_FILLED_IN'));
			if ((!array_key_exists('type',$field)) || ($field['type']=='')) warn_exit(do_lang_tempcode('IMPROPERLY_FILLED_IN'));

			if ($field['order']=='') $field['order']=0; else $field['order']=intval($field['order']);
			$defines_order=array_key_exists('defines_order',$field)?intval($field['defines_order']):0;
			$visible=array_key_exists('visible',$field)?intval($field['visible']):0;
			$searchable=array_key_exists('searchable',$field)?intval($field['searchable']):0;
			$required=array_key_exists('required',$field)?intval($field['required']):0;
			$put_in_category=array_key_exists('put_in_category',$field)?intval($field['put_in_category']):0;
			$put_in_search=array_key_exists('put_in_search',$field)?intval($field['put_in_search']):0;
			if ($field['name']!='') actual_add_catalogue_field($name,$field['name'],$field['description'],$field['type'],$field['order'],$defines_order,$visible,$searchable,$field['default'],$required,$put_in_category,$put_in_search);
		}

		// Auto-fill feature
		$auto_fill=post_param('auto_fill');
		if ($auto_fill!='')
		{
			$categories=array();
			$to_do=explode('|',$auto_fill);
			foreach ($to_do as $doing)
			{
				$bits=explode('\\',$doing);
				$parent_id=$category_id;
				foreach ($bits as $bit)
				{
					if (array_key_exists($bit,$categories))
					{
						if (!is_null($parent_id)) $parent_id=$categories[$bit];
					} else
					{
						$_parent_id=actual_add_catalogue_category($name,$bit,'','',$parent_id,'');
						if (!is_null($parent_id)) $parent_id=$_parent_id;
						require_code('permissions2');
						set_category_permissions_from_environment('catalogues_category',strval($parent_id),$this->permission_page);
						$categories[$bit]=$parent_id;
					}
				}
			}
		}

		if ($is_tree==0) $this->do_next_description=paragraph(do_lang_tempcode('SUGGEST_ADD_CATEGORY_NEXT'));

		return $name;
	}

	/**
	 * Standard aed_module edit actualiser.
	 *
	 * @param  ID_TEXT		The entry being edited
	 */
	function edit_actualisation($old_name)
	{
		$name=post_param('name',$old_name);
		$title=post_param('title',STRING_MAGIC_NULL);
		$description=post_param('description',STRING_MAGIC_NULL);
		$display_type=post_param_integer('display_type',fractional_edit()?INTEGER_MAGIC_NULL:0);
		$notes=post_param('notes',STRING_MAGIC_NULL);
		$submit_points=post_param_integer('submit_points',fractional_edit()?INTEGER_MAGIC_NULL:0);
		$ecommerce=post_param_integer('ecommerce',fractional_edit()?INTEGER_MAGIC_NULL:0);
		$send_view_reports=post_param('send_view_reports');
		$catalogue_owner_name=post_param('catalogue_owner',NULL);
		$c_owner=$GLOBALS['FORUM_DB']->query_value_null_ok('f_members','id',array('m_username'=>$catalogue_owner_name));
		if (!fractional_edit())
		{
			if (post_param_integer('reset_category_permissions',0)==1)
			{
				$rows=$GLOBALS['SITE_DB']->query_select('catalogue_categories',array('id'),array('c_name'=>$name));
				foreach ($rows as $row)
				{
					$this->set_permissions($row['id']);
				}
			}
		}

		$is_tree=$GLOBALS['SITE_DB']->query_value_null_ok('catalogues','c_is_tree',array('c_name'=>$old_name));
		if (is_null($is_tree)) warn_exit(do_lang_tempcode('MISSING_RESOURCE'));
		$this->is_tree_catalogue=($is_tree==1);

		if (!fractional_edit())
		{
			// What fields do we have?
			$old=array();
			$new=array();
			foreach ($_POST as $key=>$val)
			{
				if (!is_string($val)) continue;

				if (get_magic_quotes_gpc()) $val=stripslashes($val);

				$matches=array();
				if (preg_match('#new\_field\_(\d+)\_(.*)#A',$key,$matches)!=0)
				{
					$new[$matches[1]][$matches[2]]=$val;
				}
				elseif (preg_match('#existing\_field\_(\d+)\_(.*)#A',$key,$matches)!=0)
				{
					$old[$matches[1]][$matches[2]]=$val;
				}
			}
			$num_fields=0;
			foreach ($new as $field)
			{
				if ($field['name']!='') $num_fields++;
			}
			foreach ($old as $field)
			{
				if (!((array_key_exists('delete',$field)) && ($field['delete']=='1'))) $num_fields++;
			}
			if ($num_fields==0) warn_exit(do_lang_tempcode('NO_FIELDS'));
		}

		actual_edit_catalogue($old_name,$name,$title,$description,$display_type,$notes,$submit_points,$ecommerce,$send_view_reports,$c_owner);

		$this->new_id=$name;

		if (!fractional_edit())
		{
			$this->set_permissions($name);
		}

		if (!fractional_edit())
		{
			// Now onto the fields
			//  First we must rationalise the ordering
			$o=0;
			$orderings=array();
			foreach ($new as $current)
			{
				if (!array_key_exists('default',$current)) warn_exit(do_lang_tempcode('IMPROPERLY_FILLED_IN'));
				if (!array_key_exists('description',$current)) warn_exit(do_lang_tempcode('IMPROPERLY_FILLED_IN'));
				if (!array_key_exists('name',$current)) warn_exit(do_lang_tempcode('IMPROPERLY_FILLED_IN'));
				if (!array_key_exists('order',$current)) warn_exit(do_lang_tempcode('IMPROPERLY_FILLED_IN'));
				if ((!array_key_exists('type',$current)) || ($current['type']=='')) warn_exit(do_lang_tempcode('IMPROPERLY_FILLED_IN'));

				if ($current['name']!='')
				{
					if ((!array_key_exists('order',$current)) || ($current['order']=='')) $current['order']=strval(count($new)+count($old));
					$orderings['new_'.strval($o)]=$current['order'];
				}
				$o++;
			}
			$o=0;
			foreach ($old as $current)
			{
				if (!((array_key_exists('delete',$current)) && ($current['delete']=='1'))) // If not deleting
				{
					if ((!array_key_exists('order',$current)) || ($current['order']=='')) $current['order']=strval(count($new)+count($old));
					$orderings['old_'.strval($o)]=$current['order'];
				}
				$o++;
			}
			asort($orderings);
			//  Now add/edit them
			$o=0;
			foreach ($new as $field)
			{
				$p=0;
				foreach (array_keys($orderings) as $key)
				{
					if ($key=='new_'.strval($o)) $order=$p;
					$p++;
				}
				$defines_order=array_key_exists('defines_order',$field)?intval($field['defines_order']):0;
				$visible=array_key_exists('visible',$field)?intval($field['visible']):0;
				$searchable=array_key_exists('searchable',$field)?intval($field['searchable']):0;
				$required=array_key_exists('required',$field)?intval($field['required']):0;
				$put_in_category=array_key_exists('put_in_category',$field)?intval($field['put_in_category']):0;
				$put_in_search=array_key_exists('put_in_search',$field)?intval($field['put_in_search']):0;
				if ($field['name']!='') actual_add_catalogue_field($name,$field['name'],$field['description'],$field['type'],$order,$defines_order,$visible,$searchable,$field['default'],$required,$put_in_category,$put_in_search);
				$o++;
			}
			$o=0;
			foreach ($old as $id=>$field)
			{
				if ((array_key_exists('delete',$field)) && ($field['delete']=='1'))
				{
					actual_delete_catalogue_field($id);
				} else
				{
					$p=0;
					foreach (array_keys($orderings) as $key)
					{
						if ($key=='old_'.strval($o)) $order=$p;
						$p++;
					}
					$defines_order=array_key_exists('defines_order',$field)?$field['defines_order']:0;
					$visible=array_key_exists('visible',$field)?$field['visible']:0;
					$searchable=array_key_exists('searchable',$field)?$field['searchable']:0;
					$required=array_key_exists('required',$field)?$field['required']:0;
					$put_in_category=array_key_exists('put_in_category',$field)?$field['put_in_category']:0;
					$put_in_search=array_key_exists('put_in_search',$field)?$field['put_in_search']:0;
					$field_type=array_key_exists('type',$field)?$field['type']:NULL;
					actual_edit_catalogue_field($id,$name,$field['name'],$field['description'],$order,$defines_order,$visible,$searchable,$field['default'],$required,$put_in_category,$put_in_search,$field_type);
				}
				$o++;
			}
		}
	}

	/**
	 * Standard aed_module delete actualiser.
	 *
	 * @param  ID_TEXT		The entry being deleted
	 */
	function delete_actualisation($id)
	{
		actual_delete_catalogue($id);
	}

	/**
	 * The do-next manager for after catalogue content management.
	 *
	 * @param  tempcode		The title (output of get_page_title)
	 * @param  tempcode		Some description to show, saying what happened
	 * @param  ?ID_TEXT		The catalogue name we were working with (NULL: deleted)
	 * @return tempcode		The UI
	 */
	function do_next_manager($title,$description,$name)
	{
		breadcrumb_set_self(do_lang_tempcode('DONE'));

		if (!is_null($name))
		{
			$cat_count=$GLOBALS['SITE_DB']->query_value('catalogue_categories','COUNT(*)',array('c_name'=>$name));
			$has_categories=($cat_count!=0);
		} else $has_categories=false;

		require_code('templates_donext');
		return do_next_manager($title,$description,
					NULL,
					NULL,
					/*		TYPED-ORDERED LIST OF 'LINKS'		*/
					/*	 page	 params				  zone	  */
					(is_null($name)||(!$has_categories))?NULL:array('_SELF',array('type'=>'add_entry','catalogue_name'=>$name),'_SELF'),								  // Add one
					NULL,				 // Edit this
					(is_null($name)||(!$has_categories))?NULL:(has_specific_permission(get_member(),'edit_own_midrange_content','cms_catalogues')?array('_SELF',array('type'=>'edit_entry','catalogue_name'=>$name),'_SELF'):NULL), // Edit one
					NULL,																						// View this
					NULL,																						// View archive
					NULL,																						// Add to category
					is_null($name)?NULL:array('_SELF',array('type'=>'add_category','catalogue_name'=>$name),'_SELF'), // Add one category
					(is_null($name)||(!$has_categories))?NULL:array('_SELF',array('type'=>'edit_category','catalogue_name'=>$name),'_SELF'),// Edit one category
					NULL,																						// Edit this category
					NULL,																						// View this category
					/*	  SPECIALLY TYPED 'LINKS'				  */
					array(),
					array(),
					array(
						/*	 type							  page	 params													 zone	  */
						array('add_one_catalogue',array('_SELF',array('type'=>'add_catalogue'),'_SELF')),
						is_null($name)?NULL:array('edit_this_catalogue',array('_SELF',array('type'=>'_edit_catalogue','id'=>$name),'_SELF')),
						array('edit_one_catalogue',array('_SELF',array('type'=>'edit_catalogue'),'_SELF')),
						is_null($name)?NULL:array('view_this',array('catalogues',$this->is_tree_catalogue?array('type'=>'category','catalogue_name'=>$name):array('type'=>'index','id'=>$name),get_module_zone('catalogues')))
					),
					do_lang('MANAGE_CATALOGUES')
		);
	}
}


