<?php /*

 ocPortal 
 Copyright (c) ocProducts, 2004-2009

 See text/en/licence.txt for full licencing information.

*/

/**
 * @license		http://opensource.org/licenses/cpal_1.0 Common Public Attribution License
 * @copyright	ocProducts Ltd
 * @package		ecommerce
 */

class Hook_mpi
{
	/**
	* Get some title/description about payment method
	*
	* @return SHORT_TEXT		Title/Description about paypal
	*/
	function info()
	{
		return do_lang('EPDQ_DESCRIPTION');
	}

	/**
	 * Function to install barclays config options
	 */
	function install_config()
	{
		add_config_option('EPDQ_ENABLE','epdq_enable','special_group__tick_control__epdq__tick','return \'0\';','ECOMMERCE','PAYMENT_GATEWAY');
		add_config_option('EPDQ_API_ID','epdq_api_id','special_group__tick_control__epdqe__line','return \'\';','ECOMMERCE','PAYMENT_GATEWAY');
		add_config_option('EPDQ_API_LOGIN','epdq_api_login','special_group__tick_control__epdq__line','return \'\';','ECOMMERCE','PAYMENT_GATEWAY');
		add_config_option('EPDQ_API_PASSWORD','epdq_api_password','special_group__tick_control__epdq__line','return \'\';','ECOMMERCE','PAYMENT_GATEWAY');
		add_config_option('EPDQ_API_HOST','epdq_api_host','special_group__tick_control__epdq__line','return \'\';','ECOMMERCE','PAYMENT_GATEWAY');
		add_config_option('EPDQ_API_PORT','epdq_api_port','special_group__tick_control__epdq__line','return \'\';','ECOMMERCE','PAYMENT_GATEWAY');
	}

	/**
	 * Function to unistall barclays config options
	 */
	function uninstall_config()
	{
		delete_config_option('epdq_enable');
		delete_config_option('epdq_api_id');
		delete_config_option('epdq_api_login');
		delete_config_option('epdq_api_password');
		delete_config_option('epdq_api_host');
		delete_config_option('epdq_api_port');
	}

	/**
	 * Find the payment gateway enabled or not
	 *
	 * @return BINARY Enabled/Disabled
	 */
	function is_enabled()
	{
		return get_option('epdq_enable');
	}

	/**
	 * Get gateway CPFs
	 *
	 * @return array
	 */
	function get_gateway_cpfs()
	{
		return array('epdq_enable','epdq_api_id','epdq_api_login','epdq_api_password','epdq_api_host','epdq_api_port');
	}

	/**
	 * Whether the cpf is of this gateway
	 *
	 * @param SHORT_TEXT The cpf
	 * @return bool
	 */
	function is_gateway_cpf($cpf)
	{
		return in_array($cpf,$this->get_gateway_cpfs());
	}

	/**
	 * Show option button.
	 *
	 *	@return tempcode	Form for barclays option
	 */
	function show_option_button()
	{
		//return do_template('EPDQ_OPTION_BUTTON',array('URL'=>get_self_url()));
	}

	/**
	 * Get form fields for local payment.
	 *
	 * @param  ?ID_TEXT		The transaction ID (NULL: auto-generate)
	 * @param  ID_TEXT		The purchase ID
	 * @param  SHORT_TEXT	The item name
	 * @param  SHORT_TEXT	The amount
	 * @param  ?integer		The length (NULL: not a subscription)
	 * @param  ID_TEXT		The length units
	 * @return tempcode		The form fields
	 */
	function get_transaction_form_fields($trans_id,$purchase_id,$item_name,$amount,$length,$length_units)
	{
		require_code('ecommerce');
 		if (is_null($trans_id))	$trans_id=$this->generate_trans_id();

		list($card_det,$hidden)	=	get_general_transaction_form_fields($trans_id,$purchase_id,$item_name,$amount,$length,$length_units,$this);

		/*	General transaction fields returned from ecommerce api, attaching the details to the form.
			The fields can add/remove here, depending on the customised payment gateway features.
		*/
		$fields	=	new ocp_tempcode();
		foreach($card_det as $field)
		{
			$fields->attach($field);
		}

		$hidden->attach(form_input_hidden('payment_gateway','mpi'));

		$verified_account_logo	=	$this->get_confidence_logos();
		return array($fields,$hidden,$verified_account_logo);
	}

	/**
	 * Whether the member could enter the subscription start date in the payment form
	 *
	 * @return boolean
	 */
	function is_allow_subscription_start_date()
	{
		return true;
	}

	/**
	 * Whether to allow shipping detail in the local payment form
	 *
	 * @return boolean
	 */
	function is_allow_shipping()
	{
		return true;
	}

	/**
	 * Encrypt the fields
	 *
	 * @param	?ID_TEXT 	Member Id
	 * @param	?STRING		Curency Code
	 * @param	?integer		No of times of payment when in subscription
	 * @param	?STRING		Time units of payments when in subscription
	 * @return	STRING		Encrypted string
	 */
	function epdq_encrypt($member_id=NULL,$currency = NULL,$length=NULL,$length_units=NULL)
	{

		//got from : http://www.barclaycard.co.uk/business/documents/zips/Scripts.zip
		
		require_code('files');
		$server="secure2.epdq.co.uk";
		$url="/cgi-bin/CcxBarclaysEpdqEncTool.e";

		$access_detail=$this->get_access_detail($member_id);
		$loginid=$access_detail[0];
		$password=$access_detail[2];

		$post_params = array();
		$post_params['clientid'] = $loginid;
		$post_params['password'] = $password;
		$post_params['oid'] =$purchase_id ;
		$post_params['chargetype'] = 'Auth';
		$post_params['currencycode'] = $this->get_currency_code($currency);

		if(!is_null($length) && !is_null($length_units))
		{
			if($length_units=='m')	$length_units	=	'M';
			if($length_units=='d')	$length_units	=	'D';
			if($length_units=='y')
			{
				$length_units	=	'M';
				$freq			=	12;
			}
			else
				$freq = 1;
			$post_params['orderfrequencycycle'] = $length_units;
			$post_params['orderfrequencyinterval'] = $freq;
			$post_params['ordertype'] = 0;
			$post_params['totalnumberpayments'] = $length;
			
		}

		$post_params['total'] = $amount;
		
		$url = $server.$url;
		$response = http_download_file($url,NULL,true,false,'ocPortal',$post_params);


		#perform the HTTP Post
		//$response = pullpage( $server,$url,$params );

		#split the response into separate lines
		$response_lines=explode("\n",$response);

		#for each line in the response check for the presence of the string 'epdqdata'
		#this line contains the encrypted string
		$response_line_count=count($response_lines);
		for ($i=0;$i<$response_line_count;$i++)
		{
			if (preg_match('/epdqdata/',$response_lines[$i]))
			{
				  $strEPDQ=$response_lines[$i];
			}
		}
		return $strEPDQ;
	}

	/**
	 * Make a transaction (payment) button.
	 *
	 * @param  ID_TEXT		The product codename.
	 * @param  SHORT_TEXT	The human-readable product title.
	 * @param  ID_TEXT		The purchase ID.
	 * @param  float			A transaction amount.
	 * @param  ID_TEXT		The currency to use.
	 * @param  ID_TEXT		Member Id
	 * @return tempcode		The button
	 */
	function make_transaction_button($product,$item_name,$purchase_id,$amount,$currency='',$member_id=NULL)
	{
		$encrypted_str = $this->epdq_encrypt($member_id, $currency);

		$return_url = get_option('epdq_url_return');

		$object=find_product($product);
		$product_det=$object->product_id;
		$module=get_param('page','purchase');

		$form_url=$this->get_form_url();
		$return_url=build_url(array('page'=>$module,'type'=>'finish','from'=>'epdq'),get_module_zone($module),NULL,false,false,true);
		$return_url=$return_url->evaluate();

		if(get_option('ecommerce_test_mode')=="1" && get_option('relay_url_return')!='')
			$return_url	=	get_option('relay_url_return')."?from=epdq&to=".$return_url;

		return do_template('ECOM_BUTTON_VIA_EPDQ',array('FORM_URL'=>$form_url,'RELAY_URL'=>$return_url,'STR_EPDQ'=>$encrypted_str));
	}

	/**
	 * Make a subscription (payment) button.
	 *
	 * @param  ID_TEXT		The product codename.
	 * @param  SHORT_TEXT	The human-readable product title.
	 * @param  ID_TEXT		The purchase ID.
	 * @param  float			A transaction amount.
	 * @param  integer		The subscription length in the units.
	 * @param  ID_TEXT		The length units.
	 * @set    d w m y
	 * @param  ID_TEXT		The currency to use.
	 * @return tempcode		The button
	 */
	function make_subscription_button($product,$item_name,$purchase_id,$amount,$length,$length_units,$currency)
	{
		$encrypted_str = $this->epdq_encrypt($member_id, $currency,$length,$length_units);

		$return_url = get_option('epdq_url_return');

		$object=find_product($product);
		$product_det=$object->product_id;
		$module=get_param('page','purchase');

		$form_url=$this->get_form_url();
		$return_url=build_url(array('page'=>$module,'type'=>'finish','from'=>'epdq'),get_module_zone($module),NULL,false,false,true);
		$return_url=$return_url->evaluate();

		if(get_option('ecommerce_test_mode')=="1" && get_option('relay_url_return')!='')
			$return_url	=	get_option('relay_url_return')."?from=epdq&to=".$return_url;


		return do_template('ECOM_BUTTON_VIA_EPDQ',array('FORM_URL'=>$form_url,'RELAY_URL'=>$return_url,'STR_EPDQ'=>$encrypted_str));
	}

	/**
	 * Calculate fingerprint.
	 *
	 * @param  SHORT_TEXT	Login Id.
	 * @param  SHORT_TEXT	Transaction key
	 * @param  float			A transaction amount.
	 * @param  integer		Timestamp
	 * @param  ID_TEXT		The currency to use.
	 * @return ARRAY
	 */
	function get_finger_print($loginid,$x_tran_key,$amount,$sequence,$tstamp,$currency = "")
	{
		return $this->hmac($x_tran_key,$loginid . "^" . $sequence . "^" . $tstamp . "^" . $amount . "^" . $currency);
	}

	/**
	 * Inserts the hidden variables in the HTML FORM required for SIM
	 * Invokes hmac function to calculate fingerprint.
	 *
	 * @param  SHORT_TEXT	Transaction key
	 * @param  SHORT_TEXT	Data
	 * @return TEXT
	 */
	function hmac($key,$data)
	{
	   // RFC 2104 HMAC implementation for php.
	   // Creates an md5 HMAC.
	   // Eliminates the need to install mhash to compute a HMAC
	   // Hacked by Lance Rushing
	   $b = 64; // byte length for md5
	   if (strlen($key) > $b)
	   {
	       $key = pack("H*",md5($key));
	   }
	   $key  = str_pad($key, $b, chr(0x00));
	   $ipad = str_pad('', $b, chr(0x36));
	   $opad = str_pad('', $b, chr(0x5c));
	   $k_ipad = $key ^ $ipad ;
	   $k_opad = $key ^ $opad;

	   return md5($k_opad  . pack("H*",md5($k_ipad . $data)));
	}

	/**
	 * Get the form URL.
	 *
	 * @return URLPATH The form url.
	 */
	function get_form_url()
	{
		return "https://secure2.epdq.co.uk/cgi-bin/CcxBarclaysEpdq.e";
	}

	/**
	 * Generate a transaction ID.
	 *
	 * @return string			A transaction ID.
	 */
	function generate_trans_id()
	{
		return md5(uniqid(strval((mt_rand(0,32000))),true));
	}

	/**
	 * Perform a transaction.
	 *
	 * @param  ?ID_TEXT		The transaction ID (NULL: generate one)
	 * @param  SHORT_TEXT	Cardholder first name
	 * @param  SHORT_TEXT	Cardholder last name
	 * @param  SHORT_TEXT	Card number
	 * @param  SHORT_TEXT	Transaction amount
	 * @param  SHORT_TEXT	Card Expiry date
	 * @param  integer		Card Issue number
	 * @param  SHORT_TEXT	Card Start date
	 * @param  SHORT_TEXT	Card Type
	 * @set    "Visa" "Master Card" "Switch" "UK Maestro" "Maestro" "Solo" "Delta" "American Express" "Diners Card" "JCB"
	 * @param  SHORT_TEXT	Card CV2 number (security number)
	 * @param  ?integer		The subscription length in the units. (NULL: not a subscription)
	 * @param  ?ID_TEXT		The length units. (NULL: not a subscription)
	 * @set    d w m y
	 * @param  SHORT_TEXT	Description for the transaction
	 * @param  ?MEMBER		Member making the transaction (NULL: current member)
	 * @param  SHORT_TEXT	Address for card
	 * @param  SHORT_TEXT	City for card
	 * @param  SHORT_TEXT	State for card
	 * @param  SHORT_TEXT	Zip/post-code for card
	 * @param  SHORT_TEXT	Country for card
	 * @return array			A tuple: success (boolean), trans-id (string), message (string), raw message (string)
	 */
	function do_transaction($trans_id,$first_name,$last_name,$card_number,$amount,$expiry_date,$issue_number,$start_date,$card_type,$cv2,$length=NULL,$length_units=NULL,$description='',$member_id=NULL,$address='',$city='',$state='',$zip='',$country='',$shipping_first_name='',$shipping_last_name='',$shipping_address='',$shipping_city='',$shipping_state='',$shipping_zip='',$shipping_country='',$total_occurrences=9999,$trial_occurrences=0,$trial_amount=0)
	{
		$card_number  = str_replace('-','',str_replace(' ','',$card_number));
		$purchase_id  =	post_param_integer('customfld1','-1');
		$description  = ($purchase_id != '-1') ? $purchase_id : $description;
		$expiry_month = substr($expiry_date,5,2);
		$expiry_year  = substr($expiry_date,0,4);

      if (!is_null($length))
		{
			warn_exit(do_lang_tempcode('NO_MPI_SUBSCRIPTION'));
		}
		else
		{
         //Set AIM Parameters
         $this->set_mpi_parameters($card_number,$cv2,$expiry_month,$expiry_year,$description,$amount,$card_type,$first_name,$last_name,$address,$city,$state,$zip,$country,$shipping_first_name,$shipping_last_name,$shipping_address,$shipping_city,$shipping_state,$shipping_zip,$shipping_country,$member_id);

         require_code('files');
			$response_data = http_download_file($this->url,NULL,true,false,'ocPortal',$this->api_parameters,NULL,NULL,NULL,NULL,NULL,NULL,NULL,12.0);

			$response_result = explode($this->api_parameters['x_delim_char'],$response_data);

         /*Response Detail
         $response_result[0] = Response Code. 1=Approved, 2=Declined, 3=Error
         $response_result[1] = Response Subcode.A code used by the system for internal transaction tracking.
			$response_result[2] = Response Reason Code.A code representing more details about the result of the transaction.
			$response_result[3] = Response Reason Text.Brief description of the result, which corresponds with the Response Reason Code.
			$response_result[6] = Transaction ID*/

			if (!array_key_exists(1,$response_result)) $response_result[1]='';
			if (!array_key_exists(2,$response_result)) $response_result[2]='';
			if (!array_key_exists(3,$response_result)) $response_result[3]='';
			if (!array_key_exists(6,$response_result)) $response_result[6]='';

         $success = ($response_result[0] == 1) ? true : false;
         $code = ($success) ? $response_result[6] : $response_result[2];
         $response_reason_text = ($success) ? 'success' : $response_result[3];
         $result = array($success,$code,$response_result[1],$response_reason_text);
      }

		if(addon_installed('shopping'))
		{
			$this->store_shipping_address($purchase_id);
		}

		return $result;
	}

	/**
	 * Find a transaction fee from a transaction amount. Regular fees aren't taken into account.
	 *
	 * @param  float	A transaction amount.
	 * @return float	The fee
	 */
	function get_transaction_fee($amount)
	{
		//return round(0.30+0.0212*$amount,2);
		return round(floatval(get_option('epdq_transaction_flat_cost'))+floatval(get_option('epdq_transaction_percentage_cost'))*$amount,2);
	}

   /**
	 * Find whether the hook auto-cancels (if it does, auto cancel the given trans-id).
	 *
	 * @param ID Subscription Id
	 * @param ID Member Id (NULL: Admin)
	 * @return ?boolean	True: yes. False: no. (NULL: cancels via a user-URL-directioning)
	 */
   function auto_cancel($subscription_id,$member_id=NULL) //PENDING
   {
   	$temp	= $GLOBALS['SITE_DB']->query_value('subscriptions','s_auto_fund_key',array('id'=>$subscription_id));
   	$data	= unserialize($temp);
		$authorize_subscription_id	= $data['id'];

      $this->set_cancelation_api_parameters($authorize_subscription_id,$member_id);

      $response = http_download_file($this->url,NULL,true,false,'ocPortal',array($this->api_parameters),NULL,NULL,NULL,NULL,NULL,NULL,NULL,12.0,true);

      if(!is_null($response))
      {
         $response_result = $this->parse_return($response);
         $success = ($response_result[0] == 'OK') ? true : false;
         $result  = array($success,$response_result[1],$response_result[2],$response_result[3]);
      }
      else
      {
      	$success = false;
         $result  = array(false,NULL,NULL,NULL);
      }

      return $success;
   }

  /**
	 * Helper function for parsing response
	 */
   function substring_between($haystack,$start,$end)
   {
      if (strpos($haystack,$start) === false || strpos($haystack,$end) === false)
      {
         return false;
      }
      else
      {
         $start_position = strpos($haystack,$start)+strlen($start);
         $end_position   = strpos($haystack,$end);

         return substr($haystack,$start_position,$end_position-$start_position);
      }
   }

	/**
	 * Store shipping address for orders
	 *
	 * @param  AUTO_LINK		Order id
	 * @return ?mixed			Address id (NULL : No address record found)
	 */
	function store_shipping_address($order_id)
	{
		//RETURNED FROM PAYMENT GATEWAY AS XML, CONTAINS SHIPPING DETAILS BUT NOT ORDER_ID. CONFIRM AND CONTINUE	
		/*
		if(is_null($GLOBALS['SITE_DB']->query_value_null_ok('shopping_order_addresses','id',array('order_id'=>$order_id))))
		{
			$shipping_address['order_id']			=	$order_id;
			$shipping_address['address_name']	=	post_param('ship_first_name','')." ".post_param('ship_last_name','');
			$shipping_address['address_street']	=	post_param('ship_addr_1','');
			$shipping_address['address_city']	=	post_param('ship_city','');
			$shipping_address['address_state']	=	post_param('ship_state','');
			$shipping_address['address_zip']		=	post_param('ship_post_code','');
			$shipping_address['address_country']=	post_param('shipping_country','');
			$shipping_address['receiver_email']	=	'';//post_param('receiver_email','');

			$cpfs_to_save = array
										(
											'ocp_shipping_firstname' => post_param('ship_first_name',''),
											'ocp_shipping_lastname'	 => post_param('ship_last_name',''),
											'ocp_shipping_street'	 => post_param('ship_addr_1',''),
											'ocp_shipping_country'   => post_param('shipping_country',''),
											'ocp_shipping_state'		 => post_param('ship_state',''),
											'ocp_shipping_city'	    => post_param('ship_city',''),
											'ocp_shipping_zip'		 => post_param('ship_post_code','')
										);
			set_ocp_cpfs($cpfs_to_save);

			return $GLOBALS['SITE_DB']->query_insert('shopping_order_addresses',$shipping_address,true);
		}*/
	}


	/**
	 * This function defines the parameters needed to make an AIM call.
	 *
	 * @param NUMBER CREDIT CARD NUMBER (only numbers)
	 * @param NUMBER CREDIT CARD EXPIRY MONTH (MM)
	 * @param NUMBER CREDIT CARD EXPIRY YEAR (YYYY)
	 * @param SHORT_TEXT SHORT DESCRIPTION ABOUT THE TRANSACTION
	 * @param SHORT_TEXT Card Type.
	 * @param INTEGER AMOUNT
	 * @param ID MEMBER ID
	 */
	function set_mpi_parameters($card_number,$cv2,$expiry_month,$expiry_year,$description,$amount,$card_type,$first_name=NULL,$last_name=NULL,$address=NULL,$city=NULL,$state=NULL,$zip=NULL,$country=NULL,$shipping_first_name=NULL,$shipping_last_name=NULL,$shipping_address=NULL,$shipping_city=NULL,$shipping_state=NULL,$shipping_zip=NULL,$shipping_country=NULL,$member_id=NULL)
	{

		$access_detail = $this->get_access_detail($member_id);
		$order_id = generate_trans_id();
		$currency_code = $this->get_currency_code();
		$mpi_xml = "<EngineDocList>".
							"<DocVersion>1.0</DocVersion>".
							"<EngineDoc>".
								"<ContentType>OrderFormDoc</ContentType>".
								"<User>".
									"<Name>".$access_detail[1]."</Name>".
									"<Password>".$access_detail[2]."</Password>".
									"<ClientId DataType=\"S32\">".$access_detail[0]."</ClientId>".
								"</User>".
								"<Instructions>".
									"<Pipeline>Payment</Pipeline>".
								"</Instructions>".
								"<OrderFormDoc>".
									"<Mode>P</Mode>".
									"<Id>".$order_id."</Id>".
									"<Consumer>";
		$email = $GLOBALS['FORUM_DRIVER']->get_member_email_address($member_id);
		if($email != '') $mpi_xml .= "<Email>Customer's email address</Email>";

									$mpi_xml .=
										"<BillTo>".
											"<Location>".
												"<Address>";

		 $address_bits=explode(',',$address,3);
		 if ($first_name) $mpi_xml .= "<FirstName>". $first_name . "</FirstName>";
       if ($last_name) $mpi_xml .= "<LastName>". $last_name . "</LastName>";
       if (array_key_exists(0, $address_bits)) $mpi_xml .= "<Street1>" . trim($address_bits[0]) . "</Street1>";
       if (array_key_exists(1, $address_bits)) $mpi_xml .= "<Street2>" . trim($address_bits[1]) . "</Street2>";
       if (array_key_exists(2, $address_bits)) $mpi_xml .= "<Street3>" . trim($address_bits[2]) . "</Street3>";
       if ($city) $mpi_xml .=  "<City>" . $city . "</City>";
       if ($state) $mpi_xml .= "<StateProv>" . $state . "</StateProv>";
       if ($zip) $mpi_xml .= "<PostalCode>" . $zip . "</PostalCode>";
       if ($country) $mpi_xml .= "<Country>" . $country . "</Country>";

										$mpi_xml .=
											"</Address>".
											"</Location>".
										"</BillTo>".
										"<ShipTo>".
											"<Location>".
												"<Address>";
		 $address_bits=explode(',',$shipping_address,3);
       if ($shipping_first_name) $mpi_xml .= "<FirstName>". $shipping_first_name . "</FirstName>";
       if ($shipping_last_name) $mpi_xml .= "<LastName>". $shipping_last_name . "</LastName>";
       if (array_key_exists(0, $address_bits)) $mpi_xml .= "<Street1>" . trim($address_bits[0]) . "</Street1>";
       if (array_key_exists(1, $address_bits)) $mpi_xml .= "<Street2>" . trim($address_bits[1]) . "</Street2>";
       if (array_key_exists(2, $address_bits)) $mpi_xml .= "<Street3>" . trim($address_bits[2]) . "</Street3>";
       if ($shipping_city) $mpi_xml .=  "<City>" . $shipping_city . "</City>";
       if ($shipping_state) $mpi_xml .= "<StateProv>" . $shipping_state . "</StateProv>";
       if ($shipping_zip) $mpi_xml .= "<PostalCode>" . $shipping_zip . "</PostalCode>";
       if ($shipping_country) $mpi_xml .= "<Country>" . $shipping_country . "</Country>";

										$mpi_xml .=
												"</Address>".
											"</Location>".
										"</ShipTo>".
										"<PaymentMech>".
											"<CreditCard>".
												"<Type DataType=\"S32\">".get_card_type_code($card_type)."</Type>".
												"<Number>".$card_number."</Number>".
												"<Expires DataType=\"ExpirationDate\" Locale=\"826\">".$expiry_month."/".$expiry_year."</Expires>".
												"<IssueNum></IssueNum>".
												"<StartDate DataType=\"StartDate\"></StartDate>".
												"<Cvv2Indicator>1</Cvv2Indicator>".
												"<Cvv2Val>".$cv2."</Cvv2Val>".
											"</CreditCard>".
										"</PaymentMech>".
									"</Consumer>".
									"<Transaction>".
										"<Type>Auth</Type>".
										"<CurrentTotals>".
											"<Totals>".
												"<Total DataType=\"Money\" Currency=\"".$currency_code."\">".$amount."</Total>".
											"</Totals>".
										"</CurrentTotals>".
										"<CardholderPresentCode DataType=\"S32\"></CardholderPresentCode>".
										"<PayerSecurityLevel DataType=\"S32\"></PayerSecurityLevel>".
										"<PayerAuthenticationCode></PayerAuthenticationCode>".
										"<PayerTxnId></PayerTxnId>".
									"</Transaction>".
								"</OrderFormDoc>".
							"</EngineDoc>".
						"</EngineDocList>";

	}

   /**
	* This function defines the parameters needed to make a cancelation API call.
	*
	* @param ID Subscription Id
	* @param ID Member Id
	*/
	function set_cancelation_api_parameters($subscription_id,$member_id=NULL) // PENDING
	{
		$access_detail = $this->get_access_detail($member_id);

		if (get_option('ecommerce_test_mode')=='1')
		{
			//URL for test account
			$this->url = "https://apitest.authorize.net/xml/v1/request.api";
		} else
		{
			//URL for live account
			$this->url = "https://api.authorize.net/xml/v1/request.api";
		}

      $this->api_parameters =
        "<ARBCancelSubscriptionRequest xmlns=\"AnetApi/xml/v1/schema/AnetApiSchema.xsd\">".
        "<merchantAuthentication>".
        "<name>" . $access_detail[0] . "</name>".
        "<transactionKey>" . $access_detail[1] . "</transactionKey>".
        "</merchantAuthentication>" .
        "<subscriptionId>" . $subscription_id . "</subscriptionId>".
        "</ARBCancelSubscriptionRequest>";
	}

   /**
	 * Handle IPN's. The function may produce output, which would be returned to the Payment Gateway. The function may do transaction verification.
	 *
	 * @return array	A long tuple of collected data.
	 */
	function handle_transaction()
	{
		require_code('ecommerce');
		require_code('ocf_referral');

		$success         = (post_param('x_response_code','') == 1) ? true : false;
		$response_text   = post_param('x_response_reason_text','');
		$subscription_id = post_param('x_subscription_id','');
		$transaction_id  = post_param('x_trans_id','');
		$purchase_id     = post_param('x_description','');
		$reason_code     = post_param('x_response_reason_code','');
		$amount          = post_param('x_amount',0);
		$currency        = post_param('x_currency_code',get_option('currency'));
		$parent_txn_id   = '';
		$pending_reason  = '';
		$memo				  = '';
		$item_name		  = ($subscription_id) ? find_item_name_of_subscription($purchase_id) : find_item_name_of_order($purchase_id);
		$payment_status  = ($success) ? 'Completed' : 'Failed';
		$txn_id 			  = ($subscription_id) ? $subscription_id : $transaction_id;

		return array($purchase_id,$item_name,$payment_status,$reason_code,$pending_reason,$memo,$amount,$currency,$txn_id,$parent_txn_id);
   }

	function get_confidence_logos()
	{
		require_css('ecommerce');
		$logo1 = find_theme_image('epdq_logo1');
		$logo2 = find_theme_image('epdq_logo2');

		$logo = "<img src='$logo1' /><img src='$logo2' />";

		return do_template('VERIFIED_MERCHANT_LOGO_SCREEN',array('LOGO'=>$logo,));
	}

	/**
	 * Get authorize access detail
	 *
	 * @param ID Member Id
	 * @return array
	 */
	function get_access_detail($member_id=NULL)
	{
		$api_login				=	NULL;
		$api_transaction_key	=	NULL;

		if(!is_null($member_id) && get_option('enable_member_payments'))
		{
			$api_id  			 =	get_ocp_cpf('epdq_api_id',$member_id);
			$api_login  			 =	get_ocp_cpf('epdq_api_login',$member_id);
			$api_password  =	get_ocp_cpf('epdq_password',$member_id);
		}

		if (is_null($api_login) || $api_login=='' || is_null($api_id) || $api_id=='' || is_null($api_password) || $api_password=='')
		{
			$api_id	=	get_option('epdq_api_id');
			$api_login	=	get_option('epdq_api_login');
			$api_password	=	get_option('epdq_transaction_key');
		}

		return array($api_id,$api_login,$api_password);
	}

	/** Function to return local payment method for subscriptions
	 *
	 * @return ID_TEXT	automatic|local_controlled
	 */
	function get_subscription_local_payment_method()
	{
		return get_option('epdq_local_subscription_methods');
	}

	

	/**
	 * Function to return profile ids and types in serialized array format
	 *
	 *	@param	AUTO_LINK	Transaction id
	 * @param	AUTO_LINK 	Parent transaction id (Subscription profile id of a particular subscription)
	 * @return	?SHORT_TEXT	Type, profile id data in serialized string
	 */
	function	map_payment_ids($txn_id,$parent_txn_id)
	{
		$data = array('id'=>$parent_txn_id,'type'=>'arb');
		return serialize($data);
	}

	/**
	 * Make a subscription update button.
	 *
	 * @param  ID_TEXT		The purchase ID.
	 *	@param	ID_TEXT		Serialized data which stores transaction keys
	 * @return tempcode		The button
	 */
	function make_update_button($purchase_id,$auto_found_key=NULL)
	{
		$url	=	build_url(array('page'=>'subscriptions','type'=>'update','id'=>$purchase_id),get_module_zone('subscriptions'));
		$url	=	$url->evaluate();

		return do_template('ECOM_UPDATE_BUTTON_VIA_AUTHORIZE',array('URL'=>$url));
	}

	/**
	 * Get form fields.
	 *
	 * @param ID Subscription Id
	 * @return tempcode The form fields
	 */
	function get_update_form_fields($subscription_id)
	{
		require_code('ecommerce');

		list($card_det,$hidden)	= get_general_update_form_fields($subscription_id,$this);

		/*	General transaction fields returned from ecommerce api, attaching the details to the form.
			The fields can add/remove here, depending on the customised payment gateway features.
		*/
		$fields	=	new ocp_tempcode();
		foreach($card_det as $field)
		{
			$fields->attach($field);
		}

		$verified_account_logo	=	$this->get_confidence_logos();
		return array($fields,$hidden,$verified_account_logo);
	}

	/**
	 * Update subscription
	 *
	 * @param  ID           Subscription Id
	 * @param  SHORT_TEXT	Cardholder first name
	 * @param  SHORT_TEXT	Cardholder last name
	 * @param  SHORT_TEXT	Card number
	 * @param  SHORT_TEXT	Card Type
	 * @param  SHORT_TEXT	Card Expiry date
	 * @param  SHORT_TEXT	Card CV2 number (security number)
	 * @param  SHORT_TEXT	Address for card
	 * @param  SHORT_TEXT	City for card
	 * @param  SHORT_TEXT	State for card
	 * @param  SHORT_TEXT	Zip/post-code for card
	 * @param  SHORT_TEXT	Country for card
	 * @param  ID Member Id (NULL: Admin)
	 * @return ARRAY
	 */
   function update_subscription($subscription_id,$first_name,$last_name,$card_number,$card_type,$expiry_date,$cv2,$address1,$city,$state,$zip,$country,$member_id=NULL)
   {
   	$temp	= $GLOBALS['SITE_DB']->query_value('subscriptions','s_auto_fund_key',array('id'=>$subscription_id));
   	$data	= unserialize($temp);
		$authorize_subscription_id	= $data['id'];

		$expiry_month = substr($expiry_date,5,2);
		$expiry_year  = substr($expiry_date,0,4);
		$card_number  = str_replace('-','',str_replace(' ','',$card_number));

      $this->set_update_api_parameters($authorize_subscription_id,$first_name,$last_name,$card_number,$card_type,$expiry_month,$expiry_year,$cv2,$address1,$city,$state,$zip,$country,$member_id);

      $response = http_download_file($this->url,NULL,true,false,'ocPortal',array($this->api_parameters),NULL,NULL,NULL,NULL,NULL,NULL,NULL,12.0,true);

      if(!is_null($response))
      {
         $response_result = $this->parse_return($response);
         $success = ($response_result[0] == 'OK') ? true : false;
         $result  = array($success,$response_result[1],$response_result[2],$response_result[2]);
      }
      else
      {
         $result  = array(false,NULL,NULL,NULL);
      }

      return $result;
   }

   /**
    * This function defines the parameters needed to make an API call.
	 *
	 * @param  ID           Subscription Id
	 * @param  SHORT_TEXT	Cardholder first name
	 * @param  SHORT_TEXT	Cardholder last name
	 * @param  SHORT_TEXT	Card number
	 * @param  SHORT_TEXT	Card Type
	 * @param  SHORT_TEXT	Card Expiry date
	 * @param  SHORT_TEXT	Card CV2 number (security number)
	 * @param  SHORT_TEXT	Address for card
	 * @param  SHORT_TEXT	City for card
	 * @param  SHORT_TEXT	State for card
	 * @param  SHORT_TEXT	Zip/post-code for card
	 * @param  SHORT_TEXT	Country for card
	 * @param  ID Member Id (NULL: Admin)
	 */
	//STILL PENDING
	function set_update_api_parameters($authorize_subscription_id,$first_name,$last_name,$card_number,$card_type,$expiry_month,$expiry_year,$cv2,$address1,$city,$state,$zip,$country,$member_id=NULL)
	{
		$access_detail = $this->get_access_detail($member_id);

		if (get_option('ecommerce_test_mode')=='1')
		{
			//URL for test account
			$this->url = "https://apitest.authorize.net/xml/v1/request.api";
		} else
		{
			//URL for live account
			$this->url = "https://api.authorize.net/xml/v1/request.api";
		}

      $this->api_parameters =
        "<ARBUpdateSubscriptionRequest xmlns=\"AnetApi/xml/v1/schema/AnetApiSchema.xsd\">".
        "<merchantAuthentication>".
        "<name>" . $access_detail[0] . "</name>".
        "<transactionKey>" . $access_detail[1] . "</transactionKey>".
        "</merchantAuthentication>" .
        "<subscriptionId>" . $authorize_subscription_id . "</subscriptionId>".
        "<subscription>".
        "<payment>".
        "<creditCard>".
        "<cardNumber>" . $card_number ."</cardNumber>".
        "<expirationDate>" . $expiry_year.'-'.$expiry_month . "</expirationDate>".
        "</creditCard>".
        "</payment>".
        "<billTo>";

		 if ($first_name) $this->api_parameters .= "<firstName>". $first_name . "</firstName>";
       if ($last_name) $this->api_parameters .= "<lastName>". $last_name . "</lastName>";
       if ($address1) $this->api_parameters .= "<address>" . $address1 . "</address>";
       if ($city) $this->api_parameters .=  "<city>" . $city . "</city>";
       if ($state) $this->api_parameters .= "<state>" . $state . "</state>";
       if($zip) $this->api_parameters .= "<zip>" . $zip . "</zip>";
       if($country) $this->api_parameters .= "<country>" . $country . "</country>";

       $this->api_parameters .=
        "</billTo>".
        "</subscription>".
        "</ARBUpdateSubscriptionRequest>";
	}

	/**
	* Find whether the hook auto-cancels (if it does, auto cancel the given trans-id).
	*
	* @param ID Subscription Id
	* @param ID Member Id (NULL: Admin)
	* @return ?boolean	True: yes. False: no. (NULL: cancels via a user-URL-directioning)
	*/
	function cancel_subscription($subscription_id,$member_id)
	{
		$this->set_cancelation_api_parameters($subscription_id,$member_id);

		$response = http_download_file($this->url,NULL,true,false,'ocPortal',array($this->api_parameters),NULL,NULL,NULL,NULL,NULL,NULL,NULL,12.0,true);

		if(!is_null($response))
		{
			$response_result = $this->parse_return($response);
			$success = ($response_result[0] == 'OK') ? true : false;
			$result  = array($success,$response_result[1],$response_result[2],$response_result[3]);
		}
		else
		{
			$success = false;
			$result  = array(false,NULL,NULL,NULL);
		}

		return $success;
	}

	/**
	* Get the Currency Code for the MPI method.
	*
	* @param ?STRING	Normal Currency Code.
	* @return INTEGER	New Currency Code
	*/
	function get_currency_code($key=NULL)
	{
		if($key == NULL) $key = get_option('currency');
		$currency_code_map = array();

		$currency_code_map['AED'] = 	784;
		$currency_code_map['AFN'] = 	971;
		$currency_code_map['ALL'] = 	8;
		$currency_code_map['AMD'] = 	51;
		$currency_code_map['ANG'] = 	532;
		$currency_code_map['AOA'] = 	973;
		$currency_code_map['ARS'] = 	32;
		$currency_code_map['AUD'] = 	36;
		$currency_code_map['AWG'] = 	533;
		$currency_code_map['AZN'] = 	944;
		$currency_code_map['BAM'] = 	977;
		$currency_code_map['BBD'] = 	52;
		$currency_code_map['BDT'] = 	50;
		$currency_code_map['BGN'] = 	975;
		$currency_code_map['BHD'] = 	48;
		$currency_code_map['BIF'] = 	108;
		$currency_code_map['BMD'] = 	60;
		$currency_code_map['BND'] = 	96;
		$currency_code_map['BOB'] = 	68;
		$currency_code_map['BOV'] = 	984;
		$currency_code_map['BRL'] = 	986;
		$currency_code_map['BSD'] = 	44;
		$currency_code_map['BTN'] = 	64;
		$currency_code_map['BWP'] = 	72;
		$currency_code_map['BYR'] = 	974;
		$currency_code_map['BZD'] = 	84;
		$currency_code_map['CAD'] = 	124;
		$currency_code_map['CDF'] = 	976;
		$currency_code_map['CHE'] = 	947;
		$currency_code_map['CHF'] = 	756;
		$currency_code_map['CHW'] = 	948;
		$currency_code_map['CLF'] = 	990;
		$currency_code_map['CLP'] = 	152;
		$currency_code_map['CNY'] = 	156;
		$currency_code_map['COP'] = 	170;
		$currency_code_map['COU'] = 	970;
		$currency_code_map['CRC'] = 	188;
		$currency_code_map['CUC'] = 	931;
		$currency_code_map['CUP'] = 	192;
		$currency_code_map['CVE'] = 	132;
		$currency_code_map['CZK'] = 	203;
		$currency_code_map['DJF'] = 	262;
		$currency_code_map['DKK'] = 	208;
		$currency_code_map['DOP'] = 	214;
		$currency_code_map['DZD'] = 	12;
		$currency_code_map['EEK'] = 	233;
		$currency_code_map['EGP'] = 	818;
		$currency_code_map['ERN'] = 	232;
		$currency_code_map['ETB'] = 	230;
		$currency_code_map['EUR'] = 	978;
		$currency_code_map['FJD'] = 	242;
		$currency_code_map['FKP'] = 	238;
		$currency_code_map['GBP'] = 	826;
		$currency_code_map['GEL'] = 	981;
		$currency_code_map['GHS'] = 	936;
		$currency_code_map['GIP'] = 	292;
		$currency_code_map['GMD'] = 	270;
		$currency_code_map['GNF'] = 	324;
		$currency_code_map['GTQ'] = 	320;
		$currency_code_map['GYD'] = 	328;
		$currency_code_map['HKD'] = 	344;
		$currency_code_map['HNL'] = 	340;
		$currency_code_map['HRK'] = 	191;
		$currency_code_map['HTG'] = 	332;
		$currency_code_map['HUF'] = 	348;
		$currency_code_map['IDR'] = 	360;
		$currency_code_map['ILS'] = 	376;
		$currency_code_map['INR'] = 	356;
		$currency_code_map['IQD'] = 	368;
		$currency_code_map['IRR'] = 	364;
		$currency_code_map['ISK'] = 	352;
		$currency_code_map['JMD'] = 	388;
		$currency_code_map['JOD'] = 	400;
		$currency_code_map['JPY'] = 	392;
		$currency_code_map['KES'] = 	404;
		$currency_code_map['KGS'] = 	417;
		$currency_code_map['KHR'] = 	116;
		$currency_code_map['KMF'] = 	174;
		$currency_code_map['KPW'] = 	408;
		$currency_code_map['KRW'] = 	410;
		$currency_code_map['KWD'] = 	414;
		$currency_code_map['KYD'] = 	136;
		$currency_code_map['KZT'] = 	398;
		$currency_code_map['LAK'] = 	418;
		$currency_code_map['LBP'] = 	422;
		$currency_code_map['LKR'] = 	144;
		$currency_code_map['LRD'] = 	430;
		$currency_code_map['LSL'] = 	426;
		$currency_code_map['LTL'] = 	440;
		$currency_code_map['LVL'] = 	428;
		$currency_code_map['LYD'] = 	434;
		$currency_code_map['MAD'] = 	504;
		$currency_code_map['MDL'] = 	498;
		$currency_code_map['MGA'] = 	969;
		$currency_code_map['MKD'] = 	807;
		$currency_code_map['MMK'] = 	104;
		$currency_code_map['MNT'] = 	496;
		$currency_code_map['MOP'] = 	446;
		$currency_code_map['MRO'] = 	478;
		$currency_code_map['MUR'] = 	480;
		$currency_code_map['MVR'] = 	462;
		$currency_code_map['MWK'] = 	454;
		$currency_code_map['MXN'] = 	484;
		$currency_code_map['MXV'] = 	979;
		$currency_code_map['MYR'] = 	458;
		$currency_code_map['MZN'] = 	943;
		$currency_code_map['NAD'] = 	516;
		$currency_code_map['NGN'] = 	566;
		$currency_code_map['NIO'] = 	558;
		$currency_code_map['NOK'] = 	578;
		$currency_code_map['NPR'] = 	524;
		$currency_code_map['NZD'] = 	554;
		$currency_code_map['OMR'] = 	512;
		$currency_code_map['PAB'] = 	590;
		$currency_code_map['PEN'] = 	604;
		$currency_code_map['PGK'] = 	598;
		$currency_code_map['PHP'] = 	608;
		$currency_code_map['PKR'] = 	586;
		$currency_code_map['PLN'] = 	985;
		$currency_code_map['PYG'] = 	600;
		$currency_code_map['QAR'] = 	634;
		$currency_code_map['RON'] = 	946;
		$currency_code_map['RSD'] = 	941;
		$currency_code_map['RUB'] = 	643;
		$currency_code_map['RWF'] = 	646;
		$currency_code_map['SAR'] = 	682;
		$currency_code_map['SBD'] = 	90;
		$currency_code_map['SCR'] = 	690;
		$currency_code_map['SDG'] = 	938;
		$currency_code_map['SEK'] = 	752;
		$currency_code_map['SGD'] = 	702;
		$currency_code_map['SHP'] = 	654;
		$currency_code_map['SLL'] = 	694;
		$currency_code_map['SOS'] = 	706;
		$currency_code_map['SRD'] = 	968;
		$currency_code_map['STD'] = 	678;
		$currency_code_map['SYP'] = 	760;
		$currency_code_map['SZL'] = 	748;
		$currency_code_map['THB'] = 	764;
		$currency_code_map['TJS'] = 	972;
		$currency_code_map['TMT'] = 	934;
		$currency_code_map['TND'] = 	788;
		$currency_code_map['TOP'] = 	776;
		$currency_code_map['TRY'] = 	949;
		$currency_code_map['TTD'] = 	780;
		$currency_code_map['TWD'] = 	901;
		$currency_code_map['TZS'] = 	834;
		$currency_code_map['UAH'] = 	980;
		$currency_code_map['UGX'] = 	800;
		$currency_code_map['USD'] = 	840;
		$currency_code_map['USN'] = 	997;
		$currency_code_map['USS'] = 	998;
		$currency_code_map['UYU'] = 	858;
		$currency_code_map['UZS'] = 	860;
		$currency_code_map['VEF'] = 	937;
		$currency_code_map['VND'] = 	704;
		$currency_code_map['VUV'] = 	548;
		$currency_code_map['WST'] = 	882;
		$currency_code_map['XAF'] = 	950;
		$currency_code_map['XAG'] = 	961;
		$currency_code_map['XAU'] = 	959;
		$currency_code_map['XBA'] = 	955;
		$currency_code_map['XBB'] = 	956;
		$currency_code_map['XBC'] = 	957;
		$currency_code_map['XBD'] = 	958;
		$currency_code_map['XCD'] = 	951;
		$currency_code_map['XDR'] = 	960;
		$currency_code_map['XFU'] = 	NULL;
		$currency_code_map['XOF'] = 	952;
		$currency_code_map['XPD'] = 	964;
		$currency_code_map['XPF'] = 	953;
		$currency_code_map['XPT'] = 	962;
		$currency_code_map['XTS'] = 	963;
		$currency_code_map['XXX'] = 	999;
		$currency_code_map['YER'] = 	886;
		$currency_code_map['ZAR'] = 	710;
		$currency_code_map['ZMK'] = 	894;
		$currency_code_map['ZWL'] = 	932;

		return array_key_exists($key, $currency_code_map) ? $currency_code_map[$key] : NULL ;
	}

	/**
	* Get the type code from card type.
	*
	* @param 	STRING	Card Type.
	* @return	INTEGER	Card Code.
	*/
	function get_card_type_code($card_type)
	{
		$card_code = array();
		$card_code['Electron'] = 1;
		$card_code['American Express'] = 2;
		$card_code['Solo'] = 4;
		$card_code['Maestro'] = 8;
		$card_code['JCB'] = 16;
		$card_code['Master Card'] = 32;
		$card_code['Visa'] = 64;

		return array_key_exists($card_type, $card_code) ? $card_code[$card_type] : NULL;
	}
}
