<?php /*

 ocPortal
 Copyright (c) ocProducts, 2004-2009

 See text/en/licence.txt for full licencing information.

*/

/**
 * @license		http://opensource.org/licenses/cpal_1.0 Common Public Attribution License
 * @copyright	ocProducts Ltd
 * @package		ecommerce
 */

class Hook_authorize
{
	/**
	* Get some title/description about payment method
	*
	* @return SHORT_TEXT		Title/Description about paypal
	*/
	function info()
	{
		return do_lang('AUTHORIZE_DESCRIPTION');
	}

	/**
	 * Function to install authorize config options
	 */
	function install_config()
	{
		add_config_option('AUTHORIZE_ENABLE','authorize_enable','special_group__tick_control__authorize__tick','return \'0\';','ECOMMERCE','PAYMENT_GATEWAY');
		add_config_option('AUTHORIZE_API_LOGIN','authorize_api_login','special_group__tick_control__authorize__line','return \'\';','ECOMMERCE','PAYMENT_GATEWAY');
		add_config_option('AUTHORIZE_TRANSACTION_KEY','authorize_transaction_key','special_group__tick_control__authorize__line','return \'\';','ECOMMERCE','PAYMENT_GATEWAY');
		add_config_option('AUTHORIZE_TEST_API_LOGIN','authorize_test_api_login','special_group__tick_control__authorize__line','return \'\';','ECOMMERCE','PAYMENT_GATEWAY');
		add_config_option('AUTHORIZE_TEST_TRANSACTION_KEY','authorize_test_transaction_key','special_group__tick_control__authorize__line','return \'\';','ECOMMERCE','PAYMENT_GATEWAY');
		add_config_option('LOCAL_SUBSCRIPTION_METHODS','authorize_local_subscription_methods','special_group__tick_control__authorize__list','return \'automatic\';','ECOMMERCE','PAYMENT_GATEWAY',0,'automatic|local_controlled');
	}

	/**
	 * Function to unistall authorize config options
	 */
	function uninstall_config()
	{
		delete_config_option('authorize_api_login');
		delete_config_option('authorize_transaction_key');
		delete_config_option('authorize_test_api_login');
		delete_config_option('authorize_test_transaction_key');
		delete_config_option('authorize_enable');
		delete_config_option('authorize_local_subscription_methods');
	}

	/**
	 * Find the payment gateway enabled or not
	 *
	 * @return BINARY Enabled/Disabled
	 */
	function is_enabled()
	{
		return get_option('authorize_enable');
	}

	/**
	 * Get gateway CPFs
	 *
	 * @return array
	 */
	function get_gateway_cpfs()
	{
		return array('authorize_api_login','authorize_transaction_key','authorize_cim_profile_id','authorize_cim_payment_profile_id','authorize_cim_shipping_address_id');
	}

	/**
	 * Whether the cpf is of this gateway
	 *
	 * @param SHORT_TEXT The cpf
	 * @return bool
	 */
	function is_gateway_cpf($cpf)
	{
		return in_array($cpf,$this->get_gateway_cpfs());
	}

	/**
	 * Show option buton
	 *
	 *	@return tempcode	Form for authorize option
	 */
	function show_option_button()
	{
		return do_template('AUTHORIZE_OPTION_BUTTON',array('URL'=>get_self_url()));
	}

	/**
	 * Get form fields for local payment.
	 *
	 * @param  ?ID_TEXT		The transaction ID (NULL: auto-generate)
	 * @param  ID_TEXT		The purchase ID
	 * @param  SHORT_TEXT	The item name
	 * @param  SHORT_TEXT	The amount
	 * @param  ?integer		The length (NULL: not a subscription)
	 * @param  ID_TEXT		The length units
	 * @return tempcode		The form fields
	 */
	function get_transaction_form_fields($trans_id,$purchase_id,$item_name,$amount,$length,$length_units)
	{
		require_code('ecommerce');
 		if (is_null($trans_id))	$trans_id=$this->generate_trans_id();

		list($card_det,$hidden)	=	get_general_transaction_form_fields($trans_id,$purchase_id,$item_name,$amount,$length,$length_units,$this);

		/*	General transaction fields returned from ecommerce api, attaching the details to the form.
			The fields can add/remove here, depending on the customised payment gateway features.
		*/
		$fields	=	new ocp_tempcode();
		foreach($card_det as $field)
		{
			$fields->attach($field);
		}

		$hidden->attach(form_input_hidden('payment_gateway','authorize'));

		$verified_account_logo	=	$this->get_confidence_logos();
		return array($fields,$hidden,$verified_account_logo);
	}

	/**
	 * Whether the member could enter the subscription start date in the payment form
	 *
	 * @return boolean
	 */
	function is_allow_subscription_start_date()
	{
		return true;
	}

	/**
	 * Whether to allow shipping detail in the local payment form
	 *
	 * @return boolean
	 */
	function is_allow_shipping()
	{
		return true;
	}

	/**
	 * Make a transaction (payment) button.
	 *
	 * @param  ID_TEXT		The product codename.
	 * @param  SHORT_TEXT	The human-readable product title.
	 * @param  ID_TEXT		The purchase ID.
	 * @param  float			A transaction amount.
	 * @param  ID_TEXT		The currency to use.
	 * @param  ID_TEXT		Member Id
	 * @return tempcode		The button
	 */
	function make_transaction_button($product,$item_name,$purchase_id,$amount,$currency='',$member_id=NULL)
	{
		$object=find_product($product);
		$product_det=$object->product_id;
		$module=get_param('page','purchase');

		$access_detail=$this->get_access_detail($member_id);
		$loginid=$access_detail[0];
		$transaction_key=$access_detail[1];

		$form_url=$this->get_form_url();
		$return_url=build_url(array('page'=>$module,'type'=>'finish','from'=>'authorize'),get_module_zone($module),NULL,false,false,true);
		$return_url=$return_url->evaluate();

		if(get_option('ecommerce_test_mode')=="1" && get_option('relay_url_return')!='')
			$return_url	=	get_option('relay_url_return')."?from=authorize&to=".$return_url;

		$time_stamp = time();
		$description = $purchase_id;

		$sequence = rand(1, 1000);

		// Insert the form elements required for SIM
		$fingerprint = $this->get_finger_print($loginid,$transaction_key,$amount,$sequence,$time_stamp,$currency);

		return do_template('ECOM_BUTTON_VIA_AUTHORIZE',array('PRODUCT'=>$product,'FORM_URL'=>$form_url,'RELAY_URL'=>$return_url,'SEQUENCE'=>strval($sequence),'TIMESTAMP'=>strval($time_stamp),'HASH'=>$fingerprint,'DESCRIPTION'=>strval($description),'LOGIN'=>$loginid,'AMOUNT'=>float_to_raw_string($amount),'IS_TEST'=>(get_option('ecommerce_test_mode')=='1')?'TRUE':'FALSE','CUST_ID'=>(get_member())?strval(get_member()):'GUEST','CURRENCY'=>$currency));
	}

	/**
	 * Calculate fingerprint.
	 *
	 * @param  SHORT_TEXT	Login Id.
	 * @param  SHORT_TEXT	Transaction key
	 * @param  float			A transaction amount.
	 * @param  integer		Timestamp
	 * @param  ID_TEXT		The currency to use.
	 * @return ARRAY
	 */
	function get_finger_print($loginid,$x_tran_key,$amount,$sequence,$tstamp,$currency = "")
	{
		return $this->hmac($x_tran_key,$loginid . "^" . $sequence . "^" . $tstamp . "^" . $amount . "^" . $currency);
	}

	/**
	 * Inserts the hidden variables in the HTML FORM required for SIM
	 * Invokes hmac function to calculate fingerprint.
	 *
	 * @param  SHORT_TEXT	Transaction key
	 * @param  SHORT_TEXT	Data
	 * @return TEXT
	 */
	function hmac($key,$data)
	{
	   // RFC 2104 HMAC implementation for php.
	   // Creates an md5 HMAC.
	   // Eliminates the need to install mhash to compute a HMAC
	   // Hacked by Lance Rushing
	   $b = 64; // byte length for md5
	   if (strlen($key) > $b)
	   {
	       $key = pack("H*",md5($key));
	   }
	   $key  = str_pad($key, $b, chr(0x00));
	   $ipad = str_pad('', $b, chr(0x36));
	   $opad = str_pad('', $b, chr(0x5c));
	   $k_ipad = $key ^ $ipad ;
	   $k_opad = $key ^ $opad;

	   return md5($k_opad  . pack("H*",md5($k_ipad . $data)));
	}

	/**
	 * Get the form URL.
	 *
	 * @return URLPATH The form url.
	 */
	function get_form_url()
	{
		return ecommerce_test_mode()?'https://test.authorize.net/gateway/transact.dll':'https://secure.authorize.net/gateway/transact.dll';
	}

	/**
	 * Generate a transaction ID.
	 *
	 * @return string			A transaction ID.
	 */
	function generate_trans_id()
	{
		return md5(uniqid(strval((mt_rand(0,32000))),true));
	}

	/**
	 * Perform a transaction.
	 *
	 * @param  ?ID_TEXT		The transaction ID (NULL: generate one)
	 * @param  SHORT_TEXT	Cardholder first name
	 * @param  SHORT_TEXT	Cardholder last name
	 * @param  SHORT_TEXT	Card number
	 * @param  SHORT_TEXT	Transaction amount
	 * @param  SHORT_TEXT	Card Expiry date
	 * @param  integer		Card Issue number
	 * @param  SHORT_TEXT	Card Start date
	 * @param  SHORT_TEXT	Card Type
	 * @set    "Visa" "Master Card" "Switch" "UK Maestro" "Maestro" "Solo" "Delta" "American Express" "Diners Card" "JCB"
	 * @param  SHORT_TEXT	Card CV2 number (security number)
	 * @param  ?integer		The subscription length in the units. (NULL: not a subscription)
	 * @param  ?ID_TEXT		The length units. (NULL: not a subscription)
	 * @set    d w m y
	 * @param  SHORT_TEXT	Description for the transaction
	 * @param  ?MEMBER		Member making the transaction (NULL: current member)
	 * @param  SHORT_TEXT	Address for card
	 * @param  SHORT_TEXT	City for card
	 * @param  SHORT_TEXT	State for card
	 * @param  SHORT_TEXT	Zip/post-code for card
	 * @param  SHORT_TEXT	Country for card
	 * @return array			A tuple: success (boolean), trans-id (string), message (string), raw message (string)
	 */
	function do_transaction($trans_id,$first_name,$last_name,$card_number,$amount,$expiry_date,$issue_number,$start_date,$card_type,$cv2,$length=NULL,$length_units=NULL,$description='',$member_id=NULL,$address='',$city='',$state='',$zip='',$country='',$shipping_first_name='',$shipping_last_name='',$shipping_address='',$shipping_city='',$shipping_state='',$shipping_zip='',$shipping_country='',$total_occurrences=9999,$trial_occurrences=0,$trial_amount=0)
	{
		$card_number  = str_replace('-','',str_replace(' ','',$card_number));
		$purchase_id  =	post_param_integer('customfld1','-1');
		$description  = ($purchase_id != '-1') ? $purchase_id : $description;
		$expiry_month = substr($expiry_date,5,2);
		$expiry_year  = substr($expiry_date,0,4);

      if (!is_null($length))
		{
			if($length_units=='m')	$length_units	=	'months';
			if($length_units=='d')	$length_units	=	'days';
			if($length_units=='y')
			{
				$length_units	=	'months';
				$length			=	12*$length;
			}
         //Set ARB Parameters
         $this->set_arb_parameters($card_number,$cv2,$expiry_year.'-'.str_pad($expiry_month,2,'0',STR_PAD_LEFT),$start_date,$length,$length_units,$total_occurrences,$trial_occurrences,$description,$amount,$trial_amount,$first_name,$last_name,$address,$city,$state,$zip,$country,$shipping_first_name,$shipping_last_name,$shipping_address,$shipping_city,$shipping_state,$shipping_zip,$shipping_country,$member_id);

         $response = http_download_file($this->url,NULL,true,false,'ocPortal',array($this->api_parameters),NULL,NULL,NULL,NULL,NULL,NULL,NULL,30.0,true);

			//$this->send_request_via_curl($this->url,NULL,$this->api_parameters);

         if($response)
         {
            $response_result = $this->parse_return($response);

            $success = ($response_result[0] == 'OK') ? true : false;
            $code = ($success) ? $response_result[3] : $response_result[1];
            $response_reason_text = ($success) ? 'success' : $response_result[2];
            $result = array($success,$code,$response_result[0],$response_reason_text);
         } else
         {
            $result = array(false,NULL,NULL,NULL);
         }
		} else
		{
         //Set AIM Parameters
         $this->set_aim_parameters($card_number,$cv2,$expiry_month,$expiry_year,$description,$amount,$first_name,$last_name,$address,$city,$state,$zip,$country,$shipping_first_name,$shipping_last_name,$shipping_address,$shipping_city,$shipping_state,$shipping_zip,$shipping_country,$member_id);

         require_code('files');
			$response_data = http_download_file($this->url,NULL,true,false,'ocPortal',$this->api_parameters,NULL,NULL,NULL,NULL,NULL,NULL,NULL,12.0);

			$response_result = explode($this->api_parameters['x_delim_char'],$response_data);

         /*Response Detail
         $response_result[0] = Response Code. 1=Approved, 2=Declined, 3=Error
         $response_result[1] = Response Subcode.A code used by the system for internal transaction tracking.
			$response_result[2] = Response Reason Code.A code representing more details about the result of the transaction.
			$response_result[3] = Response Reason Text.Brief description of the result, which corresponds with the Response Reason Code.
			$response_result[6] = Transaction ID*/

			if (!array_key_exists(1,$response_result)) $response_result[1]='';
			if (!array_key_exists(2,$response_result)) $response_result[2]='';
			if (!array_key_exists(3,$response_result)) $response_result[3]='';
			if (!array_key_exists(6,$response_result)) $response_result[6]='';

         $success = ($response_result[0] == 1) ? true : false;
         $code = ($success) ? $response_result[6] : $response_result[2];
         $response_reason_text = ($success) ? 'success' : $response_result[3];
         $result = array($success,$code,$response_result[1],$response_reason_text);
      }

		if(addon_installed('shopping'))
		{
			$this->store_shipping_address($purchase_id);
		}

		return $result;
	}

	/**
	 * Find a transaction fee from a transaction amount. Regular fees aren't taken into account.
	 *
	 * @param  float	A transaction amount.
	 * @return float	The fee
	 */
	function get_transaction_fee($amount)
	{
		//return round(0.30+0.0212*$amount,2);
		return round(floatval(get_option('authorize_transaction_flat_cost'))+floatval(get_option('authorize_transaction_percentage_cost'))*$amount,2);
	}

   /**
    * Get subscription error
    *
    * @param  SHORT_TEXT   The error code
    * @param  SHORT_TEXT   The error
    * @return SHORT_TEXT   Error message
    */
   function get_subscription_error($code,$error)
   {
      require_lang('ecommerce');

      $error_msg = NULL;

      if(strpos($error,"CARDNUMBER' ELEMENT IS INVALID") !== false)
      {
         $error_msg = do_lang_tempcode('CARDNUMBER_INVALID');
      }
      elseif(strpos($error,"THE CREDIT CARD HAS EXPIRED") !== false)
      {
         $error_msg = do_lang_tempcode('CARD_EXPIRED');
      }

      if (is_null($error_msg))
      {
	      switch($code)
	      {
	      	case 'E00001':
	      		$error_msg = do_lang_tempcode('ARB_UNEXPECTED_ERROR');
	      		break;

	      	case 'E00002':
	      		$error_msg = do_lang_tempcode('ARB_CONTENT_TYPE_ERROR');
	      		break;

	      	case 'E00003':
	      		$error_msg = do_lang_tempcode('ARB_XML_PARSE_ERROR');
	      		break;

	      	case 'E00004':
	      		$error_msg = do_lang_tempcode('ARB_API_METHOD_INVALID');
	      		break;

	      	case 'E00005':
	      		$error_msg = do_lang_tempcode('ARB_TRANSACTION_KEY_ERROR');
	      		break;

	      	case 'E00006':
	      		$error_msg = do_lang_tempcode('ARB_LOGIN_ID_ERROR');
	      		break;

	      	case 'E00007':
	      		$error_msg = do_lang_tempcode('ARB_AUTHENTICATION_ERROR');
	      		break;

	      	case 'E00008':
	      		$error_msg = do_lang_tempcode('ARB_ACCOUNT_INACTIVE');
	      		break;

	      	case 'E00009':
	      		$error_msg = do_lang_tempcode('ARB_ACCOUNT_TEST_MODE');
	      		break;

	      	case 'E00010':
	      		$error_msg = do_lang_tempcode('ARB_PERMISSION_DENIED');
	      		break;

	      	case 'E00011':
	      		$error_msg = do_lang_tempcode('ARB_PERMISSION_DENIED');
	      		break;

	      	case 'E00012':
	      		$error_msg = do_lang_tempcode('ARB_DUPLICATE_SUBSCRIPTION');
	      		break;

	      	case 'E00013':
	      		$error_msg = do_lang_tempcode('ARB_FIELD_INVALID');
	      		break;

	      	case 'E00014':
	      		$error_msg = do_lang_tempcode('ARB_REQUIRED_FIELD_MISSING');
	      		break;

	      	case 'E00015':
	      		$error_msg = do_lang_tempcode('ARB_FIELD_LENGTH_INVALID');
	      		break;

	      	case 'E00016':
	      		$error_msg = do_lang_tempcode('ARB_FIELD_TYPE_INVALID');
	      		break;

	      	case 'E00017':
	      		$error_msg = do_lang_tempcode('ARB_PAST_START_DATE_ERROR');
	      		break;

	      	case 'E00018':
	      		$error_msg = do_lang_tempcode('ARB_CARD_EXPIRED_START_DATE_ERROR');
	      		break;

	      	case 'E00019':
	      		$error_msg = do_lang_tempcode('ARB_REQUIRED_TAX_LICENSE');
	      		break;

	      	case 'E00020':
	      		$error_msg = do_lang_tempcode('ARB_ECHECK_NOT_ENABLED');
	      		break;

	      	case 'E00021':
	      		$error_msg = do_lang_tempcode('ARB_CARD_NOT_ENABLED');
	      		break;

	      	case 'E00022':
	      		$error_msg = do_lang_tempcode('ARB_INTERVAL_ERROR');
	      		break;

	      	case 'E00024':
	      		$error_msg = do_lang_tempcode('ARB_TRIAL_OCCURRENCE_ZERO_ERROR');
	      		break;

	      	case 'E00025':
	      		$error_msg = do_lang_tempcode('ARB_NOT_ENABLED');
	      		break;

	      	case 'E00026':
	      		$error_msg = do_lang_tempcode('ARB_TRIAL_AMOUNT_OCCURRENCE_REQUIRED');
	      		break;

	      	case 'E00027':
	      		$error_msg = do_lang_tempcode('ARB_TRANSACTION_UNSUCCESSFUL');
	      		break;

	      	case 'E00028':
	      		$error_msg = do_lang_tempcode('ARB_TRIAL_TOTAL_OCCURRENCE_ERROR');
	      		break;

	      	case 'E00029':
	      		$error_msg = do_lang_tempcode('ARB_PAYMENT_INFO_REQUIRED');
	      		break;

	      	case 'E00030':
	      		$error_msg = do_lang_tempcode('ARB_PAYMENT_SCHEDULE_REQUIRED');
	      		break;

	      	case 'E00031':
	      		$error_msg = do_lang_tempcode('ARB_AMOUNT_REQUIRED');
	      		break;

	      	case 'E00032':
	      		$error_msg = do_lang_tempcode('ARB_START_DATE_REQUIRED');
	      		break;

	      	case 'E00033':
	      		$error_msg = do_lang_tempcode('ARB_START_DATE_CHANGE_ERROR');
	      		break;

	      	case 'E00034':
	      		$error_msg = do_lang_tempcode('ARB_INTERVAL_CHANGE_ERROR');
	      		break;

	      	case 'E00035':
	      		$error_msg = do_lang_tempcode('ARB_SUBSCRIPTION_NOT_FOUND');
	      		break;

	      	case 'E00036':
	      		$error_msg = do_lang_tempcode('ARB_PAYMENT_TYPE_CHANGE_ERROR');
	      		break;

	      	case 'E00037':
	      		$error_msg = do_lang_tempcode('ARB_UPDATE_ERROR');
	      		break;

	      	case 'E00038':
	      		$error_msg = do_lang_tempcode('ARB_CANCEL_ERROR');
	      		break;

	      	case 'E00045':
	      		$error_msg = do_lang_tempcode('ARB_NAMESPACE_ERROR');
	      		break;

	      	default:
	      		$error_msg = do_lang_tempcode('ARB_GENERAL_ERROR');
	      }
      }

      return $error_msg;
   }

   /**
	 * Find whether the hook auto-cancels (if it does, auto cancel the given trans-id).
	 *
	 * @param ID Subscription Id
	 * @param ID Member Id (NULL: Admin)
	 * @return ?boolean	True: yes. False: no. (NULL: cancels via a user-URL-directioning)
	 */
   function auto_cancel($subscription_id,$member_id=NULL)
   {
   	$temp	= $GLOBALS['SITE_DB']->query_value('subscriptions','s_auto_fund_key',array('id'=>$subscription_id));
   	$data	= unserialize($temp);
		$authorize_subscription_id	= $data['id'];

      $this->set_cancelation_api_parameters($authorize_subscription_id,$member_id);

      $response = http_download_file($this->url,NULL,true,false,'ocPortal',array($this->api_parameters),NULL,NULL,NULL,NULL,NULL,NULL,NULL,12.0,true);

      if(!is_null($response))
      {
         $response_result = $this->parse_return($response);
         $success = ($response_result[0] == 'OK') ? true : false;
         $result  = array($success,$response_result[1],$response_result[2],$response_result[3]);
      }
      else
      {
      	$success = false;
         $result  = array(false,NULL,NULL,NULL);
      }

      return $success;
   }

   /**
	 * Function to parse Authorize.net response
	 */
   function parse_return($response)
   {
      $result_code     = $this->substring_between($response,'<resultCode>','</resultCode>');
      $code            = $this->substring_between($response,'<code>','</code>');
      $text            = $this->substring_between($response,'<text>','</text>');
      $subscription_id = $this->substring_between($response,'<subscriptionId>','</subscriptionId>');

      return array (strtoupper($result_code), $code, strtoupper($text), $subscription_id);
   }


   /**
	 * Helper function for parsing response
	 */
   function substring_between($haystack,$start,$end)
   {
      if (strpos($haystack,$start) === false || strpos($haystack,$end) === false)
      {
         return false;
      }
      else
      {
         $start_position = strpos($haystack,$start)+strlen($start);
         $end_position   = strpos($haystack,$end);

         return substr($haystack,$start_position,$end_position-$start_position);
      }
   }

  /**
   * Get transaction error
   *
   * @param  SHORT_TEXT   The error code
   * @param  SHORT_TEXT   The error
   * @return SHORT_TEXT   Error message
   */
   function get_transaction_error($code,$error)
   {
      require_lang('ecommerce');

      switch($code)
      {
      	case '2':
      	case '3':
      	case '4':
      		$error_msg = do_lang_tempcode('TRANSACTION_DECLINED');
      		break;

      	case '5':
      		$error_msg = do_lang_tempcode('REQUIRED_VALID_AMOUNT');
      		break;

      	case '6':
      		$error_msg = do_lang_tempcode('CARD_NUMBER_INVALID');
      		break;

      	case '7':
      		$error_msg = do_lang_tempcode('EXPIRY_DATE_FORMAT');
      		break;

      	case '8':
      		$error_msg = do_lang_tempcode('CARD_EXPIRED');
      		break;

      	default:
      		$error_msg = do_lang_tempcode('CARD_GENERAL_ERROR');
      }

      return $error_msg;
   }

	/**
	 * Store shipping address for orders
	 *
	 * @param  AUTO_LINK		Order id
	 * @return ?mixed			Address id (NULL : No address record found)
	 */
	function store_shipping_address($order_id)
	{
		if(is_null($GLOBALS['SITE_DB']->query_value_null_ok('shopping_order_addresses','id',array('order_id'=>$order_id))))
		{
			$shipping_address['order_id']			=	$order_id;
			$shipping_address['address_name']	=	post_param('ship_first_name','')." ".post_param('ship_last_name','');
			$shipping_address['address_street']	=	post_param('ship_addr_1','');
			$shipping_address['address_city']	=	post_param('ship_city','');
			$shipping_address['address_state']	=	post_param('ship_state','');
			$shipping_address['address_zip']		=	post_param('ship_post_code','');
			$shipping_address['address_country']=	post_param('shipping_country','');
			$shipping_address['receiver_email']	=	'';//post_param('receiver_email','');

			$cpfs_to_save = array
										(
											'ocp_shipping_firstname' => post_param('ship_first_name',''),
											'ocp_shipping_lastname'	 => post_param('ship_last_name',''),
											'ocp_shipping_street'	 => post_param('ship_addr_1',''),
											'ocp_shipping_country'   => post_param('shipping_country',''),
											'ocp_shipping_state'		 => post_param('ship_state',''),
											'ocp_shipping_city'	    => post_param('ship_city',''),
											'ocp_shipping_zip'		 => post_param('ship_post_code','')
										);
			set_ocp_cpfs($cpfs_to_save);

			return $GLOBALS['SITE_DB']->query_insert('shopping_order_addresses',$shipping_address,true);
		}
	}


	/**
	 * This function defines the parameters needed to make an AIM call.
	 *
	 * @param NUMBER CREDIT CARD NUMBER (only numbers)
	 * @param NUMBER CREDIT CARD EXPIRY MONTH (MM)
	 * @param NUMBER CREDIT CARD EXPIRY YEAR (YYYY)
	 * @param SHORT_TEXT SHORT DESCRIPTION ABOUT THE TRANSACTION
	 * @param INTEGER AMOUNT
	 * @param ID MEMBER ID
	 */
	function set_aim_parameters($card_number,$cv2,$expiry_month,$expiry_year,$description,$amount,$first_name=NULL,$last_name=NULL,$address=NULL,$city=NULL,$state=NULL,$zip=NULL,$country=NULL,$shipping_first_name=NULL,$shipping_last_name=NULL,$shipping_address=NULL,$shipping_city=NULL,$shipping_state=NULL,$shipping_zip=NULL,$shipping_country=NULL,$member_id=NULL)
	{
		$this->api_parameters = array();

		$access_detail = $this->get_access_detail($member_id);

		$this->url = (get_option('ecommerce_test_mode')=='1') ? 'https://test.authorize.net/gateway/transact.dll' : 'https://secure.authorize.net/gateway/transact.dll';

		$this->api_parameters['x_login']	         = $access_detail[0];
		$this->api_parameters['x_tran_key']       = $access_detail[1];
		$this->api_parameters['x_version']        = '3.1';
		$this->api_parameters['x_type']           = 'AUTH_CAPTURE';
		$this->api_parameters['x_method']         = 'CC';
		$this->api_parameters['x_card_num']       = $card_number;
		$this->api_parameters['x_exp_date']       = $expiry_month.$expiry_year;
		$this->api_parameters['x_description']    = $description;
		$this->api_parameters['x_delim_data']     = TRUE;
		$this->api_parameters['x_delim_char']     = "|";
		$this->api_parameters['x_relay_response'] = FALSE;
		$this->api_parameters['x_amount']         = $amount;
		$this->api_parameters["x_card_code"]      = $cv2;
		$this->api_parameters["x_customer_ip"]    = get_ip_address();

		if($first_name) $this->api_parameters["x_first_name"] = $first_name;
		if($last_name) $this->api_parameters["x_last_name"] = $last_name;
		if($city) $this->api_parameters["x_city"] = $city;
		if($state) $this->api_parameters["x_state"] = $state;
		if($zip) $this->api_parameters["x_zip"] = $zip;
		if($country) $this->api_parameters["x_country"] = $country;
		if($shipping_first_name) $this->api_parameters["x_first_name"] = $shipping_first_name;
		if($shipping_last_name) $this->api_parameters["x_last_name"] = $shipping_last_name;
		if($shipping_city) $this->api_parameters["x_city"] = $shipping_city;
		if($shipping_state) $this->api_parameters["x_state"] = $shipping_state;
		if($shipping_zip) $this->api_parameters["x_zip"] = $shipping_zip;
		if($shipping_country) $this->api_parameters["x_country"] = $shipping_country;

		$email = $GLOBALS['FORUM_DRIVER']->get_member_email_address($member_id);
		if ($email!='') $this->api_parameters["x_email"] = $email;
	}

  /**
	* This function defines the parameters needed to make an ARB call.
	*
	* @param NUMBER CREDIT CARD NUMBER (only numbers)
	* @param NUMBER CREDIT CARD EXPIRY MONTH (MM)
	* @param NUMBER CREDIT CARD EXPIRY YEAR (YYYY)
	* @param SHORT_TEXT SHORT DESCRIPTION ABOUT THE TRANSACTION
	* @param INTEGER AMOUNT
	* @param ID MEMBER ID
	*/
	function set_arb_parameters($card_number,$cv2,$expiration_date,$start_date,$length,$unit,$total_occurrences,$trial_occurrences,$description,$amount,$trial_amount,$first_name=NULL,$last_name=NULL,$address=NULL,$city=NULL,$state=NULL,$zip=NULL,$country=NULL,$shipping_first_name=NULL,$shipping_last_name=NULL,$shipping_address=NULL,$shipping_city=NULL,$shipping_state=NULL,$shipping_zip=NULL,$shipping_country=NULL,$member_id=NULL)
	{
		$access_detail = $this->get_access_detail($member_id);

		if (get_option('ecommerce_test_mode')=='1')
		{
			//URL for test account
			$this->url = "https://apitest.authorize.net/xml/v1/request.api";
		} else
		{
			//URL for live account
			$this->url = "https://api.authorize.net/xml/v1/request.api";
		}

      $this->api_parameters =
        "<ARBCreateSubscriptionRequest xmlns=\"AnetApi/xml/v1/schema/AnetApiSchema.xsd\">".
        "<merchantAuthentication>".
        "<name>" . $access_detail[0] . "</name>".
        "<transactionKey>" . $access_detail[1] . "</transactionKey>".
        "</merchantAuthentication>".
        "<subscription>".
        "<name>" . $description . "</name>".
        "<paymentSchedule>".
        "<interval>".
        "<length>". $length ."</length>".
        "<unit>". $unit ."</unit>".
        "</interval>".
        "<startDate>" . $start_date . "</startDate>".
        "<totalOccurrences>". $total_occurrences . "</totalOccurrences>".
        "<trialOccurrences>". $trial_occurrences . "</trialOccurrences>".
        "</paymentSchedule>".
        "<amount>". $amount ."</amount>".
        "<trialAmount>" . $trial_amount . "</trialAmount>".
        "<payment>".
        "<creditCard>".
        "<cardNumber>" . $card_number . "</cardNumber>".
        "<expirationDate>" . $expiration_date . "</expirationDate>".
        "</creditCard>".
        "</payment>".
        "<billTo>";

		 if ($first_name) $this->api_parameters .= "<firstName>". $first_name . "</firstName>";
       if ($last_name) $this->api_parameters .= "<lastName>". $last_name . "</lastName>";
       if ($address) $this->api_parameters .= "<address>" . $address . "</address>";
       if ($city) $this->api_parameters .=  "<city>" . $city . "</city>";
       if ($state) $this->api_parameters .= "<state>" . $state . "</state>";
       if($zip) $this->api_parameters .= "<zip>" . $zip . "</zip>";
       if($country) $this->api_parameters .= "<country>" . $country . "</country>";

       $this->api_parameters .=
        "</billTo>".
        "<shipTo>";

       if ($shipping_first_name) $this->api_parameters .= "<firstName>". $shipping_first_name . "</firstName>";
       if ($shipping_last_name) $this->api_parameters .= "<lastName>". $shipping_last_name . "</lastName>";
       if ($shipping_address) $this->api_parameters .= "<address>" . $shipping_address . "</address>";
       if ($shipping_city) $this->api_parameters .=  "<city>" . $shipping_city . "</city>";
       if ($shipping_state) $this->api_parameters .= "<state>" . $shipping_state . "</state>";
       if($shipping_zip) $this->api_parameters .= "<zip>" . $shipping_zip . "</zip>";
       if($shipping_country) $this->api_parameters .= "<country>" . $shipping_country . "</country>";

       $this->api_parameters .=
        "</shipTo>".
        "</subscription>".
        "</ARBCreateSubscriptionRequest>";
	}

  /**
	* This function defines the parameters needed to make a cancelation API call.
	*
	* @param ID Subscription Id
	* @param ID Member Id
	*/
	function set_cancelation_api_parameters($subscription_id,$member_id=NULL)
	{
		$access_detail = $this->get_access_detail($member_id);

		if (get_option('ecommerce_test_mode')=='1')
		{
			//URL for test account
			$this->url = "https://apitest.authorize.net/xml/v1/request.api";
		} else
		{
			//URL for live account
			$this->url = "https://api.authorize.net/xml/v1/request.api";
		}

      $this->api_parameters =
        "<ARBCancelSubscriptionRequest xmlns=\"AnetApi/xml/v1/schema/AnetApiSchema.xsd\">".
        "<merchantAuthentication>".
        "<name>" . $access_detail[0] . "</name>".
        "<transactionKey>" . $access_detail[1] . "</transactionKey>".
        "</merchantAuthentication>" .
        "<subscriptionId>" . $subscription_id . "</subscriptionId>".
        "</ARBCancelSubscriptionRequest>";
	}

   /**
	 * Handle IPN's. The function may produce output, which would be returned to the Payment Gateway. The function may do transaction verification.
	 *
	 * @return array	A long tuple of collected data.
	 */
	function handle_transaction()
	{
      require_code('ecommerce');
      require_code('ocf_referral');

      $success         = (post_param('x_response_code','') == 1) ? true : false;
      $response_text   = post_param('x_response_reason_text','');
      $subscription_id = post_param('x_subscription_id','');
		$transaction_id  = post_param('x_trans_id','');
		$purchase_id     = post_param('x_description','');
		$reason_code     = post_param('x_response_reason_code','');
		$amount          = post_param('x_amount',0);
		$currency        = post_param('x_currency_code',get_option('currency'));
		$parent_txn_id   = '';
		$pending_reason  = '';
		$memo				  = '';
		$item_name		  = ($subscription_id) ? find_item_name_of_subscription($purchase_id) : find_item_name_of_order($purchase_id);
		if ($subscription_id)
		{
			$payment_status  = ($success) ? 'Completed' : 'SCancelled';
		}
		else
		{
			$payment_status  = ($success) ? 'Completed' : 'Failed';
		}
		$txn_id 			  = ($subscription_id) ? $subscription_id : $transaction_id;

		return array($purchase_id,$item_name,$payment_status,$reason_code,$pending_reason,$memo,$amount,$currency,$txn_id,$parent_txn_id);
   }

	function get_confidence_logos()
	{
		require_css('ecommerce');
		$logo	=	'<!-- (c) 2005, 2010. Authorize.Net is a registered trademark of CyberSource Corporation -->
						<div class="AuthorizeNetSeal"> <script type="text/javascript"
						language="javascript">var
						ANS_customer_id="f08fe5db-fa36-4720-b7b5-60ffb989c208";</script> <script
						type="text/javascript" language="javascript"
						src="//verify.authorize.net/anetseal/seal.js" ></script></div>';

		$payment_processing_link = '<a class="link_exempt" href="http://www.authorize.net/" id="AuthorizeNetText" target="_blank">Secure payment processing</a>';

		$read_text_link	=	build_url(array('page'=>'purchase','type'=>'payment_note','screen'=>'secure_payment_authorize','wide_high'=>1),get_module_zone('purchase'));

		$read_text_link	=	$read_text_link->evaluate();

		$js	=	"function secure_shopping_notes() {
                	window.open('$read_text_link','ruleswindow','menubar=1,resizable=1,scrollbars=1,width=970,height=650');
                  return false;
               }";
		return do_template('VERIFIED_MERCHANT_LOGO_SCREEN',array('LOGO'=>$logo,'READ_TEXT_LINK'=>$read_text_link,'PAYMENT_PROCESSING_LINK'=>$payment_processing_link,'JAVASCRIPT'=>$js,'PAYMENT_TEXT_URL'=>'site:secure_payment_authorize'));
	}

	/**
	 * Get authorize access detail
	 *
	 * @param ID Member Id
	 * @return array
	 */
	function get_access_detail($member_id=NULL)
	{
		$api_login				=	NULL;
		$api_transaction_key	=	NULL;

		if(!is_null($member_id) && get_option('enable_member_payments'))
		{
			$api_login  			 =	get_ocp_cpf('authorize_api_login',$member_id);
			$api_transaction_key  =	get_ocp_cpf('authorize_transaction_key',$member_id);
		}

		if (is_null($api_login) || $api_login=='' || is_null($api_transaction_key) || $api_transaction_key=='')
		{
			$api_login	=	get_option('ecommerce_test_mode')?get_option('authorize_test_api_login'): get_option('authorize_api_login');
			$api_transaction_key	=	get_option('ecommerce_test_mode')?get_option('authorize_test_transaction_key'): get_option('authorize_transaction_key');
		}

		return array($api_login,$api_transaction_key);
	}

	/** Function to return local payment method for subscriptions
	 *
	 * @return ID_TEXT	automatic|local_controlled
	 */
	function get_subscription_local_payment_method()
	{
		return get_option('authorize_local_subscription_methods');
	}

	/**
	 * Function to parse CIM response
	 *
	 * @param SHORT_TEXT Response
	 * @param SHORT_TEXT API type
	 * @return ARRAY
	 */
   function parse_cim_return($response,$api_type)
   {
      $result_code = $this->substring_between($response,'<resultCode>','</resultCode>');
      $code        = $this->substring_between($response,'<code>','</code>');
      $text        = $this->substring_between($response,'<text>','</text>');

      switch ($api_type)
      {
      	case 'profile':
	      	$profile_id = $this->substring_between($response,'<customerProfileId>','</customerProfileId>');
	      	$response = array (strtoupper($result_code), $code, $text, $profile_id);
	      	break;

      	case 'payment_profile':
	      	$payment_profile_id = $this->substring_between($response,'<customerPaymentProfileId>','</customerPaymentProfileId>');
	      	$response = array (strtoupper($result_code), $code, $text, $payment_profile_id);
	      	break;

      	case 'shipping':
	      	$shipping_address_id = $this->substring_between($response,'<customerAddressId>','</customerAddressId>');
	      	$response = array (strtoupper($result_code), $code, $text, $shipping_address_id);
	      	break;

	      case 'transaction':
	      	$direct_response = $this->substring_between($response,'<directResponse>','</directResponse>');
	      	$direct_response_fields = explode(",", $direct_response);

				$response_code        = $direct_response_fields[0]; // 1 = Approved 2 = Declined 3 = Error
				$response_reason_code = $direct_response_fields[2];
				$response_reason_text = $direct_response_fields[3];
				$approval_code        = $direct_response_fields[4];
				$gateway_trans_id     = $direct_response_fields[6];
				$amount               = $direct_response_fields[9];

	      	$response = array (strtoupper($result_code), $code, $text, $response_code, $response_reason_code, $response_reason_text, $approval_code, $gateway_trans_id, $amount);
	      	break;

      	default:
      		$response = array (strtoupper($result_code), $code, $text, '');
      }

      return $response;
   }

   /**
   * Get CIM duplicate profile id
   *
   * @param SHORT_TEXT Response
   * @return ID_TEXT
   */
   function get_cim_duplicate_profile_id($response)
   {
   	$profile_id = $this->substring_between($response,'A duplicate record with id ',' already exists');

   	return $profile_id;
   }

  /**
   * Get CIM error
   *
   * @param SHORT_TEXT Error code
   * @return SHORT_TEXT Error
   */
   function get_cim_error($code)
   {
      require_lang('ecommerce');

      switch($code)
      {
      	case '100003':
      		$error_msg = do_lang_tempcode('CIM_RECORD_DELETED');
      		break;

      	case 'E00001':
      		$error_msg = do_lang_tempcode('CIM_UNEXPECTED_ERROR');
      		break;

      	case 'E00002':
      		$error_msg = do_lang_tempcode('CIM_CONTENT_TYPE_ERROR');
      		break;

      	case 'E00003':
      		$error_msg = do_lang_tempcode('CIM_XML_PARSE_ERROR');
      		break;

      	case 'E00004':
      		$error_msg = do_lang_tempcode('CIM_API_METHOD_INVALID');
      		break;

      	case 'E00005':
      		$error_msg = do_lang_tempcode('CIM_TRANSACTION_KEY_ERROR');
      		break;

      	case 'E00006':
      		$error_msg = do_lang_tempcode('CIM_LOGIN_ID_ERROR');
      		break;

      	case 'E00007':
      		$error_msg = do_lang_tempcode('CIM_AUTHENTICATION_ERROR');
      		break;

      	case 'E00008':
      		$error_msg = do_lang_tempcode('CIM_ACCOUNT_INACTIVE');
      		break;

      	case 'E00009':
      		$error_msg = do_lang_tempcode('CIM_ACCOUNT_TEST_MODE');
      		break;

      	case 'E00010':
      		$error_msg = do_lang_tempcode('CIM_PERMISSION_DENIED');
      		break;

      	case 'E00011':
      		$error_msg = do_lang_tempcode('CIM_PERMISSION_DENIED');
      		break;

      	case 'E00013':
      		$error_msg = do_lang_tempcode('CIM_FIELD_INVALID');
      		break;

      	case 'E00014':
      		$error_msg = do_lang_tempcode('CIM_REQUIRED_FIELD_MISSING');
      		break;

      	case 'E00015':
      		$error_msg = do_lang_tempcode('CIM_FIELD_LENGTH_INVALID');
      		break;

      	case 'E00016':
      		$error_msg = do_lang_tempcode('CIM_FIELD_TYPE_INVALID');
      		break;

      	case 'E00019':
      		$error_msg = do_lang_tempcode('CIM_REQUIRED_TAX_LICENSE');
      		break;

      	case 'E00027':
      		$error_msg = do_lang_tempcode('CIM_TRANSACTION_UNSUCCESSFUL');
      		break;

      	case 'E00029':
      		$error_msg = do_lang_tempcode('CIM_PAYMENT_INFO_REQUIRED');
      		break;

      	case 'E00039':
      		$error_msg = do_lang_tempcode('CIM_DUPLICATE');
      		break;

      	case 'E00040':
      		$error_msg = do_lang_tempcode('CIM_NO_RECORED');
      		break;

      	case 'E00041':
      		$error_msg = do_lang_tempcode('CIM_NO_FIELD_VALUE');
      		break;

      	case 'E00042':
      		$error_msg = do_lang_tempcode('CIM_MAXIMUM_PAYMENT_PROFILE');
      		break;

      	case 'E00043':
      		$error_msg = do_lang_tempcode('CIM_MAXIMUM_SHIPPING_ADDRESS');
      		break;

      	case 'E00044':
      		$error_msg = do_lang_tempcode('CIM_NOT_ENABLED');
      		break;

      	case 'E00051':
      		$error_msg = do_lang_tempcode('CIM_NO_ORIGINAL_TRANSACTION');
      		break;

      	default:
      		$error_msg = do_lang_tempcode('CIM_GENERAL_ERROR');
      }

      return $error_msg;
   }

  /**
	* This function defines the parameters needed to make an API call for profile creation.
	*
	* @param SHORT_TEXT Email Id
	* @param SHORT_TEXT Description
	*/
	function set_cim_profile_parameters($email,$description=NULL)
	{
		$access_detail = $this->get_access_detail();

		if (get_option('ecommerce_test_mode')=='1')
		{
			$this->url = "https://apitest.authorize.net/xml/v1/request.api";
		} else
		{
			$this->url = "https://api.authorize.net/xml/v1/request.api";
		}

      $this->api_parameters =
			"<createCustomerProfileRequest xmlns=\"AnetApi/xml/v1/schema/AnetApiSchema.xsd\">" .
			"<merchantAuthentication>".
			"<name>".$access_detail[0]."</name>".
			"<transactionKey>".$access_detail[1]."</transactionKey>".
			"</merchantAuthentication>".
			"<profile>".
			"<description>".$description."</description>".
			"<email>".$email."</email>".
			"</profile>".
			"</createCustomerProfileRequest>";
	}

	/**
	 * Create CIM profile.
	 *
	 * @param SHORT_TEXT Email Id
	 * @param SHORT_TEXT Description
	 * @param ARRAY
	 */
	function create_cim_profile($email,$description=NULL)
	{
	 	$this->set_cim_profile_parameters($email,$description);

      $response = http_download_file($this->url,NULL,true,false,'ocPortal',array($this->api_parameters),NULL,NULL,NULL,NULL,NULL,NULL,NULL,12.0,true);

      if ($response)
      {
         $response_result = $this->parse_cim_return($response,'profile');
         $success = ($response_result[0] == 'OK') ? true : false;
         $result = array($success,$response_result[1],$response_result[2],$response_result[3]);
      } else
      {
         $result = array(false,NULL,NULL,NULL);
      }

      return $result;
	}

   /**
	 * This function defines the parameters needed to make an API call for payment profile creation.
	 *
	 * @param SHORT_TEXT Profile Id
	 * @param SHORT_TEXT First name
	 * @param SHORT_TEXT Last name
	 * @param SHORT_TEXT Card number
	 * @param SHORT_TEXT Card expiry year
	 * @param SHORT_TEXT Card expiry month
	 * @param SHORT_TEXT Address
	 * @param SHORT_TEXT City
	 * @param SHORT_TEXT A valid two-character state code
	 * @param SHORT_TEXT Zip
	 * @param SHORT_TEXT Country
	 * @param SHORT_TEXT Last name
	 */
	function set_cim_payment_profile_parameters($profile_id,$first_name,$last_name,$card_number,$expiry_year,$expiry_month,$address=NULL,$city=NULL,$state=NULL,$zip=NULL,$country=NULL)
	{
		$access_detail = $this->get_access_detail();

		if (get_option('ecommerce_test_mode')=='1')
		{
			$this->url = "https://apitest.authorize.net/xml/v1/request.api";
			$mode = 'testMode';
		} else
		{
			$this->url = "https://api.authorize.net/xml/v1/request.api";
			$mode = 'liveMode';
		}

      $this->api_parameters =
			"<createCustomerPaymentProfileRequest xmlns=\"AnetApi/xml/v1/schema/AnetApiSchema.xsd\">".
			"<merchantAuthentication>".
			"<name>".$access_detail[0]."</name>".
			"<transactionKey>".$access_detail[1]."</transactionKey>".
			"</merchantAuthentication>".
			"<customerProfileId>".$profile_id."</customerProfileId>".
			"<paymentProfile>".
			"<billTo>".
			 "<firstName>".$first_name."</firstName>".
			 "<lastName>".$last_name."</lastName>";
			if ($address) $this->api_parameters .= "<address>".$address."</address>";
			if ($city) $this->api_parameters .= "<city>".$city."</city>";
			if ($state) $this->api_parameters .= "<state>".$state."</state>";
			if ($zip) $this->api_parameters .= "<zip>".$zip."</zip>";
			if ($country) $this->api_parameters .= "<country>".$country."</country>";
			$this->api_parameters .="</billTo>".
			"<payment>".
			 "<creditCard>".
			  "<cardNumber>".$card_number."</cardNumber>".
			  "<expirationDate>".$expiry_year."-".$expiry_month."</expirationDate>".
			 "</creditCard>".
			"</payment>".
			"</paymentProfile>".
			"<validationMode>".$mode."</validationMode>".
			"</createCustomerPaymentProfileRequest>";
	}

	/**
	* Create CIM payment profile.
	*
   * @param SHORT_TEXT Profile Id
   * @param SHORT_TEXT First name
   * @param SHORT_TEXT Last name
   * @param SHORT_TEXT Card number
   * @param SHORT_TEXT Card expiry year
   * @param SHORT_TEXT Card expiry month
   * @param SHORT_TEXT Address
	* @param SHORT_TEXT City
	* @param SHORT_TEXT A valid two-character state code
	* @param SHORT_TEXT Zip
	* @param SHORT_TEXT Country
	* @param ARRAY
	*/
	function create_cim_payment_profile($profile_id,$first_name,$last_name,$card_number,$expiry_year,$expiry_month,$address=NULL,$city=NULL,$state=NULL,$zip=NULL,$country=NULL)
	{
		$card_number = str_replace('-','',str_replace(' ','',$card_number));

	 	$this->set_cim_payment_profile_parameters($profile_id,$first_name,$last_name,$card_number,$expiry_year,$expiry_month,$address,$city,$state,$zip,$country);

      $response = http_download_file($this->url,NULL,true,false,'ocPortal',array($this->api_parameters),NULL,NULL,NULL,NULL,NULL,NULL,NULL,12.0,true);

      if ($response)
      {
         $response_result = $this->parse_cim_return($response,'payment_profile');
         $success = ($response_result[0] == 'OK') ? true : false;
         $result = array($success,$response_result[1],$response_result[2],$response_result[3]);
      } else
      {
         $result = array(false,NULL,NULL,NULL);
      }

      return $result;
	}

	/**
	 * This function defines the parameters needed to make an API call for shipping address id.
	 *
	 * @param SHORT_TEXT Profile Id
	 * @param SHORT_TEXT First name
	 * @param SHORT_TEXT Last name
	 * @param SHORT_TEXT Address
	 * @param SHORT_TEXT City
	 * @param SHORT_TEXT A valid two-character state code
	 * @param SHORT_TEXT Zip
	 * @param SHORT_TEXT Country
	 */
	function set_cim_shipping_address_parameters($profile_id,$first_name,$last_name,$address=NULL,$city=NULL,$state=NULL,$zip=NULL,$country=NULL)
	{
		$access_detail = $this->get_access_detail();

		if (get_option('ecommerce_test_mode')=='1')
		{
			$this->url = "https://apitest.authorize.net/xml/v1/request.api";
		} else
		{
			$this->url = "https://api.authorize.net/xml/v1/request.api";
		}

      $this->api_parameters =
			"<createCustomerShippingAddressRequest xmlns=\"AnetApi/xml/v1/schema/AnetApiSchema.xsd\">".
			"<merchantAuthentication>".
			"<name>".$access_detail[0]."</name>".
			"<transactionKey>".$access_detail[1]."</transactionKey>".
			"</merchantAuthentication>".
			"<customerProfileId>".$profile_id."</customerProfileId>".
			"<address>".
			"<firstName>".$first_name."</firstName>".
			"<lastName>".$last_name."</lastName>";
			if ($address) $this->api_parameters .= "<address>".$address."</address>";
			if ($city) $this->api_parameters .= "<city>".$city."</city>";
			if ($state) $this->api_parameters .= "<state>".$state."</state>";
			if ($zip) $this->api_parameters .= "<zip>".$zip."</zip>";
			if ($country) $this->api_parameters .= "<country>".$country."</country>";
			$this->api_parameters .="</address>".
			"</createCustomerShippingAddressRequest>";
	}

	/**
	* Create CIM shipping address id.
	*
   * @param SHORT_TEXT Profile Id
   * @param SHORT_TEXT First name
   * @param SHORT_TEXT Last name
   * @param SHORT_TEXT Address
	* @param SHORT_TEXT City
	* @param SHORT_TEXT A valid two-character state code
	* @param SHORT_TEXT Zip
	* @param SHORT_TEXT Country
	* @param ARRAY
	*/
	function create_cim_shipping_address_id($profile_id,$first_name,$last_name,$address=NULL,$city=NULL,$state=NULL,$zip=NULL,$country=NULL)
	{
	 	$this->set_cim_shipping_address_parameters($profile_id,$first_name,$last_name,$address,$city,$state,$zip,$country);

      $response = http_download_file($this->url,NULL,true,false,'ocPortal',array($this->api_parameters),NULL,NULL,NULL,NULL,NULL,NULL,NULL,12.0,true);

      if ($response)
      {
         $response_result = $this->parse_cim_return($response,'shipping');
         $success = ($response_result[0] == 'OK') ? true : false;
         $result = array($success,$response_result[1],$response_result[2],$response_result[3]);
      } else
      {
         $result = array(false,NULL,NULL,NULL);
      }

      return $result;
	}

  /**
	* This function defines the parameters needed to make an API call for deleting a profile.
	*
	* @param SHORT_TEXT Profile Id
	*/
	function set_cim_delete_profile_parameters($profile_id)
	{
		$access_detail = $this->get_access_detail();

		if (get_option('ecommerce_test_mode')=='1')
		{
			$this->url = "https://apitest.authorize.net/xml/v1/request.api";
		} else
		{
			$this->url = "https://api.authorize.net/xml/v1/request.api";
		}

      $this->api_parameters =
			"<deleteCustomerProfileRequest xmlns=\"AnetApi/xml/v1/schema/AnetApiSchema.xsd\">".
			"<merchantAuthentication>".
			"<name>".$access_detail[0]."</name>".
			"<transactionKey>".$access_detail[1]."</transactionKey>".
			"</merchantAuthentication>".
			"<customerProfileId>".$profile_id."</customerProfileId>".
			"</deleteCustomerProfileRequest>";
	}

	/**
	 * Delete CIM profile.
	 *
	 * @param SHORT_TEXT Profile Id
	 * @param ARRAY
	 */
	function delete_cim_profile($profile_id)
	{
	 	$this->set_cim_delete_profile_parameters($profile_id);

      $response = http_download_file($this->url,NULL,true,false,'ocPortal',array($this->api_parameters),NULL,NULL,NULL,NULL,NULL,NULL,NULL,12.0,true);

      if ($response)
      {
         $response_result = $this->parse_cim_return($response,'delete');
         $success = ($response_result[0] == 'OK') ? true : false;
         $result = array($success,$response_result[1],$response_result[2],$response_result[3]);
      } else
      {
         $result = array(false,NULL,NULL,NULL);
      }

      return $result;
	}

	/**
	 * This function defines the parameters needed to make an API call for transaction.
	 *
	 * @param SHORT_TEXT Item Profile Id
	 * @param SHORT_TEXT Item Payment profile Id
	 * @param SHORT_TEXT Item Shipping address Id
	 * @param SHORT_TEXT Transaction amount (should include tax, shipping, and everything.)
	 * @param SHORT_TEXT Shipping amount
	 * @param SHORT_TEXT Shipping name
	 * @param SHORT_TEXT Shipping description
	 * @param SHORT_TEXT Tax amount
	 * @param SHORT_TEXT Tax name
	 * @param SHORT_TEXT Tax description
	 * @param SHORT_TEXT Item name
	 * @param SHORT_TEXT Item description
	 * @param SHORT_TEXT Item quantity
	 * @param SHORT_TEXT Item unit price
	 */
	function set_cim_transaction_parameters($profile_id,$payment_profile_id,$shipping_address_id,$amount,$shipping_amount='0.00',$shipping_name='Free Shipping',$shipping_description='Free',$tax_amount='0.00',$tax_name='Tax Free',$tax_description='Tax Free',$item_id=NULL,$item_name=NULL,$item_description=NULL,$item_quantity=NULL,$item_unit_price=NULL)
	{
		$access_detail = $this->get_access_detail();

		if (get_option('ecommerce_test_mode')=='1')
		{
			$this->url = "https://apitest.authorize.net/xml/v1/request.api";
		} else
		{
			$this->url = "https://api.authorize.net/xml/v1/request.api";
		}

      $this->api_parameters =
			"<createCustomerProfileTransactionRequest xmlns=\"AnetApi/xml/v1/schema/AnetApiSchema.xsd\">".
			"<merchantAuthentication>".
			"<name>".$access_detail[0]."</name>".
			"<transactionKey>".$access_detail[1]."</transactionKey>".
			"</merchantAuthentication>".
			"<transaction>".
			"<profileTransAuthOnly>".
			"<amount>".$amount."</amount>".
			"<tax>".
			"<amount>".$tax_amount."</amount>".
			"<name>".$tax_name."</name>".
			"<description>".$tax_description."</description>".
			"</tax>".
			"<shipping>".
			"<amount>".$shipping_amount."</amount>".
			"<name>".$shipping_name."</name>".
			"<description>".$shipping_description."</description>".
			"</shipping>";

			if ($item_id && $item_name && $item_description && $item_quantity && $item_unit_price)
			{
				$this->api_parameters .= "<lineItems>".
				"<itemId>".$item_id."</itemId>".
				"<name>".$item_name."</name>".
				"<description>".$item_description."</description>".
				"<quantity>".$item_quantity."</quantity>".
				"<unitPrice>".$item_unit_price."</unitPrice>";
				"</lineItems>";
			}

			$this->api_parameters .= "<customerProfileId>".$profile_id."</customerProfileId>".
			"<customerPaymentProfileId>".$payment_profile_id."</customerPaymentProfileId>".
			"<customerShippingAddressId>".$shipping_address_id."</customerShippingAddressId>".
			"</profileTransAuthOnly>".
			"</transaction>".
			"</createCustomerProfileTransactionRequest>";
	}

  /**
	* Make a transaction using CIM
	*
	* @param SHORT_TEXT Item Profile Id
	* @param SHORT_TEXT Item Payment profile Id
	* @param SHORT_TEXT Item Shipping address Id
   * @param SHORT_TEXT Transaction amount (should include tax, shipping, and everything.)
	* @param SHORT_TEXT Shipping amount
	* @param SHORT_TEXT Shipping name
	* @param SHORT_TEXT Shipping description
	* @param SHORT_TEXT Tax amount
	* @param SHORT_TEXT Tax name
	* @param SHORT_TEXT Tax description
	* @param SHORT_TEXT Item name
	* @param SHORT_TEXT Item description
	* @param SHORT_TEXT Item quantity
	* @param SHORT_TEXT Item unit price
	* @param ARRAY
	*/
	function do_cim_transaction($profile_id,$payment_profile_id,$shipping_address_id,$amount,$shipping_amount='0.00',$shipping_name='Free Shipping',$shipping_description='Free',$tax_amount='0.00',$tax_name='Tax Free',$tax_description='Tax Free',$item_id=NULL,$item_name=NULL,$item_description=NULL,$item_quantity=NULL,$item_unit_price=NULL)
	{
	 	$this->set_cim_transaction_parameters($profile_id,$payment_profile_id,$shipping_address_id,$amount,$shipping_amount,$shipping_name,$shipping_description,$tax_amount,$tax_name,$tax_description,$item_id,$item_name,$item_description,$item_quantity,$item_unit_price);

      $response = http_download_file($this->url,NULL,true,false,'ocPortal',array($this->api_parameters),NULL,NULL,NULL,NULL,NULL,NULL,NULL,12.0,true);

      if ($response)
      {
         $response_result = $this->parse_cim_return($response,'transaction');
         $success = ($response_result[0] == 'OK') ? true : false;
         $result = array($success,$response_result[1],$response_result[2],$response_result[3],$response_result[4],$response_result[5],$response_result[6],$response_result[7],$response_result[8]);
      } else
      {
         $result = array(false,NULL,NULL,NULL,NULL,NULL,NULL,NULL,NULL);
      }

      return $result;
	}

	/**
	 * Function to return profile ids and types in serialized array format
	 *
	 *	@param	AUTO_LINK	Transaction id
	 * @param	AUTO_LINK 	Parent transaction id (Subscription profile id of a particular subscription)
	 * @return	?SHORT_TEXT	Type, profile id data in serialized string
	 */
	function	map_payment_ids($txn_id,$parent_txn_id)
	{
		$data = array('id'=>$parent_txn_id,'type'=>'arb');
		return serialize($data);
	}

	/**
	 * Make a subscription update button.
	 *
	 * @param  ID_TEXT		The purchase ID.
	 *	@param	ID_TEXT		Serialized data which stores transaction keys
	 * @return tempcode		The button
	 */
	function make_update_button($purchase_id,$auto_found_key=NULL)
	{
		$url	=	build_url(array('page'=>'subscriptions','type'=>'update','id'=>$purchase_id),get_module_zone('subscriptions'));
		$url	=	$url->evaluate();

		return do_template('ECOM_UPDATE_BUTTON_VIA_AUTHORIZE',array('URL'=>$url));
	}

	/**
	 * Get form fields.
	 *
	 * @param ID Subscription Id
	 * @return tempcode The form fields
	 */
	function get_update_form_fields($subscription_id)
	{
		require_code('ecommerce');

		list($card_det,$hidden)	= get_general_update_form_fields($subscription_id,$this);

		/*	General transaction fields returned from ecommerce api, attaching the details to the form.
			The fields can add/remove here, depending on the customised payment gateway features.
		*/
		$fields	=	new ocp_tempcode();
		foreach($card_det as $field)
		{
			$fields->attach($field);
		}

		$verified_account_logo	=	$this->get_confidence_logos();
		return array($fields,$hidden,$verified_account_logo);
	}

	/**
	 * Update subscription
	 *
	 * @param  ID           Subscription Id
	 * @param  SHORT_TEXT	Cardholder first name
	 * @param  SHORT_TEXT	Cardholder last name
	 * @param  SHORT_TEXT	Card number
	 * @param  SHORT_TEXT	Card Type
	 * @param  SHORT_TEXT	Card Expiry date
	 * @param  SHORT_TEXT	Card CV2 number (security number)
	 * @param  SHORT_TEXT	Address for card
	 * @param  SHORT_TEXT	City for card
	 * @param  SHORT_TEXT	State for card
	 * @param  SHORT_TEXT	Zip/post-code for card
	 * @param  SHORT_TEXT	Country for card
	 * @param  ID Member Id (NULL: Admin)
	 * @return ARRAY
	 */
   function update_subscription($subscription_id,$first_name,$last_name,$card_number,$card_type,$expiry_date,$cv2,$address1,$city,$state,$zip,$country,$member_id=NULL)
   {
   	$temp	= $GLOBALS['SITE_DB']->query_value('subscriptions','s_auto_fund_key',array('id'=>$subscription_id));
   	$data	= unserialize($temp);
		$authorize_subscription_id	= $data['id'];

		$expiry_month = substr($expiry_date,5,2);
		$expiry_year  = substr($expiry_date,0,4);
		$card_number  = str_replace('-','',str_replace(' ','',$card_number));

      $this->set_update_api_parameters($authorize_subscription_id,$first_name,$last_name,$card_number,$card_type,$expiry_month,$expiry_year,$cv2,$address1,$city,$state,$zip,$country,$member_id);

      $response = http_download_file($this->url,NULL,true,false,'ocPortal',array($this->api_parameters),NULL,NULL,NULL,NULL,NULL,NULL,NULL,12.0,true);

      if(!is_null($response))
      {
         $response_result = $this->parse_return($response);
         $success = ($response_result[0] == 'OK') ? true : false;
         $result  = array($success,$response_result[1],$response_result[2],$response_result[2]);
      }
      else
      {
         $result  = array(false,NULL,NULL,NULL);
      }

      return $result;
   }

   /**
    * This function defines the parameters needed to make an API call.
	 *
	 * @param  ID           Subscription Id
	 * @param  SHORT_TEXT	Cardholder first name
	 * @param  SHORT_TEXT	Cardholder last name
	 * @param  SHORT_TEXT	Card number
	 * @param  SHORT_TEXT	Card Type
	 * @param  SHORT_TEXT	Card Expiry date
	 * @param  SHORT_TEXT	Card CV2 number (security number)
	 * @param  SHORT_TEXT	Address for card
	 * @param  SHORT_TEXT	City for card
	 * @param  SHORT_TEXT	State for card
	 * @param  SHORT_TEXT	Zip/post-code for card
	 * @param  SHORT_TEXT	Country for card
	 * @param  ID Member Id (NULL: Admin)
	 */
	function set_update_api_parameters($authorize_subscription_id,$first_name,$last_name,$card_number,$card_type,$expiry_month,$expiry_year,$cv2,$address1,$city,$state,$zip,$country,$member_id=NULL)
	{
		// NOTE: The first name and last name (and possibly others) need to be
		// given (usually via a custom profile field) otherwise Authorize.net
		// won't work.
		$access_detail = $this->get_access_detail($member_id);

		if (get_option('ecommerce_test_mode')=='1')
		{
			//URL for test account
			$this->url = "https://apitest.authorize.net/xml/v1/request.api";
		} else
		{
			//URL for live account
			$this->url = "https://api.authorize.net/xml/v1/request.api";
		}

      $this->api_parameters =
        "<ARBUpdateSubscriptionRequest xmlns=\"AnetApi/xml/v1/schema/AnetApiSchema.xsd\">".
        "<merchantAuthentication>".
        "<name>" . $access_detail[0] . "</name>".
        "<transactionKey>" . $access_detail[1] . "</transactionKey>".
        "</merchantAuthentication>" .
        "<subscriptionId>" . $authorize_subscription_id . "</subscriptionId>".
        "<subscription>".
        "<payment>".
        "<creditCard>".
        "<cardNumber>" . $card_number ."</cardNumber>".
        "<expirationDate>" . $expiry_year.'-'.$expiry_month . "</expirationDate>".
        "</creditCard>".
        "</payment>".
        "<billTo>";

		 if ($first_name) $this->api_parameters .= "<firstName>". $first_name . "</firstName>";
       if ($last_name) $this->api_parameters .= "<lastName>". $last_name . "</lastName>";
       if ($address1) $this->api_parameters .= "<address>" . $address1 . "</address>";
       if ($city) $this->api_parameters .=  "<city>" . $city . "</city>";
       if ($state) $this->api_parameters .= "<state>" . $state . "</state>";
       if($zip) $this->api_parameters .= "<zip>" . $zip . "</zip>";
       if($country) $this->api_parameters .= "<country>" . $country . "</country>";

       $this->api_parameters .=
        "</billTo>".
        "</subscription>".
        "</ARBUpdateSubscriptionRequest>";
	}

	/**
	* Find whether the hook auto-cancels (if it does, auto cancel the given trans-id).
	*
	* @param ID Subscription Id
	* @param ID Member Id (NULL: Admin)
	* @return ?boolean	True: yes. False: no. (NULL: cancels via a user-URL-directioning)
	*/
	function cancel_subscription($subscription_id,$member_id)
	{
		$this->set_cancelation_api_parameters($subscription_id,$member_id);

		$response = http_download_file($this->url,NULL,true,false,'ocPortal',array($this->api_parameters),NULL,NULL,NULL,NULL,NULL,NULL,NULL,12.0,true);

		if(!is_null($response))
		{
			$response_result = $this->parse_return($response);
			$success = ($response_result[0] == 'OK') ? true : false;
			$result  = array($success,$response_result[1],$response_result[2],$response_result[3]);
		}
		else
		{
			$success = false;
			$result  = array(false,NULL,NULL,NULL);
		}

		return $success;
	}
}