<?php /*

 ocPortal
 Copyright (c) ocProducts, 2004-2012

 See text/EN/licence.txt for full licencing information.


 NOTE TO PROGRAMMERS:
   Do not edit this file. If you need to make changes, save your changed file to the appropriate *_custom folder
   **** If you ignore this advice, then your website upgrades (e.g. for bug fixes) will likely kill your changes ****

*/

/**
 * @license		http://opensource.org/licenses/cpal_1.0 Common Public Attribution License
 * @copyright	ocProducts Ltd
 * @package		core
 */

/**
 * Check RBLs to see if we need to block this user.
 *
 * @param boolean		Whether this is a page level check (i.e. we won't consider blocks or approval, just ban setting)
 */
function check_rbls($page_level=false)
{
	$user_ip=get_ip_address();

	// Check ocPortal bans / caching
	require_code('support2');
	$is_already_ip_banned=ip_banned($user_ip,true,true);
	if ($is_already_ip_banned===true) critical_error('BANNED');
	if ($is_already_ip_banned===false) return; // Cached that we're not banned

	// Check exclusions
	$exclusions=explode(',',get_option('spam_check_exclusions'));
	foreach ($exclusions as $e)
	{
		if (trim($e)==$user_ip) return;
	}

	// What are we blocking? Hard-coded settings for the particular supported block lists
	$block=array(
		'tornevall_abuse'=>true,			// TornevallRBL: Block on 'abuse'
		'tornevall_anonymous'=>true,		// TornevallRBL: Block on anonymous access (anonymizers, TOR, etc)
		'tornevall_blitzed'=>true,			// TornevallRBL: Block if host are found in the Blitzed RBL (R.I.P)
		'tornevall_checked'=>false,		// TornevallRBL: Block anything that has been checked
		'tornevall_elite'=>true,			// TornevallRBL: Block elite proxies (proxies with high anonymity)
		'tornevall_error'=>false,			// TornevallRBL: Block proxies that has been tested but failed
		'tornevall_timeout'=>false,		// TornevallRBL: Block proxies that has been tested but timed out
		'tornevall_working'=>true,			// TornevallRBL: Block proxies that has been tested and works
		'efnet_openproxy'=>true,			// EFNet: Block open proxies registered at rbl.efnet.org
		'efnet_spamtrap50'=>false,			// EFNet: Block trojan spreading client (IRC-based)
		'efnet_spamtrap666'=>false,		// EFNet: Block known trojan infected/spreading client (IRC-based)
		'efnet_tor'=>true,					// EFNet: Block TOR Proxies
		'efnet_drones'=>false,				// EFNet: Drones/Flooding (IRC-based)
		'njabl_dialup'=>false,				// Njabl: Block dialups/dynamic ip-ranges (Be careful!)
		'njabl_formmail'=>true,				// Njabl: Block systems with insecure scripts that make them into open relays
		'njabl_multi'=>false,				// Njabl: Block multi-stage open relays (Don't know what this is? Leave it alone)
		'njabl_openproxy'=>true,			// Njabl: Block open proxy servers
		'njabl_passive'=>false,				// Njabl: Block passively detected 'bad hosts' (Don't know what this is? Leave it alone)
		'njabl_relay'=>false,				// Njabl: Block open relays (as in e-mail-open-relays - be careful)
		'njabl_spam'=>false					// Njabl: lock spam sources (Again, as in e-mail
	);

	// Handle the return data for the different RBLs
	$blockdetails=mixed();
	$blockedby=mixed();
	$confidencelevel=mixed();
	$rbllist=explode(',',get_option('spam_block_lists'));
	foreach ($rbllist as $rbl)
	{
		// Blocking based on opm.tornevall.org settings (used by default because stopforumspam syndicates to this and ask us to check this first, for performance)
		// http://dnsbl.tornevall.org/?do=usage
		$rtornevall=array(
			'tornevall_checked'=>1,
			'tornevall_working'=>2,
			'tornevall_blitzed'=>4,
			'tornevall_timeout'=>8,
			'tornevall_error'=>16,
			'tornevall_elite'=>32,
			'tornevall_abuse'=>64,
			'tornevall_anonymous'=>128
		);
		if (strpos($rbl,'tornevall.org')!==false)
		{
			if (!is_null($confidencelevel)) continue; // We know better than this RBL can tell us, so stick with what we know
			$rblresponse=rblresolve($user_ip,$rbl,$page_level);
			if (is_null($rblresponse)) continue; // Error

			foreach ($rtornevall as $rbl_t=>$rbl_tc)
			{
				if ((($rblresponse[3] & $rbl_tc)!=0) && ($block[$rbl_t]))
				{
					$blockdetails=$rblresponse[3];
					$blockedby=preg_replace('#^\*\.#','',$rbl);
					break;
				}
			}
			continue;
		}

		// Blocking based on njabl.org settings (not used by default)
		// http://njabl.org/use.html
		$rnjabl=array(
			'njabl_relay'=>2,
			'njabl_dialup'=>3,
			'njabl_spam'=>4,
			'njabl_multi'=>5,
			'njabl_passive'=>6,
			'njabl_formmail'=>8,
			'njabl_openproxy'=>9
		);
		if (strpos($rbl,'njabl.org')!==false)
		{
			if (!is_null($confidencelevel)) continue; // We know better than this RBL can tell us, so stick with what we know
			$rblresponse=rblresolve($user_ip,$rbl,$page_level);
			if (is_null($rblresponse)) continue; // Error

			foreach ($rnjabl as $njcheck=>$value)
			{
				if (($rblresponse[3]==$value) && ($block[$njcheck]))
				{
					$blockdetails=$rblresponse[3];
					$blockedby=preg_replace('#^\*\.#','',$rbl);
					break;
				}
			}
			continue;
		}

		// Blocking based on efnet.org settings (not used by default)
		// http://efnetrbl.org/
		$refnet=array(
			'efnet_openproxy'=>1,
			'efnet_spamtrap666'=>2,
			'efnet_spamtrap50'=>3,
			'efnet_tor'=>4,
			'efnet_drones'=>5
		);
		if (strpos($rbl,'efnet.org')!==false)
		{
			if (!is_null($confidencelevel)) continue; // We know better than this RBL can tell us, so stick with what we know
			$rblresponse=rblresolve($user_ip,$rbl,$page_level);
			if (is_null($rblresponse)) continue; // Error

			foreach ($refnet as $efcheck=>$value) 
			{
				if (($rblresponse[3]==$value) && ($block[$njcheck]))
				{
					$blockdetails=$rblresponse[3];
					$blockedby=preg_replace('#^\*\.#','',$rbl);
					break;
				}
			}
			continue;
		}

		// Blocking based on HTTP:BL settings (not used by default, because it requires getting a key)
		// http://www.projecthoneypot.org/httpbl_api.php
		if (strpos($rbl,'dnsbl.httpbl.org')!==false)
		{
			if (strpos($rbl,'*')===false) // Fix a misconfiguration based on the admin copy and pasting the given HTTP:BL setup example
			{
				$rbl=str_replace('7.1.1.127','*',$rbl);
			}

			$rblresponse=rblresolve($user_ip,$rbl,$page_level);
			if (is_null($rblresponse)) continue; // Error

			$_confidencelevel=floatval($rblresponse[2])/255.0;
			if ($_confidencelevel!=0.0)
			{
				$spam_stale_threshold=intval(get_option('spam_stale_threshold'));

				if (intval($rblresponse[1])>$spam_stale_threshold) break; // We know this IP is stale now so don't check other RBLs as no others support stale checks

				if (($confidencelevel===NULL) || ($_confidencelevel>$confidencelevel))
				{
					//$confidencelevel=$_confidencelevel;	Actually, this is a threat level, not a confidence level. If it's not zero, it is 100% confidence.
					$confidencelevel=1.0;
					$blockdetails=$rblresponse[3];
					$blockedby='dnsbl.httpbl.org';
				}
				break;
			}
			continue;
		}

		// Unknown RBL, basic support only
		if ($rblresponse[3]!=0)
		{
			if (!is_null($confidencelevel)) continue; // We know better than this RBL can tell us, so stick with what we know
			$rblresponse=rblresolve($user_ip,$rbl,$page_level);
			if (is_null($rblresponse)) continue; // Error

			$blockdetails=$rblresponse[3];
			$blockedby=preg_replace('#^\*\.#','',$rbl);
			break;
		}
	}

	// Now deal with it
	if ($blockdetails!==NULL) // If there's a block
	{
		if ($confidencelevel===NULL) $confidencelevel=floatval(get_option('implied_spammer_confidence'))/100.0;
		handle_perceived_spammer($user_ip,$confidencelevel,$blockedby,$page_level);
	} else
	{
		require_code('failure');
		add_ip_ban($user_ip,'',time()+60*intval(get_option('spam_cache_time')),false); // Mark a negative ban (i.e. cache)
	}
}

/**
 * Do an RBL lookup.
 *
 * @param  IP			The IP address to lookup
 * @param  ID_TEXT	The RBL domain
 * @param boolean		Whether this is a page level check (i.e. we won't consider blocks or approval, just ban setting)
 * @return ?array		Return result (NULL: error)
 */
function rblresolve($ip,$rbldomain,$page_level)
{
	if (strpos($ip,'.')!==false) // ipv4
	{
		$arpa=implode('.',array_reverse(explode('.',$ip)));
	} else // ipv6
	{
		$_ip=explode(':',$ip);
		$normalised_ip='';
		$normalised_ip.=str_pad('',(4*(8-count($_ip))),'0000',STR_PAD_LEFT); // Fill out trimmed 0's on left
		foreach ($_ip as $seg) // Copy rest in
			$normalised_ip.=str_pad($seg,4,'0',STR_PAD_LEFT); // Pad out each component in full, building up $normalised_ip
		$arpa=implode('.',array_reverse(preg_split('//',$normalised_ip,NULL,PREG_SPLIT_NO_EMPTY)));
	}

	$lookup=str_replace('*',$arpa,$rbldomain);

	$_result=gethostbyname($lookup);
	$result=explode('.',$_result);

	if (implode('.',$result)==$lookup) // This is how gethostbyname indicates an error happened; however it likely actually means no block happened (as the RBL returned no data on the IP)
	{
		return NULL;
	}

	if ($result[0]!='127') // This is how the RBL indicates an error happened
	{
		if (!$page_level)
		{
			require_code('failure');
			$error=do_lang('_ERROR_CHECKING_FOR_SPAMMERS',$rbldomain,$_result);
			relay_error_notification($error,false,'error_occurred');
		}
		return NULL;
	}

	// Some kind of response
	return $result;
} 

/**
 * Deal with a perceived spammer.
 *
 * @param IP			IP address
 * @param float		Confidence level (0.0 to 1.0)
 * @param ID_TEXT		Identifier for whatever did the blocking
 * @param boolean		Whether this is a page level check (i.e. we won't consider blocks or approval, just ban setting)
 */
function handle_perceived_spammer($user_ip,$confidencelevel,$blockedby,$page_level)
{
	// Ban
	$spam_ban_threshold=intval(get_option('spam_ban_threshold'));
	if (intval($confidencelevel*100.0)>=$spam_ban_threshold)
	{
		require_code('failure');
		add_ip_ban($user_ip,do_lang('IPBAN_LOG_AUTOBAN_ANTISPAM',$blockedby),time()+60*intval(get_option('spam_cache_time')));

		require_code('notifications');
		$subject=do_lang('NOTIFICATION_SPAM_CHECK_BLOCK_SUBJECT_BAN',$user_ip,$blockedby,NULL,get_site_default_lang());
		$message=do_lang('NOTIFICATION_SPAM_CHECK_BLOCK_MESSAGE_BAN',$user_ip,$blockedby,NULL,get_site_default_lang());
		dispatch_notification('spam_check_block',NULL,$subject,$message,NULL,A_FROM_SYSTEM_PRIVILEGED);

		warn_exit(do_lang_tempcode('STOPPED_BY_ANTISPAM',escape_html($user_ip),escape_html($blockedby)));
	}

	// Block
	if (!$page_level)
	{
		$spam_block_threshold=intval(get_option('spam_block_threshold'));
		if (intval($confidencelevel*100.0)>=$spam_block_threshold)
		{
			require_code('notifications');
			$subject=do_lang('NOTIFICATION_SPAM_CHECK_BLOCK_SUBJECT_BLOCK',$user_ip,$blockedby,NULL,get_site_default_lang());
			$message=do_lang('NOTIFICATION_SPAM_CHECK_BLOCK_MESSAGE_BLOCK',$user_ip,$blockedby,NULL,get_site_default_lang());
			dispatch_notification('spam_check_block',NULL,$subject,$message,NULL,A_FROM_SYSTEM_PRIVILEGED);

			warn_exit(do_lang_tempcode('STOPPED_BY_ANTISPAM',escape_html($user_ip),escape_html($blockedby)));
		}
	}

	// Require approval
	$spam_approval_threshold=intval(get_option('spam_approval_threshold'));
	if (intval($confidencelevel*100.0)>=$spam_approval_threshold)
	{
		global $SPAM_REMOVE_VALIDATION;
		$SPAM_REMOVE_VALIDATION=true;

		require_code('notifications');
		$subject=do_lang('NOTIFICATION_SPAM_CHECK_BLOCK_SUBJECT_APPROVE',$user_ip,$blockedby,NULL,get_site_default_lang());
		$message=do_lang('NOTIFICATION_SPAM_CHECK_BLOCK_MESSAGE_APPROVE',$user_ip,$blockedby,NULL,get_site_default_lang());
		dispatch_notification('spam_check_block',NULL,$subject,$message,NULL,A_FROM_SYSTEM_PRIVILEGED);
	}
}

/**
 * Check RBLs to see if we need to block this user.
 *
 * @param ?string		Check this particular username that has just been supplied (NULL: none)
 * @param ?string		Check this particular email address that has just been supplied (NULL: none)
 */
function check_stopforumspam($username=NULL,$email=NULL)
{
	// http://www.stopforumspam.com/usage

	if (get_option('spam_block_lists')=='') return;

	// Check exclusions
	$user_ip=get_ip_address();
	$exclusions=explode(',',get_option('spam_check_exclusions'));
	foreach ($exclusions as $e)
	{
		if (trim($e)==$user_ip) return;
	}

	// Are we really going to check that username?
	if (get_option('spam_check_usernames')=='0') $username=NULL;
	$confidencelevel=mixed();

	// Do the query with every detail we have
	require_code('files');
	require_code('character_sets');
	$key=get_option('stopforumspam_api_key');
	$url='http://www.stopforumspam.com/api?f=serial&unix&confidence&ip='.urlencode($user_ip);
	if (!is_null($username)) $url.='&username='.urlencode(convert_to_internal_encoding($username,get_charset(),'utf-8'));
	if (!is_null($email)) $url.='&email='.urlencode(convert_to_internal_encoding($email,get_charset(),'utf-8'));
	if ($key!='') $url.='&api_key='.urlencode($key); // Key not needed for read requests, but give it as a courtesy
	$_result=http_download_file($url,NULL,false);

	$result=@unserialize($_result);
	if ($result!==false)
	{
		if ($result['success'])
		{
			foreach (array('username','email','ip') as $criterion)
			{
				if (array_key_exists($criterion,$result))
				{
					$c=$result[$criterion];
					if ($c['appears']==1)
					{
						$spam_stale_threshold=intval(get_option('spam_stale_threshold'));
						$days_ago=floatval(time()-intval($c['lastseen']))/(24.0*60.0*60.0);
						if ($days_ago<=floatval($spam_stale_threshold))
						{
							$_confidencelevel=$c['confidence']/100.0;
							if (($confidencelevel===NULL) || ($_confidencelevel>$confidencelevel))
							{
								$confidencelevel=$_confidencelevel;
							}
						}

						// NB: frequency figure is ignored, not used in our algorithm
					}
				}
			}
		} else
		{
			require_code('failure');
			$error=do_lang('_ERROR_CHECKING_FOR_SPAMMERS','stopforumspam.com',$result['error']);
			relay_error_notification($error,false,'error_occurred');
		}
	} else
	{
		require_code('failure');
		$error=do_lang('ERROR_CHECKING_FOR_SPAMMERS','stopforumspam.com');
		relay_error_notification($error,false,'error_occurred');
	}

	if ($confidencelevel!==NULL)
	{
		handle_perceived_spammer($user_ip,$confidencelevel,'stopforumspam.com',false);
	}
}
