// UIWebView+AFNetworking.h
//
// Copyright (c) 2013-2014 AFNetworking (http://afnetworking.com)
//
// Permission is hereby granted, free of charge, to any person obtaining a copy
// of this software and associated documentation files (the "Software"), to deal
// in the Software without restriction, including without limitation the rights
// to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
// copies of the Software, and to permit persons to whom the Software is
// furnished to do so, subject to the following conditions:
//
// The above copyright notice and this permission notice shall be included in
// all copies or substantial portions of the Software.
//
// THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
// IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
// FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
// AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
// LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
// OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
// THE SOFTWARE.
import Foundation
import Availability
#if defined(__IPHONE_OS_VERSION_MIN_REQUIRED)
import UIKit

/**
 This category adds methods to the UIKit framework's `UIWebView` class. The methods in this category provide increased control over the request cycle, including progress monitoring and success / failure handling.
 
 @discussion When using these category methods, make sure to assign `delegate` for the web view, which implements `–webView:shouldStartLoadWithRequest:navigationType:` appropriately. This allows for tapped links to be loaded through AFNetworking, and can ensure that `canGoBack` & `canGoForward` update their values correctly.
 */
extension UIWebView {
    /**
     The request serializer used to serialize requests made with the `-loadRequest:...` category methods. By default, this is an instance of `AFHTTPRequestSerializer`.
     */
    weak var requestSerializer: AFURLRequestSerialization?
    /**
     The response serializer used to serialize responses made with the `-loadRequest:...` category methods. By default, this is an instance of `AFHTTPResponseSerializer`.
     */
    weak var responseSerializer: AFURLResponseSerialization?
    /**
     Asynchronously loads the specified request.
     
     @param request A URL request identifying the location of the content to load. This must not be `nil`.
     @param progress A block object to be called when an undetermined number of bytes have been downloaded from the server. This block has no return value and takes three arguments: the number of bytes read since the last time the download progress block was called, the total bytes read, and the total bytes expected to be read during the request, as initially determined by the expected content size of the `NSHTTPURLResponse` object. This block may be called multiple times, and will execute on the main thread.
     @param success A block object to be executed when the request finishes loading successfully. This block returns the HTML string to be loaded by the web view, and takes two arguments: the response, and the response string.
     @param failure A block object to be executed when the request operation finishes unsuccessfully, or that finishes successfully, but encountered an error while parsing the response data. This block has no return value and takes a single argument: the error that occurred.
     */

    func loadRequest(request: NSURLRequest, progress: (bytesWritten: Int, totalBytesWritten: Int64, totalBytesExpectedToWrite: Int64) -> Void, success: (response: NSHTTPURLResponse, HTML: String) -> String, failure: (error: NSError) -> Void) {
        self.loadRequest(request, MIMEType: nil, textEncodingName: nil, progress: progress, success: {(response: NSHTTPURLResponse, data: NSData) -> NSData in
            var stringEncoding = NSUTF8StringEncoding
            if response.textEncodingName! {
                var encoding = CFStringConvertIANACharSetNameToEncoding((response.textEncodingName! as! CFString))
                if encoding != kCFStringEncodingInvalidId {
                    stringEncoding = CFStringConvertEncodingToNSStringEncoding(encoding)
                }
            }
            var string = String(data: data, encoding: stringEncoding)
            if success {
                string = success(response, string)
            }
            return string.dataUsingEncoding(stringEncoding)!
        }, failure: failure)
    }
    /**
     Asynchronously loads the data associated with a particular request with a specified MIME type and text encoding.
    
     @param request A URL request identifying the location of the content to load. This must not be `nil`.
     @param MIMEType The MIME type of the content. Defaults to the content type of the response if not specified.
     @param textEncodingName The IANA encoding name, as in `utf-8` or `utf-16`. Defaults to the response text encoding if not specified.
     @param progress A block object to be called when an undetermined number of bytes have been downloaded from the server. This block has no return value and takes three arguments: the number of bytes read since the last time the download progress block was called, the total bytes read, and the total bytes expected to be read during the request, as initially determined by the expected content size of the `NSHTTPURLResponse` object. This block may be called multiple times, and will execute on the main thread.
     @param success A block object to be executed when the request finishes loading successfully. This block returns the data to be loaded by the web view and takes two arguments: the response, and the downloaded data.
     @param failure A block object to be executed when the request operation finishes unsuccessfully, or that finishes successfully, but encountered an error while parsing the response data. This block has no return value and takes a single argument: the error that occurred.
     */

    func loadRequest(request: NSURLRequest, MIMEType: String, textEncodingName: String, progress: (bytesWritten: Int, totalBytesWritten: Int64, totalBytesExpectedToWrite: Int64) -> Void, success: (response: NSHTTPURLResponse, data: NSData) -> NSData, failure: (error: NSError) -> Void) {
        NSParameterAssert(request)
        if self.af_HTTPRequestOperation {
            self.af_HTTPRequestOperation.cancel()
        }
        do {
            request = try self.requestSerializer.requestBySerializingRequest(request, withParameters: nil)
        }
        catch let error {
        }
        self.af_HTTPRequestOperation = AFHTTPRequestOperation(request: request)
        self.af_HTTPRequestOperation.responseSerializer = self.responseSerializer
        weak var weakSelf = self
        self.af_HTTPRequestOperation.downloadProgressBlock = progress
        self.af_HTTPRequestOperation.setCompletionBlockWithSuccess({(operation: AFHTTPRequestOperation, responseObject: id __unused) -> Void in
            var data = success ? success(operation.response, operation.responseData) : operation.responseData
//#pragma clang diagnostic push
//#pragma clang diagnostic ignored "-Wgnu"
            var strongSelf = weakSelf
            strongSelf.loadData(data, MIMEType: (MIMEType ?? operation.response.MIMEType!), textEncodingName: (textEncodingName ?? operation.response.textEncodingName!), baseURL: operation.response.URL!)
//#pragma clang diagnostic pop
        }, failure: {(operation: AFHTTPRequestOperation, error: NSError) -> Void in
            if failure {
                failure(error)
            }
        })
        self.af_HTTPRequestOperation.start()
    }


    func requestSerializer() -> AFURLRequestSerialization {
        weak var af_defaultRequestSerializer: AFURLRequestSerialization? = nil
        var onceToken: dispatch_once_t
        dispatch_once(onceToken, {() -> Void in
            self.af_defaultRequestSerializer = AFHTTPRequestSerializer.serializer()
        })
//#pragma clang diagnostic push
//#pragma clang diagnostic ignored "-Wgnu"
        return objc_getAssociatedObject(self, #selector(self.requestSerializer)) ?? af_defaultRequestSerializer
//#pragma clang diagnostic pop
    }

    func setRequestSerializer(requestSerializer: AFURLRequestSerialization) {
        objc_setAssociatedObject(self, #selector(self.requestSerializer), requestSerializer, OBJC_ASSOCIATION_RETAIN_NONATOMIC)
    }

    func responseSerializer() -> AFURLResponseSerialization {
        weak var af_defaultResponseSerializer: AFURLResponseSerialization? = nil
        var onceToken: dispatch_once_t
        dispatch_once(onceToken, {() -> Void in
            self.af_defaultResponseSerializer = AFHTTPResponseSerializer.serializer()
        })
//#pragma clang diagnostic push
//#pragma clang diagnostic ignored "-Wgnu"
        return objc_getAssociatedObject(self, #selector(self.responseSerializer)) ?? af_defaultResponseSerializer
//#pragma clang diagnostic pop
    }

    func setResponseSerializer(responseSerializer: AFURLResponseSerialization) {
        objc_setAssociatedObject(self, #selector(self.responseSerializer), responseSerializer, OBJC_ASSOCIATION_RETAIN_NONATOMIC)
    }
// MARK: -
}
#endif
// UIWebView+AFNetworking.m
//
// Copyright (c) 2013-2014 AFNetworking (http://afnetworking.com)
//
// Permission is hereby granted, free of charge, to any person obtaining a copy
// of this software and associated documentation files (the "Software"), to deal
// in the Software without restriction, including without limitation the rights
// to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
// copies of the Software, and to permit persons to whom the Software is
// furnished to do so, subject to the following conditions:
//
// The above copyright notice and this permission notice shall be included in
// all copies or substantial portions of the Software.
//
// THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
// IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
// FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
// AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
// LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
// OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
// THE SOFTWARE.
import ObjectiveC
#if defined(__IPHONE_OS_VERSION_MIN_REQUIRED)
extension UIWebView {
    var af_HTTPRequestOperation: AFHTTPRequestOperation!


    func af_HTTPRequestOperation() -> AFHTTPRequestOperation {
        return (objc_getAssociatedObject(self, #selector(self.af_HTTPRequestOperation)) as! AFHTTPRequestOperation)
    }

    func af_setHTTPRequestOperation(operation: AFHTTPRequestOperation) {
        objc_setAssociatedObject(self, #selector(self.af_HTTPRequestOperation), operation, OBJC_ASSOCIATION_RETAIN_NONATOMIC)
    }
}
// MARK: -