// UIProgressView+AFNetworking.h
//
// Copyright (c) 2013-2014 AFNetworking (http://afnetworking.com)
//
// Permission is hereby granted, free of charge, to any person obtaining a copy
// of this software and associated documentation files (the "Software"), to deal
// in the Software without restriction, including without limitation the rights
// to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
// copies of the Software, and to permit persons to whom the Software is
// furnished to do so, subject to the following conditions:
//
// The above copyright notice and this permission notice shall be included in
// all copies or substantial portions of the Software.
//
// THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
// IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
// FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
// AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
// LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
// OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
// THE SOFTWARE.
import Foundation
import Availability
#if defined(__IPHONE_OS_VERSION_MIN_REQUIRED)
import UIKit

/**
 This category adds methods to the UIKit framework's `UIProgressView` class. The methods in this category provide support for binding the progress to the upload and download progress of a session task or request operation.
 */
extension UIProgressView {
    ///------------------------------------
    /// @name Setting Session Task Progress
    ///------------------------------------
    /**
     Binds the progress to the upload progress of the specified session task.
     
     @param task The session task.
     @param animated `YES` if the change should be animated, `NO` if the change should happen immediately.
     */
#if __IPHONE_OS_VERSION_MIN_REQUIRED >= 70000
    func setProgressWithUploadProgressOfTask(task: NSURLSessionUploadTask, animated: Bool) {
        task.addObserver(self, forKeyPath: "state", options: [], context: AFTaskCountOfBytesSentContext)
        task.addObserver(self, forKeyPath: "countOfBytesSent", options: [], context: AFTaskCountOfBytesSentContext)
        self.af_setUploadProgressAnimated(animated)
    }
#endif
    /**
     Binds the progress to the download progress of the specified session task.
    
     @param task The session task.
     @param animated `YES` if the change should be animated, `NO` if the change should happen immediately.
     */
#if __IPHONE_OS_VERSION_MIN_REQUIRED >= 70000

    func setProgressWithDownloadProgressOfTask(task: NSURLSessionDownloadTask, animated: Bool) {
        task.addObserver(self, forKeyPath: "state", options: [], context: AFTaskCountOfBytesReceivedContext)
        task.addObserver(self, forKeyPath: "countOfBytesReceived", options: [], context: AFTaskCountOfBytesReceivedContext)
        self.af_setDownloadProgressAnimated(animated)
    }
#endif
    ///------------------------------------
    /// @name Setting Session Task Progress
    ///------------------------------------
    /**
     Binds the progress to the upload progress of the specified request operation.
    
     @param operation The request operation.
     @param animated `YES` if the change should be animated, `NO` if the change should happen immediately.
     */

    func setProgressWithUploadProgressOfOperation(operation: AFURLConnectionOperation, animated: Bool) {
        weak var weakSelf = self
        var original = operation.uploadProgress
        operation.uploadProgressBlock = {(bytesWritten: Int, totalBytesWritten: Int64, totalBytesExpectedToWrite: Int64) -> Void in
            if original {
                original(bytesWritten, totalBytesWritten, totalBytesExpectedToWrite)
            }
            dispatch_async(dispatch_get_main_queue(), {() -> Void in
                if totalBytesExpectedToWrite > 0 {
                    var strongSelf = weakSelf
                    strongSelf.setProgress((totalBytesWritten / (totalBytesExpectedToWrite * 1.0)), animated: animated)
                }
            })
        }
    }
    /**
     Binds the progress to the download progress of the specified request operation.
    
     @param operation The request operation.
     @param animated `YES` if the change should be animated, `NO` if the change should happen immediately.
     */

    func setProgressWithDownloadProgressOfOperation(operation: AFURLConnectionOperation, animated: Bool) {
        weak var weakSelf = self
        var original = operation.downloadProgress
        operation.downloadProgressBlock = {(bytesRead: Int, totalBytesRead: Int64, totalBytesExpectedToRead: Int64) -> Void in
            if original {
                original(bytesRead, totalBytesRead, totalBytesExpectedToRead)
            }
            dispatch_async(dispatch_get_main_queue(), {() -> Void in
                if totalBytesExpectedToRead > 0 {
                    var strongSelf = weakSelf
                    strongSelf.setProgress((totalBytesRead / (totalBytesExpectedToRead * 1.0)), animated: animated)
                }
            })
        }
    }


    func af_uploadProgressAnimated() -> Bool {
        return CBool(Int(objc_getAssociatedObject(self, #selector(self.af_uploadProgressAnimated))))
    }

    func af_setUploadProgressAnimated(animated: Bool) {
        objc_setAssociatedObject(self, #selector(self.af_uploadProgressAnimated), (animated), OBJC_ASSOCIATION_RETAIN_NONATOMIC)
    }

    func af_downloadProgressAnimated() -> Bool {
        return CBool(Int(objc_getAssociatedObject(self, #selector(self.af_downloadProgressAnimated))))
    }

    func af_setDownloadProgressAnimated(animated: Bool) {
        objc_setAssociatedObject(self, #selector(self.af_downloadProgressAnimated), (animated), OBJC_ASSOCIATION_RETAIN_NONATOMIC)
    }
// MARK: -
#if __IPHONE_OS_VERSION_MIN_REQUIRED >= 70000
#endif
// MARK: -
// MARK: - NSKeyValueObserving

    override func observeValueForKeyPath(keyPath: String, ofObject object: AnyObject, change: __unused NSDictionary, context: UnsafeMutablePointer<Void>) {
#if __IPHONE_OS_VERSION_MIN_REQUIRED >= 70000
        if context == AFTaskCountOfBytesSentContext || context == AFTaskCountOfBytesReceivedContext {
            if (keyPath == NSStringFromSelector(#selector(self.countOfBytesSent))) {
                if object.countOfBytesExpectedToSend > 0 {
                    dispatch_async(dispatch_get_main_queue(), {() -> Void in
                        self.setProgress(object.countOfBytesSent / (object.countOfBytesExpectedToSend * 1.0), animated: self.af_uploadProgressAnimated)
                    })
                }
            }
            if (keyPath == NSStringFromSelector(#selector(self.countOfBytesReceived))) {
                if object.countOfBytesExpectedToReceive > 0 {
                    dispatch_async(dispatch_get_main_queue(), {() -> Void in
                        self.setProgress(object.countOfBytesReceived / (object.countOfBytesExpectedToReceive * 1.0), animated: self.af_downloadProgressAnimated)
                    })
                }
            }
            if (keyPath == NSStringFromSelector(#selector(self.state))) {
                if (object as! NSURLSessionTask).state == NSURLSessionTaskStateCompleted {
                                        do {
                        object.removeObserver(self, forKeyPath: NSStringFromSelector(#selector(self.state)))
                        if context == AFTaskCountOfBytesSentContext {
                            object.removeObserver(self, forKeyPath: NSStringFromSelector(#selector(self.countOfBytesSent)))
                        }
                        if context == AFTaskCountOfBytesReceivedContext {
                            object.removeObserver(self, forKeyPath: NSStringFromSelector(#selector(self.countOfBytesReceived)))
                        }
                    }                     catch let exception {
                    }
                }
            }
        }
#endif
    }
}
#endif
// UIProgressView+AFNetworking.m
//
// Copyright (c) 2013-2014 AFNetworking (http://afnetworking.com)
//
// Permission is hereby granted, free of charge, to any person obtaining a copy
// of this software and associated documentation files (the "Software"), to deal
// in the Software without restriction, including without limitation the rights
// to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
// copies of the Software, and to permit persons to whom the Software is
// furnished to do so, subject to the following conditions:
//
// The above copyright notice and this permission notice shall be included in
// all copies or substantial portions of the Software.
//
// THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
// IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
// FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
// AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
// LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
// OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
// THE SOFTWARE.
import ObjectiveC
#if defined(__IPHONE_OS_VERSION_MIN_REQUIRED)
#if __IPHONE_OS_VERSION_MIN_REQUIRED >= 70000
#endif
var AFTaskCountOfBytesSentContext = AFTaskCountOfBytesSentContext

var AFTaskCountOfBytesReceivedContext = AFTaskCountOfBytesReceivedContext

extension AFURLConnectionOperation {
    var uploadProgress = Void()
    var af_uploadProgressAnimated = false
    var downloadProgress = Void()
    var af_downloadProgressAnimated = false

    // Implemented in AFURLConnectionOperation
    // Implemented in AFURLConnectionOperation
}
// MARK: -