// UIImageView+AFNetworking.h
//
// Copyright (c) 2013-2014 AFNetworking (http://afnetworking.com)
//
// Permission is hereby granted, free of charge, to any person obtaining a copy
// of this software and associated documentation files (the "Software"), to deal
// in the Software without restriction, including without limitation the rights
// to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
// copies of the Software, and to permit persons to whom the Software is
// furnished to do so, subject to the following conditions:
//
// The above copyright notice and this permission notice shall be included in
// all copies or substantial portions of the Software.
//
// THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
// IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
// FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
// AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
// LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
// OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
// THE SOFTWARE.
import Foundation
import Availability
#if defined(__IPHONE_OS_VERSION_MIN_REQUIRED)
import UIKit

/**
 This category adds methods to the UIKit framework's `UIImageView` class. The methods in this category provide support for loading remote images asynchronously from a URL.
 */
extension UIImageView {
    ///----------------------------
    /// @name Accessing Image Cache
    ///----------------------------
    /**
     The image cache used to improve image loadiing performance on scroll views. By default, this is an `NSCache` subclass conforming to the `AFImageCache` protocol, which listens for notification warnings and evicts objects accordingly.
    */
    class func sharedImageCache() -> AFImageCache {
        var af_defaultImageCache: AFImageCache? = nil
        var oncePredicate: dispatch_once_t
        dispatch_once(oncePredicate, {() -> Void in
            self.af_defaultImageCache = AFImageCache()
            NSNotificationCenter.defaultCenter().addObserverForName(UIApplicationDidReceiveMemoryWarningNotification, object: nil, queue: NSOperationQueue.mainQueue(), usingBlock: {(notification: NSNotification) -> Void in
                af_defaultImageCache!.removeAll()
            })
        })
//#pragma clang diagnostic push
//#pragma clang diagnostic ignored "-Wgnu"
        return objc_getAssociatedObject(self, #selector(self.sharedImageCache)) ?? af_defaultImageCache
//#pragma clang diagnostic pop
    }
    /**
     Set the cache used for image loading.
     
     @param imageCache The image cache.
     */

    class func setSharedImageCache(imageCache: AFImageCache) {
        objc_setAssociatedObject(self, #selector(self.sharedImageCache), imageCache, OBJC_ASSOCIATION_RETAIN_NONATOMIC)
    }
    ///------------------------------------
    /// @name Accessing Response Serializer
    ///------------------------------------
    /**
     The response serializer used to create an image representation from the server response and response data. By default, this is an instance of `AFImageResponseSerializer`.
     
     @discussion Subclasses of `AFImageResponseSerializer` could be used to perform post-processing, such as color correction, face detection, or other effects. See https://github.com/AFNetworking/AFCoreImageSerializer
     */
    weak var imageResponseSerializer: AFURLResponseSerialization?
    ///--------------------
    /// @name Setting Image
    ///--------------------
    /**
     Asynchronously downloads an image from the specified URL, and sets it once the request is finished. Any previous image request for the receiver will be cancelled. 
     
     If the image is cached locally, the image is set immediately, otherwise the specified placeholder image will be set immediately, and then the remote image will be set once the request is finished.
    
     By default, URL requests have a `Accept` header field value of "image / *", a cache policy of `NSURLCacheStorageAllowed` and a timeout interval of 30 seconds, and are set not handle cookies. To configure URL requests differently, use `setImageWithURLRequest:placeholderImage:success:failure:`
    
     @param url The URL used for the image request.
     */

    func setImageWithURL(url: NSURL) {
        self.setImageWithURL(url, placeholderImage: nil)
    }
    /**
     Asynchronously downloads an image from the specified URL, and sets it once the request is finished. Any previous image request for the receiver will be cancelled. 
     
     If the image is cached locally, the image is set immediately, otherwise the specified placeholder image will be set immediately, and then the remote image will be set once the request is finished.
    
     By default, URL requests have a `Accept` header field value of "image / *", a cache policy of `NSURLCacheStorageAllowed` and a timeout interval of 30 seconds, and are set not handle cookies. To configure URL requests differently, use `setImageWithURLRequest:placeholderImage:success:failure:`
    
     @param url The URL used for the image request.
     @param placeholderImage The image to be set initially, until the image request finishes. If `nil`, the image view will not change its image until the image request finishes.
     */

    func setImageWithURL(url: NSURL, placeholderImage: UIImage) {
        var request = NSMutableURLRequest(URL: url)
        request.addValue("image/*", forHTTPHeaderField: "Accept")
        self.setImageWithURLRequest(request, placeholderImage: placeholderImage, success: nil, failure: nil)
    }
    /**
     Asynchronously downloads an image from the specified URL request, and sets it once the request is finished. Any previous image request for the receiver will be cancelled. 
     
     If the image is cached locally, the image is set immediately, otherwise the specified placeholder image will be set immediately, and then the remote image will be set once the request is finished.
     
     If a success block is specified, it is the responsibility of the block to set the image of the image view before returning. If no success block is specified, the default behavior of setting the image with `self.image = image` is applied.
    
     @param urlRequest The URL request used for the image request.
     @param placeholderImage The image to be set initially, until the image request finishes. If `nil`, the image view will not change its image until the image request finishes.
     @param success A block to be executed when the image request operation finishes successfully. This block has no return value and takes three arguments: the request sent from the client, the response received from the server, and the image created from the response data of request. If the image was returned from cache, the request and response parameters will be `nil`.
     @param failure A block object to be executed when the image request operation finishes unsuccessfully, or that finishes successfully. This block has no return value and takes three arguments: the request sent from the client, the response received from the server, and the error object describing the network or parsing error that occurred.
     */

    func setImageWithURLRequest(urlRequest: NSURLRequest, placeholderImage: UIImage, success: (request: NSURLRequest, response: NSHTTPURLResponse, image: UIImage) -> Void, failure: (request: NSURLRequest, response: NSHTTPURLResponse, error: NSError) -> Void) {
        self.cancelImageRequestOperation()
        var cachedImage = self.self.sharedImageCache().cachedImageForRequest(urlRequest)
        if cachedImage {
            if success {
                success(nil, nil, cachedImage)
            }
            else {
                self.image = cachedImage
            }
            self.af_imageRequestOperation = nil
        }
        else {
            if placeholderImage {
                self.image = placeholderImage
            }
            weak var weakSelf = self
            self.af_imageRequestOperation = AFHTTPRequestOperation(request: urlRequest)
            self.af_imageRequestOperation.responseSerializer = self.imageResponseSerializer
            self.af_imageRequestOperation.setCompletionBlockWithSuccess({(operation: AFHTTPRequestOperation, responseObject: AnyObject) -> Void in
                var strongSelf = weakSelf
                if urlRequest.URL.isEqual(strongSelf.af_imageRequestOperation.request.URL!) {
                    if success {
                        success(urlRequest, operation.response, responseObject)
                    }
                    else if responseObject {
                        strongSelf.image = responseObject
                    }

                    if operation == strongSelf.af_imageRequestOperation {
                        strongSelf.af_imageRequestOperation = nil
                    }
                }
                strongSelf.self.sharedImageCache().cacheImage(responseObject, forRequest: urlRequest)
            }, failure: {(operation: AFHTTPRequestOperation, error: NSError) -> Void in
                var strongSelf = weakSelf
                if urlRequest.URL.isEqual(strongSelf.af_imageRequestOperation.request.URL!) {
                    if failure {
                        failure(urlRequest, operation.response, error)
                    }
                    if operation == strongSelf.af_imageRequestOperation {
                        strongSelf.af_imageRequestOperation = nil
                    }
                }
            })
            self.self.af_sharedImageRequestOperationQueue().addOperation(self.af_imageRequestOperation)
        }
    }
    /**
     Cancels any executing image operation for the receiver, if one exists.
     */

    func cancelImageRequestOperation() {
        self.af_imageRequestOperation.cancel()
        self.af_imageRequestOperation = nil
    }


    func imageResponseSerializer() -> AFURLResponseSerialization {
        weak var af_defaultImageResponseSerializer: AFURLResponseSerialization? = nil
        var onceToken: dispatch_once_t
        dispatch_once(onceToken, {() -> Void in
            self.af_defaultImageResponseSerializer = AFImageResponseSerializer.serializer()
        })
//#pragma clang diagnostic push
//#pragma clang diagnostic ignored "-Wgnu"
        return objc_getAssociatedObject(self, #selector(self.imageResponseSerializer)) ?? af_defaultImageResponseSerializer
//#pragma clang diagnostic pop
    }

    func setImageResponseSerializer(serializer: AFURLResponseSerialization) {
        objc_setAssociatedObject(self, #selector(self.imageResponseSerializer), serializer, OBJC_ASSOCIATION_RETAIN_NONATOMIC)
    }
// MARK: -
}
// MARK: -
/**
 The `AFImageCache` protocol is adopted by an object used to cache images loaded by the AFNetworking category on `UIImageView`.
 */
protocol AFImageCache: NSObject {
    /**
     Returns a cached image for the specififed request, if available.
     
     @param request The image request.
     
     @return The cached image.
     */
    func cachedImageForRequest(request: NSURLRequest) -> UIImage
    /**
     Caches a particular image for the specified request.
     
     @param image The image to cache.
     @param request The request to be used as a cache key.
     */

    func cacheImage(image: UIImage, forRequest request: NSURLRequest)
}
#endif
// UIImageView+AFNetworking.m
//
// Copyright (c) 2013-2014 AFNetworking (http://afnetworking.com)
//
// Permission is hereby granted, free of charge, to any person obtaining a copy
// of this software and associated documentation files (the "Software"), to deal
// in the Software without restriction, including without limitation the rights
// to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
// copies of the Software, and to permit persons to whom the Software is
// furnished to do so, subject to the following conditions:
//
// The above copyright notice and this permission notice shall be included in
// all copies or substantial portions of the Software.
//
// THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
// IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
// FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
// AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
// LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
// OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
// THE SOFTWARE.
import ObjectiveC
#if defined(__IPHONE_OS_VERSION_MIN_REQUIRED)
class AFImageCache: NSCache, AFImageCache {


    func cachedImageForRequest(request: NSURLRequest) -> UIImage {
        switch request.cachePolicy {
            case NSURLRequestReloadIgnoringCacheData, NSURLRequestReloadIgnoringLocalAndRemoteCacheData:
                return nil
            default:
                break
        }

        return (self[AFImageCacheKeyFromURLRequest(request)] as! String)
    }

    func cacheImage(image: UIImage, forRequest request: NSURLRequest) {
        if image && request {
            self[AFImageCacheKeyFromURLRequest(request)] = image
        }
    }
}
// MARK: -
extension UIImageView {
    var af_imageRequestOperation: AFHTTPRequestOperation!


    class func af_sharedImageRequestOperationQueue() -> NSOperationQueue {
        var af_sharedImageRequestOperationQueue: NSOperationQueue? = nil
        var onceToken: dispatch_once_t
        dispatch_once(onceToken, {() -> Void in
            self.af_sharedImageRequestOperationQueue = NSOperationQueue()
            self.af_sharedImageRequestOperationQueue.maxConcurrentOperationCount = NSOperationQueueDefaultMaxConcurrentOperationCount
        })
        return af_sharedImageRequestOperationQueue
    }

    func af_imageRequestOperation() -> AFHTTPRequestOperation {
        return (objc_getAssociatedObject(self, #selector(self.af_imageRequestOperation)) as! AFHTTPRequestOperation)
    }

    func af_setImageRequestOperation(imageRequestOperation: AFHTTPRequestOperation) {
        objc_setAssociatedObject(self, #selector(self.af_imageRequestOperation), imageRequestOperation, OBJC_ASSOCIATION_RETAIN_NONATOMIC)
    }
}
// MARK: -

// MARK: -
func AFImageCacheKeyFromURLRequest(request: NSURLRequest) -> inline NSString {
    return request.URL!.absoluteString
}