// UIButton+AFNetworking.h
//
// Copyright (c) 2013-2014 AFNetworking (http://afnetworking.com)
//
// Permission is hereby granted, free of charge, to any person obtaining a copy
// of this software and associated documentation files (the "Software"), to deal
// in the Software without restriction, including without limitation the rights
// to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
// copies of the Software, and to permit persons to whom the Software is
// furnished to do so, subject to the following conditions:
//
// The above copyright notice and this permission notice shall be included in
// all copies or substantial portions of the Software.
//
// THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
// IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
// FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
// AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
// LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
// OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
// THE SOFTWARE.
import Foundation
import Availability
#if defined(__IPHONE_OS_VERSION_MIN_REQUIRED)
import UIKit
/**
 This category adds methods to the UIKit framework's `UIButton` class. The methods in this category provide support for loading remote images and background images asynchronously from a URL.
 */
extension UIButton {
    ///--------------------
    /// @name Setting Image
    ///--------------------
    /**
     Asynchronously downloads an image from the specified URL, and sets it as the image for the specified state once the request is finished. Any previous image request for the receiver will be cancelled.
     
      If the image is cached locally, the image is set immediately, otherwise the specified placeholder image will be set immediately, and then the remote image will be set once the request is finished.
     
     @param state The control state.
     @param url The URL used for the image request.
     */
    func setImageForState(state: UIControlState, withURL url: NSURL) {
        self.setImageForState(state, withURL: url, placeholderImage: nil)
    }
    /**
     Asynchronously downloads an image from the specified URL, and sets it as the image for the specified state once the request is finished. Any previous image request for the receiver will be cancelled.
     
     If the image is cached locally, the image is set immediately, otherwise the specified placeholder image will be set immediately, and then the remote image will be set once the request is finished.
     
     @param state The control state.
     @param url The URL used for the image request.
     @param placeholderImage The image to be set initially, until the image request finishes. If `nil`, the button will not change its image until the image request finishes.
     */

    func setImageForState(state: UIControlState, withURL url: NSURL, placeholderImage: UIImage) {
        var request = NSMutableURLRequest(URL: url)
        request.addValue("image/*", forHTTPHeaderField: "Accept")
        self.setImageForState(state, withURLRequest: request, placeholderImage: placeholderImage, success: nil, failure: nil)
    }
    /**
     Asynchronously downloads an image from the specified URL request, and sets it as the image for the specified state once the request is finished. Any previous image request for the receiver will be cancelled.
    
     If the image is cached locally, the image is set immediately, otherwise the specified placeholder image will be set immediately, and then the remote image will be set once the request is finished.
     
     If a success block is specified, it is the responsibility of the block to set the image of the button before returning. If no success block is specified, the default behavior of setting the image with `setImage:forState:` is applied.
    
     @param state The control state.
     @param urlRequest The URL request used for the image request.
     @param placeholderImage The image to be set initially, until the image request finishes. If `nil`, the button will not change its image until the image request finishes.
     @param success A block to be executed when the image request operation finishes successfully. This block has no return value and takes two arguments: the server response and the image. If the image was returned from cache, the request and response parameters will be `nil`.
     @param failure A block object to be executed when the image request operation finishes unsuccessfully, or that finishes successfully. This block has no return value and takes a single argument: the error that occurred.
     */

    func setImageForState(state: UIControlState, withURLRequest urlRequest: NSURLRequest, placeholderImage: UIImage, success: (response: NSHTTPURLResponse, image: UIImage) -> Void, failure: (error: NSError) -> Void) {
        self.cancelImageRequestOperation()
        self.setImage(placeholderImage, forState: state)
        weak var weakSelf = self
        self.af_imageRequestOperation = AFHTTPRequestOperation(request: urlRequest)
        self.af_imageRequestOperation.responseSerializer = AFImageResponseSerializer.serializer()
        self.af_imageRequestOperation.setCompletionBlockWithSuccess({(operation: AFHTTPRequestOperation, responseObject: AnyObject) -> Void in
            var strongSelf = weakSelf
            if urlRequest.URL.isEqual(operation.request.URL!) {
                if success {
                    success(operation.response, responseObject)
                }
                else if responseObject {
                    strongSelf.setImage(responseObject, forState: state)
                }
            }
        }, failure: {(operation: AFHTTPRequestOperation, error: NSError) -> Void in
            if urlRequest.URL.isEqual(operation.response.URL!) {
                if failure {
                    failure(error)
                }
            }
        })
        self.self.af_sharedImageRequestOperationQueue().addOperation(self.af_imageRequestOperation)
    }
    ///-------------------------------
    /// @name Setting Background Image
    ///-------------------------------
    /**
     Asynchronously downloads an image from the specified URL, and sets it as the background image for the specified state once the request is finished. Any previous background image request for the receiver will be cancelled.
    
     If the background image is cached locally, the background image is set immediately, otherwise the specified placeholder background image will be set immediately, and then the remote background image will be set once the request is finished.
    
     @param state The control state.
     @param url The URL used for the background image request.
     */

    func setBackgroundImageForState(state: UIControlState, withURL url: NSURL) {
        self.setBackgroundImageForState(state, withURL: url, placeholderImage: nil)
    }
    /**
     Asynchronously downloads an image from the specified URL, and sets it as the background image for the specified state once the request is finished. Any previous image request for the receiver will be cancelled.
    
     If the image is cached locally, the image is set immediately, otherwise the specified placeholder image will be set immediately, and then the remote image will be set once the request is finished.
    
     @param state The control state.
     @param url The URL used for the background image request.
     @param placeholderImage The background image to be set initially, until the background image request finishes. If `nil`, the button will not change its background image until the background image request finishes.
     */

    func setBackgroundImageForState(state: UIControlState, withURL url: NSURL, placeholderImage: UIImage) {
        var request = NSMutableURLRequest(URL: url)
        request.addValue("image/*", forHTTPHeaderField: "Accept")
        self.setBackgroundImageForState(state, withURLRequest: request, placeholderImage: placeholderImage, success: nil, failure: nil)
    }
    /**
     Asynchronously downloads an image from the specified URL request, and sets it as the image for the specified state once the request is finished. Any previous image request for the receiver will be cancelled.
    
     If the image is cached locally, the image is set immediately, otherwise the specified placeholder image will be set immediately, and then the remote image will be set once the request is finished.
    
     If a success block is specified, it is the responsibility of the block to set the image of the button before returning. If no success block is specified, the default behavior of setting the image with `setBackgroundImage:forState:` is applied.
    
     @param state The control state.
     @param urlRequest The URL request used for the image request.
     @param placeholderImage The background image to be set initially, until the background image request finishes. If `nil`, the button will not change its background image until the background image request finishes.
     */

    func setBackgroundImageForState(state: UIControlState, withURLRequest urlRequest: NSURLRequest, placeholderImage: UIImage, success: (response: NSHTTPURLResponse, image: UIImage) -> Void, failure: (error: NSError) -> Void) {
        self.cancelBackgroundImageRequestOperation()
        self.setBackgroundImage(placeholderImage, forState: state)
        weak var weakSelf = self
        self.af_backgroundImageRequestOperation = AFHTTPRequestOperation(request: urlRequest)
        self.af_backgroundImageRequestOperation.responseSerializer = AFImageResponseSerializer.serializer()
        self.af_backgroundImageRequestOperation.setCompletionBlockWithSuccess({(operation: AFHTTPRequestOperation, responseObject: AnyObject) -> Void in
            var strongSelf = weakSelf
            if urlRequest.URL.isEqual(operation.request.URL!) {
                if success {
                    success(operation.response, responseObject)
                }
                else if responseObject {
                    strongSelf.setBackgroundImage(responseObject, forState: state)
                }
            }
        }, failure: {(operation: AFHTTPRequestOperation, error: NSError) -> Void in
            if urlRequest.URL.isEqual(operation.response.URL!) {
                if failure {
                    failure(error)
                }
            }
        })
        self.self.af_sharedImageRequestOperationQueue().addOperation(self.af_backgroundImageRequestOperation)
    }
    ///------------------------------
    /// @name Canceling Image Loading
    ///------------------------------
    /**
     Cancels any executing image operation for the receiver, if one exists.
     */

    func cancelImageRequestOperation() {
        self.af_imageRequestOperation.cancel()
        self.af_imageRequestOperation = nil
    }
    /**
     Cancels any executing background image operation for the receiver, if one exists.
     */

    func cancelBackgroundImageRequestOperation() {
        self.af_backgroundImageRequestOperation.cancel()
        self.af_backgroundImageRequestOperation = nil
    }

// MARK: -
// MARK: -
}
#endif
// UIButton+AFNetworking.m
//
// Copyright (c) 2013-2014 AFNetworking (http://afnetworking.com)
//
// Permission is hereby granted, free of charge, to any person obtaining a copy
// of this software and associated documentation files (the "Software"), to deal
// in the Software without restriction, including without limitation the rights
// to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
// copies of the Software, and to permit persons to whom the Software is
// furnished to do so, subject to the following conditions:
//
// The above copyright notice and this permission notice shall be included in
// all copies or substantial portions of the Software.
//
// THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
// IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
// FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
// AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
// LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
// OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
// THE SOFTWARE.
import ObjectiveC
#if defined(__IPHONE_OS_VERSION_MIN_REQUIRED)
extension UIButton {
    var af_imageRequestOperation: AFHTTPRequestOperation!
    var af_backgroundImageRequestOperation: AFHTTPRequestOperation!


    class func af_sharedImageRequestOperationQueue() -> NSOperationQueue {
        var af_sharedImageRequestOperationQueue: NSOperationQueue? = nil
        var onceToken: dispatch_once_t
        dispatch_once(onceToken, {() -> Void in
            self.af_sharedImageRequestOperationQueue = NSOperationQueue()
            self.af_sharedImageRequestOperationQueue.maxConcurrentOperationCount = NSOperationQueueDefaultMaxConcurrentOperationCount
        })
        return af_sharedImageRequestOperationQueue
    }

    func af_imageRequestOperation() -> AFHTTPRequestOperation {
        return (objc_getAssociatedObject(self, #selector(self.af_imageRequestOperation)) as! AFHTTPRequestOperation)
    }

    func af_setImageRequestOperation(imageRequestOperation: AFHTTPRequestOperation) {
        objc_setAssociatedObject(self, #selector(self.af_imageRequestOperation), imageRequestOperation, OBJC_ASSOCIATION_RETAIN_NONATOMIC)
    }

    func af_backgroundImageRequestOperation() -> AFHTTPRequestOperation {
        return (objc_getAssociatedObject(self, #selector(self.af_backgroundImageRequestOperation)) as! AFHTTPRequestOperation)
    }

    func af_setBackgroundImageRequestOperation(imageRequestOperation: AFHTTPRequestOperation) {
        objc_setAssociatedObject(self, #selector(self.af_backgroundImageRequestOperation), imageRequestOperation, OBJC_ASSOCIATION_RETAIN_NONATOMIC)
    }
}
// MARK: -