// UIAlertView+AFNetworking.h
//
// Copyright (c) 2013-2014 AFNetworking (http://afnetworking.com)
//
// Permission is hereby granted, free of charge, to any person obtaining a copy
// of this software and associated documentation files (the "Software"), to deal
// in the Software without restriction, including without limitation the rights
// to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
// copies of the Software, and to permit persons to whom the Software is
// furnished to do so, subject to the following conditions:
//
// The above copyright notice and this permission notice shall be included in
// all copies or substantial portions of the Software.
//
// THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
// IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
// FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
// AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
// LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
// OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
// THE SOFTWARE.
import Foundation
import Availability
#if defined(__IPHONE_OS_VERSION_MIN_REQUIRED)
import UIKit

/**
 This category adds methods to the UIKit framework's `UIAlertView` class. The methods in this category provide support for automatically showing an alert if a session task or request operation finishes with an error. Alert title and message are filled from the corresponding `localizedDescription` & `localizedRecoverySuggestion` or `localizedFailureReason` of the error.
 */
extension UIAlertView {
    ///-------------------------------------
    /// @name Showing Alert for Session Task
    ///-------------------------------------
    /**
     Shows an alert view with the error of the specified session task, if any.
     
     @param task The session task.
     @param delegate The alert view delegate.
     */
#if __IPHONE_OS_VERSION_MIN_REQUIRED >= 70000
    class func showAlertViewForTaskWithErrorOnCompletion(task: NSURLSessionTask, delegate: AnyObject) {
        self.showAlertViewForTaskWithErrorOnCompletion(task, delegate: delegate, cancelButtonTitle: NSLocalizedStringFromTable("Dismiss", "AFNetworking", "UIAlertView Cancel Button Title"), otherButtonTitles: "")
    }
#endif
    /**
     Shows an alert view with the error of the specified session task, if any, with a custom cancel button title and other button titles.
     
     @param task The session task.
     @param delegate The alert view delegate.
     @param cancelButtonTitle The title of the cancel button or nil if there is no cancel button. Using this argument is equivalent to setting the cancel button index to the value returned by invoking addButtonWithTitle: specifying this title.
     @param otherButtonTitles The title of another button. Using this argument is equivalent to invoking addButtonWithTitle: with this title to add more buttons. Too many buttons can cause the alert view to scroll. For guidelines on the best ways to use an alert in an app, see "Temporary Views". Titles of additional buttons to add to the receiver, terminated with `nil`.
     */
#if __IPHONE_OS_VERSION_MIN_REQUIRED >= 70000

    class func showAlertViewForTaskWithErrorOnCompletion(task: NSURLSessionTask, delegate: AnyObject, cancelButtonTitle: String, otherButtonTitles: String) {
        var observer = NSNotificationCenter.defaultCenter().addObserverForName(AFNetworkingTaskDidCompleteNotification, object: task, queue: NSOperationQueue.mainQueue(), usingBlock: {(notification: NSNotification) -> Void in
                var error = notification.userInfo![AFNetworkingTaskDidCompleteErrorKey]
                if error {
                    var title: String
                    var message: String
                    AFGetAlertViewTitleAndMessageFromError(error, title, message)
                    UIAlertView(title: title, message: message, delegate: delegate, cancelButtonTitle: cancelButtonTitle).show()
                }
                NSNotificationCenter.defaultCenter().removeObserver(observer, name: AFNetworkingTaskDidCompleteNotification, object: notification.object!)
            })
    }
#endif
    ///------------------------------------------
    /// @name Showing Alert for Request Operation
    ///------------------------------------------
    /**
     Shows an alert view with the error of the specified request operation, if any.
    
     @param operation The request operation.
     @param delegate The alert view delegate.
     */

    class func showAlertViewForRequestOperationWithErrorOnCompletion(operation: AFURLConnectionOperation, delegate: AnyObject) {
        self.showAlertViewForRequestOperationWithErrorOnCompletion(operation, delegate: delegate, cancelButtonTitle: NSLocalizedStringFromTable("Dismiss", "AFNetworking", "UIAlert View Cancel Button Title"), otherButtonTitles: "")
    }
    /**
     Shows an alert view with the error of the specified request operation, if any, with a custom cancel button title and other button titles.
    
     @param operation The request operation.
     @param delegate The alert view delegate.
     @param cancelButtonTitle The title of the cancel button or nil if there is no cancel button. Using this argument is equivalent to setting the cancel button index to the value returned by invoking addButtonWithTitle: specifying this title.
     @param otherButtonTitles The title of another button. Using this argument is equivalent to invoking addButtonWithTitle: with this title to add more buttons. Too many buttons can cause the alert view to scroll. For guidelines on the best ways to use an alert in an app, see "Temporary Views". Titles of additional buttons to add to the receiver, terminated with `nil`.
     */

    class func showAlertViewForRequestOperationWithErrorOnCompletion(operation: AFURLConnectionOperation, delegate: AnyObject, cancelButtonTitle: String, otherButtonTitles: String) {
        var observer = NSNotificationCenter.defaultCenter().addObserverForName(AFNetworkingOperationDidFinishNotification, object: operation, queue: NSOperationQueue.mainQueue(), usingBlock: {(notification: NSNotification) -> Void in
                if notification.object! && (notification.object! is AFURLConnectionOperation) {
                    var error = (notification.object! as! AFURLConnectionOperation).error
                    if error {
                        var title: String
                        var message: String
                        AFGetAlertViewTitleAndMessageFromError(error, title, message)
                        UIAlertView(title: title, message: message, delegate: delegate, cancelButtonTitle: cancelButtonTitle).show()
                    }
                }
                NSNotificationCenter.defaultCenter().removeObserver(observer, name: AFNetworkingOperationDidFinishNotification, object: notification.object!)
            })
    }

#if __IPHONE_OS_VERSION_MIN_REQUIRED >= 70000
#endif
// MARK: -
}
#endif
// UIAlertView+AFNetworking.m
//
// Copyright (c) 2013-2014 AFNetworking (http://afnetworking.com)
//
// Permission is hereby granted, free of charge, to any person obtaining a copy
// of this software and associated documentation files (the "Software"), to deal
// in the Software without restriction, including without limitation the rights
// to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
// copies of the Software, and to permit persons to whom the Software is
// furnished to do so, subject to the following conditions:
//
// The above copyright notice and this permission notice shall be included in
// all copies or substantial portions of the Software.
//
// THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
// IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
// FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
// AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
// LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
// OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
// THE SOFTWARE.
#if defined(__IPHONE_OS_VERSION_MIN_REQUIRED)
#if __IPHONE_OS_VERSION_MIN_REQUIRED >= 70000
#endif
func AFGetAlertViewTitleAndMessageFromError(error: NSError, title: String, message: String) {
    if error.localizedDescription && (error.localizedRecoverySuggestion! || error.localizedFailureReason!) {
        title = error.localizedDescription
        if error.localizedRecoverySuggestion! {
            message = error.localizedRecoverySuggestion!
        }
        else {
            message = error.localizedFailureReason!
        }
    }
    else if error.localizedDescription {
        title = NSLocalizedStringFromTable("Error", "AFNetworking", "Fallback Error Description")
        message = error.localizedDescription
    }
    else {
        title = NSLocalizedStringFromTable("Error", "AFNetworking", "Fallback Error Description")
        message = NSLocalizedStringFromTable("\(error.domain) Error: \(Int(error.code))", "AFNetworking", "Fallback Error Failure Reason Format")
    }

}