// AFNetworkActivityIndicatorManager.h
//
// Copyright (c) 2013-2014 AFNetworking (http://afnetworking.com)
//
// Permission is hereby granted, free of charge, to any person obtaining a copy
// of this software and associated documentation files (the "Software"), to deal
// in the Software without restriction, including without limitation the rights
// to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
// copies of the Software, and to permit persons to whom the Software is
// furnished to do so, subject to the following conditions:
//
// The above copyright notice and this permission notice shall be included in
// all copies or substantial portions of the Software.
//
// THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
// IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
// FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
// AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
// LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
// OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
// THE SOFTWARE.
import Foundation
import Availability
#if defined(__IPHONE_OS_VERSION_MIN_REQUIRED)
import UIKit
/**
 `AFNetworkActivityIndicatorManager` manages the state of the network activity indicator in the status bar. When enabled, it will listen for notifications indicating that a network request operation has started or finished, and start or stop animating the indicator accordingly. The number of active requests is incremented and decremented much like a stack or a semaphore, and the activity indicator will animate so long as that number is greater than zero.

 You should enable the shared instance of `AFNetworkActivityIndicatorManager` when your application finishes launching. In `AppDelegate application:didFinishLaunchingWithOptions:` you can do so with the following code:

    [[AFNetworkActivityIndicatorManager sharedManager] setEnabled:YES];

 By setting `isNetworkActivityIndicatorVisible` to `YES` for `sharedManager`, the network activity indicator will show and hide automatically as requests start and finish. You should not ever need to call `incrementActivityCount` or `decrementActivityCount` yourself.

 See the Apple Human Interface Guidelines section about the Network Activity Indicator for more information:
 http://developer.apple.com/library/iOS/#documentation/UserExperience/Conceptual/MobileHIG/UIElementGuidelines/UIElementGuidelines.html#//apple_ref/doc/uid/TP40006556-CH13-SW44
 */
class AFNetworkActivityIndicatorManager: NSObject {
    /**
     A Boolean value indicating whether the manager is enabled.
    
     If YES, the manager will change status bar network activity indicator according to network operation notifications it receives. The default value is NO.
     */
    var enabled = false
    /**
     A Boolean value indicating whether the network activity indicator is currently displayed in the status bar.
     */
    var isNetworkActivityIndicatorVisible: Bool {
        return self.activityCount > 0
    }
    /**
     Returns the shared network activity indicator manager object for the system.
    
     @return The systemwide network activity indicator manager.
     */

    class func sharedManager() -> Self {
        var sharedManager: AFNetworkActivityIndicatorManager? = nil
        var oncePredicate: dispatch_once_t
        dispatch_once(oncePredicate, {() -> Void in
            self.sharedManager = self.init()
        })
        return sharedManager
    }
    /**
     Increments the number of active network requests. If this number was zero before incrementing, this will start animating the status bar network activity indicator.
     */

    func incrementActivityCount() {
        self.willChangeValueForKey("activityCount")
        let lockQueue = dispatch_queue_create("com.test.LockQueue")
        dispatch_sync(lockQueue) {
            activityCount += 1
        }
        self.didChangeValueForKey("activityCount")
        dispatch_async(dispatch_get_main_queue(), {() -> Void in
            self.updateNetworkActivityIndicatorVisibilityDelayed()
        })
    }
    /**
     Decrements the number of active network requests. If this number becomes zero after decrementing, this will stop animating the status bar network activity indicator.
     */

    func decrementActivityCount() {
        self.willChangeValueForKey("activityCount")
        let lockQueue = dispatch_queue_create("com.test.LockQueue")
        dispatch_sync(lockQueue) {
//#pragma clang diagnostic push
//#pragma clang diagnostic ignored "-Wgnu"
            self.activityCount = max(activityCount - 1, 0)
//#pragma clang diagnostic pop
        }
        self.didChangeValueForKey("activityCount")
        dispatch_async(dispatch_get_main_queue(), {() -> Void in
            self.updateNetworkActivityIndicatorVisibilityDelayed()
        })
    }


    class func keyPathsForValuesAffectingIsNetworkActivityIndicatorVisible() -> Set<NSObject> {
        return Set<NSObject>(["activityCount"])
    }

    override init() {
        super.init()
        if !self {
            return nil
        }
        NSNotificationCenter.defaultCenter().addObserver(self, selector: #selector(self.networkRequestDidStart), name: AFNetworkingOperationDidStartNotification, object: nil)
        NSNotificationCenter.defaultCenter().addObserver(self, selector: #selector(self.networkRequestDidFinish), name: AFNetworkingOperationDidFinishNotification, object: nil)
#if __IPHONE_OS_VERSION_MAX_ALLOWED >= 70000
        NSNotificationCenter.defaultCenter().addObserver(self, selector: #selector(self.networkRequestDidStart), name: AFNetworkingTaskDidResumeNotification, object: nil)
        NSNotificationCenter.defaultCenter().addObserver(self, selector: #selector(self.networkRequestDidFinish), name: AFNetworkingTaskDidSuspendNotification, object: nil)
        NSNotificationCenter.defaultCenter().addObserver(self, selector: #selector(self.networkRequestDidFinish), name: AFNetworkingTaskDidCompleteNotification, object: nil)
#endif

    }

    deinit {
        NSNotificationCenter.defaultCenter().removeObserver(self)
        activityIndicatorVisibilityTimer.invalidate()
    }

    func updateNetworkActivityIndicatorVisibilityDelayed() {
        if self.enabled {
            // Delay hiding of activity indicator for a short interval, to avoid flickering
            if !self.isNetworkActivityIndicatorVisible() {
                self.activityIndicatorVisibilityTimer.invalidate()
                self.activityIndicatorVisibilityTimer = NSTimer(timeInterval: kAFNetworkActivityIndicatorInvisibilityDelay, target: self, selector: #selector(self.updateNetworkActivityIndicatorVisibility), userInfo: nil, repeats: false)
                NSRunLoop.mainRunLoop().addTimer(self.activityIndicatorVisibilityTimer, forMode: NSRunLoopCommonModes)
            }
            else {
                self.performSelectorOnMainThread(#selector(self.updateNetworkActivityIndicatorVisibility), withObject: nil, waitUntilDone: false, modes: [NSRunLoopCommonModes])
            }
        }
    }

    func updateNetworkActivityIndicatorVisibility() {
        UIApplication.sharedApplication().networkActivityIndicatorVisible = self.isNetworkActivityIndicatorVisible()
    }

    func setActivityCount(activityCount: Int) {
        let lockQueue = dispatch_queue_create("com.test.LockQueue")
        dispatch_sync(lockQueue) {
            self.activityCount = activityCount
        }
        dispatch_async(dispatch_get_main_queue(), {() -> Void in
            self.updateNetworkActivityIndicatorVisibilityDelayed()
        })
    }

    func networkRequestDidStart(notification: NSNotification) {
        if AFNetworkRequestFromNotification(notification).URL() {
            self.incrementActivityCount()
        }
    }

    func networkRequestDidFinish(notification: NSNotification) {
        if AFNetworkRequestFromNotification(notification).URL() {
            self.decrementActivityCount()
        }
    }

    var activityCount: Int {
        get {
            // TODO: add getter implementation
        }
        set(activityCount) {
            let lockQueue = dispatch_queue_create("com.test.LockQueue")
            dispatch_sync(lockQueue) {
                self.activityCount = activityCount
            }
            dispatch_async(dispatch_get_main_queue(), {() -> Void in
                self.updateNetworkActivityIndicatorVisibilityDelayed()
            })
        }
    }
    var activityIndicatorVisibilityTimer: NSTimer!
    var networkActivityIndicatorVisible: Bool {
        return self.activityCount > 0
    }

    func updateNetworkActivityIndicatorVisibility() {
    }

    func updateNetworkActivityIndicatorVisibilityDelayed() {
    }
}
#endif
// AFNetworkActivityIndicatorManager.m
//
// Copyright (c) 2013-2014 AFNetworking (http://afnetworking.com)
//
// Permission is hereby granted, free of charge, to any person obtaining a copy
// of this software and associated documentation files (the "Software"), to deal
// in the Software without restriction, including without limitation the rights
// to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
// copies of the Software, and to permit persons to whom the Software is
// furnished to do so, subject to the following conditions:
//
// The above copyright notice and this permission notice shall be included in
// all copies or substantial portions of the Software.
//
// THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
// IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
// FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
// AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
// LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
// OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
// THE SOFTWARE.
#if defined(__IPHONE_OS_VERSION_MAX_ALLOWED)
#if __IPHONE_OS_VERSION_MAX_ALLOWED >= 70000
#endif
let kAFNetworkActivityIndicatorInvisibilityDelay = 0.17

func AFNetworkRequestFromNotification(notification: NSNotification) -> NSURLRequest {
    if (notification.object! is AFURLConnectionOperation) {
        return (notification.object! as! AFURLConnectionOperation).request
    }
#if __IPHONE_OS_VERSION_MAX_ALLOWED >= 70000
    if notification.object!.respondsToSelector(#selector(self.originalRequest)) {
        return (notification.object! as! NSURLSessionTask).originalRequest
    }
#endif
    return nil
}