//
//  MBProgressHUD.h
//  Version 0.7
//  Created by Matej Bukovinski on 2.4.09.
//
// This code is distributed under the terms and conditions of the MIT license. 
// Copyright (c) 2013 Matej Bukovinski
//
// Permission is hereby granted, free of charge, to any person obtaining a copy
// of this software and associated documentation files (the "Software"), to deal
// in the Software without restriction, including without limitation the rights
// to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
// copies of the Software, and to permit persons to whom the Software is
// furnished to do so, subject to the following conditions:
//
// The above copyright notice and this permission notice shall be included in
// all copies or substantial portions of the Software.
//
// THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
// IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
// FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
// AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
// LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
// OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
// THE SOFTWARE.
import Foundation
import UIKit
import CoreGraphics

enum MBProgressHUDMode : Int {
    /** Progress is shown using an UIActivityIndicatorView. This is the default. */
    case Indeterminate
    /** Progress is shown using a round, pie-chart like, progress view. */
    case Determinate
    /** Progress is shown using a horizontal progress bar */
    case DeterminateHorizontalBar
    /** Progress is shown using a ring-shaped progress view. */
    case AnnularDeterminate
    /** Shows a custom view */
    case CustomView
    /** Shows only labels */
    case Text
}

enum MBProgressHUDAnimation : Int {
    /** Opacity animation */
    case Fade
    /** Opacity + scale animation */
    case Zoom
    static let ZoomOut: MBProgressHUDAnimation = .Zoom
    case ZoomIn
}

#if !MB_INSTANCETYPE
#if __has_feature(objc_instancetype)
let MB_INSTANCETYPE = instancetype
#else
let MB_INSTANCETYPE = id
#endif
#endif
#if !MB_STRONG
#if __has_feature(objc_arc)
let MB_STRONG = strong
#else
let MB_STRONG = retain
#endif
#endif
#if !MB_WEAK
#if __has_feature(objc_arc_weak)
let MB_WEAK = weak
#elseif __has_feature(objc_arc)
let MB_WEAK = unsafe_unretained
#else
let MB_WEAK = assign
#endif
#endif
#if NS_BLOCKS_AVAILABLE
typealias MBProgressHUDCompletionBlock = () -> Void
#endif
/** 
 * Displays a simple HUD window containing a progress indicator and two optional labels for short messages.
 *
 * This is a simple drop-in class for displaying a progress HUD view similar to Apple's private UIProgressHUD class.
 * The MBProgressHUD window spans over the entire space given to it by the initWithFrame constructor and catches all
 * user input on this region, thereby preventing the user operations on components below the view. The HUD itself is
 * drawn centered as a rounded semi-transparent view which resizes depending on the user specified content.
 *
 * This view supports four modes of operation:
 * - MBProgressHUDModeIndeterminate - shows a UIActivityIndicatorView
 * - MBProgressHUDModeDeterminate - shows a custom round progress indicator
 * - MBProgressHUDModeAnnularDeterminate - shows a custom annular progress indicator
 * - MBProgressHUDModeCustomView - shows an arbitrary, user specified view (@see customView)
 *
 * All three modes can have optional labels assigned:
 * - If the labelText property is set and non-empty then a label containing the provided content is placed below the
 *   indicator view.
 * - If also the detailsLabelText property is set then another label is placed below the first label.
 */
class MBProgressHUD: UIView {
    /**
     * Creates a new HUD, adds it to provided view and shows it. The counterpart to this method is hideHUDForView:animated:.
     * 
     * @param view The view that the HUD will be added to
     * @param animated If set to YES the HUD will appear using the current animationType. If set to NO the HUD will not use
     * animations while appearing.
     * @return A reference to the created HUD.
     *
     * @see hideHUDForView:animated:
     * @see animationType
     */
    class func showHUDAddedTo(view: UIView, animated: Bool) -> MB_INSTANCETYPE {
        var hud = self.init(view: view)
        view.addSubview(hud)
        hud.show(animated)
        return MB_AUTORELEASE(hud)
    }
    /**
     * Finds the top-most HUD subview and hides it. The counterpart to this method is showHUDAddedTo:animated:.
     *
     * @param view The view that is going to be searched for a HUD subview.
     * @param animated If set to YES the HUD will disappear using the current animationType. If set to NO the HUD will not use
     * animations while disappearing.
     * @return YES if a HUD was found and removed, NO otherwise. 
     *
     * @see showHUDAddedTo:animated:
     * @see animationType
     */

    class func hideHUDForView(view: UIView, animated: Bool) -> Bool {
        var hud = self.HUDForView(view)
        if hud != nil {
            hud.removeFromSuperViewOnHide = true
            hud.hide(animated)
            return true
        }
        return false
    }
    /**
     * Finds all the HUD subviews and hides them. 
     *
     * @param view The view that is going to be searched for HUD subviews.
     * @param animated If set to YES the HUDs will disappear using the current animationType. If set to NO the HUDs will not use
     * animations while disappearing.
     * @return the number of HUDs found and removed.
     *
     * @see hideHUDForView:animated:
     * @see animationType
     */

    class func hideAllHUDsForView(view: UIView, animated: Bool) -> Int {
        var huds = MBProgressHUD.allHUDsForView(view)
        for hud: MBProgressHUD in huds {
            hud.removeFromSuperViewOnHide = true
            hud.hide(animated)
        }
        return huds.count
    }
    /**
     * Finds the top-most HUD subview and returns it. 
     *
     * @param view The view that is going to be searched.
     * @return A reference to the last HUD subview discovered.
     */

    class func HUDForView(view: UIView) -> MB_INSTANCETYPE {
        var subviewsEnum = (view.subviews as NSArray).reverseObjectEnumerator()
        for subview: UIView in subviewsEnum {
            if (subview is self) {
                return (subview as! MBProgressHUD)
            }
        }
        return nil
    }
    /**
     * Finds all HUD subviews and returns them.
     *
     * @param view The view that is going to be searched.
     * @return All found HUD views (array of MBProgressHUD objects).
     */

    class func allHUDsForView(view: UIView) -> [AnyObject] {
        var huds = [AnyObject]()
        var subviews = view.subviews
        for aView: UIView in subviews {
            if (aView is self) {
                huds.append(aView)
            }
        }
        return [AnyObject](arrayLiteral: huds)
    }
    /**
     *  Create an MBProgressHUD object globally
     *
     *  @return self returns self after initialization
     */

    class func showGlobalProgressHUD() -> MBProgressHUD {
        var window = UIApplication.sharedApplication().windows.last!
        MBProgressHUD.hideAllHUDsForView(window, animated: true)
        var hud = MBProgressHUD.showHUDAddedTo(window, animated: true)
        return hud
    }
    /**
     *  Create an MBProgressHUD object globally with title
     *
     *  @param title Title to be displayed
     *
     *  @return self returns self after initialization
     */

    class func showGlobalProgressHUDWithTitle(title: String) -> MBProgressHUD {
        var window = UIApplication.sharedApplication().windows.last!
        MBProgressHUD.hideAllHUDsForView(window, animated: true)
        var hud = MBProgressHUD.showHUDAddedTo(window, animated: true)
        hud.labelText = title
        return hud
    }
    /**
     *  Dismiss all Global HUDs created under window
     */

    class func dismissGlobalHUD() {
        var window = UIApplication.sharedApplication().windows.last!
        MBProgressHUD.hideHUDForView(window, animated: true)
    }
    /**
     * A convenience constructor that initializes the HUD with the window's bounds. Calls the designated constructor with
     * window.bounds as the parameter.
     *
     * @param window The window instance that will provide the bounds for the HUD. Should be the same instance as
     * the HUD's superview (i.e., the window that the HUD will be added to).
     */

    convenience override init(window: UIWindow) {
        return self.init(view: window)
    }
    /**
     * A convenience constructor that initializes the HUD with the view's bounds. Calls the designated constructor with
     * view.bounds as the parameter
     *
     * @param view The view instance that will provide the bounds for the HUD. Should be the same instance as
     * the HUD's superview (i.e., the view that the HUD will be added to).
     */

    convenience override init(view: UIView) {
        assert(view, "View must not be nil.")
        return self.init(frame: view.bounds)
    }
    /** 
     * Display the HUD. You need to make sure that the main thread completes its run loop soon after this method call so
     * the user interface can be updated. Call this method when your task is already set-up to be executed in a new thread
     * (e.g., when using something like NSOperation or calling an asynchronous call like NSURLRequest).
     *
     * @param animated If set to YES the HUD will appear using the current animationType. If set to NO the HUD will not use
     * animations while appearing.
     *
     * @see animationType
     */

    func show(animated: Bool) {
        useAnimation = animated
        // If the grace time is set postpone the HUD display
        if self.graceTime > 0.0 {
            self.graceTimer = NSTimer.scheduledTimerWithTimeInterval(self.graceTime, target: self, selector: #selector(self.handleGraceTimer), userInfo: nil, repeats: false)
        }
        else {
            self.setNeedsDisplay()
            self.showUsingAnimation(useAnimation)
        }
    }
    /** 
     * Hide the HUD. This still calls the hudWasHidden: delegate. This is the counterpart of the show: method. Use it to
     * hide the HUD when your task completes.
     *
     * @param animated If set to YES the HUD will disappear using the current animationType. If set to NO the HUD will not use
     * animations while disappearing.
     *
     * @see animationType
     */

    func hide(animated: Bool) {
        useAnimation = animated
        // If the minShow time is set, calculate how long the hud was shown,
        // and pospone the hiding operation if necessary
        if self.minShowTime > 0.0 && showStarted {
            var interv = NSDate().timeIntervalSinceDate(showStarted)
            if interv < self.minShowTime {
                self.minShowTimer = NSTimer.scheduledTimerWithTimeInterval((self.minShowTime - interv), target: self, selector: #selector(self.handleMinShowTimer), userInfo: nil, repeats: false)
                return
            }
        }
        // ... otherwise hide the HUD immediately
        self.hideUsingAnimation(useAnimation)
    }
    /** 
     * Hide the HUD after a delay. This still calls the hudWasHidden: delegate. This is the counterpart of the show: method. Use it to
     * hide the HUD when your task completes.
     *
     * @param animated If set to YES the HUD will disappear using the current animationType. If set to NO the HUD will not use
     * animations while disappearing.
     * @param delay Delay in seconds until the HUD is hidden.
     *
     * @see animationType
     */

    func hide(animated: Bool, afterDelay delay: NSTimeInterval) {
        self.performSelector(#selector(self.hideDelayed), withObject: Int(animated), afterDelay: delay)
    }
    /** 
     * Shows the HUD while a background task is executing in a new thread, then hides the HUD.
     *
     * This method also takes care of autorelease pools so your method does not have to be concerned with setting up a
     * pool.
     *
     * @param method The method to be executed while the HUD is shown. This method will be executed in a new thread.
     * @param target The object that the target method belongs to.
     * @param object An optional object to be passed to the method.
     * @param animated If set to YES the HUD will (dis)appear using the current animationType. If set to NO the HUD will not use
     * animations while (dis)appearing.
     */

    func showWhileExecuting(method: Selector, onTarget target: AnyObject, withObject object: AnyObject, animated: Bool) {
        methodForExecution = method
        targetForExecution = MB_RETAIN(target)
        objectForExecution = MB_RETAIN(object)
        // Launch execution in new thread
        self.taskInProgress = true
        NSThread.detachNewThreadSelector(#selector(self.launchExecution), toTarget: self, withObject: nil)
        // Show HUD view
        self.show(animated)
    }
#if NS_BLOCKS_AVAILABLE
    /**
     * Shows the HUD while a block is executing on a background queue, then hides the HUD.
     *
     * @see showAnimated:whileExecutingBlock:onQueue:completionBlock:
     */

    func showAnimated(animated: Bool, whileExecutingBlock block: dispatch_block_t) {
        var queue = dispatch_get_global_queue(DISPATCH_QUEUE_PRIORITY_DEFAULT, 0)
        self.showAnimated(animated, whileExecutingBlock: block, onQueue: queue, completionBlock: nil)
    }
    /**
     * Shows the HUD while a block is executing on a background queue, then hides the HUD.
     *
     * @see showAnimated:whileExecutingBlock:onQueue:completionBlock:
     */

    func showAnimated(animated: Bool, whileExecutingBlock block: dispatch_block_t, completionBlock completion: () -> Void) {
        var queue = dispatch_get_global_queue(DISPATCH_QUEUE_PRIORITY_DEFAULT, 0)
        self.showAnimated(animated, whileExecutingBlock: block, onQueue: queue, completionBlock: completion)
    }
    /**
     * Shows the HUD while a block is executing on the specified dispatch queue, then hides the HUD.
     *
     * @see showAnimated:whileExecutingBlock:onQueue:completionBlock:
     */

    func showAnimated(animated: Bool, whileExecutingBlock block: dispatch_block_t, onQueue queue: dispatch_queue_t) {
        self.showAnimated(animated, whileExecutingBlock: block, onQueue: queue, completionBlock: nil)
    }
    /** 
     * Shows the HUD while a block is executing on the specified dispatch queue, executes completion block on the main queue, and then hides the HUD.
     *
     * @param animated If set to YES the HUD will (dis)appear using the current animationType. If set to NO the HUD will
     * not use animations while (dis)appearing.
     * @param block The block to be executed while the HUD is shown.
     * @param queue The dispatch queue on which the block should be executed.
     * @param completion The block to be executed on completion.
     *
     * @see completionBlock
     */

    func showAnimated(animated: Bool, whileExecutingBlock block: dispatch_block_t, onQueue queue: dispatch_queue_t, completionBlock completion: MBProgressHUDCompletionBlock) {
        self.taskInProgress = true
        self.completionBlock() = completion
        dispatch_async(queue, {() -> Void in
            block()
            dispatch_async(dispatch_get_main_queue(), {() -> Void in
                self.cleanUp()
            })
        })
        self.show(animated)
    }
    /**
     * A block that gets called after the HUD was completely hidden.
     */
    var completionBlock = MBProgressHUDCompletionBlock()
#endif
    /** 
     * MBProgressHUD operation mode. The default is MBProgressHUDModeIndeterminate.
     *
     * @see MBProgressHUDMode
     */
    var mode = MBProgressHUDMode()
    /**
     * The animation type that should be used when the HUD is shown and hidden. 
     *
     * @see MBProgressHUDAnimation
     */
    var animationType = MBProgressHUDAnimation()
    /**
     * The UIView (e.g., a UIImageView) to be shown when the HUD is in MBProgressHUDModeCustomView.
     * For best results use a 37 by 37 pixel view (so the bounds match the built in indicator bounds). 
     */
    var customView: UIView!
    /** 
     * The HUD delegate object. 
     *
     * @see MBProgressHUDDelegate
     */
    weak var delegate: MBProgressHUDDelegate?
    /** 
     * An optional short message to be displayed below the activity indicator. The HUD is automatically resized to fit
     * the entire text. If the text is too long it will get clipped by displaying "..." at the end. If left unchanged or
     * set to @"", then no message is displayed.
     */
    var labelText = ""
    /** 
     * An optional details message displayed below the labelText message. This message is displayed only if the labelText
     * property is also set and is different from an empty string (@""). The details text can span multiple lines. 
     */
    var detailsLabelText = ""
    /** 
     * The opacity of the HUD window. Defaults to 0.8 (80% opacity). 
     */
    var opacity: Float = 0.0
    /**
     * The color of the HUD window. Defaults to black. If this property is set, color is set using
     * this UIColor and the opacity property is not used.  using retain because performing copy on
     * UIColor base colors (like [UIColor greenColor]) cause problems with the copyZone.
     */
    var color: UIColor!
    /** 
     * The x-axis offset of the HUD relative to the centre of the superview. 
     */
    var xOffset: Float = 0.0
    /** 
     * The y-axis offset of the HUD relative to the centre of the superview. 
     */
    var yOffset: Float = 0.0
    /**
     * The amount of space between the HUD edge and the HUD elements (labels, indicators or custom views). 
     * Defaults to 20.0
     */
    var margin: Float = 0.0
    /** 
     * Cover the HUD background view with a radial gradient. 
     */
    var dimBackground = false
    /*
     * Grace period is the time (in seconds) that the invoked method may be run without 
     * showing the HUD. If the task finishes before the grace time runs out, the HUD will
     * not be shown at all. 
     * This may be used to prevent HUD display for very short tasks.
     * Defaults to 0 (no grace time).
     * Grace time functionality is only supported when the task status is known!
     * @see taskInProgress
     */
    var graceTime: Float = 0.0
    /**
     * The minimum time (in seconds) that the HUD is shown. 
     * This avoids the problem of the HUD being shown and than instantly hidden.
     * Defaults to 0 (no minimum show time).
     */
    var minShowTime: Float = 0.0
    /**
     * Indicates that the executed operation is in progress. Needed for correct graceTime operation.
     * If you don't set a graceTime (different than 0.0) this does nothing.
     * This property is automatically set when using showWhileExecuting:onTarget:withObject:animated:.
     * When threading is done outside of the HUD (i.e., when the show: and hide: methods are used directly),
     * you need to set this property when your task starts and completes in order to have normal graceTime 
     * functionality.
     */
    var taskInProgress = false
    /**
     * Removes the HUD from its parent view when hidden. 
     * Defaults to NO. 
     */
    var removeFromSuperViewOnHide = false
    /** 
     * Font to be used for the main label. Set this property if the default is not adequate. 
     */
    var labelFont: UIFont!
    /** 
     * Font to be used for the details label. Set this property if the default is not adequate. 
     */
    var detailsLabelFont: UIFont!
    /** 
     * The progress of the progress indicator, from 0.0 to 1.0. Defaults to 0.0. 
     */
    var progress: Float = 0.0
    /**
     * The minimum size of the HUD bezel. Defaults to CGSizeZero (no minimum size).
     */
    var minSize = CGSize.zero
    /**
     * Force the HUD dimensions to be equal if possible. 
     */
    var square = false
    var useAnimation = false
    var methodForExecution = Selector()
    var targetForExecution: AnyObject!
    var objectForExecution: AnyObject!
    var label: UILabel!
    var detailsLabel: UILabel!
    var isFinished = false
    var rotationTransform = CGAffineTransform()


// MARK: - Properties
#if NS_BLOCKS_AVAILABLE
#endif
// MARK: - Class methods
// MARK: Custom Initialisation Methods (Aaswini)
// MARK: - Lifecycle

    override init(frame: CGRect) {
        super.init(frame: frame)
        
        // Set default values for properties
        self.animationType = .Fade
        self.mode = .Indeterminate
        self.labelText = nil
        self.detailsLabelText = nil
        self.opacity = 0.8
        self.color = nil
        self.labelFont = UIFont.boldSystemFontOfSize(kLabelFontSize)
        self.detailsLabelFont = UIFont.boldSystemFontOfSize(kDetailsLabelFontSize)
        self.xOffset = 0.0
        self.yOffset = 0.0
        self.dimBackground = false
        self.margin = 20.0
        self.graceTime = 0.0
        self.minShowTime = 0.0
        self.removeFromSuperViewOnHide = false
        self.minSize = CGSizeZero
        self.square = false
        self.autoresizingMask = [.FlexibleTopMargin, .FlexibleBottomMargin, .FlexibleLeftMargin, .FlexibleRightMargin]
        // Transparent background
        self.opaque = false
        self.backgroundColor = UIColor.clearColor()
        // Make it invisible for now
        self.alpha() = 0.0
        taskInProgress = false
        rotationTransform = CGAffineTransformIdentity
        self.setupLabels()
        self.updateIndicators()
        self.registerForKVO()
        self.registerForNotifications()
    
    }

    deinit {
        self.unregisterFromNotifications()
        self.unregisterFromKVO()
#if !__has_feature(objc_arc)
        color.release()
        indicator.release()
        label.release()
        detailsLabel.release()
        labelText.release()
        detailsLabelText.release()
        graceTimer.release()
        minShowTimer.release()
        showStarted.release()
        customView.release()
#if NS_BLOCKS_AVAILABLE
        completionBlock.release()
#endif
        super.dealloc()
#endif
    }
// MARK: - Show & hide

    func hideDelayed(animated: Int) {
        self.hide(CBool(animated))
    }
// MARK: - Timer callbacks

    func handleGraceTimer(theTimer: NSTimer) {
        // Show the HUD only if the task is still running
        if taskInProgress {
            self.setNeedsDisplay()
            self.showUsingAnimation(useAnimation)
        }
    }

    func handleMinShowTimer(theTimer: NSTimer) {
        self.hideUsingAnimation(useAnimation)
    }
// MARK: - View Hierrarchy

    override func didMoveToSuperview() {
        // We need to take care of rotation ourselfs if we're adding the HUD to a window
        if (self.superview! is UIWindow) {
            self.transformForCurrentOrientation = false
        }
    }
// MARK: - Internal show & hide operations

    func showUsingAnimation(animated: Bool) {
        if animated && animationType == .ZoomIn {
            self.transform = CGAffineTransformConcat(rotationTransform, CGAffineTransformMakeScale(0.5, 0.5))
        }
        else if animated && animationType == .ZoomOut {
            self.transform = CGAffineTransformConcat(rotationTransform, CGAffineTransformMakeScale(1.5, 1.5))
        }

        self.showStarted = NSDate()
        // Fade in
        if animated {
            UIView.beginAnimations(nil, context: nil)
            UIView.animationDuration = 0.30
            self.alpha() = 1.0
            if animationType == .ZoomIn || animationType == .ZoomOut {
                self.transform = rotationTransform
            }
            UIView.commitAnimations()
        }
        else {
            self.alpha() = 1.0
        }
    }

    func hideUsingAnimation(animated: Bool) {
        // Fade out
        if animated && showStarted {
            UIView.beginAnimations(nil, context: nil)
            UIView.animationDuration = 0.30
            UIView.animationDelegate = self
            UIView.animationDidStopSelector = Selector("animationFinished:finished:context:")
            // 0.02 prevents the hud from passing through touches during the animation the hud will get completely hidden
            // in the done method
            if animationType == .ZoomIn {
                self.transform = CGAffineTransformConcat(rotationTransform, CGAffineTransformMakeScale(1.5, 1.5))
            }
            else if animationType == .ZoomOut {
                self.transform = CGAffineTransformConcat(rotationTransform, CGAffineTransformMakeScale(0.5, 0.5))
            }

            self.alpha() = 0.02
            UIView.commitAnimations()
        }
        else {
            self.alpha() = 0.0
            self.done()
        }
        self.showStarted = nil
    }

    func animationFinished(animationID: String, finished: Bool, context: UnsafeMutablePointer<Void>) {
        self.done()
    }

    func done() {
        isFinished = true
        self.alpha() = 0.0
        if delegate!.respondsToSelector(#selector(self.hudWasHidden)) {
            delegate!.performSelector(#selector(self.hudWasHidden), withObject: self)
        }
#if NS_BLOCKS_AVAILABLE
        if self.completionBlock() {
            self.completionBlock()()
            self.completionBlock() = nil
        }
#endif
        if removeFromSuperViewOnHide {
            self.removeFromSuperview()
        }
    }
// MARK: - Threading
#if NS_BLOCKS_AVAILABLE
#endif

    func launchExecution() {
        //#pragma clang diagnostic push
//#pragma clang diagnostic ignored "-Warc-performSelector-leaks"
            // Start executing the requested task
            targetForExecution.performSelector(methodForExecution, withObject: objectForExecution)
//#pragma clang diagnostic pop
            // Task completed, update view in main thread (note: view operations should
            // be done only in the main thread)
            self.performSelectorOnMainThread(#selector(self.cleanUp), withObject: nil, waitUntilDone: false)

    }

    func cleanUp() {
        taskInProgress = false
        self.indicator = nil
#if !__has_feature(objc_arc)
        targetForExecution.release()
        objectForExecution.release()
#else
        targetForExecution = nil
        objectForExecution = nil
#endif
        self.hide(useAnimation)
    }
// MARK: - UI

    func setupLabels() {
        label = UILabel(frame: self.bounds)
        label.adjustsFontSizeToFitWidth = false
        label.textAlignment = MBLabelAlignmentCenter
        label.opaque = false
        label.backgroundColor = UIColor.clearColor()
        label.textColor = UIColor.whiteColor()
        label.font = self.labelFont
        label.text = self.labelText
        self.addSubview(label)
        detailsLabel = UILabel(frame: self.bounds)
        detailsLabel.font = self.detailsLabelFont
        detailsLabel.adjustsFontSizeToFitWidth = false
        detailsLabel.textAlignment = MBLabelAlignmentCenter
        detailsLabel.opaque = false
        detailsLabel.backgroundColor = UIColor.clearColor()
        detailsLabel.textColor = UIColor.whiteColor()
        detailsLabel.numberOfLines = 0
        detailsLabel.font = self.detailsLabelFont
        detailsLabel.text = self.detailsLabelText
        self.addSubview(detailsLabel)
    }

    func updateIndicators() {
        var isActivityIndicator = (indicator is UIActivityIndicatorView)
        var isRoundIndicator = (indicator is MBRoundProgressView)
        if mode == .Indeterminate && !isActivityIndicator {
            // Update to indeterminate indicator
            indicator.removeFromSuperview()
            self.indicator = MB_AUTORELEASE(UIActivityIndicatorView(activityIndicatorStyle: .WhiteLarge))
            (indicator as! UIActivityIndicatorView).startAnimating()
            self.addSubview(indicator)
        }
        else if mode == .DeterminateHorizontalBar {
            // Update to bar determinate indicator
            indicator.removeFromSuperview()
            self.indicator = MB_AUTORELEASE(MBBarProgressView())
            self.addSubview(indicator)
        }
        else if mode == .Determinate || mode == .AnnularDeterminate {
            if !isRoundIndicator {
                // Update to determinante indicator
                indicator.removeFromSuperview()
                self.indicator = MB_AUTORELEASE(MBRoundProgressView())
                self.addSubview(indicator)
            }
            if mode == .AnnularDeterminate {
                (indicator as! MBRoundProgressView).annular = true
            }
        }
        else if mode == .CustomView && customView != indicator {
            // Update custom view indicator
            indicator.removeFromSuperview()
            self.indicator = customView
            self.addSubview(indicator)
        }
        else if mode == .Text {
            indicator.removeFromSuperview()
            self.indicator = nil
        }

    }
// MARK: - Layout

    override func layoutSubviews() {
            // Entirely cover the parent view
        var parent = self.superview!
        if parent {
            self.frame = parent.bounds
        }
        var bounds = self.bounds
            // Determine the total widt and height needed
        var maxWidth: CGFloat = bounds.size.width - 4 * margin
        var totalSize = CGSizeZero
        var indicatorF = indicator.bounds
        indicatorF.size.width = min(indicatorF.size.width, maxWidth)
        totalSize.width = max(totalSize.width, indicatorF.size.width)
        totalSize.height += indicatorF.size.height
        var labelSize = label.text.sizeWithFont(label.font)
        labelSize.width = min(labelSize.width, maxWidth)
        totalSize.width = max(totalSize.width, labelSize.width)
        totalSize.height += labelSize.height
        if labelSize.height > 0.0 && indicatorF.size.height > 0.0 {
            totalSize.height += kPadding
        }
        var remainingHeight: CGFloat = bounds.size.height - totalSize.height - kPadding - 4 * margin
        var maxSize = CGSizeMake(maxWidth, remainingHeight)
        var detailsLabelSize = detailsLabel.text.sizeWithFont(detailsLabel.font, constrainedToSize: maxSize, lineBreakMode: detailsLabel.lineBreakMode)
        totalSize.width = max(totalSize.width, detailsLabelSize.width)
        totalSize.height += detailsLabelSize.height
        if detailsLabelSize.height > 0.0 && (indicatorF.size.height > 0.0 || labelSize.height > 0.0) {
            totalSize.height += kPadding
        }
        totalSize.width += 2 * margin
        totalSize.height += 2 * margin
            // Position elements
        var yPos: CGFloat = roundf(((bounds.size.height - totalSize.height) / 2)) + margin + yOffset
        var xPos: CGFloat = xOffset
        indicatorF.origin.y = yPos
        indicatorF.origin.x = roundf((bounds.size.width - indicatorF.size.width) / 2) + xPos
        indicator.frame = indicatorF
        yPos += indicatorF.size.height
        if labelSize.height > 0.0 && indicatorF.size.height > 0.0 {
            yPos += kPadding
        }
        var labelF: CGRect
        labelF.origin.y = yPos
        labelF.origin.x = roundf((bounds.size.width - labelSize.width) / 2) + xPos
        labelF.size = labelSize
        label.frame = labelF
        yPos += labelF.size.height
        if detailsLabelSize.height > 0.0 && (indicatorF.size.height > 0.0 || labelSize.height > 0.0) {
            yPos += kPadding
        }
        var detailsLabelF: CGRect
        detailsLabelF.origin.y = yPos
        detailsLabelF.origin.x = roundf((bounds.size.width - detailsLabelSize.width) / 2) + xPos
        detailsLabelF.size = detailsLabelSize
        detailsLabel.frame = detailsLabelF
        // Enforce minsize and quare rules
        if square {
            var max: CGFloat = max(totalSize.width, totalSize.height)
            if max <= bounds.size.width - 2 * margin {
                totalSize.width = max
            }
            if max <= bounds.size.height - 2 * margin {
                totalSize.height = max
            }
        }
        if totalSize.width < minSize.width {
            totalSize.width = minSize.width
        }
        if totalSize.height < minSize.height {
            totalSize.height = minSize.height
        }
        self.size = totalSize
    }
// MARK: BG Drawing

    override func drawRect(rect: CGRect) {
        var context = UIGraphicsGetCurrentContext()
        UIGraphicsPushContext(context)
        if self.dimBackground {
                //Gradient colours
            var gradLocationsNum = 2
            var gradLocations = [0.0, 1.0]
            var gradColors = [0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.75]
            var colorSpace = CGColorSpaceCreateDeviceRGB()
            var gradient = CGGradientCreateWithColorComponents(colorSpace, gradColors, gradLocations, gradLocationsNum)
            CGColorSpaceRelease(colorSpace)
                //Gradient center
            var gradCenter = CGPointMake(self.bounds.size.width / 2, self.bounds.size.height / 2)
                //Gradient radius
            var gradRadius: Float = min(self.bounds.size.width, self.bounds.size.height)
            //Gradient draw
            CGContextDrawRadialGradient(context, gradient, gradCenter, 0, gradCenter, gradRadius, kCGGradientDrawsAfterEndLocation)
            CGGradientRelease(gradient)
        }
        // Set background rect color
        if self.color {
            CGContextSetFillColorWithColor(context, self.color.CGColor)
        }
        else {
            CGContextSetGrayFillColor(context, 0.0, self.opacity)
        }
            // Center HUD
        var allRect = self.bounds
            // Draw rounded HUD backgroud rect
        var boxRect = CGRectMake(roundf((allRect.size.width - size.width) / 2) + self.xOffset, roundf((allRect.size.height - size.height) / 2) + self.yOffset, size.width, size.height)
        var radius: Float = 10.0
        CGContextBeginPath(context)
        CGContextMoveToPoint(context, CGRectGetMinX(boxRect) + radius, CGRectGetMinY(boxRect))
        CGContextAddArc(context, CGRectGetMaxX(boxRect) - radius, CGRectGetMinY(boxRect) + radius, radius, 3 * Float(M_PI) / 2, 0, 0)
        CGContextAddArc(context, CGRectGetMaxX(boxRect) - radius, CGRectGetMaxY(boxRect) - radius, radius, 0, Float(M_PI) / 2, 0)
        CGContextAddArc(context, CGRectGetMinX(boxRect) + radius, CGRectGetMaxY(boxRect) - radius, radius, Float(M_PI) / 2, Float(M_PI), 0)
        CGContextAddArc(context, CGRectGetMinX(boxRect) + radius, CGRectGetMinY(boxRect) + radius, radius, Float(M_PI), 3 * Float(M_PI) / 2, 0)
        CGContextClosePath(context)
        CGContextFillPath(context)
        UIGraphicsPopContext()
    }
// MARK: - KVO

    func registerForKVO() {
        for keyPath: String in self.observableKeypaths() {
            self.addObserver(self, forKeyPath: keyPath, options: .New, context: nil)
        }
    }

    func unregisterFromKVO() {
        for keyPath: String in self.observableKeypaths() {
            self.removeObserver(self, forKeyPath: keyPath)
        }
    }

    func observableKeypaths() -> [AnyObject] {
        return ["mode", "customView", "labelText", "labelFont", "detailsLabelText", "detailsLabelFont", "progress"]
    }

    override func observeValueForKeyPath(keyPath: String, ofObject object: AnyObject, change: [NSObject : AnyObject], context: UnsafeMutablePointer<Void>) {
        if !NSThread.isMainThread {
            self.performSelectorOnMainThread(#selector(self.updateUIForKeypath), withObject: keyPath, waitUntilDone: false)
        }
        else {
            self.updateUIForKeypath(keyPath)
        }
    }

    func updateUIForKeypath(keyPath: String) {
        if (keyPath == "mode") || (keyPath == "customView") {
            self.updateIndicators()
        }
        else if (keyPath == "labelText") {
            label.text = self.labelText
        }
        else if (keyPath == "labelFont") {
            label.font = self.labelFont
        }
        else if (keyPath == "detailsLabelText") {
            detailsLabel.text = self.detailsLabelText
        }
        else if (keyPath == "detailsLabelFont") {
            detailsLabel.font = self.detailsLabelFont
        }
        else if (keyPath == "progress") {
            if indicator.respondsToSelector(#selector(self.setProgress)) {
                (indicator as! AnyObject).progress = progress
            }
            return
        }

        self.setNeedsLayout()
        self.setNeedsDisplay()
    }
// MARK: - Notifications

    func registerForNotifications() {
        var nc = NSNotificationCenter.defaultCenter()
        nc.addObserver(self, selector: #selector(self.deviceOrientationDidChange), name: .DidChangeNotification, object: nil)
    }

    func unregisterFromNotifications() {
        NSNotificationCenter.defaultCenter().removeObserver(self)
    }

    func deviceOrientationDidChange(notification: NSNotification) {
        var superview = self.superview!
        if !superview {
            return
        }
        else if (superview is UIWindow) {
            self.transformForCurrentOrientation = true
        }
        else {
            self.bounds = self.superview!.bounds
            self.setNeedsDisplay()
        }

    }

    func setTransformForCurrentOrientation(animated: Bool) {
        // Stay in sync with the superview
        if self.superview! {
            self.bounds = self.superview!.bounds
            self.setNeedsDisplay()
        }
        var orientation = UIApplication.sharedApplication().statusBarOrientation
        var radians: CGFloat = 0
        if .IsLandscape(orientation) {
            if orientation == .LandscapeLeft {
                radians = -CGFloat(M_PI_2)
            }
            else {
                radians = CGFloat(M_PI_2)
            }
            // Window coordinates differ!
            self.bounds = CGRectMake(0, 0, self.bounds.size.height, self.bounds.size.width)
        }
        else {
            if orientation == .PortraitUpsideDown {
                radians = CGFloat(M_PI)
            }
            else {
                radians = 0
            }
        }
        rotationTransform = CGAffineTransformMakeRotation(radians)
        if animated {
            UIView.beginAnimations(nil, context: nil)
        }
        self.transform = rotationTransform
        if animated {
            UIView.commitAnimations()
        }
    }

    func setupLabels() {
    }

    func registerForKVO() {
    }

    func unregisterFromKVO() {
    }

    func observableKeypaths() -> [AnyObject] {
    }

    func registerForNotifications() {
    }

    func unregisterFromNotifications() {
    }

    func updateUIForKeypath(keyPath: String) {
    }

    func hideUsingAnimation(animated: Bool) {
    }

    func showUsingAnimation(animated: Bool) {
    }

    func done() {
    }

    func updateIndicators() {
    }

    func handleGraceTimer(theTimer: NSTimer) {
    }

    func handleMinShowTimer(theTimer: NSTimer) {
    }

    func setTransformForCurrentOrientation(animated: Bool) {
    }

    func cleanUp() {
    }

    func launchExecution() {
    }

    func deviceOrientationDidChange(notification: NSNotification) {
    }

    func hideDelayed(animated: Int) {
    }
    var indicator: UIView!
    var graceTimer: NSTimer!
    var minShowTimer: NSTimer!
    var showStarted: NSDate!
    var size = CGSize.zero
}
protocol MBProgressHUDDelegate: NSObject {
    /** 
     * Called after the HUD was fully hidden from the screen. 
     */
    func hudWasHidden(hud: MBProgressHUD)
}
/**
 * A progress view for showing definite progress by filling up a circle (pie chart).
 */
class MBRoundProgressView: UIView {
    /**
     * Progress (0.0 to 1.0)
     */
    var progress: Float = 0.0
    /**
     * Indicator progress color.
     * Defaults to white [UIColor whiteColor]
     */
    var progressTintColor: UIColor!
    /**
     * Indicator background (non-progress) color.
     * Defaults to translucent white (alpha 0.1)
     */
    var backgroundTintColor: UIColor!
    /*
     * Display mode - NO = round or YES = annular. Defaults to round.
     */
    var annular = false

// MARK: - Lifecycle

    convenience override init() {
        return self.init(frame: CGRectMake(0.0, 0.0, 37.0, 37.0))
    }

    override init(frame: CGRect) {
        super.init(frame: frame)
        
        self.backgroundColor = UIColor.clearColor()
        self.opaque = false
        self.progress = 0.0
        self.annular = false
        self.progressTintColor = UIColor(white: 1.0, alpha: 1.0)
        self.backgroundTintColor = UIColor(white: 1.0, alpha: 0.1)
        self.registerForKVO()
    
    }

    deinit {
        self.unregisterFromKVO()
#if !__has_feature(objc_arc)
        progressTintColor.release()
        backgroundTintColor.release()
        super.dealloc()
#endif
    }
// MARK: - Drawing

    override func drawRect(rect: CGRect) {
        var allRect = self.bounds
        var circleRect = CGRectInset(allRect, 2.0, 2.0)
        var context = UIGraphicsGetCurrentContext()
        if annular {
                // Draw background
            var lineWidth: CGFloat = 5.0
            var processBackgroundPath = UIBezierPath.bezierPath()
            processBackgroundPath.lineWidth = lineWidth
            processBackgroundPath.lineCapStyle = kCGLineCapRound
            var center = CGPointMake(self.bounds.size.width / 2, self.bounds.size.height / 2)
            var radius: CGFloat = (self.bounds.size.width - lineWidth) / 2
            var startAngle: CGFloat = -(Float(M_PI) / 2)
                // 90 degrees
            var endAngle: CGFloat = (2 * Float(M_PI)) + startAngle
            processBackgroundPath.addArcWithCenter(center, radius: radius, startAngle: startAngle, endAngle: endAngle, clockwise: true)
            Set<NSObject>()
            processBackgroundPath.stroke()
                // Draw progress
            var processPath = UIBezierPath.bezierPath()
            processPath.lineCapStyle = kCGLineCapRound
            processPath.lineWidth = lineWidth
            endAngle = (self.progress * 2 * Float(M_PI)) + startAngle
            processPath.addArcWithCenter(center, radius: radius, startAngle: startAngle, endAngle: endAngle, clockwise: true)
            Set<NSObject>()
            processPath.stroke()
        }
        else {
            // Draw background
            progressTintColor.setStroke()
            backgroundTintColor.setFill()
            CGContextSetLineWidth(context, 2.0)
            CGContextFillEllipseInRect(context, circleRect)
            CGContextStrokeEllipseInRect(context, circleRect)
                // Draw progress
            var center = CGPointMake(allRect.size.width / 2, allRect.size.height / 2)
            var radius: CGFloat = (allRect.size.width - 4) / 2
            var startAngle: CGFloat = -(Float(M_PI) / 2)
                // 90 degrees
            var endAngle: CGFloat = (self.progress * 2 * Float(M_PI)) + startAngle
            CGContextSetRGBFillColor(context, 1.0, 1.0, 1.0, 1.0)
            // white
            CGContextMoveToPoint(context, center.x, center.y)
            CGContextAddArc(context, center.x, center.y, radius, startAngle, endAngle, 0)
            CGContextClosePath(context)
            CGContextFillPath(context)
        }
    }
// MARK: - KVO

    func registerForKVO() {
        for keyPath: String in self.observableKeypaths() {
            self.addObserver(self, forKeyPath: keyPath, options: .New, context: nil)
        }
    }

    func unregisterFromKVO() {
        for keyPath: String in self.observableKeypaths() {
            self.removeObserver(self, forKeyPath: keyPath)
        }
    }

    func observableKeypaths() -> [AnyObject] {
        return ["progressTintColor", "backgroundTintColor", "progress", "annular"]
    }

    override func observeValueForKeyPath(keyPath: String, ofObject object: AnyObject, change: [NSObject : AnyObject], context: UnsafeMutablePointer<Void>) {
        self.setNeedsDisplay()
    }
}
/**
 * A flat bar progress view. 
 */
class MBBarProgressView: UIView {
    /**
     * Progress (0.0 to 1.0)
     */
    var progress: Float = 0.0
    /**
     * Bar border line color.
     * Defaults to white [UIColor whiteColor].
     */
    var lineColor: UIColor!
    /**
     * Bar background color.
     * Defaults to clear [UIColor clearColor];
     */
    var progressRemainingColor: UIColor!
    /**
     * Bar progress color.
     * Defaults to white [UIColor whiteColor].
     */
    var progressColor: UIColor!

// MARK: - Lifecycle

    convenience override init() {
        return self.init(frame: CGRectMake(0.0, 0.0, 120.0, 20.0))
    }

    override init(frame: CGRect) {
        super.init(frame: frame)
        
        self.progress = 0.0
        self.lineColor = UIColor.whiteColor()
        self.progressColor = UIColor.whiteColor()
        self.progressRemainingColor = UIColor.clearColor()
        self.backgroundColor = UIColor.clearColor()
        self.opaque = false
        self.registerForKVO()
    
    }

    deinit {
        self.unregisterFromKVO()
#if !__has_feature(objc_arc)
        lineColor.release()
        progressColor.release()
        progressRemainingColor.release()
        super.dealloc()
#endif
    }
// MARK: - Drawing

    override func drawRect(rect: CGRect) {
        var context = UIGraphicsGetCurrentContext()
        // setup properties
        CGContextSetLineWidth(context, 2)
        CGContextSetStrokeColorWithColor(context, lineColor.CGColor)
        CGContextSetFillColorWithColor(context, progressRemainingColor.CGColor)
            // draw line border
        var radius: Float = (rect.size.height / 2) - 2
        CGContextMoveToPoint(context, 2, rect.size.height / 2)
        CGContextAddArcToPoint(context, 2, 2, radius + 2, 2, radius)
        CGContextAddLineToPoint(context, rect.size.width - radius - 2, 2)
        CGContextAddArcToPoint(context, rect.size.width - 2, 2, rect.size.width - 2, rect.size.height / 2, radius)
        CGContextAddArcToPoint(context, rect.size.width - 2, rect.size.height - 2, rect.size.width - radius - 2, rect.size.height - 2, radius)
        CGContextAddLineToPoint(context, radius + 2, rect.size.height - 2)
        CGContextAddArcToPoint(context, 2, rect.size.height - 2, 2, rect.size.height / 2, radius)
        CGContextFillPath(context)
        // draw progress background
        CGContextMoveToPoint(context, 2, rect.size.height / 2)
        CGContextAddArcToPoint(context, 2, 2, radius + 2, 2, radius)
        CGContextAddLineToPoint(context, rect.size.width - radius - 2, 2)
        CGContextAddArcToPoint(context, rect.size.width - 2, 2, rect.size.width - 2, rect.size.height / 2, radius)
        CGContextAddArcToPoint(context, rect.size.width - 2, rect.size.height - 2, rect.size.width - radius - 2, rect.size.height - 2, radius)
        CGContextAddLineToPoint(context, radius + 2, rect.size.height - 2)
        CGContextAddArcToPoint(context, 2, rect.size.height - 2, 2, rect.size.height / 2, radius)
        CGContextStrokePath(context)
        // setup to draw progress color
        CGContextSetFillColorWithColor(context, progressColor.CGColor)
        radius = radius - 2
        var amount: Float = self.progress * rect.size.width
        // if progress is in the middle area
        if amount >= radius + 4 && amount <= (rect.size.width - radius - 4) {
            // top
            CGContextMoveToPoint(context, 4, rect.size.height / 2)
            CGContextAddArcToPoint(context, 4, 4, radius + 4, 4, radius)
            CGContextAddLineToPoint(context, amount, 4)
            CGContextAddLineToPoint(context, amount, radius + 4)
            // bottom
            CGContextMoveToPoint(context, 4, rect.size.height / 2)
            CGContextAddArcToPoint(context, 4, rect.size.height - 4, radius + 4, rect.size.height - 4, radius)
            CGContextAddLineToPoint(context, amount, rect.size.height - 4)
            CGContextAddLineToPoint(context, amount, radius + 4)
            CGContextFillPath(context)
        }
        else if amount > radius + 4 {
            var x: Float = amount - (rect.size.width - radius - 4)
            // top
            CGContextMoveToPoint(context, 4, rect.size.height / 2)
            CGContextAddArcToPoint(context, 4, 4, radius + 4, 4, radius)
            CGContextAddLineToPoint(context, rect.size.width - radius - 4, 4)
            var angle: Float = -Float(acos(x / radius))
            if isnan(angle) {
                angle = 0
            }
            CGContextAddArc(context, rect.size.width - radius - 4, rect.size.height / 2, radius, Float(M_PI), angle, 0)
            CGContextAddLineToPoint(context, amount, rect.size.height / 2)
            // bottom
            CGContextMoveToPoint(context, 4, rect.size.height / 2)
            CGContextAddArcToPoint(context, 4, rect.size.height - 4, radius + 4, rect.size.height - 4, radius)
            CGContextAddLineToPoint(context, rect.size.width - radius - 4, rect.size.height - 4)
            angle = Float(acos(x / radius))
            if isnan(angle) {
                angle = 0
            }
            CGContextAddArc(context, rect.size.width - radius - 4, rect.size.height / 2, radius, Float(-M_PI), angle, 1)
            CGContextAddLineToPoint(context, amount, rect.size.height / 2)
            CGContextFillPath(context)
        }
        else if amount < radius + 4 && amount > 0 {
            // top
            CGContextMoveToPoint(context, 4, rect.size.height / 2)
            CGContextAddArcToPoint(context, 4, 4, radius + 4, 4, radius)
            CGContextAddLineToPoint(context, radius + 4, rect.size.height / 2)
            // bottom
            CGContextMoveToPoint(context, 4, rect.size.height / 2)
            CGContextAddArcToPoint(context, 4, rect.size.height - 4, radius + 4, rect.size.height - 4, radius)
            CGContextAddLineToPoint(context, radius + 4, rect.size.height / 2)
            CGContextFillPath(context)
        }

    }
// MARK: - KVO

    func registerForKVO() {
        for keyPath: String in self.observableKeypaths() {
            self.addObserver(self, forKeyPath: keyPath, options: .New, context: nil)
        }
    }

    func unregisterFromKVO() {
        for keyPath: String in self.observableKeypaths() {
            self.removeObserver(self, forKeyPath: keyPath)
        }
    }

    func observableKeypaths() -> [AnyObject] {
        return ["lineColor", "progressRemainingColor", "progressColor", "progress"]
    }

    override func observeValueForKeyPath(keyPath: String, ofObject object: AnyObject, change: [NSObject : AnyObject], context: UnsafeMutablePointer<Void>) {
        self.setNeedsDisplay()
    }
}
//
// MBProgressHUD.m
// Version 0.7
// Created by Matej Bukovinski on 2.4.09.
//
#if __has_feature(objc_arc)
//#define MB_AUTORELEASE(exp) exp
//#define MB_RELEASE(exp) exp
//#define MB_RETAIN(exp) exp
#else
//#define MB_AUTORELEASE(exp) [exp autorelease]
//#define MB_RELEASE(exp) [exp release]
//#define MB_RETAIN(exp) [exp retain]
#endif
#if __IPHONE_OS_VERSION_MIN_REQUIRED >= 60000
let MBLabelAlignmentCenter = .Center
#else
let MBLabelAlignmentCenter = .Center
#endif
let kPadding: CGFloat = 4.0

let kLabelFontSize: CGFloat = 16.0

let kDetailsLabelFontSize: CGFloat = 12.0