// AFURLSessionManager.h
// 
// Copyright (c) 2013-2014 AFNetworking (http://afnetworking.com)
//
// Permission is hereby granted, free of charge, to any person obtaining a copy
// of this software and associated documentation files (the "Software"), to deal
// in the Software without restriction, including without limitation the rights
// to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
// copies of the Software, and to permit persons to whom the Software is
// furnished to do so, subject to the following conditions:
// 
// The above copyright notice and this permission notice shall be included in
// all copies or substantial portions of the Software.
// 
// THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
// IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
// FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
// AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
// LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
// OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
// THE SOFTWARE.
import UIKit
import Foundation
/**
 `AFURLSessionManager` creates and manages an `NSURLSession` object based on a specified `NSURLSessionConfiguration` object, which conforms to `<NSURLSessionTaskDelegate>`, `<NSURLSessionDataDelegate>`, `<NSURLSessionDownloadDelegate>`, and `<NSURLSessionDelegate>`.
 
 ## Subclassing Notes
 
 This is the base class for `AFHTTPSessionManager`, which adds functionality specific to making HTTP requests. If you are looking to extend `AFURLSessionManager` specifically for HTTP, consider subclassing `AFHTTPSessionManager` instead.
 
 ## NSURLSession & NSURLSessionTask Delegate Methods
 
 `AFURLSessionManager` implements the following delegate methods:
 
 ### `NSURLSessionDelegate`
 
 - `URLSession:didBecomeInvalidWithError:`
 - `URLSession:didReceiveChallenge:completionHandler:`
 - `URLSessionDidFinishEventsForBackgroundURLSession:`

 ### `NSURLSessionTaskDelegate`
 
 - `URLSession:willPerformHTTPRedirection:newRequest:completionHandler:`
 - `URLSession:task:didReceiveChallenge:completionHandler:`
 - `URLSession:task:didSendBodyData:totalBytesSent:totalBytesExpectedToSend:`
 - `URLSession:task:didCompleteWithError:`

 ### `NSURLSessionDataDelegate`
 
 - `URLSession:dataTask:didReceiveResponse:completionHandler:`
 - `URLSession:dataTask:didBecomeDownloadTask:`
 - `URLSession:dataTask:didReceiveData:`
 - `URLSession:dataTask:willCacheResponse:completionHandler:`

 ### `NSURLSessionDownloadDelegate`

 - `URLSession:downloadTask:didFinishDownloadingToURL:`
 - `URLSession:downloadTask:didWriteData:totalBytesWritten:totalBytesWritten:totalBytesExpectedToWrite:`
 - `URLSession:downloadTask:didResumeAtOffset:expectedTotalBytes:`
 
 If any of these methods are overridden in a subclass, they _must_ call the `super` implementation first.
 
 ## Network Reachability Monitoring

 Network reachability status and change monitoring is available through the `reachabilityManager` property. Applications may choose to monitor network reachability conditions in order to prevent or suspend any outbound requests. See `AFNetworkReachabilityManager` for more details.
 
 ## NSCoding Caveats
 
 - Encoded managers do not include any block properties. Be sure to set delegate callback blocks when using `-initWithCoder:` or `NSKeyedUnarchiver`.

 ## NSCopying Caveats

 - `-copy` and `-copyWithZone:` return a new manager with a new `NSURLSession` created from the configuration of the original.
 - Operation copies do not include any delegate callback blocks, as they often strongly captures a reference to `self`, which would otherwise have the unintuitive side-effect of pointing to the _original_ session manager when copied.
 */
#if (defined(__IPHONE_OS_VERSION_MAX_ALLOWED) && __IPHONE_OS_VERSION_MAX_ALLOWED >= 70000) || (defined(__MAC_OS_X_VERSION_MAX_ALLOWED) && __MAC_OS_X_VERSION_MAX_ALLOWED >= 1090)
class AFURLSessionManager: NSObject, NSURLSessionDelegate, NSURLSessionTaskDelegate, NSURLSessionDataDelegate, NSURLSessionDownloadDelegate, NSSecureCoding, NSCopying {
    /**
     The managed session.
     */
    private(set) var session: NSURLSession!
    /**
     The operation queue on which delegate callbacks are run.
     */
    private(set) var operationQueue: NSOperationQueue!
    /**
     Responses sent from the server in data tasks created with `dataTaskWithRequest:success:failure:` and run using the `GET` / `POST` / et al. convenience methods are automatically validated and serialized by the response serializer. By default, this property is set to an instance of `AFJSONResponseSerializer`.
    
     @warning `responseSerializer` must not be `nil`.
     */
    weak var responseSerializer: AFURLResponseSerialization?
    ///-------------------------------
    /// @name Managing Security Policy
    ///-------------------------------
    /**
     The security policy used by created request operations to evaluate server trust for secure connections. `AFURLSessionManager` uses the `defaultPolicy` unless otherwise specified.
     */
    var securityPolicy: AFSecurityPolicy!
    ///--------------------------------------
    /// @name Monitoring Network Reachability
    ///--------------------------------------
    /**
     The network reachability manager. `AFURLSessionManager` uses the `sharedManager` by default.
     */
    var reachabilityManager: AFNetworkReachabilityManager!
    ///----------------------------
    /// @name Getting Session Tasks
    ///----------------------------
    /**
     The data, upload, and download tasks currently run by the managed session.
     */
    private(set) var tasks = [AnyObject]()
    /**
     The data tasks currently run by the managed session.
     */
    private(set) var dataTasks = [AnyObject]()
    /**
     The upload tasks currently run by the managed session.
     */
    private(set) var uploadTasks = [AnyObject]()
    /**
     The download tasks currently run by the managed session.
     */
    private(set) var downloadTasks = [AnyObject]()
    ///-------------------------------
    /// @name Managing Callback Queues
    ///-------------------------------
    /**
     The dispatch queue for `completionBlock`. If `NULL` (default), the main queue is used.
     */
    var completionQueue = dispatch_queue_t()
    /**
     The dispatch group for `completionBlock`. If `NULL` (default), a private dispatch group is used.
     */
    var completionGroup = dispatch_group_t()
    ///---------------------------------
    /// @name Working Around System Bugs
    ///---------------------------------
    /**
     Whether to attempt to retry creation of upload tasks for background sessions when initial call returns `nil`. `NO` by default.
    
     @bug As of iOS 7.0, there is a bug where upload tasks created for background tasks are sometimes `nil`. As a workaround, if this property is `YES`, AFNetworking will follow Apple's recommendation to try creating the task again.
    
     @see https://github.com/AFNetworking/AFNetworking/issues/1675
     */
    var attemptsToRecreateUploadTasksForBackgroundSessions = false
    ///---------------------
    /// @name Initialization
    ///---------------------
    /**
     Creates and returns a manager for a session created with the specified configuration. This is the designated initializer.
     
     @param configuration The configuration used to create the managed session.
     
     @return A manager for a newly-created session.
     */

    override init(sessionConfiguration configuration: NSURLSessionConfiguration) {
        super.init()
        if !self {
            return nil
        }
        if !configuration {
            configuration = NSURLSessionConfiguration.defaultSessionConfiguration()
        }
        self.sessionConfiguration = configuration
        self.operationQueue! = NSOperationQueue()
        self.operationQueue!.maxConcurrentOperationCount = 1
        self.session = NSURLSession(configuration: self.sessionConfiguration, delegate: self, delegateQueue: self.operationQueue!)
        self.responseSerializer = AFJSONResponseSerializer.serializer()
        self.securityPolicy = AFSecurityPolicy.defaultPolicy()
        self.reachabilityManager = AFNetworkReachabilityManager.sharedManager()
        self.mutableTaskDelegatesKeyedByTaskIdentifier = [NSObject : AnyObject]()
        self.lock() = NSLock()
        self.lock().name! = AFURLSessionManagerLockName
        self.session.getTasksWithCompletionHandler({(dataTasks: [AnyObject], uploadTasks: [AnyObject], downloadTasks: [AnyObject]) -> Void in
            for task: NSURLSessionDataTask in dataTasks {
                self.addDelegateForDataTask(task, completionHandler: nil)
            }
            for uploadTask: NSURLSessionUploadTask in uploadTasks {
                self.addDelegateForUploadTask(uploadTask, progress: nil, completionHandler: nil)
            }
            for downloadTask: NSURLSessionDownloadTask in downloadTasks {
                self.addDelegateForDownloadTask(downloadTask, progress: nil, destination: nil, completionHandler: nil)
            }
        })
    }
    /**
     Invalidates the managed session, optionally canceling pending tasks.
     
     @param cancelPendingTasks Whether or not to cancel pending tasks.
     */

    func invalidateSessionCancelingTasks(cancelPendingTasks: Bool) {
        dispatch_async(dispatch_get_main_queue(), {() -> Void in
            if cancelPendingTasks {
                self.session.invalidateAndCancel()
            }
            else {
                self.session.finishTasksAndInvalidate()
            }
        })
    }
    ///-------------------------
    /// @name Running Data Tasks
    ///-------------------------
    /**
     Creates an `NSURLSessionDataTask` with the specified request.
    
     @param request The HTTP request for the request.
     @param completionHandler A block object to be executed when the task finishes. This block has no return value and takes three arguments: the server response, the response object created by that serializer, and the error that occurred, if any.
     */

    override func dataTaskWithRequest(request: NSURLRequest, completionHandler: (response: NSURLResponse, responseObject: AnyObject, error: NSError) -> Void) -> NSURLSessionDataTask {
        var dataTask: NSURLSessionDataTask? = nil
        dispatch_sync(url_session_manager_creation_queue(), {() -> Void in
            dataTask = self.session.dataTaskWithRequest(request)
        })
        self.addDelegateForDataTask(dataTask, completionHandler: completionHandler)
        return dataTask
    }
    ///---------------------------
    /// @name Running Upload Tasks
    ///---------------------------
    /**
     Creates an `NSURLSessionUploadTask` with the specified request for a local file.
    
     @param request The HTTP request for the request.
     @param fileURL A URL to the local file to be uploaded.
     @param progress A progress object monitoring the current upload progress.
     @param completionHandler A block object to be executed when the task finishes. This block has no return value and takes three arguments: the server response, the response object created by that serializer, and the error that occurred, if any.
     
     @see `attemptsToRecreateUploadTasksForBackgroundSessions`
     */

    func uploadTaskWithRequest(request: NSURLRequest, fromFile fileURL: NSURL, progress: NSProgress, completionHandler: (response: NSURLResponse, responseObject: AnyObject, error: NSError) -> Void) -> NSURLSessionUploadTask {
        var uploadTask: NSURLSessionUploadTask? = nil
        dispatch_sync(url_session_manager_creation_queue(), {() -> Void in
            uploadTask = self.session.uploadTaskWithRequest(request, fromFile: fileURL)
        })
        if !uploadTask && self.attemptsToRecreateUploadTasksForBackgroundSessions && self.session.configuration.identifier {
            var attempts = 0
            while !uploadTask && attempts < AFMaximumNumberOfAttemptsToRecreateBackgroundSessionUploadTask {
                uploadTask = self.session.uploadTaskWithRequest(request, fromFile: fileURL)
                attempts += 1
            }
        }
        self.addDelegateForUploadTask(uploadTask, progress: progress, completionHandler: completionHandler)
        return uploadTask
    }
    /**
     Creates an `NSURLSessionUploadTask` with the specified request for an HTTP body.
    
     @param request The HTTP request for the request.
     @param bodyData A data object containing the HTTP body to be uploaded.
     @param progress A progress object monitoring the current upload progress.
     @param completionHandler A block object to be executed when the task finishes. This block has no return value and takes three arguments: the server response, the response object created by that serializer, and the error that occurred, if any.
     */

    func uploadTaskWithRequest(request: NSURLRequest, fromData bodyData: NSData, progress: NSProgress, completionHandler: (response: NSURLResponse, responseObject: AnyObject, error: NSError) -> Void) -> NSURLSessionUploadTask {
        var uploadTask: NSURLSessionUploadTask? = nil
        dispatch_sync(url_session_manager_creation_queue(), {() -> Void in
            uploadTask = self.session.uploadTaskWithRequest(request, fromData: bodyData)
        })
        self.addDelegateForUploadTask(uploadTask, progress: progress, completionHandler: completionHandler)
        return uploadTask
    }
    /**
     Creates an `NSURLSessionUploadTask` with the specified streaming request.
    
     @param request The HTTP request for the request.
     @param progress A progress object monitoring the current upload progress.
     @param completionHandler A block object to be executed when the task finishes. This block has no return value and takes three arguments: the server response, the response object created by that serializer, and the error that occurred, if any.
     */

    func uploadTaskWithStreamedRequest(request: NSURLRequest, progress: NSProgress, completionHandler: (response: NSURLResponse, responseObject: AnyObject, error: NSError) -> Void) -> NSURLSessionUploadTask {
        var uploadTask: NSURLSessionUploadTask? = nil
        dispatch_sync(url_session_manager_creation_queue(), {() -> Void in
            uploadTask = self.session.uploadTaskWithStreamedRequest(request)
        })
        self.addDelegateForUploadTask(uploadTask, progress: progress, completionHandler: completionHandler)
        return uploadTask
    }
    ///-----------------------------
    /// @name Running Download Tasks
    ///-----------------------------
    /**
     Creates an `NSURLSessionDownloadTask` with the specified request.
    
     @param request The HTTP request for the request.
     @param progress A progress object monitoring the current download progress.
     @param destination A block object to be executed in order to determine the destination of the downloaded file. This block takes two arguments, the target path & the server response, and returns the desired file URL of the resulting download. The temporary file used during the download will be automatically deleted after being moved to the returned URL.
     @param completionHandler A block to be executed when a task finishes. This block has no return value and takes three arguments: the server response, the path of the downloaded file, and the error describing the network or parsing error that occurred, if any.
     
     @warning If using a background `NSURLSessionConfiguration` on iOS, these blocks will be lost when the app is terminated. Background sessions may prefer to use `-setDownloadTaskDidFinishDownloadingBlock:` to specify the URL for saving the downloaded file, rather than the destination block of this method.
     */

    func downloadTaskWithRequest(request: NSURLRequest, progress: NSProgress, destination: (targetPath: NSURL, response: NSURLResponse) -> NSURL, completionHandler: (response: NSURLResponse, filePath: NSURL, error: NSError) -> Void) -> NSURLSessionDownloadTask {
        var downloadTask: NSURLSessionDownloadTask? = nil
        dispatch_sync(url_session_manager_creation_queue(), {() -> Void in
            downloadTask = self.session.downloadTaskWithRequest(request)
        })
        self.addDelegateForDownloadTask(downloadTask, progress: progress, destination: destination, completionHandler: completionHandler)
        return downloadTask
    }
    /**
     Creates an `NSURLSessionDownloadTask` with the specified resume data.
    
     @param resumeData The data used to resume downloading.
     @param progress A progress object monitoring the current download progress.
     @param destination A block object to be executed in order to determine the destination of the downloaded file. This block takes two arguments, the target path & the server response, and returns the desired file URL of the resulting download. The temporary file used during the download will be automatically deleted after being moved to the returned URL.
     @param completionHandler A block to be executed when a task finishes. This block has no return value and takes three arguments: the server response, the path of the downloaded file, and the error describing the network or parsing error that occurred, if any.
     */

    func downloadTaskWithResumeData(resumeData: NSData, progress: NSProgress, destination: (targetPath: NSURL, response: NSURLResponse) -> NSURL, completionHandler: (response: NSURLResponse, filePath: NSURL, error: NSError) -> Void) -> NSURLSessionDownloadTask {
        var downloadTask: NSURLSessionDownloadTask? = nil
        dispatch_sync(url_session_manager_creation_queue(), {() -> Void in
            downloadTask = self.session.downloadTaskWithResumeData(resumeData)
        })
        self.addDelegateForDownloadTask(downloadTask, progress: progress, destination: destination, completionHandler: completionHandler)
        return downloadTask
    }
    ///---------------------------------
    /// @name Getting Progress for Tasks
    ///---------------------------------
    /**
     Returns the upload progress of the specified task.
    
     @param uploadTask The session upload task. Must not be `nil`.
    
     @return An `NSProgress` object reporting the upload progress of a task, or `nil` if the progress is unavailable.
     */

    func uploadProgressForTask(uploadTask: NSURLSessionUploadTask) -> NSProgress {
        return self.delegateForTask(uploadTask).progress
    }
    /**
     Returns the download progress of the specified task.
     
     @param downloadTask The session download task. Must not be `nil`.
     
     @return An `NSProgress` object reporting the download progress of a task, or `nil` if the progress is unavailable.
     */

    func downloadProgressForTask(downloadTask: NSURLSessionDownloadTask) -> NSProgress {
        return self.delegateForTask(downloadTask).progress
    }
    ///-----------------------------------------
    /// @name Setting Session Delegate Callbacks
    ///-----------------------------------------
    /**
     Sets a block to be executed when the managed session becomes invalid, as handled by the `NSURLSessionDelegate` method `URLSession:didBecomeInvalidWithError:`.
     
     @param block A block object to be executed when the managed session becomes invalid. The block has no return value, and takes two arguments: the session, and the error related to the cause of invalidation.
     */

    func setSessionDidBecomeInvalidBlock(block: (session: NSURLSession, error: NSError) -> Void) {
        self.sessionDidBecomeInvalid = block
    }
    /**
     Sets a block to be executed when a connection level authentication challenge has occurred, as handled by the `NSURLSessionDelegate` method `URLSession:didReceiveChallenge:completionHandler:`.
    
     @param block A block object to be executed when a connection level authentication challenge has occurred. The block returns the disposition of the authentication challenge, and takes three arguments: the session, the authentication challenge, and a pointer to the credential that should be used to resolve the challenge.
     */

    func setSessionDidReceiveAuthenticationChallengeBlock(block: (session: NSURLSession, challenge: NSURLAuthenticationChallenge, credential: NSURLCredential) -> NSURLSessionAuthChallengeDisposition) {
        self.sessionDidReceiveAuthenticationChallenge = block
    }
    ///--------------------------------------
    /// @name Setting Task Delegate Callbacks
    ///--------------------------------------
    /**
     Sets a block to be executed when a task requires a new request body stream to send to the remote server, as handled by the `NSURLSessionTaskDelegate` method `URLSession:task:needNewBodyStream:`.
    
     @param block A block object to be executed when a task requires a new request body stream.
     */

    func setTaskNeedNewBodyStreamBlock(block: (session: NSURLSession, task: NSURLSessionTask) -> NSInputStream) {
        self.taskNeedNewBodyStream = block
    }
    /**
     Sets a block to be executed when an HTTP request is attempting to perform a redirection to a different URL, as handled by the `NSURLSessionTaskDelegate` method `URLSession:willPerformHTTPRedirection:newRequest:completionHandler:`.
     
     @param block A block object to be executed when an HTTP request is attempting to perform a redirection to a different URL. The block returns the request to be made for the redirection, and takes four arguments: the session, the task, the redirection response, and the request corresponding to the redirection response.
     */

    func setTaskWillPerformHTTPRedirectionBlock(block: (session: NSURLSession, task: NSURLSessionTask, response: NSURLResponse, request: NSURLRequest) -> NSURLRequest) {
        self.taskWillPerformHTTPRedirection = block
    }
    /**
     Sets a block to be executed when a session task has received a request specific authentication challenge, as handled by the `NSURLSessionTaskDelegate` method `URLSession:task:didReceiveChallenge:completionHandler:`.
     
     @param block A block object to be executed when a session task has received a request specific authentication challenge. The block returns the disposition of the authentication challenge, and takes four arguments: the session, the task, the authentication challenge, and a pointer to the credential that should be used to resolve the challenge.
     */

    func setTaskDidReceiveAuthenticationChallengeBlock(block: (session: NSURLSession, task: NSURLSessionTask, challenge: NSURLAuthenticationChallenge, credential: NSURLCredential) -> NSURLSessionAuthChallengeDisposition) {
        self.taskDidReceiveAuthenticationChallenge = block
    }
    /**
     Sets a block to be executed periodically to track upload progress, as handled by the `NSURLSessionTaskDelegate` method `URLSession:task:didSendBodyData:totalBytesSent:totalBytesExpectedToSend:`.
     
     @param block A block object to be called when an undetermined number of bytes have been uploaded to the server. This block has no return value and takes five arguments: the session, the task, the number of bytes written since the last time the upload progress block was called, the total bytes written, and the total bytes expected to be written during the request, as initially determined by the length of the HTTP body. This block may be called multiple times, and will execute on the main thread.
     */

    func setTaskDidSendBodyDataBlock(block: (session: NSURLSession, task: NSURLSessionTask, bytesSent: Int64, totalBytesSent: Int64, totalBytesExpectedToSend: Int64) -> Void) {
        self.taskDidSendBodyData = block
    }
    /**
     Sets a block to be executed as the last message related to a specific task, as handled by the `NSURLSessionTaskDelegate` method `URLSession:task:didCompleteWithError:`.
     
     @param block A block object to be executed when a session task is completed. The block has no return value, and takes three arguments: the session, the task, and any error that occurred in the process of executing the task.
     */

    func setTaskDidCompleteBlock(block: (session: NSURLSession, task: NSURLSessionTask, error: NSError) -> Void) {
        self.taskDidComplete = block
    }
    ///-------------------------------------------
    /// @name Setting Data Task Delegate Callbacks
    ///-------------------------------------------
    /**
     Sets a block to be executed when a data task has received a response, as handled by the `NSURLSessionDataDelegate` method `URLSession:dataTask:didReceiveResponse:completionHandler:`.
    
     @param block A block object to be executed when a data task has received a response. The block returns the disposition of the session response, and takes three arguments: the session, the data task, and the received response.
     */

    func setDataTaskDidReceiveResponseBlock(block: (session: NSURLSession, dataTask: NSURLSessionDataTask, response: NSURLResponse) -> NSURLSessionResponseDisposition) {
        self.dataTaskDidReceiveResponse = block
    }
    /**
     Sets a block to be executed when a data task has become a download task, as handled by the `NSURLSessionDataDelegate` method `URLSession:dataTask:didBecomeDownloadTask:`.
     
     @param block A block object to be executed when a data task has become a download task. The block has no return value, and takes three arguments: the session, the data task, and the download task it has become.
     */

    func setDataTaskDidBecomeDownloadTaskBlock(block: (session: NSURLSession, dataTask: NSURLSessionDataTask, downloadTask: NSURLSessionDownloadTask) -> Void) {
        self.dataTaskDidBecomeDownloadTask = block
    }
    /**
     Sets a block to be executed when a data task receives data, as handled by the `NSURLSessionDataDelegate` method `URLSession:dataTask:didReceiveData:`.
     
     @param block A block object to be called when an undetermined number of bytes have been downloaded from the server. This block has no return value and takes three arguments: the session, the data task, and the data received. This block may be called multiple times, and will execute on the session manager operation queue.
     */

    func setDataTaskDidReceiveDataBlock(block: (session: NSURLSession, dataTask: NSURLSessionDataTask, data: NSData) -> Void) {
        self.dataTaskDidReceiveData = block
    }
    /**
     Sets a block to be executed to determine the caching behavior of a data task, as handled by the `NSURLSessionDataDelegate` method `URLSession:dataTask:willCacheResponse:completionHandler:`.
     
     @param block A block object to be executed to determine the caching behavior of a data task. The block returns the response to cache, and takes three arguments: the session, the data task, and the proposed cached URL response.
     */

    func setDataTaskWillCacheResponseBlock(block: (session: NSURLSession, dataTask: NSURLSessionDataTask, proposedResponse: NSCachedURLResponse) -> NSCachedURLResponse) {
        self.dataTaskWillCacheResponse = block
    }
    /**
     Sets a block to be executed once all messages enqueued for a session have been delivered, as handled by the `NSURLSessionDataDelegate` method `URLSessionDidFinishEventsForBackgroundURLSession:`.
     
     @param block A block object to be executed once all messages enqueued for a session have been delivered. The block has no return value and takes a single argument: the session.
     */

    func setDidFinishEventsForBackgroundURLSessionBlock(block: (session: NSURLSession) -> Void) {
        self.didFinishEventsForBackgroundURLSession = block
    }
    ///-----------------------------------------------
    /// @name Setting Download Task Delegate Callbacks
    ///-----------------------------------------------
    /**
     Sets a block to be executed when a download task has completed a download, as handled by the `NSURLSessionDownloadDelegate` method `URLSession:downloadTask:didFinishDownloadingToURL:`.
     
     @param block A block object to be executed when a download task has completed. The block returns the URL the download should be moved to, and takes three arguments: the session, the download task, and the temporary location of the downloaded file. If the file manager encounters an error while attempting to move the temporary file to the destination, an `AFURLSessionDownloadTaskDidFailToMoveFileNotification` will be posted, with the download task as its object, and the user info of the error.
     */

    func setDownloadTaskDidFinishDownloadingBlock(block: (session: NSURLSession, downloadTask: NSURLSessionDownloadTask, location: NSURL) -> NSURL) {
        self.downloadTaskDidFinishDownloading = block
    }
    /**
     Sets a block to be executed periodically to track download progress, as handled by the `NSURLSessionDownloadDelegate` method `URLSession:downloadTask:didWriteData:totalBytesWritten:totalBytesWritten:totalBytesExpectedToWrite:`.
     
     @param block A block object to be called when an undetermined number of bytes have been downloaded from the server. This block has no return value and takes five arguments: the session, the download task, the number of bytes read since the last time the download progress block was called, the total bytes read, and the total bytes expected to be read during the request, as initially determined by the expected content size of the `NSHTTPURLResponse` object. This block may be called multiple times, and will execute on the session manager operation queue.
     */

    func setDownloadTaskDidWriteDataBlock(block: (session: NSURLSession, downloadTask: NSURLSessionDownloadTask, bytesWritten: Int64, totalBytesWritten: Int64, totalBytesExpectedToWrite: Int64) -> Void) {
        self.downloadTaskDidWriteData = block
    }
    /**
     Sets a block to be executed when a download task has been resumed, as handled by the `NSURLSessionDownloadDelegate` method `URLSession:downloadTask:didResumeAtOffset:expectedTotalBytes:`.
     
     @param block A block object to be executed when a download task has been resumed. The block has no return value and takes four arguments: the session, the download task, the file offset of the resumed download, and the total number of bytes expected to be downloaded.
     */

    func setDownloadTaskDidResumeBlock(block: (session: NSURLSession, downloadTask: NSURLSessionDownloadTask, fileOffset: Int64, expectedTotalBytes: Int64) -> Void) {
        self.downloadTaskDidResume = block
    }


    convenience override init() {
        return self.init(sessionConfiguration: nil)
    }
// MARK: -

    func delegateForTask(task: NSURLSessionTask) -> AFURLSessionManagerTaskDelegate {
        NSParameterAssert(task)
        var delegate: AFURLSessionManagerTaskDelegate? = nil
        self.lock().lock()
        delegate = self.mutableTaskDelegatesKeyedByTaskIdentifier[(task.taskIdentifier)]
        self.lock().unlock()
        return delegate
    }

    func setDelegate(delegate: AFURLSessionManagerTaskDelegate, forTask task: NSURLSessionTask) {
        NSParameterAssert(task)
        NSParameterAssert(delegate)
        task.addObserver(self, forKeyPath: NSStringFromSelector(#selector(self.state)), options: [.Old, .New], context: AFTaskStateChangedContext)
        self.lock().lock()
        self.mutableTaskDelegatesKeyedByTaskIdentifier[(task.taskIdentifier)] = delegate
        self.lock().unlock()
    }

    func addDelegateForDataTask(dataTask: NSURLSessionDataTask, completionHandler: (response: NSURLResponse, responseObject: AnyObject, error: NSError) -> Void) {
        var delegate = AFURLSessionManagerTaskDelegate()
        delegate.manager = self
        delegate.completionHandler = completionHandler
        self.setDelegate(delegate, forTask: dataTask)
    }

    func addDelegateForUploadTask(uploadTask: NSURLSessionUploadTask, progress: NSProgress, completionHandler: (response: NSURLResponse, responseObject: AnyObject, error: NSError) -> Void) {
        var delegate = AFURLSessionManagerTaskDelegate()
        delegate.manager = self
        delegate.completionHandler = completionHandler
        var totalUnitCount: Int64 = uploadTask!.countOfBytesExpectedToSend
        if totalUnitCount == NSURLSessionTransferSizeUnknown {
            var contentLength = uploadTask!.originalRequest.valueForHTTPHeaderField("Content-Length")!
            if contentLength != "" {
                totalUnitCount = Int64(contentLength.longLongValue)
            }
        }
        delegate.progress = NSProgress(totalUnitCount: totalUnitCount)
        delegate.progress.pausingHandler = {() -> Void in
            uploadTask!.suspend()
        }
        delegate.progress.cancellationHandler = {() -> Void in
            uploadTask!.cancel()
        }
        if progress {
            progress = delegate.progress
        }
        self.setDelegate(delegate, forTask: uploadTask)
    }

    func addDelegateForDownloadTask(downloadTask: NSURLSessionDownloadTask, progress: NSProgress, destination: (targetPath: NSURL, response: NSURLResponse) -> NSURL, completionHandler: (response: NSURLResponse, filePath: NSURL, error: NSError) -> Void) {
        var delegate = AFURLSessionManagerTaskDelegate()
        delegate.manager = self
        delegate.completionHandler = completionHandler
        if destination {
            delegate.downloadTaskDidFinishDownloading = {(session: NSURLSession, task: NSURLSessionDownloadTask, location: NSURL) -> NSURL in
                return destination(location, task.response!)
            }
        }
        if progress {
            progress = delegate.progress
        }
        self.setDelegate(delegate, forTask: downloadTask)
    }

    func removeDelegateForTask(task: NSURLSessionTask) {
        NSParameterAssert(task)
        task.removeObserver(self, forKeyPath: NSStringFromSelector(#selector(self.state)), context: AFTaskStateChangedContext)
        self.lock().lock()
        self.mutableTaskDelegatesKeyedByTaskIdentifier.removeValueForKey((task.taskIdentifier))
        self.lock().unlock()
    }

    func removeAllDelegates() {
        self.lock().lock()
        self.mutableTaskDelegatesKeyedByTaskIdentifier.removeAll()
        self.lock().unlock()
    }
// MARK: -

    func tasksForKeyPath(keyPath: String) -> [AnyObject] {
        var tasks: [AnyObject]? = nil
        var semaphore = dispatch_semaphore_create(0)
        self.session.getTasksWithCompletionHandler({(dataTasks: [AnyObject], uploadTasks: [AnyObject], downloadTasks: [AnyObject]) -> Void in
            if (keyPath == NSStringFromSelector(#selector(self.dataTasks))) {
                tasks = dataTasks
            }
            else if (keyPath == NSStringFromSelector(#selector(self.uploadTasks))) {
                tasks = uploadTasks
            }
            else if (keyPath == NSStringFromSelector(#selector(self.downloadTasks))) {
                tasks = downloadTasks
            }
            else if (keyPath == NSStringFromSelector(#selector(self.tasks))) {
                tasks = [dataTasks, uploadTasks, downloadTasks].valueForKeyPath("@unionOfArrays.self")!
            }

            dispatch_semaphore_signal(semaphore)
        })
        dispatch_semaphore_wait(semaphore, DISPATCH_TIME_FOREVER)
        return tasks
    }

    func tasks() -> [AnyObject] {
        return self.tasksForKeyPath(NSStringFromSelector(__FUNCTION__))
    }

    func dataTasks() -> [AnyObject] {
        return self.tasksForKeyPath(NSStringFromSelector(__FUNCTION__))
    }

    func uploadTasks() -> [AnyObject] {
        return self.tasksForKeyPath(NSStringFromSelector(__FUNCTION__))
    }

    func downloadTasks() -> [AnyObject] {
        return self.tasksForKeyPath(NSStringFromSelector(__FUNCTION__))
    }
// MARK: -
// MARK: -

    func setResponseSerializer(responseSerializer: AFURLResponseSerialization) {
        NSParameterAssert(responseSerializer)
        self.responseSerializer = responseSerializer
    }
// MARK: -
// MARK: -
// MARK: -
// MARK: -
// MARK: -
// MARK: -
// MARK: -
// MARK: -
// MARK: - NSObject

    override func description() -> String {
        return "<\(NSStringFromClass(self.self)): \(self), session: \(self.session), operationQueue: \(self.operationQueue!)>"
    }

    override func respondsToSelector(selector: Selector) -> Bool {
        if selector == Selector("URLSession:task:willPerformHTTPRedirection:newRequest:completionHandler:") {
            return self.taskWillPerformHTTPRedirection != nil
        }
        else if selector == Selector("URLSession:dataTask:didReceiveResponse:completionHandler:") {
            return self.dataTaskDidReceiveResponse != nil
        }
        else if selector == Selector("URLSession:dataTask:willCacheResponse:completionHandler:") {
            return self.dataTaskWillCacheResponse != nil
        }
        else if selector == (self.URLSessionDidFinishEventsForBackgroundURLSession) {
            return self.didFinishEventsForBackgroundURLSession != nil
        }

        return self.self.instancesRespondToSelector(selector)
    }
// MARK: - NSKeyValueObserving

    override func observeValueForKeyPath(keyPath: String, ofObject object: AnyObject, change: [NSObject : AnyObject], context: UnsafeMutablePointer<Void>) {
        if context == AFTaskStateChangedContext && (keyPath == "state") {
            if change[.OldKey] && change[.NewKey] && change[.NewKey].isEqual(change[.OldKey]) {
                return
            }
            var notificationName: String? = nil
            switch (object as! NSURLSessionTask).state {
                case NSURLSessionTaskStateRunning:
                    notificationName = AFNetworkingTaskDidResumeNotification
                case NSURLSessionTaskStateSuspended:
                    notificationName = AFNetworkingTaskDidSuspendNotification
            case NSURLSessionTaskStateCompleted: break                 // AFNetworkingTaskDidFinishNotification posted by task completion handlers
default:
                    break
            }

            if notificationName != nil {
                dispatch_async(dispatch_get_main_queue(), {() -> Void in
                    NSNotificationCenter.defaultCenter().postNotificationName(notificationName, object: object)
                })
            }
        }
        else {
            super.observeValueForKeyPath(keyPath, ofObject: object, change: change, context: context)
        }
    }
// MARK: - NSURLSessionDelegate

    func URLSession(session: NSURLSession, didBecomeInvalidWithError error: NSError?) {
        if self.sessionDidBecomeInvalid {
            self.sessionDidBecomeInvalid(session, error)
        }
        self.session.getTasksWithCompletionHandler({(dataTasks: [AnyObject], uploadTasks: [AnyObject], downloadTasks: [AnyObject]) -> Void in
            var tasks = [dataTasks, uploadTasks, downloadTasks].valueForKeyPath("@unionOfArrays.self")!
            for task: NSURLSessionTask in tasks {
                task.removeObserver(self, forKeyPath: NSStringFromSelector((self.state)), context: AFTaskStateChangedContext)
            }
            self.removeAllDelegates()
        })
        NSNotificationCenter.defaultCenter().postNotificationName(AFURLSessionDidInvalidateNotification, object: session)
    }

    func URLSession(session: NSURLSession, didReceiveChallenge challenge: NSURLAuthenticationChallenge, completionHandler: (disposition: NSURLSessionAuthChallengeDisposition, credential: NSURLCredential) -> Void) {
        var disposition = NSURLSessionAuthChallengePerformDefaultHandling
        var credential: NSURLCredential? = nil
        if self.sessionDidReceiveAuthenticationChallenge {
            disposition = self.sessionDidReceiveAuthenticationChallenge(session, challenge, credential)
        }
        else {
            if (challenge.protectionSpace.authenticationMethod! == NSURLAuthenticationMethodServerTrust) {
                if self.securityPolicy.evaluateServerTrust(challenge.protectionSpace.serverTrust!, forDomain: challenge.protectionSpace.host) {
                    credential = NSURLCredential(forTrust: challenge.protectionSpace.serverTrust!)
                    if credential != nil {
                        disposition = NSURLSessionAuthChallengeUseCredential
                    }
                    else {
                        disposition = NSURLSessionAuthChallengePerformDefaultHandling
                    }
                }
                else {
                    disposition = NSURLSessionAuthChallengeCancelAuthenticationChallenge
                }
            }
            else {
                disposition = NSURLSessionAuthChallengePerformDefaultHandling
            }
        }
        if completionHandler {
            completionHandler(disposition, credential)
        }
    }
// MARK: - NSURLSessionTaskDelegate

    func URLSession(session: NSURLSession, task: NSURLSessionTask, willPerformHTTPRedirection response: NSHTTPURLResponse, newRequest request: NSURLRequest, completionHandler: () -> Void) {
        var redirectRequest = request
        if self.taskWillPerformHTTPRedirection {
            redirectRequest = self.taskWillPerformHTTPRedirection(session, task, response, request)
        }
        if completionHandler {
            completionHandler(redirectRequest)
        }
    }

    func URLSession(session: NSURLSession, task: NSURLSessionTask, didReceiveChallenge challenge: NSURLAuthenticationChallenge, completionHandler: (disposition: NSURLSessionAuthChallengeDisposition, credential: NSURLCredential) -> Void) {
        var disposition = NSURLSessionAuthChallengePerformDefaultHandling
        var credential: NSURLCredential? = nil
        if self.taskDidReceiveAuthenticationChallenge {
            disposition = self.taskDidReceiveAuthenticationChallenge(session, task, challenge, credential)
        }
        else {
            if (challenge.protectionSpace.authenticationMethod! == NSURLAuthenticationMethodServerTrust) {
                if self.securityPolicy.evaluateServerTrust(challenge.protectionSpace.serverTrust!, forDomain: challenge.protectionSpace.host) {
                    disposition = NSURLSessionAuthChallengeUseCredential
                    credential = NSURLCredential(forTrust: challenge.protectionSpace.serverTrust!)
                }
                else {
                    disposition = NSURLSessionAuthChallengeCancelAuthenticationChallenge
                }
            }
            else {
                disposition = NSURLSessionAuthChallengePerformDefaultHandling
            }
        }
        if completionHandler {
            completionHandler(disposition, credential)
        }
    }

    func URLSession(session: NSURLSession, task: NSURLSessionTask, needNewBodyStream completionHandler: (bodyStream: NSInputStream) -> Void) {
        var inputStream: NSInputStream? = nil
        if self.taskNeedNewBodyStream {
            inputStream = self.taskNeedNewBodyStream(session, task)
        }
        else if task.originalRequest.HTTPBodyStream! && task.originalRequest.HTTPBodyStream! is NSCopying {
            inputStream = task.originalRequest.HTTPBodyStream!
        }

        if completionHandler {
            completionHandler(inputStream)
        }
    }

    func URLSession(session: NSURLSession, task: NSURLSessionTask, didSendBodyData bytesSent: Int64, totalBytesSent: Int64, totalBytesExpectedToSend: Int64) {
        var totalUnitCount: Int64 = totalBytesExpectedToSend
        if totalUnitCount == NSURLSessionTransferSizeUnknown {
            var contentLength = task.originalRequest.valueForHTTPHeaderField("Content-Length")!
            if contentLength != "" {
                totalUnitCount = Int64(contentLength.longLongValue)
            }
        }
        var delegate = self.delegateForTask(task)
        delegate.URLSession(session, task: task, didSendBodyData: bytesSent, totalBytesSent: totalBytesSent, totalBytesExpectedToSend: totalUnitCount)
        if self.taskDidSendBodyData {
            self.taskDidSendBodyData(session, task, bytesSent, totalBytesSent, totalUnitCount)
        }
    }

    func URLSession(session: NSURLSession, task: NSURLSessionTask, didCompleteWithError error: NSError?) {
        var delegate = self.delegateForTask(task)
        // delegate may be nil when completing a task in the background
        if delegate {
            delegate.URLSession(session, task: task, didCompleteWithError: error)
            self.removeDelegateForTask(task)
        }
        if self.taskDidComplete {
            self.taskDidComplete(session, task, error)
        }
    }
// MARK: - NSURLSessionDataDelegate

    func URLSession(session: NSURLSession, dataTask: NSURLSessionDataTask, didReceiveResponse response: NSURLResponse, completionHandler: (disposition: NSURLSessionResponseDisposition) -> Void) {
        var disposition = NSURLSessionResponseAllow
        if self.dataTaskDidReceiveResponse {
            disposition = self.dataTaskDidReceiveResponse(session, dataTask, response)
        }
        if completionHandler {
            completionHandler(disposition)
        }
    }

    func URLSession(session: NSURLSession, dataTask: NSURLSessionDataTask, didBecomeDownloadTask downloadTask: NSURLSessionDownloadTask) {
        var delegate = self.delegateForTask(dataTask)
        if delegate {
            self.removeDelegateForTask(dataTask)
            self.setDelegate(delegate, forTask: downloadTask)
        }
        if self.dataTaskDidBecomeDownloadTask {
            self.dataTaskDidBecomeDownloadTask(session, dataTask, downloadTask)
        }
    }

    func URLSession(session: NSURLSession, dataTask: NSURLSessionDataTask, didReceiveData data: NSData) {
        var delegate = self.delegateForTask(dataTask)
        delegate.URLSession(session, dataTask: dataTask, didReceiveData: data)
        if self.dataTaskDidReceiveData {
            self.dataTaskDidReceiveData(session, dataTask, data)
        }
    }

    func URLSession(session: NSURLSession, dataTask: NSURLSessionDataTask, willCacheResponse proposedResponse: NSCachedURLResponse, completionHandler: (cachedResponse: NSCachedURLResponse) -> Void) {
        var cachedResponse = proposedResponse
        if self.dataTaskWillCacheResponse {
            cachedResponse = self.dataTaskWillCacheResponse(session, dataTask, proposedResponse)
        }
        if completionHandler {
            completionHandler(cachedResponse)
        }
    }

    func URLSessionDidFinishEventsForBackgroundURLSession(session: NSURLSession) {
        if self.didFinishEventsForBackgroundURLSession {
            dispatch_async(dispatch_get_main_queue(), {() -> Void in
                self.didFinishEventsForBackgroundURLSession(session)
            })
        }
    }
// MARK: - NSURLSessionDownloadDelegate

    func URLSession(session: NSURLSession, downloadTask: NSURLSessionDownloadTask, didFinishDownloadingToURL location: NSURL) {
        if self.downloadTaskDidFinishDownloading {
            var fileURL = self.downloadTaskDidFinishDownloading(session, downloadTask, location)
            if fileURL {
                var error: NSError? = nil
                do {
                    try NSFileManager.defaultManager().moveItemAtURL(location, toURL: fileURL)
                }
                catch let error {
                }
                if error != nil {
                    NSNotificationCenter.defaultCenter().postNotificationName(AFURLSessionDownloadTaskDidFailToMoveFileNotification, object: downloadTask, userInfo: error!.userInfo!)
                }
                return
            }
        }
        var delegate = self.delegateForTask(downloadTask)
        if delegate {
            delegate.URLSession(session, downloadTask: downloadTask, didFinishDownloadingToURL: location)
        }
    }

    func URLSession(session: NSURLSession, downloadTask: NSURLSessionDownloadTask, didWriteData bytesWritten: Int64, totalBytesWritten: Int64, totalBytesExpectedToWrite: Int64) {
        var delegate = self.delegateForTask(downloadTask)
        delegate.URLSession(session, downloadTask: downloadTask, didWriteData: bytesWritten, totalBytesWritten: totalBytesWritten, totalBytesExpectedToWrite: totalBytesExpectedToWrite)
        if self.downloadTaskDidWriteData {
            self.downloadTaskDidWriteData(session, downloadTask, bytesWritten, totalBytesWritten, totalBytesExpectedToWrite)
        }
    }

    func URLSession(session: NSURLSession, downloadTask: NSURLSessionDownloadTask, didResumeAtOffset fileOffset: Int64, expectedTotalBytes: Int64) {
        var delegate = self.delegateForTask(downloadTask)
        delegate.URLSession(session, downloadTask: downloadTask, didResumeAtOffset: fileOffset, expectedTotalBytes: expectedTotalBytes)
        if self.downloadTaskDidResume {
            self.downloadTaskDidResume(session, downloadTask, fileOffset, expectedTotalBytes)
        }
    }
// MARK: - NSSecureCoding

    class func supportsSecureCoding() -> Bool {
        return true
    }

    convenience required init?(coder decoder: NSCoder) {
        var configuration = decoder.decodeObjectOfClass(NSURLSessionConfiguration.self, forKey: "sessionConfiguration")!
        self.init(sessionConfiguration: configuration)
        if !self {
            return nil
        }
    }

    override func encodeWithCoder(coder: NSCoder) {
        coder.encodeObject(self.session.configuration, forKey: "sessionConfiguration")
    }
// MARK: - NSCopying

    override func copyWithZone(zone: NSZone) -> AnyObject {
        return self.self.allocWithZone(zone)(sessionConfiguration: self.session.configuration)
    }

    var sessionConfiguration: NSURLSessionConfiguration!
    var operationQueue: NSOperationQueue!
    var session: NSURLSession!
    var mutableTaskDelegatesKeyedByTaskIdentifier = [NSObject : AnyObject]()
    var lock: NSLock!
    var sessionDidBecomeInvalid = AFURLSessionDidBecomeInvalidBlock()
    var sessionDidReceiveAuthenticationChallenge = AFURLSessionDidReceiveAuthenticationChallengeBlock()
    var didFinishEventsForBackgroundURLSession = AFURLSessionDidFinishEventsForBackgroundURLSessionBlock()
    var taskWillPerformHTTPRedirection = AFURLSessionTaskWillPerformHTTPRedirectionBlock()
    var taskDidReceiveAuthenticationChallenge = AFURLSessionTaskDidReceiveAuthenticationChallengeBlock()
    var taskNeedNewBodyStream = AFURLSessionTaskNeedNewBodyStreamBlock()
    var taskDidSendBodyData = AFURLSessionTaskDidSendBodyDataBlock()
    var taskDidComplete = AFURLSessionTaskDidCompleteBlock()
    var dataTaskDidReceiveResponse = AFURLSessionDataTaskDidReceiveResponseBlock()
    var dataTaskDidBecomeDownloadTask = AFURLSessionDataTaskDidBecomeDownloadTaskBlock()
    var dataTaskDidReceiveData = AFURLSessionDataTaskDidReceiveDataBlock()
    var dataTaskWillCacheResponse = AFURLSessionDataTaskWillCacheResponseBlock()
    var downloadTaskDidFinishDownloading = AFURLSessionDownloadTaskDidFinishDownloadingBlock()
    var downloadTaskDidWriteData = AFURLSessionDownloadTaskDidWriteDataBlock()
    var downloadTaskDidResume = AFURLSessionDownloadTaskDidResumeBlock()
}
#endif
///--------------------
/// @name Notifications
///--------------------
/**
 Posted when a task begins executing.
 
 @deprecated Use `AFNetworkingTaskDidResumeNotification` instead.
 */
let DEPRECATED_ATTRIBUTE = ""

/**
 Posted when a task resumes.
 */
let AFNetworkingTaskDidResumeNotification = ""

/**
 Posted when a task finishes executing. Includes a userInfo dictionary with additional information about the task.
 
 @deprecated Use `AFNetworkingTaskDidCompleteNotification` instead.
 */
let DEPRECATED_ATTRIBUTE = ""

/**
 Posted when a task finishes executing. Includes a userInfo dictionary with additional information about the task.
 */
let AFNetworkingTaskDidCompleteNotification = ""

/**
 Posted when a task suspends its execution.
 */
let AFNetworkingTaskDidSuspendNotification = ""

/**
 Posted when a session is invalidated.
 */
let AFURLSessionDidInvalidateNotification = ""

/**
 Posted when a session download task encountered an error when moving the temporary download file to a specified destination.
 */
let AFURLSessionDownloadTaskDidFailToMoveFileNotification = ""

/**
 The raw response data of the task. Included in the userInfo dictionary of the `AFNetworkingTaskDidFinishNotification` if response data exists for the task.
 
 @deprecated Use `AFNetworkingTaskDidCompleteResponseDataKey` instead.
 */
let DEPRECATED_ATTRIBUTE = ""

/**
 The raw response data of the task. Included in the userInfo dictionary of the `AFNetworkingTaskDidFinishNotification` if response data exists for the task.
 */
let AFNetworkingTaskDidCompleteResponseDataKey = ""

/**
 The serialized response object of the task. Included in the userInfo dictionary of the `AFNetworkingTaskDidFinishNotification` if the response was serialized.
 
 @deprecated Use `AFNetworkingTaskDidCompleteSerializedResponseKey` instead.
 */
let DEPRECATED_ATTRIBUTE = ""

/**
 The serialized response object of the task. Included in the userInfo dictionary of the `AFNetworkingTaskDidFinishNotification` if the response was serialized.
 */
let AFNetworkingTaskDidCompleteSerializedResponseKey = ""

/**
 The response serializer used to serialize the response. Included in the userInfo dictionary of the `AFNetworkingTaskDidFinishNotification` if the task has an associated response serializer.
 
 @deprecated Use `AFNetworkingTaskDidCompleteResponseSerializerKey` instead.
 */
let DEPRECATED_ATTRIBUTE = ""

/**
 The response serializer used to serialize the response. Included in the userInfo dictionary of the `AFNetworkingTaskDidFinishNotification` if the task has an associated response serializer.
 */
let AFNetworkingTaskDidCompleteResponseSerializerKey = ""

/**
 The file path associated with the download task. Included in the userInfo dictionary of the `AFNetworkingTaskDidFinishNotification` if an the response data has been stored directly to disk.
 
 @deprecated Use `AFNetworkingTaskDidCompleteAssetPathKey` instead.
 */
let DEPRECATED_ATTRIBUTE = ""

/**
 The file path associated with the download task. Included in the userInfo dictionary of the `AFNetworkingTaskDidFinishNotification` if an the response data has been stored directly to disk.
 */
let AFNetworkingTaskDidCompleteAssetPathKey = ""

/**
 Any error associated with the task, or the serialization of the response. Included in the userInfo dictionary of the `AFNetworkingTaskDidFinishNotification` if an error exists.
 
 @deprecated Use `AFNetworkingTaskDidCompleteErrorKey` instead.
 */
let DEPRECATED_ATTRIBUTE = ""

/**
 Any error associated with the task, or the serialization of the response. Included in the userInfo dictionary of the `AFNetworkingTaskDidFinishNotification` if an error exists.
 */
let AFNetworkingTaskDidCompleteErrorKey = ""

// AFURLSessionManager.m
// 
// Copyright (c) 2013-2014 AFNetworking (http://afnetworking.com)
//
// Permission is hereby granted, free of charge, to any person obtaining a copy
// of this software and associated documentation files (the "Software"), to deal
// in the Software without restriction, including without limitation the rights
// to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
// copies of the Software, and to permit persons to whom the Software is
// furnished to do so, subject to the following conditions:
// 
// The above copyright notice and this permission notice shall be included in
// all copies or substantial portions of the Software.
// 
// THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
// IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
// FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
// AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
// LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
// OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
// THE SOFTWARE.
#if (defined(__IPHONE_OS_VERSION_MAX_ALLOWED) && __IPHONE_OS_VERSION_MAX_ALLOWED >= 70000) || (defined(__MAC_OS_X_VERSION_MAX_ALLOWED) && __MAC_OS_X_VERSION_MAX_ALLOWED >= 1090)
func url_session_manager_creation_queue() -> dispatch_queue_t {
    var af_url_session_manager_creation_queue: dispatch_queue_t
    var onceToken: dispatch_once_t
    dispatch_once(onceToken, {() -> Void in
        af_url_session_manager_creation_queue = dispatch_queue_create("com.alamofire.networking.session.manager.creation", DISPATCH_QUEUE_SERIAL)
    })
    return af_url_session_manager_creation_queue
}

func url_session_manager_processing_queue() -> dispatch_queue_t {
    var af_url_session_manager_processing_queue: dispatch_queue_t
    var onceToken: dispatch_once_t
    dispatch_once(onceToken, {() -> Void in
        af_url_session_manager_processing_queue = dispatch_queue_create("com.alamofire.networking.session.manager.processing", DISPATCH_QUEUE_CONCURRENT)
    })
    return af_url_session_manager_processing_queue
}

func url_session_manager_completion_group() -> dispatch_group_t {
    var af_url_session_manager_completion_group: dispatch_group_t
    var onceToken: dispatch_once_t
    dispatch_once(onceToken, {() -> Void in
        af_url_session_manager_completion_group = dispatch_group_create()
    })
    return af_url_session_manager_completion_group
}

let AFNetworkingTaskDidResumeNotification = "com.alamofire.networking.task.resume"

let AFNetworkingTaskDidCompleteNotification = "com.alamofire.networking.task.complete"

let AFNetworkingTaskDidSuspendNotification = "com.alamofire.networking.task.suspend"

let AFURLSessionDidInvalidateNotification = "com.alamofire.networking.session.invalidate"

let AFURLSessionDownloadTaskDidFailToMoveFileNotification = "com.alamofire.networking.session.download.file-manager-error"

let AFNetworkingTaskDidStartNotification = "com.alamofire.networking.task.resume"

// Deprecated
let AFNetworkingTaskDidFinishNotification = "com.alamofire.networking.task.complete"

// Deprecated
let AFNetworkingTaskDidCompleteSerializedResponseKey = "com.alamofire.networking.task.complete.serializedresponse"

let AFNetworkingTaskDidCompleteResponseSerializerKey = "com.alamofire.networking.task.complete.responseserializer"

let AFNetworkingTaskDidCompleteResponseDataKey = "com.alamofire.networking.complete.finish.responsedata"

let AFNetworkingTaskDidCompleteErrorKey = "com.alamofire.networking.task.complete.error"

let AFNetworkingTaskDidCompleteAssetPathKey = "com.alamofire.networking.task.complete.assetpath"

let AFNetworkingTaskDidFinishSerializedResponseKey = "com.alamofire.networking.task.complete.serializedresponse"

// Deprecated
let AFNetworkingTaskDidFinishResponseSerializerKey = "com.alamofire.networking.task.complete.responseserializer"

// Deprecated
let AFNetworkingTaskDidFinishResponseDataKey = "com.alamofire.networking.complete.finish.responsedata"

// Deprecated
let AFNetworkingTaskDidFinishErrorKey = "com.alamofire.networking.task.complete.error"

// Deprecated
let AFNetworkingTaskDidFinishAssetPathKey = "com.alamofire.networking.task.complete.assetpath"

// Deprecated
let AFURLSessionManagerLockName = "com.alamofire.networking.session.manager.lock"

let AFMaximumNumberOfAttemptsToRecreateBackgroundSessionUploadTask = 3

var AFTaskStateChangedContext = AFTaskStateChangedContext

typealias AFURLSessionDidBecomeInvalidBlock = (session: NSURLSession, error: NSError) -> Void
typealias AFURLSessionDidReceiveAuthenticationChallengeBlock = (session: NSURLSession, challenge: NSURLAuthenticationChallenge, credential: NSURLCredential) -> NSURLSessionAuthChallengeDisposition
typealias AFURLSessionTaskWillPerformHTTPRedirectionBlock = (session: NSURLSession, task: NSURLSessionTask, response: NSURLResponse, request: NSURLRequest) -> NSURLRequest
typealias AFURLSessionTaskDidReceiveAuthenticationChallengeBlock = (session: NSURLSession, task: NSURLSessionTask, challenge: NSURLAuthenticationChallenge, credential: NSURLCredential) -> NSURLSessionAuthChallengeDisposition
typealias AFURLSessionDidFinishEventsForBackgroundURLSessionBlock = (session: NSURLSession) -> Void
typealias AFURLSessionTaskNeedNewBodyStreamBlock = (session: NSURLSession, task: NSURLSessionTask) -> NSInputStream
typealias AFURLSessionTaskDidSendBodyDataBlock = (session: NSURLSession, task: NSURLSessionTask, bytesSent: Int64, totalBytesSent: Int64, totalBytesExpectedToSend: Int64) -> Void
typealias AFURLSessionTaskDidCompleteBlock = (session: NSURLSession, task: NSURLSessionTask, error: NSError) -> Void
typealias AFURLSessionDataTaskDidReceiveResponseBlock = (session: NSURLSession, dataTask: NSURLSessionDataTask, response: NSURLResponse) -> NSURLSessionResponseDisposition
typealias AFURLSessionDataTaskDidBecomeDownloadTaskBlock = (session: NSURLSession, dataTask: NSURLSessionDataTask, downloadTask: NSURLSessionDownloadTask) -> Void
typealias AFURLSessionDataTaskDidReceiveDataBlock = (session: NSURLSession, dataTask: NSURLSessionDataTask, data: NSData) -> Void
typealias AFURLSessionDataTaskWillCacheResponseBlock = (session: NSURLSession, dataTask: NSURLSessionDataTask, proposedResponse: NSCachedURLResponse) -> NSCachedURLResponse
typealias AFURLSessionDownloadTaskDidFinishDownloadingBlock = (session: NSURLSession, downloadTask: NSURLSessionDownloadTask, location: NSURL) -> NSURL
typealias AFURLSessionDownloadTaskDidWriteDataBlock = (session: NSURLSession, downloadTask: NSURLSessionDownloadTask, bytesWritten: Int64, totalBytesWritten: Int64, totalBytesExpectedToWrite: Int64) -> Void
typealias AFURLSessionDownloadTaskDidResumeBlock = (session: NSURLSession, downloadTask: NSURLSessionDownloadTask, fileOffset: Int64, expectedTotalBytes: Int64) -> Void
typealias AFURLSessionTaskCompletionHandler = (response: NSURLResponse, responseObject: AnyObject, error: NSError) -> Void
// MARK: -
class AFURLSessionManagerTaskDelegate: NSObject, NSURLSessionTaskDelegate, NSURLSessionDataDelegate, NSURLSessionDownloadDelegate {
    weak var manager: AFURLSessionManager?
    var mutableData: NSMutableData!
    var progress: NSProgress!
    var downloadFileURL: NSURL!
    var downloadTaskDidFinishDownloading = AFURLSessionDownloadTaskDidFinishDownloadingBlock()
    var completionHandler = AFURLSessionTaskCompletionHandler()


    override init() {
        super.init()
        if !self {
            return nil
        }
        self.mutableData = NSMutableData.data
        self.progress = NSProgress(totalUnitCount: 0)
    }
// MARK: - NSURLSessionTaskDelegate

    func URLSession(session: __unused NSURLSession, task: __unused NSURLSessionTask, didSendBodyData bytesSent: __unused int64_t, totalBytesSent: Int64, totalBytesExpectedToSend: Int64) {
        self.progress.totalUnitCount = totalBytesExpectedToSend
        self.progress.completedUnitCount = totalBytesSent
    }

    func URLSession(session: __unused NSURLSession, task: NSURLSessionTask, didCompleteWithError error: NSError?) {
//#pragma clang diagnostic push
//#pragma clang diagnostic ignored "-Wgnu"
        var manager = self.manager
        var responseObject: AnyObject? = nil
        var userInfo = [NSObject : AnyObject]()
        userInfo[AFNetworkingTaskDidCompleteResponseSerializerKey] = manager.responseSerializer
        if self.downloadFileURL {
            userInfo[AFNetworkingTaskDidCompleteAssetPathKey] = self.downloadFileURL
        }
        else if self.mutableData {
            userInfo[AFNetworkingTaskDidCompleteResponseDataKey] = NSData(data: self.mutableData)
        }

        if error != nil {
            userInfo[AFNetworkingTaskDidCompleteErrorKey] = error
            dispatch_group_async(manager.completionGroup ?? url_session_manager_completion_group(), manager.completionQueue ?? dispatch_get_main_queue(), {() -> Void in
                if self.completionHandler {
                    self.completionHandler(task.response!, responseObject, error)
                }
                dispatch_async(dispatch_get_main_queue(), {() -> Void in
                    NSNotificationCenter.defaultCenter().postNotificationName(AFNetworkingTaskDidCompleteNotification, object: task, userInfo: userInfo)
                })
            })
        }
        else {
            dispatch_async(url_session_manager_processing_queue(), {() -> Void in
                var serializationError: NSError? = nil
                do {
                    responseObject = try manager.responseSerializer.responseObjectForResponse(task.response!, data: NSData(data: self.mutableData))
                }
                catch let error {
                }
                if self.downloadFileURL {
                    responseObject = self.downloadFileURL
                }
                if responseObject != nil {
                    userInfo[AFNetworkingTaskDidCompleteSerializedResponseKey] = responseObject
                }
                if serializationError != nil {
                    userInfo[AFNetworkingTaskDidCompleteErrorKey] = serializationError
                }
                dispatch_group_async(manager.completionGroup ?? url_session_manager_completion_group(), manager.completionQueue ?? dispatch_get_main_queue(), {() -> Void in
                    if self.completionHandler {
                        self.completionHandler(task.response!, responseObject, serializationError)
                    }
                    dispatch_async(dispatch_get_main_queue(), {() -> Void in
                        NSNotificationCenter.defaultCenter().postNotificationName(AFNetworkingTaskDidCompleteNotification, object: task, userInfo: userInfo)
                    })
                })
            })
        }
//#pragma clang diagnostic pop
    }
// MARK: - NSURLSessionDataTaskDelegate

    func URLSession(session: __unused NSURLSession, dataTask: __unused NSURLSessionDataTask, didReceiveData data: NSData) {
        self.mutableData.appendData(data)
    }
// MARK: - NSURLSessionDownloadTaskDelegate

    func URLSession(session: NSURLSession, downloadTask: NSURLSessionDownloadTask, didFinishDownloadingToURL location: NSURL) {
        var fileManagerError: NSError? = nil
        self.downloadFileURL = nil
        if self.downloadTaskDidFinishDownloading {
            self.downloadFileURL = self.downloadTaskDidFinishDownloading(session, downloadTask, location)
            if self.downloadFileURL {
                do {
                    try NSFileManager.defaultManager().moveItemAtURL(location, toURL: self.downloadFileURL)
                }
                catch let error {
                }
                if fileManagerError != nil {
                    NSNotificationCenter.defaultCenter().postNotificationName(AFURLSessionDownloadTaskDidFailToMoveFileNotification, object: downloadTask, userInfo: fileManagerError!.userInfo!)
                }
            }
        }
    }

    func URLSession(session: __unused NSURLSession, downloadTask: __unused NSURLSessionDownloadTask, didWriteData bytesWritten: __unused int64_t, totalBytesWritten: Int64, totalBytesExpectedToWrite: Int64) {
        self.progress.totalUnitCount = totalBytesExpectedToWrite
        self.progress.completedUnitCount = totalBytesWritten
    }

    func URLSession(session: __unused NSURLSession, downloadTask: __unused NSURLSessionDownloadTask, didResumeAtOffset fileOffset: Int64, expectedTotalBytes: Int64) {
        self.progress.totalUnitCount = expectedTotalBytes
        self.progress.completedUnitCount = fileOffset
    }
}
// MARK: -