// AFSerialization.h
//
// Copyright (c) 2013-2014 AFNetworking (http://afnetworking.com)
//
// Permission is hereby granted, free of charge, to any person obtaining a copy
// of this software and associated documentation files (the "Software"), to deal
// in the Software without restriction, including without limitation the rights
// to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
// copies of the Software, and to permit persons to whom the Software is
// furnished to do so, subject to the following conditions:
// 
// The above copyright notice and this permission notice shall be included in
// all copies or substantial portions of the Software.
// 
// THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
// IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
// FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
// AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
// LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
// OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
// THE SOFTWARE.
import Foundation
import CoreGraphics
/**
 The `AFURLResponseSerialization` protocol is adopted by an object that decodes data into a more useful object representation, according to details in the server response. Response serializers may additionally perform validation on the incoming response and data.

 For example, a JSON response serializer may check for an acceptable status code (`2XX` range) and content type (`application/json`), decoding a valid JSON response into an object.
 */
protocol AFURLResponseSerialization: NSObject, NSSecureCoding, NSCopying {
    /**
     The response object decoded from the data associated with a specified response.
    
     @param response The response to be processed.
     @param data The response data to be decoded.
     @param error The error that occurred while attempting to decode the response data.
    
     @return The object decoded from the specified response data.
     */
    func responseObjectForResponse(response: NSURLResponse, data: NSData, error: NSError?) -> AnyObject
}
// MARK: -
/**
 `AFHTTPResponseSerializer` conforms to the `AFURLRequestSerialization` & `AFURLResponseSerialization` protocols, offering a concrete base implementation of query string / URL form-encoded parameter serialization and default request headers, as well as response status code and content type validation.

 Any request or response serializer dealing with HTTP is encouraged to subclass `AFHTTPResponseSerializer` in order to ensure consistent default behavior.
 */
class AFHTTPResponseSerializer: NSObject, AFURLResponseSerialization {
    /**
     The string encoding used to serialize parameters.
     */
    var stringEncoding = NSStringEncoding()
    /**
     Creates and returns a serializer with default configuration.
     */

    class func serializer() -> Self {
        return self.init()
    }
    ///-----------------------------------------
    /// @name Configuring Response Serialization
    ///-----------------------------------------
    /**
     The acceptable HTTP status codes for responses. When non-`nil`, responses with status codes not contained by the set will result in an error during validation.
    
     See http://www.w3.org/Protocols/rfc2616/rfc2616-sec10.html
     */
    var acceptableStatusCodes: NSIndexSet!
    /**
     The acceptable MIME types for responses. When non-`nil`, responses with a `Content-Type` with MIME types that do not intersect with the set will result in an error during validation.
     */
    var acceptableContentTypes = Set<NSObject>()
    /**
     Validates the specified response and data.
    
     In its base implementation, this method checks for an acceptable status code and content type. Subclasses may wish to add other domain-specific checks.
    
     @param response The response to be validated.
     @param data The data associated with the response.
     @param error The error that occurred while attempting to validate the response.
    
     @return `YES` if the response is valid, otherwise `NO`.
     */

    func validateResponse(response: NSHTTPURLResponse, data: NSData, error: NSError?) -> Bool {
        var responseIsValid = true
        var validationError: NSError? = nil
        if response && (response is NSHTTPURLResponse) {
            if self.acceptableContentTypes && !self.acceptableContentTypes.contains(response.MIMEType!) {
                if data.length > 0 {
                    var userInfo = [NSLocalizedDescriptionKey: NSLocalizedStringFromTable("Request failed: unacceptable content-type: \(response.MIMEType!)", "AFNetworking"), NSURLErrorFailingURLErrorKey: response.URL!, AFNetworkingOperationFailingURLResponseErrorKey: response, AFNetworkingOperationFailingURLResponseDataErrorKey: data]
                    validationError = AFErrorWithUnderlyingError(NSError(domain: AFURLResponseSerializationErrorDomain, code: NSURLErrorCannotDecodeContentData, userInfo: userInfo), validationError)
                }
                responseIsValid = false
            }
            if self.acceptableStatusCodes && !self.acceptableStatusCodes.containsIndex(Int(response.statusCode)) {
                var userInfo = [NSLocalizedDescriptionKey: NSLocalizedStringFromTable("Request failed: \(NSHTTPURLResponse.localizedStringForStatusCode(response.statusCode)) (\(Int(response.statusCode)))", "AFNetworking"), NSURLErrorFailingURLErrorKey: response.URL!, AFNetworkingOperationFailingURLResponseErrorKey: response, AFNetworkingOperationFailingURLResponseDataErrorKey: data]
                validationError = AFErrorWithUnderlyingError(NSError(domain: AFURLResponseSerializationErrorDomain, code: NSURLErrorBadServerResponse, userInfo: userInfo), validationError)
                responseIsValid = false
            }
        }
        if error && !responseIsValid {
            error = validationError
        }
        return responseIsValid
    }


    override init() {
        super.init()
        if !self {
            return nil
        }
        self.stringEncoding = NSUTF8StringEncoding
        self.acceptableStatusCodes = NSIndexSet(indexesInRange: NSMakeRange(200, 100))
        self.acceptableContentTypes = nil
    }
// MARK: -
// MARK: - AFURLResponseSerialization

    func responseObjectForResponse(response: NSURLResponse, data: NSData, error: NSError?) -> AnyObject {
        do {
            try self.validateResponse((response as! NSHTTPURLResponse), data: data)
        }
        catch let error {
        }
        return data
    }
// MARK: - NSSecureCoding

    class func supportsSecureCoding() -> Bool {
        return true
    }

    convenience required init?(coder decoder: NSCoder) {
        self.init()
        if !self {
            return nil
        }
        self.acceptableStatusCodes = decoder.decodeObjectOfClass(NSIndexSet.self, forKey: NSStringFromSelector(self.acceptableStatusCodes))!
        self.acceptableContentTypes = decoder.decodeObjectOfClass(NSIndexSet.self, forKey: NSStringFromSelector(self.acceptableContentTypes))!
    }

    override func encodeWithCoder(coder: NSCoder) {
        coder.encodeObject(self.acceptableStatusCodes, forKey: NSStringFromSelector(self.acceptableStatusCodes))
        coder.encodeObject(self.acceptableContentTypes, forKey: NSStringFromSelector((self.acceptableContentTypes)))
    }
// MARK: - NSCopying

    override func copyWithZone(zone: NSZone) -> AnyObject {
        var serializer = self.self.allocWithZone(zone)()
        serializer.acceptableStatusCodes = self.acceptableStatusCodes.copyWithZone(zone)
        serializer.acceptableContentTypes = self.acceptableContentTypes.copyWithZone(zone)
        return serializer
    }
}
// MARK: -
/**
 `AFJSONResponseSerializer` is a subclass of `AFHTTPResponseSerializer` that validates and decodes JSON responses.

 By default, `AFJSONResponseSerializer` accepts the following MIME types, which includes the official standard, `application/json`, as well as other commonly-used types:

 - `application/json`
 - `text/json`
 - `text/javascript`
 */
class AFJSONResponseSerializer: AFHTTPResponseSerializer {
    /**
     Options for reading the response JSON data and creating the Foundation objects. For possible values, see the `NSJSONSerialization` documentation section "NSJSONReadingOptions". `0` by default.
     */
    var readingOptions = NSJSONReadingOptions()
    /**
     Whether to remove keys with `NSNull` values from response JSON. Defaults to `NO`.
     */
    var removesKeysWithNullValues = false
    /**
     Creates and returns a JSON serializer with specified reading and writing options.
    
     @param readingOptions The specified JSON reading options.
     */

    class func serializerWithReadingOptions(readingOptions: NSJSONReadingOptions) -> Self {
        var serializer = self.init()
        serializer.readingOptions = readingOptions
        return serializer
    }


    class func serializer() -> Self {
        return self.serializerWithReadingOptions(0)
    }

    override init() {
        super.init()
        if !self {
            return nil
        }
        self.acceptableContentTypes = Set<NSObject>(["application/json", "text/json", "text/javascript"])
    }
// MARK: - AFURLResponseSerialization

    func responseObjectForResponse(response: NSURLResponse, data: NSData, error: NSError?) -> AnyObject {
        do {
            if !try self.validateResponse((response as! NSHTTPURLResponse), data: data) {
                    if !error || AFErrorOrUnderlyingErrorHasCodeInDomain(error, NSURLErrorCannotDecodeContentData, AFURLResponseSerializationErrorDomain) {
                        return nil
                    }
                }
        }
        catch let error {
        }
            // Workaround for behavior of Rails to return a single space for `head :ok` (a workaround for a bug in Safari), which is not interpreted as valid input by NSJSONSerialization.
            // See https://github.com/rails/rails/issues/1742
        var stringEncoding = self.stringEncoding
        if response.textEncodingName! {
            var encoding = CFStringConvertIANACharSetNameToEncoding((response.textEncodingName! as! CFString))
            if encoding != kCFStringEncodingInvalidId {
                stringEncoding = CFStringConvertEncodingToNSStringEncoding(encoding)
            }
        }
        var responseObject: AnyObject? = nil
        var serializationError: NSError? = nil
                    var responseString = String(data: data, encoding: stringEncoding)
            if responseString && !(responseString == " ") {
                // Workaround for a bug in NSJSONSerialization when Unicode character escape codes are used instead of the actual character
                // See http://stackoverflow.com/a/12843465/157142
                data = responseString.dataUsingEncoding(NSUTF8StringEncoding)!
                if data {
                    if data.length > 0 {
                        do {
                            responseObject = try NSJSONSerialization.JSONObjectWithData(data, options: self.readingOptions)!
                        }
                        catch let error {
                        }
                    }
                    else {
                        return nil
                    }
                }
                else {
                    var userInfo = [NSLocalizedDescriptionKey: NSLocalizedStringFromTable("Data failed decoding as a UTF-8 string", nil, "AFNetworking"), NSLocalizedFailureReasonErrorKey: NSLocalizedStringFromTable("Could not decode string: \(responseString)", nil, "AFNetworking")]
                    serializationError = NSError(domain: AFURLResponseSerializationErrorDomain, code: NSURLErrorCannotDecodeContentData, userInfo: userInfo)
                }
            }

        if self.removesKeysWithNullValues && responseObject {
            responseObject = AFJSONObjectByRemovingKeysWithNullValues(responseObject, self.readingOptions)
        }
        if error {
            error = AFErrorWithUnderlyingError(serializationError, error)
        }
        return responseObject
    }
// MARK: - NSSecureCoding

    required init?(coder decoder: NSCoder) {
        super.init(coder: decoder)
        if !self {
            return nil
        }
        self.readingOptions = CUnsignedInt(decoder.decodeObjectOfClass(Int.self, forKey: NSStringFromSelector(#selector(self.readingOptions)))!)
        self.removesKeysWithNullValues = CBool(decoder.decodeObjectOfClass(Int.self, forKey: NSStringFromSelector(#selector(self.removesKeysWithNullValues)))!)
    }

    override func encodeWithCoder(coder: NSCoder) {
        super.encodeWithCoder(coder)
        coder.encodeObject((self.readingOptions), forKey: NSStringFromSelector(#selector(self.readingOptions)))
        coder.encodeObject((self.removesKeysWithNullValues), forKey: NSStringFromSelector(#selector(self.removesKeysWithNullValues)))
    }
// MARK: - NSCopying

    override func copyWithZone(zone: NSZone) -> AnyObject {
        var serializer = self.self.allocWithZone(zone)()
        serializer.readingOptions = self.readingOptions
        serializer.removesKeysWithNullValues = self.removesKeysWithNullValues
        return serializer
    }
}
// MARK: -
/**
 `AFXMLParserSerializer` is a subclass of `AFHTTPResponseSerializer` that validates and decodes XML responses as an `NSXMLParser` objects.

 By default, `AFXMLParserSerializer` accepts the following MIME types, which includes the official standard, `application/xml`, as well as other commonly-used types:

 - `application/xml`
 - `text/xml`
 */
class AFXMLParserResponseSerializer: AFHTTPResponseSerializer {


    class func serializer() -> Self {
        var serializer = self.init()
        return serializer
    }

    override init() {
        super.init()
        if !self {
            return nil
        }
        self.acceptableContentTypes = ["application/xml", "text/xml"]
    }
// MARK: - AFURLResponseSerialization

    func responseObjectForResponse(response: NSHTTPURLResponse, data: NSData, error: NSError?) -> AnyObject {
        do {
            if !try self.validateResponse((response as! NSHTTPURLResponse), data: data) {
                    if !error || AFErrorOrUnderlyingErrorHasCodeInDomain(error, NSURLErrorCannotDecodeContentData, AFURLResponseSerializationErrorDomain) {
                        return nil
                    }
                }
        }
        catch let error {
        }
        return NSXMLParser(data: data)
    }
}
// MARK: -
#if __MAC_OS_X_VERSION_MIN_REQUIRED
/**
 `AFXMLDocumentSerializer` is a subclass of `AFHTTPResponseSerializer` that validates and decodes XML responses as an `NSXMLDocument` objects.

 By default, `AFXMLDocumentSerializer` accepts the following MIME types, which includes the official standard, `application/xml`, as well as other commonly-used types:

 - `application/xml`
 - `text/xml`
 */
class AFXMLDocumentResponseSerializer: AFHTTPResponseSerializer {
    /**
     Input and output options specifically intended for `NSXMLDocument` objects. For possible values, see the `NSJSONSerialization` documentation section "NSJSONReadingOptions". `0` by default.
     */
    var options = 0
    /**
     Creates and returns an XML document serializer with the specified options.
    
     @param mask The XML document options.
     */

    class func serializerWithXMLDocumentOptions(mask: Int) -> Self {
        var serializer = self.init()
        serializer.options = mask
        return serializer
    }


    class func serializer() -> Self {
        return self.serializerWithXMLDocumentOptions(0)
    }

    override init() {
        super.init()
        if !self {
            return nil
        }
        self.acceptableContentTypes = ["application/xml", "text/xml"]
    }
// MARK: - AFURLResponseSerialization

    func responseObjectForResponse(response: NSURLResponse, data: NSData, error: NSError?) -> AnyObject {
        do {
            if !try self.validateResponse((response as! NSHTTPURLResponse), data: data) {
                    if !error || AFErrorOrUnderlyingErrorHasCodeInDomain(error, NSURLErrorCannotDecodeContentData, AFURLResponseSerializationErrorDomain) {
                        return nil
                    }
                }
        }
        catch let error {
        }
        var serializationError: NSError? = nil
        var document = try! NSXMLDocument(data: data, options: self.options)
        if error {
            error = AFErrorWithUnderlyingError(serializationError, error)
        }
        return document
    }
// MARK: - NSSecureCoding

    required init?(coder decoder: NSCoder) {
        super.init(coder: decoder)
        if !self {
            return nil
        }
        self.options = CUnsignedInt(decoder.decodeObjectOfClass(Int.self, forKey: NSStringFromSelector(#selector(self.options)))!)
    }

    override func encodeWithCoder(coder: NSCoder) {
        super.encodeWithCoder(coder)
        coder.encodeObject((self.options), forKey: NSStringFromSelector(#selector(self.options)))
    }
// MARK: - NSCopying

    override func copyWithZone(zone: NSZone) -> AnyObject {
        var serializer = self.self.allocWithZone(zone)()
        serializer.options = self.options
        return serializer
    }
}
#endif
// MARK: -
/**
 `AFPropertyListSerializer` is a subclass of `AFHTTPResponseSerializer` that validates and decodes XML responses as an `NSXMLDocument` objects.

 By default, `AFPropertyListSerializer` accepts the following MIME types:

 - `application/x-plist`
 */
class AFPropertyListResponseSerializer: AFHTTPResponseSerializer {
    /**
     The property list format. Possible values are described in "NSPropertyListFormat".
     */
    var format = NSPropertyListFormat()
    /**
     The property list reading options. Possible values are described in "NSPropertyListMutabilityOptions."
     */
    var readOptions = NSPropertyListReadOptions()
    /**
     Creates and returns a property list serializer with a specified format, read options, and write options.
    
     @param format The property list format.
     @param readOptions The property list reading options.
     */

    class func serializerWithFormat(format: NSPropertyListFormat, readOptions: NSPropertyListReadOptions) -> Self {
        var serializer = self.init()
        serializer.format = format
        serializer.readOptions = readOptions
        return serializer
    }


    class func serializer() -> Self {
        return self.serializerWithFormat(NSPropertyListXMLFormat_v1_0, readOptions: 0)
    }

    override init() {
        super.init()
        if !self {
            return nil
        }
        self.acceptableContentTypes = ["application/x-plist"]
    }
// MARK: - AFURLResponseSerialization

    func responseObjectForResponse(response: NSURLResponse, data: NSData, error: NSError?) -> AnyObject {
        do {
            if !try self.validateResponse((response as! NSHTTPURLResponse), data: data) {
                    if !error || AFErrorOrUnderlyingErrorHasCodeInDomain(error, NSURLErrorCannotDecodeContentData, AFURLResponseSerializationErrorDomain) {
                        return nil
                    }
                }
        }
        catch let error {
        }
        var responseObject: AnyObject?
        var serializationError: NSError? = nil
        if data {
            do {
                responseObject = try NSPropertyListSerialization.propertyListWithData(data, options: self.readOptions, format: nil)!
            }
            catch let error {
            }
        }
        if error {
            error = AFErrorWithUnderlyingError(serializationError, error)
        }
        return responseObject
    }
// MARK: - NSSecureCoding

    required init?(coder decoder: NSCoder) {
        super.init(coder: decoder)
        if !self {
            return nil
        }
        self.format = CUnsignedInt(decoder.decodeObjectOfClass(Int.self, forKey: NSStringFromSelector(#selector(self.format)))!)
        self.readOptions = CUnsignedInt(decoder.decodeObjectOfClass(Int.self, forKey: NSStringFromSelector(#selector(self.readOptions)))!)
    }

    override func encodeWithCoder(coder: NSCoder) {
        super.encodeWithCoder(coder)
        coder.encodeObject((self.format), forKey: NSStringFromSelector(#selector(self.format)))
        coder.encodeObject((self.readOptions), forKey: NSStringFromSelector(#selector(self.readOptions)))
    }
// MARK: - NSCopying

    override func copyWithZone(zone: NSZone) -> AnyObject {
        var serializer = self.self.allocWithZone(zone)()
        serializer.format = self.format
        serializer.readOptions = self.readOptions
        return serializer
    }
}
// MARK: -
/**
 `AFImageSerializer` is a subclass of `AFHTTPResponseSerializer` that validates and decodes image responses.

 By default, `AFImageSerializer` accepts the following MIME types, which correspond to the image formats supported by UIImage or NSImage:

 - `image/tiff`
 - `image/jpeg`
 - `image/gif`
 - `image/png`
 - `image/ico`
 - `image/x-icon`
 - `image/bmp`
 - `image/x-bmp`
 - `image/x-xbitmap`
 - `image/x-win-bitmap`
 */
class AFImageResponseSerializer: AFHTTPResponseSerializer {
#if defined(__IPHONE_OS_VERSION_MIN_REQUIRED)
    /**
     The scale factor used when interpreting the image data to construct `responseImage`. Specifying a scale factor of 1.0 results in an image whose size matches the pixel-based dimensions of the image. Applying a different scale factor changes the size of the image as reported by the size property. This is set to the value of scale of the main screen by default, which automatically scales images for retina displays, for instance.
     */
    var imageScale: CGFloat = 0.0
    /**
     Whether to automatically inflate response image data for compressed formats (such as PNG or JPEG). Enabling this can significantly improve drawing performance on iOS when used with `setCompletionBlockWithSuccess:failure:`, as it allows a bitmap representation to be constructed in the background rather than on the main thread. `YES` by default.
     */
    var automaticallyInflatesResponseImage = false


    override init() {
        super.init()
        if !self {
            return nil
        }
        self.acceptableContentTypes = ["image/tiff", "image/jpeg", "image/gif", "image/png", "image/ico", "image/x-icon", "image/bmp", "image/x-bmp", "image/x-xbitmap", "image/x-win-bitmap"]
#if defined(__IPHONE_OS_VERSION_MIN_REQUIRED)
        self.imageScale = UIScreen.mainScreen().scale
        self.automaticallyInflatesResponseImage = true
#endif

    }
// MARK: - AFURLResponseSerializer

    func responseObjectForResponse(response: NSURLResponse, data: NSData, error: NSError?) -> AnyObject {
        do {
            if !try self.validateResponse((response as! NSHTTPURLResponse), data: data) {
                    if !error || AFErrorOrUnderlyingErrorHasCodeInDomain(error, NSURLErrorCannotDecodeContentData, AFURLResponseSerializationErrorDomain) {
                        return nil
                    }
                }
        }
        catch let error {
        }
#if defined(__IPHONE_OS_VERSION_MIN_REQUIRED)
        if self.automaticallyInflatesResponseImage {
            return AFInflatedImageFromResponseWithDataAtScale((response as! NSHTTPURLResponse), data, self.imageScale)
        }
        else {
            return AFImageWithDataAtScale(data, self.imageScale)
        }
#elseif defined(__MAC_OS_X_VERSION_MIN_REQUIRED)
            // Ensure that the image is set to it's correct pixel width and height
        var bitimage = NSBitmapImageRep(data: data)
        var image = NSImage(size: NSMakeSize(bitimage.pixelsWide(), bitimage.pixelsHigh()))
        image.addRepresentation(bitimage)
        return image
#endif
        return nil
    }
// MARK: - NSSecureCoding

    required init?(coder decoder: NSCoder) {
        super.init(coder: decoder)
        if !self {
            return nil
        }
#if defined(__IPHONE_OS_VERSION_MIN_REQUIRED)
        var imageScale = decoder.decodeObjectOfClass(Int.self, forKey: NSStringFromSelector(#selector(self.imageScale)))!
#if CGFLOAT_IS_DOUBLE
        self.imageScale = CDouble(imageScale)
#else
        self.imageScale = CFloat(imageScale)
#endif
        self.automaticallyInflatesResponseImage = decoder.decodeBoolForKey(NSStringFromSelector(#selector(self.automaticallyInflatesResponseImage)))
#endif

    }

    override func encodeWithCoder(coder: NSCoder) {
        super.encodeWithCoder(coder)
#if defined(__IPHONE_OS_VERSION_MIN_REQUIRED)
        coder.encodeObject((self.imageScale), forKey: NSStringFromSelector(#selector(self.imageScale)))
        coder.encodeBool(self.automaticallyInflatesResponseImage, forKey: NSStringFromSelector(#selector(self.automaticallyInflatesResponseImage)))
#endif
    }
// MARK: - NSCopying

    override func copyWithZone(zone: NSZone) -> AnyObject {
        var serializer = self.self.allocWithZone(zone)()
#if defined(__IPHONE_OS_VERSION_MIN_REQUIRED)
        serializer.imageScale = self.imageScale
        serializer.automaticallyInflatesResponseImage = self.automaticallyInflatesResponseImage
#endif
        return serializer
    }
}
// MARK: -
/**
 `AFCompoundSerializer` is a subclass of `AFHTTPResponseSerializer` that delegates the response serialization to the first `AFHTTPResponseSerializer` object that returns an object for `responseObjectForResponse:data:error:`, falling back on the default behavior of `AFHTTPResponseSerializer`. This is useful for supporting multiple potential types and structures of server responses with a single serializer.
 */
class AFCompoundResponseSerializer: AFHTTPResponseSerializer {
    /**
     The component response serializers.
     */
    private(set) var responseSerializers = [AnyObject]()
    /**
     Creates and returns a compound serializer comprised of the specified response serializers.
    
     @warning Each response serializer specified must be a subclass of `AFHTTPResponseSerializer`, and response to `-validateResponse:data:error:`.
     */

    class func compoundSerializerWithResponseSerializers(responseSerializers: [AnyObject]) -> Self {
        var serializer = self.init()
        serializer.responseSerializers = responseSerializers
        return serializer
    }

// MARK: - AFURLResponseSerialization

    func responseObjectForResponse(response: NSURLResponse, data: NSData, error: NSError?) -> AnyObject {
        do {
            for serializer: AFURLResponseSerialization in self.responseSerializers {
                    if !(serializer is AFHTTPResponseSerializer) {
        
                    }
                    var serializerError: NSError? = nil
                    var responseObject = try serializer.responseObjectForResponse(response, data: data)
                    if responseObject {
                        if error {
                            error = AFErrorWithUnderlyingError(serializerError, error)
                        }
                        return responseObject
                    }
                }
        }
        catch let error {
        }
        do {
            return try super.responseObjectForResponse(response, data: data)
        }
        catch let error {
        }
    }
// MARK: - NSSecureCoding

    required init?(coder decoder: NSCoder) {
        super.init(coder: decoder)
        if !self {
            return nil
        }
        self.responseSerializers = decoder.decodeObjectOfClass([AnyObject].self, forKey: NSStringFromSelector(#selector(self.responseSerializers)))!
    }

    override func encodeWithCoder(coder: NSCoder) {
        super.encodeWithCoder(coder)
        coder.encodeObject(self.responseSerializers, forKey: NSStringFromSelector(#selector(self.responseSerializers)))
    }
// MARK: - NSCopying

    override func copyWithZone(zone: NSZone) -> AnyObject {
        var serializer = self.self.allocWithZone(zone)()
        serializer.responseSerializers = self.responseSerializers
        return serializer
    }

    var responseSerializers = [AnyObject]()
}
///----------------
/// @name Constants
///----------------
/**
 ## Error Domains

 The following error domain is predefined.

 - `NSString * const AFURLResponseSerializationErrorDomain`

 ### Constants

 `AFURLResponseSerializationErrorDomain`
 AFURLResponseSerializer errors. Error codes for `AFURLResponseSerializationErrorDomain` correspond to codes in `NSURLErrorDomain`.
 */
let AFURLResponseSerializationErrorDomain = ""

/**
 ## User info dictionary keys

 These keys may exist in the user info dictionary, in addition to those defined for NSError.

 - `NSString * const AFNetworkingOperationFailingURLResponseErrorKey`
 - `NSString * const AFNetworkingOperationFailingURLResponseDataErrorKey`

 ### Constants

 `AFNetworkingOperationFailingURLResponseErrorKey`
 The corresponding value is an `NSURLResponse` containing the response of the operation associated with an error. This key is only present in the `AFURLResponseSerializationErrorDomain`.
 
 `AFNetworkingOperationFailingURLResponseDataErrorKey`
 The corresponding value is an `NSData` containing the original data of the operation associated with an error. This key is only present in the `AFURLResponseSerializationErrorDomain`.
 */
let AFNetworkingOperationFailingURLResponseErrorKey = ""

let AFNetworkingOperationFailingURLResponseDataErrorKey = ""

// AFSerialization.h
//
// Copyright (c) 2013-2014 AFNetworking (http://afnetworking.com)
//
// Permission is hereby granted, free of charge, to any person obtaining a copy
// of this software and associated documentation files (the "Software"), to deal
// in the Software without restriction, including without limitation the rights
// to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
// copies of the Software, and to permit persons to whom the Software is
// furnished to do so, subject to the following conditions:
// 
// The above copyright notice and this permission notice shall be included in
// all copies or substantial portions of the Software.
// 
// THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
// IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
// FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
// AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
// LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
// OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
// THE SOFTWARE.
#if defined(__IPHONE_OS_VERSION_MIN_REQUIRED)
import UIKit
#elseif defined(__MAC_OS_X_VERSION_MIN_REQUIRED)
import Cocoa
#endif
let AFURLResponseSerializationErrorDomain = "com.alamofire.error.serialization.response"

let AFNetworkingOperationFailingURLResponseErrorKey = "com.alamofire.serialization.response.error.response"

let AFNetworkingOperationFailingURLResponseDataErrorKey = "com.alamofire.serialization.response.error.data"

func AFErrorWithUnderlyingError(error: NSError, underlyingError: NSError) -> NSError {
    if !error {
        return underlyingError
    }
    if !underlyingError || error.userInfo![NSUnderlyingErrorKey] {
        return error
    }
    var mutableUserInfo = error.userInfo!
    mutableUserInfo[NSUnderlyingErrorKey] = underlyingError
    return NSError(domain: error.domain, code: error.code, userInfo: mutableUserInfo)
}

func AFErrorOrUnderlyingErrorHasCodeInDomain(error: NSError, code: Int, domain: String) -> Bool {
    if (error.domain == domain) && error.code == code {
        return true
    }
    else if error.userInfo![NSUnderlyingErrorKey] {
        return AFErrorOrUnderlyingErrorHasCodeInDomain(error.userInfo![NSUnderlyingErrorKey], code, domain)
    }

    return false
}

func AFJSONObjectByRemovingKeysWithNullValues(JSONObject: AnyObject, readingOptions: NSJSONReadingOptions) -> AnyObject {
    if (JSONObject is [AnyObject]) {
        var mutableArray = [AnyObject]() /* capacity: (JSONObject as! [AnyObject]).count */
        for value: AnyObject in (JSONObject as! [AnyObject]) {
            mutableArray.append(AFJSONObjectByRemovingKeysWithNullValues(value, readingOptions))
        }
        return (readingOptions & NSJSONReadingMutableContainers) ? mutableArray : [AnyObject](arrayLiteral: mutableArray)
    }
    else if (JSONObject is [NSObject : AnyObject]) {
        var mutableDictionary = JSONObject
        for key: NSCopying in (JSONObject as! [NSObject : AnyObject]).allKeys() {
            var value = ((JSONObject as! [NSObject : AnyObject])[key] as! String)
            if !value || value.isEqual(NSNull()) {
                mutableDictionary.removeValueForKey(key)
            }
            else if (value is [AnyObject]) || (value is [NSObject : AnyObject]) {
                mutableDictionary[key] = AFJSONObjectByRemovingKeysWithNullValues(value, readingOptions)
            }
        }
        return (readingOptions & NSJSONReadingMutableContainers) ? mutableDictionary : mutableDictionary
    }

    return JSONObject
}

// MARK: -

// MARK: -

// MARK: -
#if __MAC_OS_X_VERSION_MIN_REQUIRED

#endif
// MARK: -

// MARK: -
#if defined(__IPHONE_OS_VERSION_MIN_REQUIRED)
import CoreGraphics
func AFImageWithDataAtScale(data: NSData, scale: CGFloat) -> UIImage {
    var image = UIImage(data: data)!
    return UIImage(CGImage: image.CGImage!, scale: scale, orientation: image.imageOrientation)
}

func AFInflatedImageFromResponseWithDataAtScale(response: NSHTTPURLResponse, data: NSData, scale: CGFloat) -> UIImage {
    if !data || data.length == 0 {
        return nil
    }
    var imageRef: CGImageRef? = nil
    var dataProvider = CGDataProviderCreateWithCFData((data as! CFDataRef))
    if (response.MIMEType! == "image/png") {
        imageRef = CGImageCreateWithPNGDataProvider(dataProvider, nil, true, kCGRenderingIntentDefault)
    }
    else if (response.MIMEType! == "image/jpeg") {
        imageRef = CGImageCreateWithJPEGDataProvider(dataProvider, nil, true, kCGRenderingIntentDefault)
        // CGImageCreateWithJPEGDataProvider does not properly handle CMKY, so if so, fall back to AFImageWithDataAtScale
        if imageRef != nil {
            var imageColorSpace = CGImageGetColorSpace(imageRef)
            var imageColorSpaceModel = CGColorSpaceGetModel(imageColorSpace)
            if imageColorSpaceModel == kCGColorSpaceModelCMYK {
                CGImageRelease(imageRef)
                imageRef = nil
            }
        }
    }

    CGDataProviderRelease(dataProvider)
    var image = AFImageWithDataAtScale(data, scale)
    if imageRef == nil {
        if image.images || !image {
            return image
        }
        imageRef = CGImageCreateCopy(image.CGImage!)
        if imageRef == nil {
            return nil
        }
    }
    var width = CGImageGetWidth(imageRef)
    var height = CGImageGetHeight(imageRef)
    var bitsPerComponent = CGImageGetBitsPerComponent(imageRef)
    if width * height > 1024 * 1024 || bitsPerComponent > 8 {
        CGImageRelease(imageRef)
        return image
    }
    var bytesPerRow = 0
        // CGImageGetBytesPerRow() calculates incorrectly in iOS 5.0, so defer to CGBitmapContextCreate
    var colorSpace = CGColorSpaceCreateDeviceRGB()
    var colorSpaceModel = CGColorSpaceGetModel(colorSpace)
    var bitmapInfo = CGImageGetBitmapInfo(imageRef)
    if colorSpaceModel == kCGColorSpaceModelRGB {
        var alpha = (bitmapInfo & kCGBitmapAlphaInfoMask)
        if alpha == kCGImageAlphaNone {
            bitmapInfo &= ~kCGBitmapAlphaInfoMask
            bitmapInfo |= kCGImageAlphaNoneSkipFirst
        }
        else if !(alpha == kCGImageAlphaNoneSkipFirst || alpha == kCGImageAlphaNoneSkipLast) {
            bitmapInfo &= ~kCGBitmapAlphaInfoMask
            bitmapInfo |= kCGImageAlphaPremultipliedFirst
        }
    }
    var context = CGBitmapContextCreate(nil, width, height, bitsPerComponent, bytesPerRow, colorSpace, bitmapInfo)
    CGColorSpaceRelease(colorSpace)
    if !context {
        CGImageRelease(imageRef)
        return image
    }
    CGContextDrawImage(context, CGRectMake(0.0, 0.0, width, height), imageRef)
    var inflatedImageRef = CGBitmapContextCreateImage(context)
    CGContextRelease(context)
    var inflatedImage = UIImage(CGImage: inflatedImageRef, scale: scale, orientation: image.imageOrientation)
    CGImageRelease(inflatedImageRef)
    CGImageRelease(imageRef)
    return inflatedImage
}

#endif

// MARK: -