// AFSerialization.h
//
// Copyright (c) 2013-2014 AFNetworking (http://afnetworking.com)
//
// Permission is hereby granted, free of charge, to any person obtaining a copy
// of this software and associated documentation files (the "Software"), to deal
// in the Software without restriction, including without limitation the rights
// to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
// copies of the Software, and to permit persons to whom the Software is
// furnished to do so, subject to the following conditions:
// 
// The above copyright notice and this permission notice shall be included in
// all copies or substantial portions of the Software.
// 
// THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
// IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
// FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
// AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
// LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
// OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
// THE SOFTWARE.
import Foundation
#if defined(__IPHONE_OS_VERSION_MIN_REQUIRED)
import UIKit
#endif
/**
 The `AFURLRequestSerialization` protocol is adopted by an object that encodes parameters for a specified HTTP requests. Request serializers may encode parameters as query strings, HTTP bodies, setting the appropriate HTTP header fields as necessary.

 For example, a JSON request serializer may set the HTTP body of the request to a JSON representation, and set the `Content-Type` HTTP header field value to `application/json`.
 */
protocol AFURLRequestSerialization: NSObject, NSSecureCoding, NSCopying {
    /**
     Returns a request with the specified parameters encoded into a copy of the original request.
    
     @param request The original request.
     @param parameters The parameters to be encoded.
     @param error The error that occurred while attempting to encode the request parameters.
    
     @return A serialized request.
     */
    func requestBySerializingRequest(request: NSURLRequest, withParameters parameters: AnyObject, error: NSError?) -> NSURLRequest
}
// MARK: -
/**

 */
enum AFHTTPRequestQueryStringSerializationStyle : Int {
    case AFHTTPRequestQueryStringDefaultStyle = 0
}

/**
 `AFHTTPRequestSerializer` conforms to the `AFURLRequestSerialization` & `AFURLResponseSerialization` protocols, offering a concrete base implementation of query string / URL form-encoded parameter serialization and default request headers, as well as response status code and content type validation.

 Any request or response serializer dealing with HTTP is encouraged to subclass `AFHTTPRequestSerializer` in order to ensure consistent default behavior.
 */
class AFHTTPRequestSerializer: NSObject, AFURLRequestSerialization {
    /**
     The string encoding used to serialize parameters. `NSUTF8StringEncoding` by default.
     */
    var stringEncoding = NSStringEncoding()
    /**
     Whether created requests can use the device’s cellular radio (if present). `YES` by default.
     
     @see NSMutableURLRequest -setAllowsCellularAccess:
     */
    var allowsCellularAccess = false
    /**
     The cache policy of created requests. `NSURLRequestUseProtocolCachePolicy` by default.
     
     @see NSMutableURLRequest -setCachePolicy:
     */
    var cachePolicy = NSURLRequestCachePolicy()
    /**
     Whether created requests should use the default cookie handling. `YES` by default.
     
     @see NSMutableURLRequest -setHTTPShouldHandleCookies:
     */
    var HTTPShouldHandleCookies = false
    /**
     Whether created requests can continue transmitting data before receiving a response from an earlier transmission. `NO` by default
     
     @see NSMutableURLRequest -setHTTPShouldUsePipelining:
     */
    var HTTPShouldUsePipelining = false
    /**
     The network service type for created requests. `NSURLNetworkServiceTypeDefault` by default.
     
     @see NSMutableURLRequest -setNetworkServiceType:
     */
    var networkServiceType = NSURLRequestNetworkServiceType()
    /**
     The timeout interval, in seconds, for created requests. The default timeout interval is 60 seconds.
     
     @see NSMutableURLRequest -setTimeoutInterval:
     */
    var timeoutInterval = NSTimeInterval()
    ///---------------------------------------
    /// @name Configuring HTTP Request Headers
    ///---------------------------------------
    /**
     Default HTTP header field values to be applied to serialized requests.
     */
    private(set) var HTTPRequestHeaders = [NSObject : AnyObject]()
    /**
     Creates and returns a serializer with default configuration.
     */

    class func serializer() -> Self {
        return self.init()
    }
    /**
     Sets the value for the HTTP headers set in request objects made by the HTTP client. If `nil`, removes the existing value for that header.
    
     @param field The HTTP header to set a default value for
     @param value The value set as default for the specified header, or `nil`
     */

    override func setValue(value: String, forHTTPHeaderField field: String) {
        self.mutableHTTPRequestHeaders[field] = value
    }
    /**
     Returns the value for the HTTP headers set in the request serializer.
    
     @param field The HTTP header to retrieve the default value for
     
     @return The value set as default for the specified header, or `nil`
     */

    override func valueForHTTPHeaderField(field: String) -> String {
        return (self.mutableHTTPRequestHeaders.valueForKey(field) as! String)
    }
    /**
     Sets the "Authorization" HTTP header set in request objects made by the HTTP client to a basic authentication value with Base64-encoded username and password. This overwrites any existing value for this header.
    
     @param username The HTTP basic auth username
     @param password The HTTP basic auth password
     */

    func setAuthorizationHeaderFieldWithUsername(username: String, password: String) {
        var basicAuthCredentials = "\(username):\(password)"
        self.setValue("Basic \(AFBase64EncodedStringFromString(basicAuthCredentials))", forHTTPHeaderField: "Authorization")
    }
    /**
     @deprecated This method has been deprecated. Use -setValue:forHTTPHeaderField: instead.
     */

    func setAuthorizationHeaderFieldWithToken(token: String) {
        self.setValue("Token token=\"\(token)\"", forHTTPHeaderField: "Authorization")
    }
    /**
     Clears any existing value for the "Authorization" HTTP header.
     */

    func clearAuthorizationHeader() {
        self.mutableHTTPRequestHeaders.removeValueForKey("Authorization")
    }
    ///-------------------------------------------------------
    /// @name Configuring Query String Parameter Serialization
    ///-------------------------------------------------------
    /**
     HTTP methods for which serialized requests will encode parameters as a query string. `GET`, `HEAD`, and `DELETE` by default.
     */
    var HTTPMethodsEncodingParametersInURI = Set<NSObject>()
    /**
     Set the method of query string serialization according to one of the pre-defined styles.
    
     @param style The serialization style.
    
     @see AFHTTPRequestQueryStringSerializationStyle
     */

    func setQueryStringSerializationWithStyle(style: AFHTTPRequestQueryStringSerializationStyle) {
        self.queryStringSerializationStyle = style
        self.queryStringSerialization = nil
    }
    /**
     Set the a custom method of query string serialization according to the specified block.
    
     @param block A block that defines a process of encoding parameters into a query string. This block returns the query string and takes three arguments: the request, the parameters to encode, and the error that occurred when attempting to encode parameters for the given request.
     */

    func setQueryStringSerializationWithBlock(block: () -> String) {
        self.queryStringSerialization = block
    }
    ///-------------------------------
    /// @name Creating Request Objects
    ///-------------------------------
    /**
     @deprecated This method has been deprecated. Use -requestWithMethod:URLString:parameters:error: instead.
     */

    override func requestWithMethod(method: String, URLString: String, parameters: AnyObject) -> NSMutableURLRequest {
        do {
            return try self.request(method: method, URLString: URLString, parameters: parameters)
        }
        catch let error {
        }
    }
    /**
     Creates an `NSMutableURLRequest` object with the specified HTTP method and URL string.
    
     If the HTTP method is `GET`, `HEAD`, or `DELETE`, the parameters will be used to construct a url-encoded query string that is appended to the request's URL. Otherwise, the parameters will be encoded according to the value of the `parameterEncoding` property, and set as the request body.
    
     @param method The HTTP method for the request, such as `GET`, `POST`, `PUT`, or `DELETE`. This parameter must not be `nil`.
     @param URLString The URL string used to create the request URL.
     @param parameters The parameters to be either set as a query string for `GET` requests, or the request HTTP body.
     @param error The error that occured while constructing the request.
    
     @return An `NSMutableURLRequest` object.
     */

    override func requestWithMethod(method: String, URLString: String, parameters: AnyObject, error: NSError?) -> NSMutableURLRequest {
        NSParameterAssert(method)
        NSParameterAssert(URLString)
        var url = NSURL(string: URLString)!
        NSParameterAssert(url)
        var mutableRequest = NSMutableURLRequest(URL: url)!
        mutableRequest.HTTPMethod = method
        for keyPath: String in AFHTTPRequestSerializerObservedKeyPaths() {
            if self.mutableObservedChangedKeyPaths.contains(keyPath) {
                mutableRequest[keyPath] = self.valueForKeyPath(keyPath)
            }
        }
        do {
            mutableRequest = try self.requestBySerializingRequest(mutableRequest, withParameters: parameters)
        }
        catch let error {
        }
        return mutableRequest
    }
    /**
     @deprecated This method has been deprecated. Use -multipartFormRequestWithMethod:URLString:parameters:constructingBodyWithBlock:error: instead.
     */

    func multipartFormRequestWithMethod(method: String, URLString: String, parameters: [NSObject : AnyObject], constructingBodyWithBlock block: (formData: AFMultipartFormData) -> Void) -> NSMutableURLRequest {
        do {
            return try self.multipartFormRequestWithMethod(method, URLString: URLString, parameters: parameters, constructingBodyWithBlock: block)
        }
        catch let error {
        }
    }
    /**
     Creates an `NSMutableURLRequest` object with the specified HTTP method and URLString, and constructs a `multipart/form-data` HTTP body, using the specified parameters and multipart form data block. See http://www.w3.org/TR/html4/interact/forms.html#h-17.13.4.2
    
     Multipart form requests are automatically streamed, reading files directly from disk along with in-memory data in a single HTTP body. The resulting `NSMutableURLRequest` object has an `HTTPBodyStream` property, so refrain from setting `HTTPBodyStream` or `HTTPBody` on this request object, as it will clear out the multipart form body stream.
    
     @param method The HTTP method for the request. This parameter must not be `GET` or `HEAD`, or `nil`.
     @param URLString The URL string used to create the request URL.
     @param parameters The parameters to be encoded and set in the request HTTP body.
     @param block A block that takes a single argument and appends data to the HTTP body. The block argument is an object adopting the `AFMultipartFormData` protocol.
     @param error The error that occured while constructing the request.
    
     @return An `NSMutableURLRequest` object
     */

    func multipartFormRequestWithMethod(method: String, URLString: String, parameters: [NSObject : AnyObject], constructingBodyWithBlock block: (formData: AFMultipartFormData) -> Void, error: NSError?) -> NSMutableURLRequest {
        NSParameterAssert(method)
        NSParameterAssert(!(method == "GET") && !(method == "HEAD"))
        var mutableRequest = try! self.request(method: method, URLString: URLString, parameters: nil)
        var formData = AFStreamingMultipartFormData(URLRequest: mutableRequest, stringEncoding: NSUTF8StringEncoding)
        if parameters {
            for pair: AFQueryStringPair in AFQueryStringPairsFromDictionary(parameters) {
                var data: NSData? = nil
                if (pair.value is NSData) {
                    data = pair.value
                }
                else if pair.value.isEqual(NSNull()) {
                    data = NSData.data
                }
                else {
                    data = pair.value.description.dataUsingEncoding(self.stringEncoding)!
                }

                if data != nil {
                    formData.appendPartWithFormData(data, name: pair.field.description)
                }
            }
        }
        if block {
            block(formData)
        }
        return formData.requestByFinalizingMultipartFormData()
    }
    /**
     Creates an `NSMutableURLRequest` by removing the `HTTPBodyStream` from a request, and asynchronously writing its contents into the specified file, invoking the completion handler when finished.
     
     @param request The multipart form request.
     @param fileURL The file URL to write multipart form contents to.
     @param handler A handler block to execute.
     
     @discussion There is a bug in `NSURLSessionTask` that causes requests to not send a `Content-Length` header when streaming contents from an HTTP body, which is notably problematic when interacting with the Amazon S3 webservice. As a workaround, this method takes a request constructed with `multipartFormRequestWithMethod:URLString:parameters:constructingBodyWithBlock:error:`, or any other request with an `HTTPBodyStream`, writes the contents to the specified file and returns a copy of the original request with the `HTTPBodyStream` property set to `nil`. From here, the file can either be passed to `AFURLSessionManager -uploadTaskWithRequest:fromFile:progress:completionHandler:`, or have its contents read into an `NSData` that's assigned to the `HTTPBody` property of the request.
    
     @see https://github.com/AFNetworking/AFNetworking/issues/1398
     */

    override func requestWithMultipartFormRequest(request: NSURLRequest, writingStreamContentsToFile fileURL: NSURL, completionHandler handler: (error: NSError) -> Void) -> NSMutableURLRequest {
        if !request.HTTPBodyStream! {
            return request
        }
        NSParameterAssert(fileURL.isFileURL())
        var inputStream = request.HTTPBodyStream!
        var outputStream = NSOutputStream(URL: fileURL, append: false)!
        var error: NSError? = nil
        dispatch_async(dispatch_get_global_queue(DISPATCH_QUEUE_PRIORITY_DEFAULT, 0), {() -> Void in
            inputStream.scheduleInRunLoop(NSRunLoop.currentRunLoop(), forMode: NSDefaultRunLoopMode)
            outputStream.scheduleInRunLoop(NSRunLoop.currentRunLoop(), forMode: NSDefaultRunLoopMode)
            inputStream.open()
            outputStream.open()
            while inputStream.hasBytesAvailable && outputStream.hasSpaceAvailable {
                var buffer = [UInt8](count: 1024, repeatedValue: 0)
                var bytesRead = inputStream.read(buffer, maxLength: 1024)
                if inputStream.streamEcharacterAtIndexrror! || bytesRead < 0 {
                    error = inputStream.streamEcharacterAtIndexrror!
                }
                var bytesWritten = outputStream.write(buffer, maxLength: Int(bytesRead))
                if outputStream.streamEcharacterAtIndexrror! || bytesWritten < 0 {
                    error = outputStream.streamEcharacterAtIndexrror!
                }
                if bytesRead == 0 && bytesWritten == 0 {

                }
            }
            outputStream.close()
            inputStream.close()
            if handler {
                dispatch_async(dispatch_get_main_queue(), {() -> Void in
                    handler(error)
                })
            }
        })
        var mutableRequest = request
        mutableRequest.HTTPBodyStream! = nil
        return mutableRequest
    }


    override init() {
        super.init()
        if !self {
            return nil
        }
        self.stringEncoding = NSUTF8StringEncoding
        self.mutableHTTPRequestHeaders = [NSObject : AnyObject]()
            // Accept-Language HTTP Header; see http://www.w3.org/Protocols/rfc2616/rfc2616-sec14.html#sec14.4
        var acceptLanguagesComponents = [AnyObject]()
        (NSLocale.preferredLanguages() as NSArray).enumerateObjectsUsingBlock({(obj: AnyObject, idx: Int, stop: Bool) -> Void in
            var q: Float = 1.0 - (idx * 0.1)
            acceptLanguagesComponents.append("\(obj);q=%0.1g")
            stop = q <= 0.5
        })
        self.setValue((acceptLanguagesComponents as NSArray).componentsJoinedByString(", "), forHTTPHeaderField: "Accept-Language")
        var userAgent: String? = nil
//#pragma clang diagnostic push
//#pragma clang diagnostic ignored "-Wgnu"
#if defined(__IPHONE_OS_VERSION_MIN_REQUIRED)
        // User-Agent Header; see http://www.w3.org/Protocols/rfc2616/rfc2616-sec14.html#sec14.43
        userAgent = "\(NSBundle.mainBundle().infoDictionary![(kCFBundleExecutableKey as! String)] as! String) ?? (NSBundle.mainBundle().infoDictionary![(kCFBundleIdentifierKey as! String)] as! String)/\(CFBundleGetValueForInfoDictionaryKey(CFBundleGetMainBundle(), kCFBundleVersionKey) as! AnyObject) ?? (NSBundle.mainBundle().infoDictionary![(kCFBundleVersionKey as! String)] as! String) (\(UIDevice.currentDevice().model); iOS \(UIDevice.currentDevice().systemVersion); Scale/%0.2f)"
#elseif defined(__MAC_OS_X_VERSION_MIN_REQUIRED)
        userAgent = "\(NSBundle.mainBundle().infoDictionary![(kCFBundleExecutableKey as! String)] as! String) ?? (NSBundle.mainBundle().infoDictionary![(kCFBundleIdentifierKey as! String)] as! String)/\(NSBundle.mainBundle().infoDictionary!["CFBundleShortVersionString"] as! String) ?? (NSBundle.mainBundle().infoDictionary![(kCFBundleVersionKey as! String)] as! String) (Mac OS X \(NSProcessInfo.processInfo().operatingSystemVersionString))"
#endif
//#pragma clang diagnostic pop
        if userAgent != nil {
            if !userAgent!.canBeConvertedToEncoding(NSASCIIStringEncoding) {
                var mutableUserAgent = userAgent!
                if CFStringTransform(((mutableUserAgent) as! CFMutableStringRef), nil, ("Any-Latin; Latin-ASCII; [:^ASCII:] Remove" as! CFString), false) {
                    userAgent = mutableUserAgent
                }
            }
            self.setValue(userAgent, forHTTPHeaderField: "User-Agent")
        }
        // HTTP Method Definitions; see http://www.w3.org/Protocols/rfc2616/rfc2616-sec9.html
        self.HTTPMethodsEncodingParametersInURI = Set<NSObject>(["GET", "HEAD", "DELETE"])
        self.mutableObservedChangedKeyPaths = Set<NSObject>()
        for keyPath: String in AFHTTPRequestSerializerObservedKeyPaths() {
            self.addObserver(self, forKeyPath: keyPath, options: .New, context: AFHTTPRequestSerializerObserverContext)
        }
    }

    deinit {
        for keyPath: String in AFHTTPRequestSerializerObservedKeyPaths() {
            self.removeObserver(self, forKeyPath: keyPath, context: AFHTTPRequestSerializerObserverContext)
        }
    }
// MARK: -

    func HTTPRequestHeaders() -> [NSObject : AnyObject] {
        return self.mutableHTTPRequestHeaders
    }
// MARK: -
// MARK: -
// MARK: - AFURLRequestSerialization

    func requestBySerializingRequest(request: NSURLRequest, withParameters parameters: AnyObject, error: NSError?) -> NSURLRequest {
        NSParameterAssert(request)
        var mutableRequest = request
        self.HTTPRequestHeaders.enumerateKeysAndObjectsUsingBlock({(field: AnyObject, value: AnyObject, stop: Bool) -> Void in
            if !request.valueForHTTPHeaderField(field)! {
                mutableRequest.setValue(value, forHTTPHeaderField: field)
            }
        })
        if parameters {
            var query: String? = nil
            if self.queryStringSerialization {
                var serializationError: NSError?
                query = self.queryStringSerialization(request, parameters, serializationError)
                if serializationError != nil {
                    if error != nil {
                        error = serializationError
                    }
                    return nil
                }
            }
            else {
                switch self.queryStringSerializationStyle {
                    case AFHTTPRequestQueryStringDefaultStyle:
                        query = AFQueryStringFromParametersWithEncoding(parameters, self.stringEncoding)
                }
            }
            if self.HTTPMethodsEncodingParametersInURI.contains(request.HTTPMethod!.uppercaseString) {
                mutableRequest.URL! = NSURL(string: mutableRequest.URL!.absoluteString.stringByAppendingFormat(mutableRequest.URL!.query! ? "&%@" : "?%@", query))!
            }
            else {
                if !mutableRequest.valueForHTTPHeaderField("Content-Type")! {
                    mutableRequest.setValue("application/x-www-form-urlencoded", forHTTPHeaderField: "Content-Type")
                }
                mutableRequest.HTTPBody = query!.dataUsingEncoding(self.stringEncoding)!
            }
        }
        return mutableRequest
    }
// MARK: - NSKeyValueObserving

    override func observeValueForKeyPath(keyPath: String, ofObject object: __unused id, change: [NSObject : AnyObject], context: UnsafeMutablePointer<Void>) {
        if context == AFHTTPRequestSerializerObserverContext {
            if change[.NewKey].isEqual(NSNull()) {
                self.mutableObservedChangedKeyPaths.removeAtIndex(self.mutableObservedChangedKeyPaths.indexOf(keyPath)!)
            }
            else {
                self.mutableObservedChangedKeyPaths.append(keyPath)
            }
        }
    }
// MARK: - NSSecureCoding

    class func supportsSecureCoding() -> Bool {
        return true
    }

    convenience required init?(coder decoder: NSCoder) {
        self.init()
        if !self {
            return nil
        }
        self.mutableHTTPRequestHeaders = decoder.decodeObjectOfClass([NSObject : AnyObject].self, forKey: NSStringFromSelector(#selector(self.mutableHTTPRequestHeaders)))!
        self.queryStringSerializationStyle = CUnsignedInt(decoder.decodeObjectOfClass(Int.self, forKey: NSStringFromSelector(#selector(self.queryStringSerializationStyle)))!)
    }

    override func encodeWithCoder(coder: NSCoder) {
        coder.encodeObject(self.mutableHTTPRequestHeaders, forKey: NSStringFromSelector(#selector(self.mutableHTTPRequestHeaders)))
        coder.encodeInteger(self.queryStringSerializationStyle, forKey: NSStringFromSelector(#selector(self.queryStringSerializationStyle)))
    }
// MARK: - NSCopying

    override func copyWithZone(zone: NSZone) -> AnyObject {
        var serializer = self.self.allocWithZone(zone)()
        serializer.mutableHTTPRequestHeaders = self.mutableHTTPRequestHeaders.mutableCopyWithZone(zone)!
        serializer.queryStringSerializationStyle = self.queryStringSerializationStyle
        serializer.queryStringSerialization = self.queryStringSerialization
        return serializer
    }

    var mutableObservedChangedKeyPaths = Set<NSObject>()
    var mutableHTTPRequestHeaders = [NSObject : AnyObject]()
    var queryStringSerializationStyle = AFHTTPRequestQueryStringSerializationStyle()
    var queryStringSerialization = AFQueryStringSerializationBlock()
}
// MARK: -
/**
 The `AFMultipartFormData` protocol defines the methods supported by the parameter in the block argument of `AFHTTPRequestSerializer -multipartFormRequestWithMethod:URLString:parameters:constructingBodyWithBlock:`.
 */
protocol AFMultipartFormData: class {
    /**
     Appends the HTTP header `Content-Disposition: file; filename=#{generated filename}; name=#{name}"` and `Content-Type: #{generated mimeType}`, followed by the encoded file data and the multipart form boundary.
    
     The filename and MIME type for this data in the form will be automatically generated, using the last path component of the `fileURL` and system associated MIME type for the `fileURL` extension, respectively.
    
     @param fileURL The URL corresponding to the file whose content will be appended to the form. This parameter must not be `nil`.
     @param name The name to be associated with the specified data. This parameter must not be `nil`.
     @param error If an error occurs, upon return contains an `NSError` object that describes the problem.
    
     @return `YES` if the file data was successfully appended, otherwise `NO`.
     */
    func appendPartWithFileURL(fileURL: NSURL, name: String, error: NSError?) -> Bool
    /**
     Appends the HTTP header `Content-Disposition: file; filename=#{filename}; name=#{name}"` and `Content-Type: #{mimeType}`, followed by the encoded file data and the multipart form boundary.
    
     @param fileURL The URL corresponding to the file whose content will be appended to the form. This parameter must not be `nil`.
     @param name The name to be associated with the specified data. This parameter must not be `nil`.
     @param fileName The file name to be used in the `Content-Disposition` header. This parameter must not be `nil`.
     @param mimeType The declared MIME type of the file data. This parameter must not be `nil`.
     @param error If an error occurs, upon return contains an `NSError` object that describes the problem.
    
     @return `YES` if the file data was successfully appended otherwise `NO`.
     */

    func appendPartWithFileURL(fileURL: NSURL, name: String, fileName: String, mimeType: String, error: NSError?) -> Bool
    /**
     Appends the HTTP header `Content-Disposition: file; filename=#{filename}; name=#{name}"` and `Content-Type: #{mimeType}`, followed by the data from the input stream and the multipart form boundary.
    
     @param inputStream The input stream to be appended to the form data
     @param name The name to be associated with the specified input stream. This parameter must not be `nil`.
     @param fileName The filename to be associated with the specified input stream. This parameter must not be `nil`.
     @param length The length of the specified input stream in bytes.
     @param mimeType The MIME type of the specified data. (For example, the MIME type for a JPEG image is image/jpeg.) For a list of valid MIME types, see http://www.iana.org/assignments/media-types/. This parameter must not be `nil`.
     */

    func appendPartWithInputStream(inputStream: NSInputStream, name: String, fileName: String, length: Int64, mimeType: String)
    /**
     Appends the HTTP header `Content-Disposition: file; filename=#{filename}; name=#{name}"` and `Content-Type: #{mimeType}`, followed by the encoded file data and the multipart form boundary.
    
     @param data The data to be encoded and appended to the form data.
     @param name The name to be associated with the specified data. This parameter must not be `nil`.
     @param fileName The filename to be associated with the specified data. This parameter must not be `nil`.
     @param mimeType The MIME type of the specified data. (For example, the MIME type for a JPEG image is image/jpeg.) For a list of valid MIME types, see http://www.iana.org/assignments/media-types/. This parameter must not be `nil`.
     */

    func appendPartWithFileData(data: NSData, name: String, fileName: String, mimeType: String)
    /**
     Appends the HTTP headers `Content-Disposition: form-data; name=#{name}"`, followed by the encoded data and the multipart form boundary.
    
     @param data The data to be encoded and appended to the form data.
     @param name The name to be associated with the specified data. This parameter must not be `nil`.
     */

    func appendPartWithFormData(data: NSData, name: String)
    /**
     Appends HTTP headers, followed by the encoded data and the multipart form boundary.
    
     @param headers The HTTP headers to be appended to the form data.
     @param body The data to be encoded and appended to the form data. This parameter must not be `nil`.
     */

    func appendPartWithHeaders(headers: [NSObject : AnyObject], body: NSData)
    /**
     Throttles request bandwidth by limiting the packet size and adding a delay for each chunk read from the upload stream.
    
     When uploading over a 3G or EDGE connection, requests may fail with "request body stream exhausted". Setting a maximum packet size and delay according to the recommended values (`kAFUploadStream3GSuggestedPacketSize` and `kAFUploadStream3GSuggestedDelay`) lowers the risk of the input stream exceeding its allocated bandwidth. Unfortunately, there is no definite way to distinguish between a 3G, EDGE, or LTE connection over `NSURLConnection`. As such, it is not recommended that you throttle bandwidth based solely on network reachability. Instead, you should consider checking for the "request body stream exhausted" in a failure block, and then retrying the request with throttled bandwidth.
    
     @param numberOfBytes Maximum packet size, in number of bytes. The default packet size for an input stream is 16kb.
     @param delay Duration of delay each time a packet is read. By default, no delay is set.
     */

    func throttleBandwidthWithPacketSize(numberOfBytes: Int, delay: NSTimeInterval)
}
// MARK: -
class AFJSONRequestSerializer: AFHTTPRequestSerializer {
    /**
     Options for writing the request JSON data from Foundation objects. For possible values, see the `NSJSONSerialization` documentation section "NSJSONWritingOptions". `0` by default.
     */
    var writingOptions = NSJSONWritingOptions()
    /**
     Creates and returns a JSON serializer with specified reading and writing options.
    
     @param writingOptions The specified JSON writing options.
     */

    class func serializerWithWritingOptions(writingOptions: NSJSONWritingOptions) -> Self {
        var serializer = self.init()
        serializer.writingOptions = writingOptions
        return serializer
    }


    class func serializer() -> Self {
        return self.serializerWithWritingOptions(0)
    }
// MARK: - AFURLRequestSerialization

    func requestBySerializingRequest(request: NSURLRequest, withParameters parameters: AnyObject, error: NSError?) -> NSURLRequest {
        NSParameterAssert(request)
        if self.HTTPMethodsEncodingParametersInURI.contains(request.HTTPMethod!.uppercaseString) {
            do {
                return try super.requestBySerializingRequest(request, withParameters: parameters)
            }
            catch let error {
            }
        }
        var mutableRequest = request
        self.HTTPRequestHeaders.enumerateKeysAndObjectsUsingBlock({(field: AnyObject, value: AnyObject, stop: Bool) -> Void in
            if !request.valueForHTTPHeaderField(field)! {
                mutableRequest.setValue(value, forHTTPHeaderField: field)
            }
        })
        if parameters {
            if !mutableRequest.valueForHTTPHeaderField("Content-Type")! {
                var charset = (CFStringConvertEncodingToIANACharSetName(CFStringConvertNSStringEncodingToEncoding(NSUTF8StringEncoding)) as! String)
                mutableRequest.setValue("application/json; charset=\(charset)", forHTTPHeaderField: "Content-Type")
            }
            do {
                mutableRequest.HTTPBody = try NSJSONSerialization.dataWithJSONObject(parameters, options: self.writingOptions)!
            }
            catch let error {
            }
        }
        return mutableRequest
    }
// MARK: - NSSecureCoding

    required init?(coder decoder: NSCoder) {
        super.init(coder: decoder)
        if !self {
            return nil
        }
        self.writingOptions = CUnsignedInt(decoder.decodeObjectOfClass(Int.self, forKey: NSStringFromSelector(#selector(self.writingOptions)))!)
    }

    override func encodeWithCoder(coder: NSCoder) {
        super.encodeWithCoder(coder)
        coder.encodeInteger(self.writingOptions, forKey: NSStringFromSelector(#selector(self.writingOptions)))
    }
// MARK: - NSCopying

    override func copyWithZone(zone: NSZone) -> AnyObject {
        var serializer = super.copyWithZone(zone)
        serializer.writingOptions = self.writingOptions
        return serializer
    }
}
class AFPropertyListRequestSerializer: AFHTTPRequestSerializer {
    /**
     The property list format. Possible values are described in "NSPropertyListFormat".
     */
    var format = NSPropertyListFormat()
    /**
     @warning The `writeOptions` property is currently unused.
     */
    var writeOptions = NSPropertyListWriteOptions()
    /**
     Creates and returns a property list serializer with a specified format, read options, and write options.
    
     @param format The property list format.
     @param writeOptions The property list write options.
     
     @warning The `writeOptions` property is currently unused.
     */

    class func serializerWithFormat(format: NSPropertyListFormat, writeOptions: NSPropertyListWriteOptions) -> Self {
        var serializer = self.init()
        serializer.format = format
        serializer.writeOptions = writeOptions
        return serializer
    }


    class func serializer() -> Self {
        return self.serializerWithFormat(NSPropertyListXMLFormat_v1_0, writeOptions: 0)
    }
// MARK: - AFURLRequestSerializer

    func requestBySerializingRequest(request: NSURLRequest, withParameters parameters: AnyObject, error: NSError?) -> NSURLRequest {
        NSParameterAssert(request)
        if self.HTTPMethodsEncodingParametersInURI.contains(request.HTTPMethod!.uppercaseString) {
            do {
                return try super.requestBySerializingRequest(request, withParameters: parameters)
            }
            catch let error {
            }
        }
        var mutableRequest = request
        self.HTTPRequestHeaders.enumerateKeysAndObjectsUsingBlock({(field: AnyObject, value: AnyObject, stop: Bool) -> Void in
            if !request.valueForHTTPHeaderField(field)! {
                mutableRequest.setValue(value, forHTTPHeaderField: field)
            }
        })
        if parameters {
            if !mutableRequest.valueForHTTPHeaderField("Content-Type")! {
                var charset = (CFStringConvertEncodingToIANACharSetName(CFStringConvertNSStringEncodingToEncoding(NSUTF8StringEncoding)) as! String)
                mutableRequest.setValue("application/x-plist; charset=\(charset)", forHTTPHeaderField: "Content-Type")
            }
            do {
                mutableRequest.HTTPBody = try NSPropertyListSerialization.dataWithPropertyList(parameters, format: self.format, options: self.writeOptions)!
            }
            catch let error {
            }
        }
        return mutableRequest
    }
// MARK: - NSSecureCoding

    required init?(coder decoder: NSCoder) {
        super.init(coder: decoder)
        if !self {
            return nil
        }
        self.format = CUnsignedInt(decoder.decodeObjectOfClass(Int.self, forKey: NSStringFromSelector(#selector(self.format)))!)
        self.writeOptions = CUnsignedInt(decoder.decodeObjectOfClass(Int.self, forKey: NSStringFromSelector(#selector(self.writeOptions)))!)
    }

    override func encodeWithCoder(coder: NSCoder) {
        super.encodeWithCoder(coder)
        coder.encodeInteger(self.format, forKey: NSStringFromSelector(#selector(self.format)))
        coder.encodeObject((self.writeOptions), forKey: NSStringFromSelector(#selector(self.writeOptions)))
    }
// MARK: - NSCopying

    override func copyWithZone(zone: NSZone) -> AnyObject {
        var serializer = super.copyWithZone(zone)
        serializer.format = self.format
        serializer.writeOptions = self.writeOptions
        return serializer
    }
}
///----------------
/// @name Constants
///----------------
/**
 ## Error Domains

 The following error domain is predefined.

 - `NSString * const AFURLRequestSerializationErrorDomain`

 ### Constants

 `AFURLRequestSerializationErrorDomain`
 AFURLRequestSerializer errors. Error codes for `AFURLRequestSerializationErrorDomain` correspond to codes in `NSURLErrorDomain`.
 */
let AFURLRequestSerializationErrorDomain = ""

/**
 ## User info dictionary keys

 These keys may exist in the user info dictionary, in addition to those defined for NSError.

 - `NSString * const AFNetworkingOperationFailingURLResponseErrorKey`

 ### Constants

 `AFNetworkingOperationFailingURLRequestErrorKey`
 The corresponding value is an `NSURLRequest` containing the request of the operation associated with an error. This key is only present in the `AFURLRequestSerializationErrorDomain`.
 */
let AFNetworkingOperationFailingURLRequestErrorKey = ""

/**
 ## Throttling Bandwidth for HTTP Request Input Streams

 @see -throttleBandwidthWithPacketSize:delay:

 ### Constants

 `kAFUploadStream3GSuggestedPacketSize`
 Maximum packet size, in number of bytes. Equal to 16kb.

 `kAFUploadStream3GSuggestedDelay`
 Duration of delay each time a packet is read. Equal to 0.2 seconds.
 */
let kAFUploadStream3GSuggestedPacketSize = 0

let kAFUploadStream3GSuggestedDelay = NSTimeInterval()

// AFSerialization.h
//
// Copyright (c) 2013-2014 AFNetworking (http://afnetworking.com)
//
// Permission is hereby granted, free of charge, to any person obtaining a copy
// of this software and associated documentation files (the "Software"), to deal
// in the Software without restriction, including without limitation the rights
// to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
// copies of the Software, and to permit persons to whom the Software is
// furnished to do so, subject to the following conditions:
// 
// The above copyright notice and this permission notice shall be included in
// all copies or substantial portions of the Software.
// 
// THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
// IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
// FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
// AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
// LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
// OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
// THE SOFTWARE.
#if __IPHONE_OS_VERSION_MIN_REQUIRED
import MobileCoreServices
#else
import CoreServices
#endif
let AFURLRequestSerializationErrorDomain = "com.alamofire.error.serialization.request"

let AFNetworkingOperationFailingURLRequestErrorKey = "com.alamofire.serialization.request.error.response"

typealias AFQueryStringSerializationBlock = (request: NSURLRequest, parameters: [NSObject : AnyObject], error: NSError) -> String
func AFBase64EncodedStringFromString(string: String) -> String {
    var data = NSData(bytes: string.UTF8String, length: string.lengthOfBytesUsingEncoding(NSUTF8StringEncoding))
    var length = data.length
    var mutableData = NSMutableData(length: ((length + 2) / 3) * 4)
    var input: UInt8 = UInt8(data.bytes)
    var output: UInt8 = UInt8(mutableData.mutableBytes)
    var i = 0
    while i < length {
        var value = 0
        for j in i..<(i + 3) {
            value <<= 8
            if j < length {
                value |= (0xFF & input[j])
            }
        }
        let kAFBase64EncodingTable = "ABCDEFGHIJKLMNOPQRSTUVWXYZabcdefghijklmnopqrstuvwxyz0123456789+/"
        var idx = (i / 3) * 4
        output[idx + 0] = kAFBase64EncodingTable[(value >> 18) & 0x3F]
        output[idx + 1] = kAFBase64EncodingTable[(value >> 12) & 0x3F]
        output[idx + 2] = (i + 1) < length ? kAFBase64EncodingTable[(value >> 6) & 0x3F] : "="
        output[idx + 3] = (i + 2) < length ? kAFBase64EncodingTable[(value >> 0) & 0x3F] : "="
        i += 3
    }
    return String(data: mutableData, encoding: NSASCIIStringEncoding)
}

let kAFCharactersToBeEscapedInQueryString = ":/?&=;+!@#$()',*"

func AFPercentEscapedQueryStringKeyFromStringWithEncoding(string: String, encoding: NSStringEncoding) -> String {
    let kAFCharactersToLeaveUnescapedInQueryStringPairKey = "[]."
    return (CFURLCreateStringByAddingPercentEscapes(kCFAllocatorDefault, (string as! CFString), (kAFCharactersToLeaveUnescapedInQueryStringPairKey as! CFString), (kAFCharactersToBeEscapedInQueryString as! CFString), CFStringConvertNSStringEncodingToEncoding(encoding)) as!  NSString)
}

func AFPercentEscapedQueryStringValueFromStringWithEncoding(string: String, encoding: NSStringEncoding) -> String {
    return (CFURLCreateStringByAddingPercentEscapes(kCFAllocatorDefault, (string as! CFString), nil, (kAFCharactersToBeEscapedInQueryString as! CFString), CFStringConvertNSStringEncodingToEncoding(encoding)) as!  NSString)
}

// MARK: -
class AFQueryStringPair: NSObject {
    var field: AnyObject!
    var value: AnyObject!

    override init(field: AnyObject, value: AnyObject) {
        super.init()
        if !self {
            return nil
        }
        self.field = field
        self.value = value
    }

    func URLEncodedStringValueWithEncoding(stringEncoding: NSStringEncoding) -> String {
        if !self.value || self.value.isEqual(NSNull()) {
            return AFPercentEscapedQueryStringKeyFromStringWithEncoding(self.field.description, stringEncoding)
        }
        else {
            return "\(AFPercentEscapedQueryStringKeyFromStringWithEncoding(self.field.description, stringEncoding))=\(AFPercentEscapedQueryStringValueFromStringWithEncoding(self.value.description, stringEncoding))"
        }
    }
}
// MARK: -
var AFQueryStringPairsFromDictionary = [AnyObject]()

var AFQueryStringPairsFromKeyAndValue = [AnyObject]()

func AFQueryStringFromParametersWithEncoding(parameters: [NSObject : AnyObject], stringEncoding: NSStringEncoding) -> String {
    var mutablePairs = [AnyObject]()
    for pair: AFQueryStringPair in AFQueryStringPairsFromDictionary(parameters) {
        mutablePairs.append(pair.URLEncodedStringValueWithEncoding(stringEncoding))
    }
    return (mutablePairs as NSArray).componentsJoinedByString("&")
}

func AFQueryStringPairsFromDictionary(dictionary: [NSObject : AnyObject]) -> [AnyObject] {
    return AFQueryStringPairsFromKeyAndValue(nil, dictionary)
}

func AFQueryStringPairsFromKeyAndValue(key: String, value: AnyObject) -> [AnyObject] {
    var mutableQueryStringComponents = [AnyObject]()
    var sortDescriptor = NSSortDescriptor(key: "description", ascending: true, selector: #selector(self.compare))
    if (value is [NSObject : AnyObject]) {
        var dictionary = value
        // Sort dictionary keys to ensure consistent ordering in query string, which is important when deserializing potentially ambiguous sequences, such as an array of dictionaries
        for nestedKey: AnyObject in (dictionary.allKeys() as NSArray).sortedArrayUsingDescriptors([sortDescriptor]) {
            var nestedValue = (dictionary[nestedKey] as! String)
            if nestedValue {
                mutableQueryStringComponents += AFQueryStringPairsFromKeyAndValue((key ? "\(key)[\(nestedKey)]" : nestedKey), nestedValue)
            }
        }
    }
    else if (value is [AnyObject]) {
        var array = value
        for nestedValue: AnyObject in array {
            mutableQueryStringComponents += AFQueryStringPairsFromKeyAndValue("\(key)[]", nestedValue)
        }
    }
    else if (value is Set<NSObject>) {
        var set = value
        for obj: AnyObject in (set as NSArray).sortedArrayUsingDescriptors([sortDescriptor]) {
            mutableQueryStringComponents += AFQueryStringPairsFromKeyAndValue(key, obj)
        }
    }
    else {
        mutableQueryStringComponents.append(AFQueryStringPair(field: key, value: value))
    }

    return mutableQueryStringComponents
}

// MARK: -
class AFStreamingMultipartFormData: NSObject, AFMultipartFormData {
    override init(URLRequest urlRequest: NSMutableURLRequest, stringEncoding encoding: NSStringEncoding) {
        super.init()
        if !self {
            return nil
        }
        self.request = urlRequest
        self.stringEncoding = encoding
        self.boundary = AFCreateMultipartFormBoundary()
        self.bodyStream = AFMultipartBodyStream(stringEncoding: encoding)
    }

    func requestByFinalizingMultipartFormData() -> NSMutableURLRequest {
        if self.bodyStream.isEmpty() {
            return self.request
        }
        // Reset the initial and final boundaries to ensure correct Content-Length
        self.bodyStream.setInitialAndFinalBoundaries()
        self.request.HTTPBodyStream = self.bodyStream
        self.request.setValue("multipart/form-data; boundary=\(self.boundary)", forHTTPHeaderField: "Content-Type")
        self.request.setValue("\(self.bodyStream.contentLength)", forHTTPHeaderField: "Content-Length")
        return self.request
    }


    func appendPartWithFileURL(fileURL: NSURL, name: String, error: NSError?) -> Bool {
        NSParameterAssert(fileURL)
        NSParameterAssert(name)
        var fileName = fileURL.lastPathComponent
        var mimeType = AFContentTypeForPathExtension(fileURL.pathExtension)
        do {
            return try self.appendPartWithFileURL(fileURL, name: name, fileName: fileName, mimeType: mimeType)
        }
        catch let error {
        }
    }

    func appendPartWithFileURL(fileURL: NSURL, name: String, fileName: String, mimeType: String, error: NSError?) -> Bool {
        NSParameterAssert(fileURL)
        NSParameterAssert(name)
        NSParameterAssert(fileName)
        NSParameterAssert(mimeType)
        if !fileURL.isFileURL() {
            var userInfo = [NSLocalizedFailureReasonErrorKey: NSLocalizedStringFromTable("Expected URL to be a file URL", "AFNetworking")]
            if error != nil {
                error = NSError(domain: AFURLRequestSerializationErrorDomain, code: NSURLErrorBadURL, userInfo: userInfo)
            }
            return false
        }
        else if fileURL.checkResourceIsReachableAndReturnError(error) == false {
            var userInfo = [NSLocalizedFailureReasonErrorKey: NSLocalizedStringFromTable("File URL not reachable.", "AFNetworking")]
            if error != nil {
                error = NSError(domain: AFURLRequestSerializationErrorDomain, code: NSURLErrorBadURL, userInfo: userInfo)
            }
            return false
        }

        var fileAttributes = try! NSFileManager.defaultManager().attributesOfItemAtPath(fileURL.path!)!
        if fileAttributes.isEmpty {
            return false
        }
        var mutableHeaders = [NSObject : AnyObject]()
        mutableHeaders["Content-Disposition"] = "form-data; name=\"\(name)\"; filename=\"\(fileName)\""
        mutableHeaders["Content-Type"] = mimeType
        var bodyPart = AFHTTPBodyPart()
        bodyPart.stringEncoding = self.stringEncoding
        bodyPart.headers = mutableHeaders
        bodyPart.boundary = self.boundary
        bodyPart.body = fileURL
        bodyPart.bodyContentLength = CUnsignedLongLong((fileAttributes[NSFileSize] as! String))
        self.bodyStream.appendHTTPBodyPart(bodyPart)
        return true
    }

    func appendPartWithInputStream(inputStream: NSInputStream, name: String, fileName: String, length: Int64, mimeType: String) {
        NSParameterAssert(name)
        NSParameterAssert(fileName)
        NSParameterAssert(mimeType)
        var mutableHeaders = [NSObject : AnyObject]()
        mutableHeaders["Content-Disposition"] = "form-data; name=\"\(name)\"; filename=\"\(fileName)\""
        mutableHeaders["Content-Type"] = mimeType
        var bodyPart = AFHTTPBodyPart()
        bodyPart.stringEncoding = self.stringEncoding
        bodyPart.headers = mutableHeaders
        bodyPart.boundary = self.boundary
        bodyPart.body = inputStream
        bodyPart.bodyContentLength = UInt64(length)
        self.bodyStream.appendHTTPBodyPart(bodyPart)
    }

    func appendPartWithFileData(data: NSData, name: String, fileName: String, mimeType: String) {
        NSParameterAssert(name)
        NSParameterAssert(fileName)
        NSParameterAssert(mimeType)
        var mutableHeaders = [NSObject : AnyObject]()
        mutableHeaders["Content-Disposition"] = "form-data; name=\"\(name)\"; filename=\"\(fileName)\""
        mutableHeaders["Content-Type"] = mimeType
        self.appendPartWithHeaders(mutableHeaders, body: data)
    }

    func appendPartWithFormData(data: NSData, name: String) {
        NSParameterAssert(name)
        var mutableHeaders = [NSObject : AnyObject]()
        mutableHeaders["Content-Disposition"] = "form-data; name=\"\(name)\""
        self.appendPartWithHeaders(mutableHeaders, body: data)
    }

    func appendPartWithHeaders(headers: [NSObject : AnyObject], body: NSData) {
        NSParameterAssert(body)
        var bodyPart = AFHTTPBodyPart()
        bodyPart.stringEncoding = self.stringEncoding
        bodyPart.headers = headers
        bodyPart.boundary = self.boundary
        bodyPart.bodyContentLength = body.length
        bodyPart.body = body
        self.bodyStream.appendHTTPBodyPart(bodyPart)
    }

    func throttleBandwidthWithPacketSize(numberOfBytes: Int, delay: NSTimeInterval) {
        self.bodyStream.numberOfBytesInPacket = numberOfBytes
        self.bodyStream.delay = delay
    }

    var request: NSMutableURLRequest!
    var stringEncoding = NSStringEncoding()
    var boundary = ""
    var bodyStream: AFMultipartBodyStream!
}
// MARK: -
func AFHTTPRequestSerializerObservedKeyPaths() -> [AnyObject] {
    var AFHTTPRequestSerializerObservedKeyPaths: [AnyObject]? = nil
    var onceToken: dispatch_once_t
    dispatch_once(onceToken, {() -> Void in
        self.AFHTTPRequestSerializerObservedKeyPaths = [NSStringFromSelector(#selector(self.allowsCellularAccess)), NSStringFromSelector(#selector(self.cachePolicy)), NSStringFromSelector(#selector(self.HTTPShouldHandleCookies)), NSStringFromSelector(#selector(self.HTTPShouldUsePipelining)), NSStringFromSelector(#selector(self.networkServiceType)), NSStringFromSelector(#selector(self.timeoutInterval))]
    })
    return AFHTTPRequestSerializerObservedKeyPaths
}

var AFHTTPRequestSerializerObserverContext = AFHTTPRequestSerializerObserverContext

// MARK: -
func AFCreateMultipartFormBoundary() -> String {
    return String(format: "Boundary+%08X%08X", arc4random(), arc4random())
}

let kAFMultipartFormCRLF = "\r\n"

func AFMultipartFormInitialBoundary(boundary: String) -> inline NSString {
    return "--\(boundary)\(kAFMultipartFormCRLF)"
}

func AFMultipartFormEncapsulationBoundary(boundary: String) -> inline NSString {
    return "\(kAFMultipartFormCRLF)--\(boundary)\(kAFMultipartFormCRLF)"
}

func AFMultipartFormFinalBoundary(boundary: String) -> inline NSString {
    return "\(kAFMultipartFormCRLF)--\(boundary)--\(kAFMultipartFormCRLF)"
}

func AFContentTypeForPathExtension(extension: String) -> inline NSString {
#if __UTTYPE__
    var UTI = (UTTypeCreatePreferredIdentifierForTag(kUTTagClassFilenameExtension, (extension as! CFString), nil) as!  NSString)
    var contentType = (UTTypeCopyPreferredTagWithClass((UTI as! CFString), kUTTagClassMIMEType) as!  NSString)
    if contentType == "" {
        return "application/octet-stream"
    }
    else {
        return contentType
    }
#else
//#pragma unused (extension)
    return "application/octet-stream"
#endif
}

let kAFUploadStream3GSuggestedPacketSize = 1024 * 16

let kAFUploadStream3GSuggestedDelay = 0.2

class AFHTTPBodyPart: NSObject {
    var stringEncoding = NSStringEncoding()
    var headers = [NSObject : AnyObject]()
    var boundary = ""
    var body: AnyObject!
    var bodyContentLength: UInt64 = 0
    var inputStream: NSInputStream!
    var hasInitialBoundary = false
    var hasFinalBoundary = false
    private(set) var bytesAvailable = false
    private(set) var contentLength: UInt64 = 0

    override func read(buffer: UInt8, maxLength length: Int) -> Int {
        var totalNumberOfBytesRead = 0
        if phase == AFEncapsulationBoundaryPhase {
            var encapsulationBoundaryData = (self.hasInitialBoundary() ? AFMultipartFormInitialBoundary(self.boundary) : AFMultipartFormEncapsulationBoundary(self.boundary)).dataUsingEncoding(self.stringEncoding)!
            totalNumberOfBytesRead += self.readData(encapsulationBoundaryData, intoBuffer: buffer[totalNumberOfBytesRead], maxLength: (length - Int(totalNumberOfBytesRead)))
        }
        if phase == AFHeaderPhase {
            var headersData = self.stringForHeaders().dataUsingEncoding(self.stringEncoding)!
            totalNumberOfBytesRead += self.readData(headersData, intoBuffer: buffer[totalNumberOfBytesRead], maxLength: (length - Int(totalNumberOfBytesRead)))
        }
        if phase == AFBodyPhase {
            var numberOfBytesRead = 0
            numberOfBytesRead = self.inputStream.read(buffer[totalNumberOfBytesRead], maxLength: (length - Int(totalNumberOfBytesRead)))
            if numberOfBytesRead == -1 {
                return -1
            }
            else {
                totalNumberOfBytesRead += numberOfBytesRead
                if self.inputStream.streamStatus >= .AtEnd {
                    self.transitionToNextPhase()
                }
            }
        }
        if phase == AFFinalBoundaryPhase {
            var closingBoundaryData = (self.hasFinalBoundary() ? AFMultipartFormFinalBoundary(self.boundary).dataUsingEncoding(self.stringEncoding)! : NSData.data)
            totalNumberOfBytesRead += self.readData(closingBoundaryData, intoBuffer: buffer[totalNumberOfBytesRead], maxLength: (length - Int(totalNumberOfBytesRead)))
        }
        return totalNumberOfBytesRead
    }


    override init() {
        super.init()
        if !self {
            return nil
        }
        self.transitionToNextPhase()
    }

    deinit {
        if inputStream != nil {
            inputStream.close()
            self.inputStream = nil
        }
    }

    override func inputStream() -> NSInputStream {
        if inputStream == nil {
            if (self.body is NSData) {
                self.inputStream = NSInputStream(data: self.body)
            }
            else if (self.body is NSURL) {
                self.inputStream = NSInputStream(URL: self.body)
            }
            else if (self.body is NSInputStream) {
                self.inputStream = self.body
            }
            else {
                self.inputStream = NSInputStream(data: NSData.data)
            }
        }
        return inputStream
    }

    func stringForHeaders() -> String {
        var headerString = String()
        for field: String in self.headers.allKeys() {
            headerString += "\(field): \(self.headers.valueForKey(field) as! String)\(kAFMultipartFormCRLF)"
        }
        headerString += kAFMultipartFormCRLF
        return String = headerString
    }

    override func contentLength() -> UInt64 {
        var length: UInt64 = 0
        var encapsulationBoundaryData = (self.hasInitialBoundary() ? AFMultipartFormInitialBoundary(self.boundary) : AFMultipartFormEncapsulationBoundary(self.boundary)).dataUsingEncoding(self.stringEncoding)!
        length += encapsulationBoundaryData.length
        var headersData = self.stringForHeaders().dataUsingEncoding(self.stringEncoding)!
        length += headersData.length
        length += bodyContentLength
        var closingBoundaryData = (self.hasFinalBoundary() ? AFMultipartFormFinalBoundary(self.boundary).dataUsingEncoding(self.stringEncoding)! : NSData.data)
        length += closingBoundaryData.length
        return length
    }

    override func hasBytesAvailable() -> Bool {
        // Allows `read:maxLength:` to be called again if `AFMultipartFormFinalBoundary` doesn't fit into the available buffer
        if phase == AFFinalBoundaryPhase {
            return true
        }
//#pragma clang diagnostic push
//#pragma clang diagnostic ignored "-Wcovered-switch-default"
        switch self.inputStream.streamStatus {
            case .NotOpen, .Opening, .Open, .Reading, .Writing:
                return true
            default:
                return false
        }

//#pragma clang diagnostic pop
    }

    func readData(data: NSData, intoBuffer buffer: UInt8, maxLength length: Int) -> Int {
//#pragma clang diagnostic push
//#pragma clang diagnostic ignored "-Wgnu"
        var range = NSMakeRange(Int(phaseReadOffset), min(data.length - (Int(phaseReadOffset)), length))
        data.getBytes(buffer, range: range)
//#pragma clang diagnostic pop
        self.phaseReadOffset += range.length
        if (Int(phaseReadOffset)) >= data.length {
            self.transitionToNextPhase()
        }
        return Int(range.length)
    }

    func transitionToNextPhase() -> Bool {
        if !NSThread.currentThread().isMainThread {
            self.performSelectorOnMainThread(#selector(self.transitionToNextPhase), withObject: nil, waitUntilDone: true)
            return true
        }
//#pragma clang diagnostic push
//#pragma clang diagnostic ignored "-Wcovered-switch-default"
        switch phase {
            case AFEncapsulationBoundaryPhase:
                self.phase = AFHeaderPhase
            case AFHeaderPhase:
                self.inputStream.scheduleInRunLoop(NSRunLoop.currentRunLoop(), forMode: NSRunLoopCommonModes)
                self.inputStream.open()
                self.phase = AFBodyPhase
            case AFBodyPhase:
                self.inputStream.close()
                self.phase = AFFinalBoundaryPhase
            default:
                self.phase = AFEncapsulationBoundaryPhase
        }

        self.phaseReadOffset = 0
//#pragma clang diagnostic pop
        return true
    }
// MARK: - NSCopying

    override func copyWithZone(zone: NSZone) -> AnyObject {
        var bodyPart = self.self.allocWithZone(zone)()
        bodyPart.stringEncoding = self.stringEncoding
        bodyPart.headers = self.headers
        bodyPart.bodyContentLength = self.bodyContentLength
        bodyPart.body = self.body
        bodyPart.boundary = self.boundary
        return bodyPart
    }
    var phase = AFHTTPBodyPartReadPhase()
    var inputStream: NSInputStream!
    var phaseReadOffset: UInt64 = 0


    func transitionToNextPhase() -> Bool {
    }

    func readData(data: NSData, intoBuffer buffer: UInt8, maxLength length: Int) -> Int {
    }
}
class AFMultipartBodyStream: NSInputStream, NSStreamDelegate {
    var numberOfBytesInPacket = 0
    var delay = NSTimeInterval()
    var inputStream: NSInputStream!
    private(set) var contentLength: UInt64 = 0
    private(set) var empty = false

    override init(stringEncoding encoding: NSStringEncoding) {
        super.init()
        if !self {
            return nil
        }
        self.stringEncoding = encoding
        self.HTTPBodyParts = [AnyObject]()
        self.numberOfBytesInPacket = NSIntegerMax
    }

    func setInitialAndFinalBoundaries() {
        if self.HTTPBodyParts.count > 0 {
            for bodyPart: AFHTTPBodyPart in self.HTTPBodyParts {
                bodyPart.hasInitialBoundary = false
                bodyPart.hasFinalBoundary = false
            }
            self.HTTPBodyParts[0].hasInitialBoundary = true
            self.HTTPBodyParts.last!.hasFinalBoundary = true
        }
    }

    func appendHTTPBodyPart(bodyPart: AFHTTPBodyPart) {
        self.HTTPBodyParts.append(bodyPart)
    }

//#pragma clang diagnostic push
//#pragma clang diagnostic ignored "-Wimplicit-atomic-properties"
#if (defined(__IPHONE_OS_VERSION_MAX_ALLOWED) && __IPHONE_OS_VERSION_MAX_ALLOWED >= 80000) || (defined(__MAC_OS_X_VERSION_MAX_ALLOWED) && __MAC_OS_X_VERSION_MAX_ALLOWED >= 1100)
#endif
//#pragma clang diagnostic pop

    func isEmpty() -> Bool {
        return self.HTTPBodyParts.count == 0
    }
// MARK: - NSInputStream

    override func read(buffer: UInt8, maxLength length: Int) -> Int {
        if self.streamStatus == .Closed {
            return 0
        }
        var totalNumberOfBytesRead = 0
//#pragma clang diagnostic push
//#pragma clang diagnostic ignored "-Wgnu"
        while Int(totalNumberOfBytesRead) < min(length, self.numberOfBytesInPacket) {
            if !self.currentHTTPBodyPart || !self.currentHTTPBodyPart.hasBytesAvailable {
                if !(self.currentHTTPBodyPart = self.HTTPBodyPartEnumerator.nextObject()!) {

                }
            }
            else {
                var maxLength = length - Int(totalNumberOfBytesRead)
                var numberOfBytesRead = self.currentHTTPBodyPart.read(buffer[totalNumberOfBytesRead], maxLength: maxLength)
                if numberOfBytesRead == -1 {
                    self.streamEcharacterAtIndexrror! = self.currentHTTPBodyPart.inputStream.streamEcharacterAtIndexrror!
                }
                else {
                    totalNumberOfBytesRead += numberOfBytesRead
                    if self.delay > 0.0 {
                        NSThread.sleepForTimeInterval(self.delay)
                    }
                }
            }
        }
//#pragma clang diagnostic pop
        return totalNumberOfBytesRead
    }

    override func getBuffer(buffer: __unused uint8_t, length len: __unused NSUInteger) -> Bool {
        return false
    }

    override func hasBytesAvailable() -> Bool {
        return self.streamStatus == .Open
    }
// MARK: - NSStream

    override func open() {
        if self.streamStatus == .Open {
            return
        }
        self.streamStatus = .Open
        self.setInitialAndFinalBoundaries()
        self.HTTPBodyPartEnumerator = (self.HTTPBodyParts as NSArray).objectEnumerator()
    }

    override func close() {
        self.streamStatus = .Closed
    }

    override func propertyForKey(key: __unused NSString) -> AnyObject {
        return nil
    }

    override func setProperty(property: __unused id, forKey key: __unused NSString) -> Bool {
        return false
    }

    override func scheduleInRunLoop(aRunLoop: __unused NSRunLoop, forMode mode: __unused NSString) {
    }

    override func removeFromRunLoop(aRunLoop: __unused NSRunLoop, forMode mode: __unused NSString) {
    }

    override func contentLength() -> UInt64 {
        var length: UInt64 = 0
        for bodyPart: AFHTTPBodyPart in self.HTTPBodyParts {
            length += bodyPart.contentLength
        }
        return length
    }
// MARK: - Undocumented CFReadStream Bridged Methods

    func _scheduleInCFRunLoop(aRunLoop: __unused CFRunLoopRef, forMode aMode: __unused CFStringRef) {
    }

    func _unscheduleFromCFRunLoop(aRunLoop: __unused CFRunLoopRef, forMode aMode: __unused CFStringRef) {
    }

    func _setCFClientFlags(inFlags: __unused CFOptionFlags, callback inCallback: __unused CFReadStreamClientCallBack, context inContext: __unused CFStreamClientContext) -> Bool {
        return false
    }
// MARK: - NSCopying

    override func copyWithZone(zone: NSZone) -> AnyObject {
        var bodyStreamCopy = self.self.allocWithZone(zone)(stringEncoding: self.stringEncoding)
        for bodyPart: AFHTTPBodyPart in self.HTTPBodyParts {
            bodyStreamCopy.appendHTTPBodyPart(bodyPart)
        }
        bodyStreamCopy.setInitialAndFinalBoundaries()
        return bodyStreamCopy
    }

    var stringEncoding = NSStringEncoding()
    var HTTPBodyParts = [AnyObject]()
    var HTTPBodyPartEnumerator: NSEnumerator!
    var currentHTTPBodyPart: AFHTTPBodyPart!
    var outputStream: NSOutputStream!
    var buffer: NSMutableData!
}
// MARK: -

// MARK: -

// MARK: -
enum AFHTTPBodyPartReadPhase : Int {
    case AFEncapsulationBoundaryPhase = 1
    case AFHeaderPhase = 2
    case AFBodyPhase = 3
    case AFFinalBoundaryPhase = 4
}

// MARK: -

// MARK: -