// AFURLConnectionOperation.h
//
// Copyright (c) 2013-2014 AFNetworking (http://afnetworking.com)
//
// Permission is hereby granted, free of charge, to any person obtaining a copy
// of this software and associated documentation files (the "Software"), to deal
// in the Software without restriction, including without limitation the rights
// to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
// copies of the Software, and to permit persons to whom the Software is
// furnished to do so, subject to the following conditions:
//
// The above copyright notice and this permission notice shall be included in
// all copies or substantial portions of the Software.
//
// THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
// IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
// FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
// AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
// LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
// OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
// THE SOFTWARE.
import Foundation
import Availability
/**
 `AFURLConnectionOperation` is a subclass of `NSOperation` that implements `NSURLConnection` delegate methods.

 ## Subclassing Notes

 This is the base class of all network request operations. You may wish to create your own subclass in order to implement additional `NSURLConnection` delegate methods (see "`NSURLConnection` Delegate Methods" below), or to provide additional properties and/or class constructors.

 If you are creating a subclass that communicates over the HTTP or HTTPS protocols, you may want to consider subclassing `AFHTTPRequestOperation` instead, as it supports specifying acceptable content types or status codes.

 ## NSURLConnection Delegate Methods

 `AFURLConnectionOperation` implements the following `NSURLConnection` delegate methods:

 - `connection:didReceiveResponse:`
 - `connection:didReceiveData:`
 - `connectionDidFinishLoading:`
 - `connection:didFailWithError:`
 - `connection:didSendBodyData:totalBytesWritten:totalBytesExpectedToWrite:`
 - `connection:willCacheResponse:`
 - `connectionShouldUseCredentialStorage:`
 - `connection:needNewBodyStream:` 
 - `connection:willSendRequestForAuthenticationChallenge:`

 If any of these methods are overridden in a subclass, they _must_ call the `super` implementation first.

 ## Callbacks and Completion Blocks

 The built-in `completionBlock` provided by `NSOperation` allows for custom behavior to be executed after the request finishes. It is a common pattern for class constructors in subclasses to take callback block parameters, and execute them conditionally in the body of its `completionBlock`. Make sure to handle cancelled operations appropriately when setting a `completionBlock` (i.e. returning early before parsing response data). See the implementation of any of the `AFHTTPRequestOperation` subclasses for an example of this.

 Subclasses are strongly discouraged from overriding `setCompletionBlock:`, as `AFURLConnectionOperation`'s implementation includes a workaround to mitigate retain cycles, and what Apple rather ominously refers to as ["The Deallocation Problem"](http://developer.apple.com/library/ios/#technotes/tn2109/).
 
 ## SSL Pinning
 
 Relying on the CA trust model to validate SSL certificates exposes your app to security vulnerabilities, such as man-in-the-middle attacks. For applications that connect to known servers, SSL certificate pinning provides an increased level of security, by checking server certificate validity against those specified in the app bundle.
 
 SSL with certificate pinning is strongly recommended for any application that transmits sensitive information to an external webservice.

 Connections will be validated on all matching certificates with a `.cer` extension in the bundle root.
 
 ## App Extensions
 
 When using AFNetworking in an App Extension, `#define AF_APP_EXTENSIONS` to avoid using unavailable APIs.

 ## NSCoding & NSCopying Conformance

 `AFURLConnectionOperation` conforms to the `NSCoding` and `NSCopying` protocols, allowing operations to be archived to disk, and copied in memory, respectively. However, because of the intrinsic limitations of capturing the exact state of an operation at a particular moment, there are some important caveats to keep in mind:

 ### NSCoding Caveats

 - Encoded operations do not include any block or stream properties. Be sure to set `completionBlock`, `outputStream`, and any callback blocks as necessary when using `-initWithCoder:` or `NSKeyedUnarchiver`.
 - Operations are paused on `encodeWithCoder:`. If the operation was encoded while paused or still executing, its archived state will return `YES` for `isReady`. Otherwise, the state of an operation when encoding will remain unchanged.

 ### NSCopying Caveats

 - `-copy` and `-copyWithZone:` return a new operation with the `NSURLRequest` of the original. So rather than an exact copy of the operation at that particular instant, the copying mechanism returns a completely new instance, which can be useful for retrying operations.
 - A copy of an operation will not include the `outputStream` of the original.
 - Operation copies do not include `completionBlock`, as it often strongly captures a reference to `self`, which would otherwise have the unintuitive side-effect of pointing to the _original_ operation when copied.
 */
class AFURLConnectionOperation: NSOperation, NSURLConnectionDelegate, NSURLConnectionDataDelegate, NSSecureCoding, NSCopying {
    ///-------------------------------
    /// @name Accessing Run Loop Modes
    ///-------------------------------
    /**
     The run loop modes in which the operation will run on the network thread. By default, this is a single-member set containing `NSRunLoopCommonModes`.
     */
    var runLoopModes = Set<NSObject>()
    ///-----------------------------------------
    /// @name Getting URL Connection Information
    ///-----------------------------------------
    /**
     The request used by the operation's connection.
     */
    private(set) var request: NSURLRequest!
    /**
     The last response received by the operation's connection.
     */
    private(set) var response: NSURLResponse!
    /**
     The error, if any, that occurred in the lifecycle of the request.
     */
    private(set) var error: NSError!
    ///----------------------------
    /// @name Getting Response Data
    ///----------------------------
    /**
     The data received during the request.
     */
    private(set) var responseData: NSData!
    /**
     The string representation of the response data.
     */
    var responseString: String {
        self.lock().lock()
            if !responseString && self.response && self.responseData {
                self.responseString = String(data: self.responseData, encoding: self.responseStringEncoding)
            }
            self.lock().unlock()
            return responseString
    }
    /**
     The string encoding of the response.
    
     If the response does not specify a valid string encoding, `responseStringEncoding` will return `NSUTF8StringEncoding`.
     */
    var responseStringEncoding: NSStringEncoding {
        self.lock().lock()
            if !responseStringEncoding && self.response {
                var stringEncoding = NSUTF8StringEncoding
                if self.response.textEncodingName! {
                    var IANAEncoding = CFStringConvertIANACharSetNameToEncoding((self.response.textEncodingName! as! CFString))
                    if IANAEncoding != kCFStringEncodingInvalidId {
                        stringEncoding = CFStringConvertEncodingToNSStringEncoding(IANAEncoding)
                    }
                }
                self.responseStringEncoding = stringEncoding
            }
            self.lock().unlock()
            return responseStringEncoding
    }
    ///-------------------------------
    /// @name Managing URL Credentials
    ///-------------------------------
    /**
     Whether the URL connection should consult the credential storage for authenticating the connection. `YES` by default.
    
     This is the value that is returned in the `NSURLConnectionDelegate` method `-connectionShouldUseCredentialStorage:`.
     */
    var shouldUseCredentialStorage = false
    /**
     The credential used for authentication challenges in `-connection:didReceiveAuthenticationChallenge:`.
    
     This will be overridden by any shared credentials that exist for the username or password of the request URL, if present.
     */
    var credential: NSURLCredential!
    ///-------------------------------
    /// @name Managing Security Policy
    ///-------------------------------
    /**
     The security policy used to evaluate server trust for secure connections.
     */
    var securityPolicy: AFSecurityPolicy!
    ///------------------------
    /// @name Accessing Streams
    ///------------------------
    /**
     The input stream used to read data to be sent during the request.
    
     This property acts as a proxy to the `HTTPBodyStream` property of `request`.
     */
    var inputStream: NSInputStream? {
        get {
            return self.request.HTTPBodyStream!
        }
        set(inputStream) {
            var mutableRequest = self.request
            mutableRequest.HTTPBodyStream! = inputStream
            self.request = mutableRequest
        }
    }
    /**
     The output stream that is used to write data received until the request is finished.
    
     By default, data is accumulated into a buffer that is stored into `responseData` upon completion of the request, with the intermediary `outputStream` property set to `nil`. When `outputStream` is set, the data will not be accumulated into an internal buffer, and as a result, the `responseData` property of the completed request will be `nil`. The output stream will be scheduled in the network thread runloop upon being set.
     */
    var outputStream: NSOutputStream? {
        get {
            if !outputStream {
                self.outputStream = NSOutputStream.outputStreamToMemory()
            }
            return outputStream
        }
        set(outputStream) {
            self.lock().lock()
            if outputStream != outputStream {
                if outputStream {
                    outputStream.close()
                }
                self.outputStream = outputStream
            }
            self.lock().unlock()
        }
    }
    ///---------------------------------
    /// @name Managing Callback Queues
    ///---------------------------------
    /**
     The dispatch queue for `completionBlock`. If `NULL` (default), the main queue is used.
     */
    var completionQueue = dispatch_queue_t()
    /**
     The dispatch group for `completionBlock`. If `NULL` (default), a private dispatch group is used.
     */
    var completionGroup = dispatch_group_t()
    ///---------------------------------------------
    /// @name Managing Request Operation Information
    ///---------------------------------------------
    /**
     The user info dictionary for the receiver.
     */
    var userInfo = [NSObject : AnyObject]()
    ///------------------------------------------------------
    /// @name Initializing an AFURLConnectionOperation Object
    ///------------------------------------------------------
    /**
     Initializes and returns a newly allocated operation object with a url connection configured with the specified url request.
     
     This is the designated initializer.
     
     @param urlRequest The request object to be used by the operation connection.
     */

    override init(request urlRequest: NSURLRequest) {
        NSParameterAssert(urlRequest)
        super.init()
        if !self {
            return nil
        }
        self.state = AFOperationReadyState
        self.lock() = NSRecursiveLock()
        self.lock().name! = kAFNetworkingLockName
        self.runLoopModes = Set<NSObject>([NSRunLoopCommonModes])
        self.request = urlRequest
        self.shouldUseCredentialStorage = true
        self.securityPolicy = AFSecurityPolicy.defaultPolicy()
    }
    ///----------------------------------
    /// @name Pausing / Resuming Requests
    ///----------------------------------
    /**
     Pauses the execution of the request operation.
    
     A paused operation returns `NO` for `-isReady`, `-isExecuting`, and `-isFinished`. As such, it will remain in an `NSOperationQueue` until it is either cancelled or resumed. Pausing a finished, cancelled, or paused operation has no effect.
     */

    override func pause() {
        if self.isPaused() || self.isFinished() || self.isCancelled() {
            return
        }
        self.lock().lock()
        if self.isExecuting() {
            self.performSelector(#selector(self.operationDidPause), onThread: self.self.networkRequestThread(), withObject: nil, waitUntilDone: false, modes: self.runLoopModes.allObjects())
            dispatch_async(dispatch_get_main_queue(), {() -> Void in
                var notificationCenter = NSNotificationCenter.defaultCenter()
                notificationCenter.postNotificationName(AFNetworkingOperationDidFinishNotification, object: self)
            })
        }
        self.state = AFOperationPausedState
        self.lock().unlock()
    }
    /**
     Whether the request operation is currently paused.
    
     @return `YES` if the operation is currently paused, otherwise `NO`.
     */

    func isPaused() -> Bool {
        return self.state == AFOperationPausedState
    }
    /**
     Resumes the execution of the paused request operation.
    
     Pause/Resume behavior varies depending on the underlying implementation for the operation class. In its base implementation, resuming a paused requests restarts the original request. However, since HTTP defines a specification for how to request a specific content range, `AFHTTPRequestOperation` will resume downloading the request from where it left off, instead of restarting the original request.
     */

    override func resume() {
        if !self.isPaused() {
            return
        }
        self.lock().lock()
        self.state = AFOperationReadyState
        self.start()
        self.lock().unlock()
    }
    ///----------------------------------------------
    /// @name Configuring Backgrounding Task Behavior
    ///----------------------------------------------
    /**
     Specifies that the operation should continue execution after the app has entered the background, and the expiration handler for that background task.
    
     @param handler A handler to be called shortly before the application’s remaining background time reaches 0. The handler is wrapped in a block that cancels the operation, and cleans up and marks the end of execution, unlike the `handler` parameter in `UIApplication -beginBackgroundTaskWithExpirationHandler:`, which expects this to be done in the handler itself. The handler is called synchronously on the main thread, thus blocking the application’s suspension momentarily while the application is notified.
      */
#if defined(__IPHONE_OS_VERSION_MIN_REQUIRED) && !defined(AF_APP_EXTENSIONS)

    func setShouldExecuteAsBackgroundTaskWithExpirationHandler(handler: () -> Void) {
        self.lock().lock()
        if !self.backgroundTaskIdentifier {
            var application = UIApplication.sharedApplication()
            weak var weakSelf = self
            self.backgroundTaskIdentifier = application.beginBackgroundTaskWithExpirationHandler({() -> Void in
                var strongSelf = weakSelf
                if handler {
                    handler()
                }
                if strongSelf {
                    strongSelf.cancel()
                    application.endBackgroundTask(strongSelf.backgroundTaskIdentifier)
                    strongSelf.backgroundTaskIdentifier = UIBackgroundTaskInvalid
                }
            })
        }
        self.lock().unlock()
    }
#endif
    ///---------------------------------
    /// @name Setting Progress Callbacks
    ///---------------------------------
    /**
     Sets a callback to be called when an undetermined number of bytes have been uploaded to the server.
    
     @param block A block object to be called when an undetermined number of bytes have been uploaded to the server. This block has no return value and takes three arguments: the number of bytes written since the last time the upload progress block was called, the total bytes written, and the total bytes expected to be written during the request, as initially determined by the length of the HTTP body. This block may be called multiple times, and will execute on the main thread.
     */

    func setUploadProgressBlock(block: (bytesWritten: Int, totalBytesWritten: Int64, totalBytesExpectedToWrite: Int64) -> Void) {
        self.uploadProgress = block
    }
    /**
     Sets a callback to be called when an undetermined number of bytes have been downloaded from the server.
    
     @param block A block object to be called when an undetermined number of bytes have been downloaded from the server. This block has no return value and takes three arguments: the number of bytes read since the last time the download progress block was called, the total bytes read, and the total bytes expected to be read during the request, as initially determined by the expected content size of the `NSHTTPURLResponse` object. This block may be called multiple times, and will execute on the main thread.
     */

    func setDownloadProgressBlock(block: (bytesRead: Int, totalBytesRead: Int64, totalBytesExpectedToRead: Int64) -> Void) {
        self.downloadProgress = block
    }
    ///-------------------------------------------------
    /// @name Setting NSURLConnection Delegate Callbacks
    ///-------------------------------------------------
    /**
     Sets a block to be executed when the connection will authenticate a challenge in order to download its request, as handled by the `NSURLConnectionDelegate` method `connection:willSendRequestForAuthenticationChallenge:`.
     
     @param block A block object to be executed when the connection will authenticate a challenge in order to download its request. The block has no return type and takes two arguments: the URL connection object, and the challenge that must be authenticated. This block must invoke one of the challenge-responder methods (NSURLAuthenticationChallengeSender protocol).
     
     If `allowsInvalidSSLCertificate` is set to YES, `connection:willSendRequestForAuthenticationChallenge:` will attempt to have the challenge sender use credentials with invalid SSL certificates.
     */

    func setWillSendRequestForAuthenticationChallengeBlock(block: (connection: NSURLConnection, challenge: NSURLAuthenticationChallenge) -> Void) {
        self.authenticationChallenge = block
    }
    /**
     Sets a block to be executed when the server redirects the request from one URL to another URL, or when the request URL changed by the `NSURLProtocol` subclass handling the request in order to standardize its format, as handled by the `NSURLConnectionDataDelegate` method `connection:willSendRequest:redirectResponse:`.
    
     @param block A block object to be executed when the request URL was changed. The block returns an `NSURLRequest` object, the URL request to redirect, and takes three arguments: the URL connection object, the the proposed redirected request, and the URL response that caused the redirect.
     */

    func setRedirectResponseBlock(block: (connection: NSURLConnection, request: NSURLRequest, redirectResponse: NSURLResponse) -> NSURLRequest) {
        self.redirectResponse = block
    }
    /**
     Sets a block to be executed to modify the response a connection will cache, if any, as handled by the `NSURLConnectionDelegate` method `connection:willCacheResponse:`.
    
     @param block A block object to be executed to determine what response a connection will cache, if any. The block returns an `NSCachedURLResponse` object, the cached response to store in memory or `nil` to prevent the response from being cached, and takes two arguments: the URL connection object, and the cached response provided for the request.
     */

    func setCacheResponseBlock(block: (connection: NSURLConnection, cachedResponse: NSCachedURLResponse) -> NSCachedURLResponse) {
        self.cacheResponse = block
    }
    ///
    /**
    
     */

    class func batchOfRequestOperations(operations: [AnyObject], progressBlock: (numberOfFinishedOperations: Int, totalNumberOfOperations: Int) -> Void, completionBlock: (operations: [AnyObject]) -> Void) -> [AnyObject] {
        if !operations || operations.count == 0 {
            return [NSBlockOperation.blockOperationWithBlock({() -> Void in
                dispatch_async(dispatch_get_main_queue(), {() -> Void in
                    if completionBlock {
                        completionBlock([])
                    }
                })
            })]
        }
        var group = dispatch_group_create()
        var batchedOperation = NSBlockOperation.blockOperationWithBlock({() -> Void in
                dispatch_group_notify(group, dispatch_get_main_queue(), {() -> Void in
                    if completionBlock {
                        completionBlock(operations)
                    }
                })
            })
        for operation: AFURLConnectionOperation in operations {
            operation.completionGroup = group
            var originalCompletionBlock = operation.completionBlock()
            weak var weakOperation = operation
            operation.completionBlock() = {() -> Void in
                var strongOperation = weakOperation
//#pragma clang diagnostic push
//#pragma clang diagnostic ignored "-Wgnu"
                var queue = strongOperation.completionQueue ?? dispatch_get_main_queue()
//#pragma clang diagnostic pop
                dispatch_group_async(group, queue, {() -> Void in
                    if originalCompletionBlock {
                        originalCompletionBlock()
                    }
                    var numberOfFinishedOperations = (operations as NSArray).indexesOfObjectsPassingTest({(op: AnyObject, idx: NSUInteger __unused, stop: BOOL __unused) -> BOOL in
                            return op.isFinished()
                        }).count
                    if progressBlock {
                        progressBlock(numberOfFinishedOperations, operations.count)
                    }
                    dispatch_group_leave(group)
                })
            }
            dispatch_group_enter(group)
            batchedOperation.addDependency(operation)
        }
        return operations + [batchedOperation]
    }


    class func networkRequestThreadEntryPoint(__unused: AnyObject) {
                    NSThread.currentThread().name = "AFNetworking"
            var runLoop = NSRunLoop.currentRunLoop()
            runLoop.addPort(NSMachPort.port, forMode: NSDefaultRunLoopMode)
            runLoop.run()

    }

    class func networkRequestThread() -> NSThread {
        var networkRequestThread: NSThread? = nil
        var oncePredicate: dispatch_once_t
        dispatch_once(oncePredicate, {() -> Void in
            self.networkRequestThread = NSThread(target: self, selector: #selector(self.networkRequestThreadEntryPoint), object: nil)
            networkRequestThread!.start()
        })
        return networkRequestThread
    }

    deinit {
        if outputStream != nil {
            outputStream!.close()
            self.outputStream = nil
        }
#if defined(__IPHONE_OS_VERSION_MIN_REQUIRED) && !defined(AF_APP_EXTENSIONS)
        if backgroundTaskIdentifier {
            UIApplication.sharedApplication().endBackgroundTask(backgroundTaskIdentifier)
            self.backgroundTaskIdentifier = UIBackgroundTaskInvalid
        }
#endif
    }
// MARK: -

    func setResponseData(responseData: NSData) {
        self.lock().lock()
        if responseData == nil {
            self.responseData = nil
        }
        else {
            self.responseData = NSData(bytes: responseData.bytes, length: responseData.length)
        }
        self.lock().unlock()
    }
#if defined(__IPHONE_OS_VERSION_MIN_REQUIRED) && !defined(AF_APP_EXTENSIONS)
#endif
// MARK: -

    override func setState(state: AFOperationState) {
        if !AFStateTransitionIsValid(self.state, state, self.isCancelled()) {
            return
        }
        self.lock().lock()
        var oldStateKey = AFKeyPathFromOperationState(self.state)
        var newStateKey = AFKeyPathFromOperationState(state)
        self.willChangeValueForKey(newStateKey)
        self.willChangeValueForKey(oldStateKey)
        self.state = state
        self.didChangeValueForKey(oldStateKey)
        self.didChangeValueForKey(newStateKey)
        self.lock().unlock()
    }

    func operationDidPause() {
        self.lock().lock()
        self.connection.cancel()
        self.lock().unlock()
    }
// MARK: -
// MARK: - NSOperation

    override func setCompletionBlock(block: () -> Void) {
        self.lock().lock()
        if !block {
            super.completionBlock = nil
        }
        else {
            weak var weakSelf = self
            super.completionBlock = {() -> Void in
                var strongSelf = weakSelf
//#pragma clang diagnostic push
//#pragma clang diagnostic ignored "-Wgnu"
                var group = strongSelf.completionGroup ?? url_request_operation_completion_group()
                var queue = strongSelf.completionQueue ?? dispatch_get_main_queue()
//#pragma clang diagnostic pop
                dispatch_group_async(group, queue, {() -> Void in
                    block()
                })
                dispatch_group_notify(group, url_request_operation_completion_queue(), {() -> Void in
                    strongSelf.completionBlock = nil
                })
            }
        }
        self.lock().unlock()
    }

    func isReady() -> Bool {
        return self.state == AFOperationReadyState && super.isReady()
    }

    func isExecuting() -> Bool {
        return self.state == AFOperationExecutingState
    }

    func isFinished() -> Bool {
        return self.state == AFOperationFinishedState
    }

    func isConcurrent() -> Bool {
        return true
    }

    override func start() {
        self.lock().lock()
        if self.isCancelled() {
            self.performSelector(#selector(self.cancelConnection), onThread: self.self.networkRequestThread(), withObject: nil, waitUntilDone: false, modes: self.runLoopModes.allObjects())
        }
        else if self.isReady() {
            self.state = AFOperationExecutingState
            self.performSelector(#selector(self.operationDidStart), onThread: self.self.networkRequestThread(), withObject: nil, waitUntilDone: false, modes: self.runLoopModes.allObjects())
        }

        self.lock().unlock()
    }

    func operationDidStart() {
        self.lock().lock()
        if !self.isCancelled() {
            self.connection = NSURLConnection(request: self.request, delegate: self, startImmediately: false)
            var runLoop = NSRunLoop.currentRunLoop()
            for runLoopMode: String in self.runLoopModes {
                self.connection.scheduleInRunLoop(runLoop, forMode: runLoopMode)
                self.outputStream.scheduleInRunLoop(runLoop, forMode: runLoopMode)
            }
            self.connection.start()
        }
        self.lock().unlock()
        dispatch_async(dispatch_get_main_queue(), {() -> Void in
            NSNotificationCenter.defaultCenter().postNotificationName(AFNetworkingOperationDidStartNotification, object: self)
        })
    }

    func finish() {
        self.lock().lock()
        self.state = AFOperationFinishedState
        self.lock().unlock()
        dispatch_async(dispatch_get_main_queue(), {() -> Void in
            NSNotificationCenter.defaultCenter().postNotificationName(AFNetworkingOperationDidFinishNotification, object: self)
        })
    }

    override func cancel() {
        self.lock().lock()
        if !self.isFinished() && !self.isCancelled() {
            super.cancel()
            if self.isExecuting() {
                self.performSelector(#selector(self.cancelConnection), onThread: self.self.networkRequestThread(), withObject: nil, waitUntilDone: false, modes: self.runLoopModes.allObjects())
            }
        }
        self.lock().unlock()
    }

    func cancelConnection() {
        var userInfo: [NSObject : AnyObject]? = nil
        if self.request.URL! {
            userInfo = [ NSURLErrorFailingURLErrorKey : self.request.URL! ]
        }
        var error = NSError(domain: NSURLErrorDomain, code: NSURLErrorCancelled, userInfo: userInfo)
        if !self.isFinished() {
            if self.connection {
                self.connection.cancel()
                self.performSelector(Selector("connection:didFailWithError:"), withObject: self.connection, withObject: error)
            }
            else {
                // Accomodate race condition where `self.connection` has not yet been set before cancellation
                self.error = error
                self.finish()
            }
        }
    }
// MARK: -
// MARK: - NSObject

    override func description() -> String {
        return "<\(NSStringFromClass(self.self)): \(self), state: \(AFKeyPathFromOperationState(self.state)), cancelled: \(self.isCancelled() ? "YES" : "NO") request: \(self.request), response: \(self.response)>"
    }
// MARK: - NSURLConnectionDelegate

    func connection(connection: NSURLConnection, willSendRequestForAuthenticationChallenge challenge: NSURLAuthenticationChallenge) {
        if self.authenticationChallenge {
            self.authenticationChallenge(connection, challenge)
            return
        }
        if (challenge.protectionSpace.authenticationMethod! == NSURLAuthenticationMethodServerTrust) {
            if self.securityPolicy.evaluateServerTrust(challenge.protectionSpace.serverTrust!, forDomain: challenge.protectionSpace.host) {
                var credential = NSURLCredential(forTrust: challenge.protectionSpace.serverTrust!)
                challenge.sender.useCredential(credential, forAuthenticationChallenge: challenge)
            }
            else {
                challenge.sender.cancelAuthenticationChallenge(challenge)
            }
        }
        else {
            if challenge.previousFailureCount == 0 {
                if self.credential {
                    challenge.sender.useCredential(self.credential, forAuthenticationChallenge: challenge)
                }
                else {
                    challenge.sender.continueWithoutCredentialForAuthenticationChallenge(challenge)
                }
            }
            else {
                challenge.sender.continueWithoutCredentialForAuthenticationChallenge(challenge)
            }
        }
    }

    func connectionShouldUseCredentialStorage(connection: NSURLConnection __unused) -> Bool {
        return self.shouldUseCredentialStorage
    }

    func connection(connection: NSURLConnection, willSendRequest request: NSURLRequest, redirectResponse: NSURLResponse) -> NSURLRequest {
        if self.redirectResponse {
            return self.redirectResponse(connection, request, redirectResponse)
        }
        else {
            return request
        }
    }

    func connection(connection: NSURLConnection __unused, didSendBodyData bytesWritten: Int, totalBytesWritten: Int, totalBytesExpectedToWrite: Int) {
        dispatch_async(dispatch_get_main_queue(), {() -> Void in
            if self.uploadProgress {
                self.uploadProgress(Int(bytesWritten), totalBytesWritten, totalBytesExpectedToWrite)
            }
        })
    }

    func connection(connection: NSURLConnection __unused, didReceiveResponse response: NSURLResponse) {
        self.response = response
        self.outputStream.open()
    }

    func connection(connection: NSURLConnection __unused, didReceiveData data: NSData) {
        var length = data.length
        while true {
            var totalNumberOfBytesWritten = 0
            if self.outputStream.hasSpaceAvailable {
                let dataBuffer: UInt8 = UInt8(data.bytes)
                var numberOfBytesWritten = 0
                while totalNumberOfBytesWritten < Int(length) {
                    numberOfBytesWritten = self.outputStream.write(dataBuffer[Int(totalNumberOfBytesWritten)], maxLength: (length - Int(totalNumberOfBytesWritten)))
                    if numberOfBytesWritten == -1 {

                    }
                    totalNumberOfBytesWritten += numberOfBytesWritten
                }
            }
            if self.outputStream.streamEcharacterAtIndexrror! {
                self.connection.cancel()
                self.performSelector(Selector("connection:didFailWithError:"), withObject: self.connection, withObject: self.outputStream.streamEcharacterAtIndexrror!)
                return
            }
        }
        dispatch_async(dispatch_get_main_queue(), {() -> Void in
            self.totalBytesRead += Int64(length)
            if self.downloadProgress {
                self.downloadProgress(length, self.totalBytesRead, self.response.expectedContentLength)
            }
        })
    }

    func connectionDidFinishLoading(connection: NSURLConnection __unused) {
        self.responseData = self.outputStream.propertyForKey(NSStreamDataWrittenToMemoryStreamKey)!
        self.outputStream.close()
        if self.responseData {
            self.outputStream = nil
        }
        self.connection = nil
        self.finish()
    }

    func connection(connection: NSURLConnection __unused, didFailWithError error: NSError?) {
        self.error = error
        self.outputStream.close()
        if self.responseData {
            self.outputStream = nil
        }
        self.connection = nil
        self.finish()
    }

    func connection(connection: NSURLConnection, willCacheResponse cachedResponse: NSCachedURLResponse) -> NSCachedURLResponse {
        if self.cacheResponse {
            return self.cacheResponse(connection, cachedResponse)
        }
        else {
            if self.isCancelled() {
                return nil
            }
            return cachedResponse
        }
    }
// MARK: - NSSecureCoding

    class func supportsSecureCoding() -> Bool {
        return true
    }

    convenience required init?(coder decoder: NSCoder) {
        var request = decoder.decodeObjectOfClass(NSURLRequest.self, forKey: NSStringFromSelector(#selector(self.request)))!
        self.init(request: request)
        if !self {
            return nil
        }
        self.state = CInt(decoder.decodeObjectOfClass(Int.self, forKey: NSStringFromSelector(#selector(self.state)))!)
        self.response = decoder.decodeObjectOfClass(NSHTTPURLResponse.self, forKey: NSStringFromSelector(#selector(self.response)))!
        self.error = decoder.decodeObjectOfClass(NSError.self, forKey: NSStringFromSelector(#selector(self.error)))!
        self.responseData = decoder.decodeObjectOfClass(NSData.self, forKey: NSStringFromSelector(#selector(self.responseData)))!
        self.totalBytesRead = CLongLong(decoder.decodeObjectOfClass(Int.self, forKey: NSStringFromSelector(#selector(self.totalBytesRead)))!)
    }

    override func encodeWithCoder(coder: NSCoder) {
        self.pause()
        coder.encodeObject(self.request, forKey: NSStringFromSelector(#selector(self.request)))
        switch self.state {
            case AFOperationExecutingState, AFOperationPausedState:
                coder.encodeInteger(AFOperationReadyState, forKey: NSStringFromSelector(#selector(self.state)))
            default:
                coder.encodeInteger(self.state, forKey: NSStringFromSelector(#selector(self.state)))
        }

        coder.encodeObject(self.response, forKey: NSStringFromSelector(#selector(self.response)))
        coder.encodeObject(self.error, forKey: NSStringFromSelector(#selector(self.error)))
        coder.encodeObject(self.responseData, forKey: NSStringFromSelector(#selector(self.responseData)))
        coder.encodeInt64(self.totalBytesRead, forKey: NSStringFromSelector(#selector(self.totalBytesRead)))
    }
// MARK: - NSCopying

    override func copyWithZone(zone: NSZone) -> AnyObject {
        var operation = (self.self.allocWithZone(zone) as! AFURLConnectionOperation)(request: self.request)
        operation.uploadProgress = self.uploadProgress
        operation.downloadProgress = self.downloadProgress
        operation.authenticationChallenge = self.authenticationChallenge
        operation.cacheResponse = self.cacheResponse
        operation.redirectResponse = self.redirectResponse
        operation.completionQueue = self.completionQueue
        operation.completionGroup = self.completionGroup
        return operation
    }

    var state: AFOperationState {
        get {
            // TODO: add getter implementation
        }
        set(state) {
            if !AFStateTransitionIsValid(self.state, state, self.isCancelled()) {
                return
            }
            self.lock().lock()
            var oldStateKey = AFKeyPathFromOperationState(self.state)
            var newStateKey = AFKeyPathFromOperationState(state)
            self.willChangeValueForKey(newStateKey)
            self.willChangeValueForKey(oldStateKey)
            self.state = state
            self.didChangeValueForKey(oldStateKey)
            self.didChangeValueForKey(newStateKey)
            self.lock().unlock()
        }
    }
    var lock: NSRecursiveLock!
    var connection: NSURLConnection!
    var request: NSURLRequest!
    var response: NSURLResponse!
    var error: NSError!
    var responseData: NSData? {
        get {
            // TODO: add getter implementation
        }
        set(responseData) {
            self.lock().lock()
            if responseData == nil {
                self.responseData = nil
            }
            else {
                self.responseData = NSData(bytes: responseData.bytes, length: responseData.length)
            }
            self.lock().unlock()
        }
    }
    var responseString: String {
        self.lock().lock()
            if !responseString && self.response && self.responseData {
                self.responseString = String(data: self.responseData, encoding: self.responseStringEncoding)
            }
            self.lock().unlock()
            return responseString
    }
    var responseStringEncoding: NSStringEncoding {
        self.lock().lock()
            if !responseStringEncoding && self.response {
                var stringEncoding = NSUTF8StringEncoding
                if self.response.textEncodingName! {
                    var IANAEncoding = CFStringConvertIANACharSetNameToEncoding((self.response.textEncodingName! as! CFString))
                    if IANAEncoding != kCFStringEncodingInvalidId {
                        stringEncoding = CFStringConvertEncodingToNSStringEncoding(IANAEncoding)
                    }
                }
                self.responseStringEncoding = stringEncoding
            }
            self.lock().unlock()
            return responseStringEncoding
    }
    var totalBytesRead: Int64 = 0
    var backgroundTaskIdentifier = AFBackgroundTaskIdentifier()
    var uploadProgress = AFURLConnectionOperationProgressBlock()
    var downloadProgress = AFURLConnectionOperationProgressBlock()
    var authenticationChallenge = AFURLConnectionOperationAuthenticationChallengeBlock()
    var cacheResponse = AFURLConnectionOperationCacheResponseBlock()
    var redirectResponse = AFURLConnectionOperationRedirectResponseBlock()

    func operationDidStart() {
    }

    func finish() {
    }

    func cancelConnection() {
    }
}
///--------------------
/// @name Notifications
///--------------------
/**
 Posted when an operation begins executing.
 */
let AFNetworkingOperationDidStartNotification = ""

/**
 Posted when an operation finishes.
 */
let AFNetworkingOperationDidFinishNotification = ""

// AFURLConnectionOperation.m
//
// Copyright (c) 2013-2014 AFNetworking (http://afnetworking.com)
//
// Permission is hereby granted, free of charge, to any person obtaining a copy
// of this software and associated documentation files (the "Software"), to deal
// in the Software without restriction, including without limitation the rights
// to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
// copies of the Software, and to permit persons to whom the Software is
// furnished to do so, subject to the following conditions:
//
// The above copyright notice and this permission notice shall be included in
// all copies or substantial portions of the Software.
//
// THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
// IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
// FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
// AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
// LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
// OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
// THE SOFTWARE.
#if defined(__IPHONE_OS_VERSION_MIN_REQUIRED)
import UIKit
#endif
#if !__has_feature(objc_arc)
// You can turn on ARC for only AFNetworking files by adding -fobjc-arc to the build phase for each of its files.
#endif
enum AFOperationState : Int {
    case AFOperationPausedState = -1
    case AFOperationReadyState = 1
    case AFOperationExecutingState = 2
    case AFOperationFinishedState = 3
}

#if defined(__IPHONE_OS_VERSION_MIN_REQUIRED) && !defined(AF_APP_EXTENSIONS)
var AFBackgroundTaskIdentifier = UIBackgroundTaskIdentifier()

#else
var AFBackgroundTaskIdentifier: AnyObject!

#endif
func url_request_operation_completion_group() -> dispatch_group_t {
    var af_url_request_operation_completion_group: dispatch_group_t
    var onceToken: dispatch_once_t
    dispatch_once(onceToken, {() -> Void in
        af_url_request_operation_completion_group = dispatch_group_create()
    })
    return af_url_request_operation_completion_group
}

func url_request_operation_completion_queue() -> dispatch_queue_t {
    var af_url_request_operation_completion_queue: dispatch_queue_t
    var onceToken: dispatch_once_t
    dispatch_once(onceToken, {() -> Void in
        af_url_request_operation_completion_queue = dispatch_queue_create("com.alamofire.networking.operation.queue", DISPATCH_QUEUE_CONCURRENT)
    })
    return af_url_request_operation_completion_queue
}

let kAFNetworkingLockName = "com.alamofire.networking.operation.lock"

let AFNetworkingOperationDidStartNotification = "com.alamofire.networking.operation.start"

let AFNetworkingOperationDidFinishNotification = "com.alamofire.networking.operation.finish"

typealias AFURLConnectionOperationProgressBlock = (bytes: Int, totalBytes: Int64, totalBytesExpected: Int64) -> Void
typealias AFURLConnectionOperationAuthenticationChallengeBlock = (connection: NSURLConnection, challenge: NSURLAuthenticationChallenge) -> Void
typealias AFURLConnectionOperationCacheResponseBlock = (connection: NSURLConnection, cachedResponse: NSCachedURLResponse) -> NSCachedURLResponse
typealias AFURLConnectionOperationRedirectResponseBlock = (connection: NSURLConnection, request: NSURLRequest, redirectResponse: NSURLResponse) -> NSURLRequest
func AFKeyPathFromOperationState(state: AFOperationState) -> inline NSString {
    switch state {
        case AFOperationReadyState:
            return "isReady"
        case AFOperationExecutingState:
            return "isExecuting"
        case AFOperationFinishedState:
            return "isFinished"
        case AFOperationPausedState:
            return "isPaused"
        default:
            //#pragma clang diagnostic push
//#pragma clang diagnostic ignored "-Wunreachable-code"
            return "state"
//#pragma clang diagnostic pop

    }

}

func AFStateTransitionIsValid(fromState: AFOperationState, toState: AFOperationState, isCancelled: Bool) -> inline BOOL {
    switch fromState {
        case AFOperationReadyState:
            switch toState {
                case AFOperationPausedState, AFOperationExecutingState:
                    return true
                case AFOperationFinishedState:
                    return isCancelled
                default:
                    return false
            }

        case AFOperationExecutingState:
            switch toState {
                case AFOperationPausedState, AFOperationFinishedState:
                    return true
                default:
                    return false
            }

        case AFOperationFinishedState:
            return false
        case AFOperationPausedState:
            return toState == AFOperationReadyState
        default:
            //#pragma clang diagnostic push
//#pragma clang diagnostic ignored "-Wunreachable-code"
            switch toState {
                case AFOperationPausedState, AFOperationReadyState, AFOperationExecutingState, AFOperationFinishedState:
                    return true
                default:
                    return false
            }


    }

}