// AFSecurity.h
//
// Copyright (c) 2013-2014 AFNetworking (http://afnetworking.com)
//
// Permission is hereby granted, free of charge, to any person obtaining a copy
// of this software and associated documentation files (the "Software"), to deal
// in the Software without restriction, including without limitation the rights
// to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
// copies of the Software, and to permit persons to whom the Software is
// furnished to do so, subject to the following conditions:
//
// The above copyright notice and this permission notice shall be included in
// all copies or substantial portions of the Software.
//
// THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
// IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
// FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
// AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
// LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
// OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
// THE SOFTWARE.
import UIKit
import Foundation
import Security
enum AFSSLPinningMode : Int {
    case None
    case PublicKey
    case Certificate
}

/**
 `AFSecurityPolicy` evaluates server trust against pinned X.509 certificates and public keys over secure connections.
 
 Adding pinned SSL certificates to your app helps prevent man-in-the-middle attacks and other vulnerabilities. Applications dealing with sensitive customer data or financial information are strongly encouraged to route all communication over an HTTPS connection with SSL pinning configured and enabled.
 */
class AFSecurityPolicy: NSObject {
    /**
     The criteria by which server trust should be evaluated against the pinned SSL certificates. Defaults to `AFSSLPinningModeNone`.
     */
    var SSLPinningMode = AFSSLPinningMode()
    /**
     Whether to evaluate an entire SSL certificate chain, or just the leaf certificate. Defaults to `YES`.
     */
    var validatesCertificateChain = false
    /**
     The certificates used to evaluate server trust according to the SSL pinning mode. By default, this property is set to any (`.cer`) certificates included in the app bundle.
     */
    var pinnedCertificates: [AnyObject] {
        get {
            // TODO: add getter implementation
        }
        set(pinnedCertificates) {
            self.pinnedCertificates = pinnedCertificates
            if self.pinnedCertificates {
                var mutablePinnedPublicKeys = [AnyObject]() /* capacity: self.pinnedCertificates.count */
                for certificate: NSData in self.pinnedCertificates {
                    var publicKey = AFPublicKeyForCertificate(certificate)
                    if !publicKey {
    
                    }
                    mutablePinnedPublicKeys.append(publicKey)
                }
                self.pinnedPublicKeys = [AnyObject](arrayLiteral: mutablePinnedPublicKeys)
            }
            else {
                self.pinnedPublicKeys = nil
            }
        }
    }
    /**
     Whether or not to trust servers with an invalid or expired SSL certificates. Defaults to `NO`.
     */
    var allowInvalidCertificates = false
    /**
     Whether or not to validate the domain name in the certificates CN field. Defaults to `YES` for `AFSSLPinningModePublicKey` or `AFSSLPinningModeCertificate`, otherwise `NO`.
     */
    var validatesDomainName = false
    ///-----------------------------------------
    /// @name Getting Specific Security Policies
    ///-----------------------------------------
    /**
     Returns the shared default security policy, which does not accept invalid certificates, and does not validate against pinned certificates or public keys.
     
     @return The default security policy.
     */

    class func defaultPolicy() -> Self {
        var securityPolicy = self.init()
        securityPolicy.SSLPinningMode = .None
        return securityPolicy
    }
    ///---------------------
    /// @name Initialization
    ///---------------------
    /**
     Creates and returns a security policy with the specified pinning mode.
     
     @param pinningMode The SSL pinning mode.
     
     @return A new security policy.
     */

    class func policyWithPinningMode(pinningMode: AFSSLPinningMode) -> Self {
        var securityPolicy = self.init()
        securityPolicy.SSLPinningMode = pinningMode
        securityPolicy.validatesDomainName = true
        securityPolicy.pinnedCertificates = self.defaultPinnedCertificates()
        return securityPolicy
    }
    ///------------------------------
    /// @name Evaluating Server Trust
    ///------------------------------
    /**
     Whether or not the specified server trust should be accepted, based on the security policy.
    
     This method should be used when responding to an authentication challenge from a server.
    
     @param serverTrust The X.509 certificate trust of the server.
    
     @return Whether or not to trust the server.
     
     @warning This method has been deprecated in favor of `-evaluateServerTrust:forDomain:`.
     */

    func evaluateServerTrust(serverTrust: SecTrustRef) -> Bool {
        return self.evaluateServerTrust(serverTrust, forDomain: nil)
    }
    /**
     Whether or not the specified server trust should be accepted, based on the security policy. 
     
     This method should be used when responding to an authentication challenge from a server.
     
     @param serverTrust The X.509 certificate trust of the server.
     @param domain The domain of serverTrust. If `nil`, the domain will not be validated.
     
     @return Whether or not to trust the server.
     */

    func evaluateServerTrust(serverTrust: SecTrustRef, forDomain domain: String) -> Bool {
        var policies = [AnyObject]()
        if self.validatesDomainName {
            policies.append((SecPolicyCreateSSL(true, (domain as! CFString)) as!  id))
        }
        else {
            policies.append((SecPolicyCreateBasicX509() as!  id))
        }
        SecTrustSetPolicies(serverTrust, (policies as! CFArrayRef))
        if !AFServerTrustIsValid(serverTrust) && !self.allowInvalidCertificates {
            return false
        }
        var serverCertificates = AFCertificateTrustChainForServerTrust(serverTrust)
        switch self.SSLPinningMode {
            case .None:
                return true
            case .Certificate:
                                var pinnedCertificates = [AnyObject]()
                for certificateData: NSData in self.pinnedCertificates {
                    pinnedCertificates.append((SecCertificateCreateWithData(nil, (certificateData as! CFDataRef)) as!  id))
                }
                SecTrustSetAnchorCertificates(serverTrust, (pinnedCertificates as! CFArrayRef))
                if !AFServerTrustIsValid(serverTrust) {
                    return false
                }
                if !self.validatesCertificateChain {
                    return true
                }
                var trustedCertificateCount = 0
                for trustChainCertificate: NSData in serverCertificates {
                    if self.pinnedCertificates.contains(trustChainCertificate) {
                        trustedCertificateCount += 1
                    }
                }
                return trustedCertificateCount == serverCertificates.count

            case .PublicKey:
                                var trustedPublicKeyCount = 0
                var publicKeys = AFPublicKeyTrustChainForServerTrust(serverTrust)
                if !self.validatesCertificateChain && publicKeys.count > 0 {
                    publicKeys = [publicKeys.first!]
                }
                for trustChainPublicKey: AnyObject in publicKeys {
                    for pinnedPublicKey: AnyObject in self.pinnedPublicKeys {
                        if AFSecKeyIsEqualToKey((trustChainPublicKey as! SecKeyRef), (pinnedPublicKey as! SecKeyRef)) {
                            trustedPublicKeyCount += 1
                        }
                    }
                }
                return trustedPublicKeyCount > 0 && (self.validatesCertificateChain && trustedPublicKeyCount == serverCertificates.count) || (!self.validatesCertificateChain && trustedPublicKeyCount >= 1)

        }

        return false
    }


    class func defaultPinnedCertificates() -> [AnyObject] {
        var defaultPinnedCertificates: [AnyObject]? = nil
        var onceToken: dispatch_once_t
        dispatch_once(onceToken, {() -> Void in
            var bundle = NSBundle(forClass: self.self)
            var paths = bundle.pathsForResourcesOfType("cer", inDirectory: ".")
            var certificates = [AnyObject]() /* capacity: paths.count */
            for path: String in paths {
                var certificateData = NSData(contentsOfFile: path)
                certificates.append(certificateData)
            }
            self.defaultPinnedCertificates = [AnyObject](arrayLiteral: certificates)
        })
        return defaultPinnedCertificates
    }

    override init() {
        super.init()
        if !self {
            return nil
        }
        self.validatesCertificateChain = true
    }
// MARK: -
// MARK: -
// MARK: - NSKeyValueObserving

    class func keyPathsForValuesAffectingPinnedPublicKeys() -> Set<NSObject> {
        return Set<NSObject>(["pinnedCertificates"])
    }

    var pinnedPublicKeys = [AnyObject]()
}
var exceptionLabel = ()

var exceptionLabel = ()

#endif
#if !defined(__IPHONE_OS_VERSION_MIN_REQUIRED)
func AFSecKeyGetData(key: SecKeyRef) -> NSData {
    var data: CFDataRef? = nil
    AF_Require_noErr(SecItemExport(key, kSecFormatUnknown, kSecItemPemArmour, nil, data), out)
    return (data as!  NSData)
        if data != nil {
        CFRelease(data)
    }
    return nil
}

#endif
func AFSecKeyIsEqualToKey(key1: SecKeyRef, key2: SecKeyRef) -> Bool {
#if defined(__IPHONE_OS_VERSION_MIN_REQUIRED)
    return (key1 as! AnyObject).isEqual((key2 as! AnyObject))
#else
    return AFSecKeyGetData(key1).isEqual(AFSecKeyGetData(key2))
#endif
}

func AFPublicKeyForCertificate(certificate: NSData) -> AnyObject {
    var allowedPublicKey: AnyObject? = nil
    var allowedCertificate: SecCertificateRef
    var allowedCertificates = [SecCertificateRef](count: 1, repeatedValue: SecCertificateRef())
    var tempCertificates: CFArrayRef? = nil
    var policy: SecPolicyRef? = nil
    var allowedTrust: SecTrustRef? = nil
    var result: SecTrustResultType
    allowedCertificate = SecCertificateCreateWithData(nil, (certificate as! CFDataRef))
    AF_Require(allowedCertificate != nil, out)
    allowedCertificates[0] = allowedCertificate
    tempCertificates = CFArrayCreate(nil, (allowedCertificates as! Void), 1, nil)
    policy = SecPolicyCreateBasicX509()
    AF_Require_noErr(SecTrustCreateWithCertificates(tempCertificates, policy, allowedTrust), out)
    AF_Require_noErr(SecTrustEvaluate(allowedTrust, result), out)
    allowedPublicKey = (SecTrustCopyPublicKey(allowedTrust) as!  id)
        if allowedTrust != nil {
        CFRelease(allowedTrust)
    }
    if policy != nil {
        CFRelease(policy)
    }
    if tempCertificates != nil {
        CFRelease(tempCertificates)
    }
    if allowedCertificate {
        CFRelease(allowedCertificate)
    }
    return allowedPublicKey
}

func AFServerTrustIsValid(serverTrust: SecTrustRef) -> Bool {
    var isValid = false
    var result: SecTrustResultType
    AF_Require_noErr(SecTrustEvaluate(serverTrust, result), out)
    isValid = (result == kSecTrustResultUnspecified || result == kSecTrustResultProceed)
        return isValid
}

func AFCertificateTrustChainForServerTrust(serverTrust: SecTrustRef) -> [AnyObject] {
    var certificateCount = SecTrustGetCertificateCount(serverTrust)
    var trustChain = [AnyObject]() /* capacity: Int(certificateCount) */
    for i in 0..<certificateCount {
        var certificate = SecTrustGetCertificateAtIndex(serverTrust, i)
        trustChain.append((SecCertificateCopyData(certificate) as!  NSData))
    }
    return [AnyObject](arrayLiteral: trustChain)
}

func AFPublicKeyTrustChainForServerTrust(serverTrust: SecTrustRef) -> [AnyObject] {
    var policy = SecPolicyCreateBasicX509()
    var certificateCount = SecTrustGetCertificateCount(serverTrust)
    var trustChain = [AnyObject]() /* capacity: Int(certificateCount) */
    for i in 0..<certificateCount {
        var certificate = SecTrustGetCertificateAtIndex(serverTrust, i)
        var someCertificates = [certificate]
        var certificates = CFArrayCreate(nil, (someCertificates as! Void), 1, nil)
        var trust: SecTrustRef
        AF_Require_noErr(SecTrustCreateWithCertificates(certificates, policy, trust), out)
        var result: SecTrustResultType
        AF_Require_noErr(SecTrustEvaluate(trust, result), out)
        trustChain.append((SecTrustCopyPublicKey(trust) as!  id))
                if trust {
            CFRelease(trust)
        }
        if certificates {
            CFRelease(certificates)
        }
    }
    CFRelease(policy)
    return [AnyObject](arrayLiteral: trustChain)
}

// MARK: -