// AFNetworkReachabilityManager.h
// 
// Copyright (c) 2013-2014 AFNetworking (http://afnetworking.com)
//
// Permission is hereby granted, free of charge, to any person obtaining a copy
// of this software and associated documentation files (the "Software"), to deal
// in the Software without restriction, including without limitation the rights
// to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
// copies of the Software, and to permit persons to whom the Software is
// furnished to do so, subject to the following conditions:
// 
// The above copyright notice and this permission notice shall be included in
// all copies or substantial portions of the Software.
// 
// THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
// IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
// FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
// AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
// LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
// OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
// THE SOFTWARE.
import UIKit
import Foundation
import SystemConfiguration
enum AFNetworkReachabilityStatus : Int {
    case Unknown = -1
    case NotReachable = 0
    case ReachableViaWWAN = 1
    case ReachableViaWiFi = 2
}

/**
 `AFNetworkReachabilityManager` monitors the reachability of domains, and addresses for both WWAN and WiFi network interfaces.
 
 Reachability can be used to determine background information about why a network operation failed, or to trigger a network operation retrying when a connection is established. It should not be used to prevent a user from initiating a network request, as it's possible that an initial request may be required to establish reachability.

 See Apple's Reachability Sample Code (https://developer.apple.com/library/ios/samplecode/reachability/)
 
 @warning Instances of `AFNetworkReachabilityManager` must be started with `-startMonitoring` before reachability status can be determined.
 */
class AFNetworkReachabilityManager: NSObject {
    /**
     The current network reachability status.
     */
    private(set) var networkReachabilityStatus = AFNetworkReachabilityStatus()
    /**
     Whether or not the network is currently reachable.
     */
    var reachable: Bool {
        return self.isReachableViaWWAN()| self.isReachableViaWiFi()
    }
    /**
     Whether or not the network is currently reachable via WWAN.
     */
    var reachableViaWWAN: Bool {
        return self.networkReachabilityStatus == .ReachableViaWWAN
    }
    /**
     Whether or not the network is currently reachable via WiFi.
     */
    var reachableViaWiFi: Bool {
        return self.networkReachabilityStatus == .ReachableViaWiFi
    }
    ///---------------------
    /// @name Initialization
    ///---------------------
    /**
     Returns the shared network reachability manager.
     */

    class func sharedManager() -> Self {
        var sharedManager: AFNetworkReachabilityManager? = nil
        var onceToken: dispatch_once_t
        dispatch_once(onceToken, {() -> Void in
            var address: structsockaddr_in
            bzero(address, sizeof(address))
            address.sin_len = sizeof(address)
            address.sin_family = AF_INET
            self.sharedManager = self.managerForAddress(address)
        })
        return sharedManager
    }
    /**
     Creates and returns a network reachability manager for the specified domain.
     
     @param domain The domain used to evaluate network reachability.
     
     @return An initialized network reachability manager, actively monitoring the specified domain.
     */

    class func managerForDomain(domain: String) -> Self {
        var reachability = SCNetworkReachabilityCreateWithName(kCFAllocatorDefault, domain.UTF8String)
        var manager = self.init(reachability: reachability)
        manager.networkReachabilityAssociation = AFNetworkReachabilityForName
        return manager
    }
    /**
     Creates and returns a network reachability manager for the socket address.
    
     @param address The socket address (`sockaddr_in`) used to evaluate network reachability.
    
     @return An initialized network reachability manager, actively monitoring the specified socket address.
     */

    class func managerForAddress(address: UnsafePointer<Void>) -> Self {
        var reachability = SCNetworkReachabilityCreateWithAddress(kCFAllocatorDefault, (address as! structsockaddr))
        var manager = self.init(reachability: reachability)
        manager.networkReachabilityAssociation = AFNetworkReachabilityForAddress
        return manager
    }
    /**
     Initializes an instance of a network reachability manager from the specified reachability object.
     
     @param reachability The reachability object to monitor.
     
     @return An initialized network reachability manager, actively monitoring the specified reachability.
     */

    override init(reachability: SCNetworkReachabilityRef) {
        super.init()
        if !self {
            return nil
        }
        self.networkReachability = reachability
        self.networkReachabilityStatus = .Unknown
    }
    ///--------------------------------------------------
    /// @name Starting & Stopping Reachability Monitoring
    ///--------------------------------------------------
    /**
     Starts monitoring for changes in network reachability status.
     */

    override func startMonitoring() {
        self.stopMonitoring()
        if !self.networkReachability {
            return
        }
        weak var weakSelf = self
        var callback = {(status: AFNetworkReachabilityStatus) -> Void in
                var strongSelf = weakSelf
                strongSelf.networkReachabilityStatus = status
                if strongSelf.networkReachabilityStatusBlock {
                    strongSelf.networkReachabilityStatusBlock(status)
                }
            }
        var context = [0, (callback as! Void), AFNetworkReachabilityRetainCallback, AFNetworkReachabilityReleaseCallback, nil]
        SCNetworkReachabilitySetCallback(self.networkReachability, AFNetworkReachabilityCallback, context)
        SCNetworkReachabilityScheduleWithRunLoop(self.networkReachability, CFRunLoopGetMain(), kCFRunLoopCommonModes)
        switch self.networkReachabilityAssociation {
            case AFNetworkReachabilityForName:
                break
            default:
                                dispatch_async(dispatch_get_global_queue(DISPATCH_QUEUE_PRIORITY_BACKGROUND, 0), {() -> Void in
                    var flags: SCNetworkReachabilityFlags
                    SCNetworkReachabilityGetFlags(self.networkReachability, flags)
                    var status = .ForFlags(flags)
                    dispatch_async(dispatch_get_main_queue(), {() -> Void in
                        callback(status)
                        var notificationCenter = NSNotificationCenter.defaultCenter()
                        notificationCenter.postNotificationName(AFNetworkingReachabilityDidChangeNotification, object: nil, userInfo: [AFNetworkingReachabilityNotificationStatusItem: (status)])
                    })
                })

        }

    }
    /**
     Stops monitoring for changes in network reachability status.
     */

    override func stopMonitoring() {
        if !self.networkReachability {
            return
        }
        SCNetworkReachabilityUnscheduleFromRunLoop(self.networkReachability, CFRunLoopGetMain(), kCFRunLoopCommonModes)
    }
    ///-------------------------------------------------
    /// @name Getting Localized Reachability Description
    ///-------------------------------------------------
    /**
     Returns a localized string representation of the current network reachability status.
     */

    func localizedNetworkReachabilityStatusString() -> String {
        return AFStringFromNetworkReachabilityStatus(self.networkReachabilityStatus)
    }
    ///---------------------------------------------------
    /// @name Setting Network Reachability Change Callback
    ///---------------------------------------------------
    /**
     Sets a callback to be executed when the network availability of the `baseURL` host changes.
    
     @param block A block object to be executed when the network availability of the `baseURL` host changes.. This block has no return value and takes a single argument which represents the various reachability states from the device to the `baseURL`.
     */

    func setReachabilityStatusChangeBlock(block: (status: AFNetworkReachabilityStatus) -> Void) {
        self.networkReachabilityStatusBlock = block
    }


    deinit {
        self.stopMonitoring()
        if networkReachability {
            CFRelease(networkReachability)
            self.networkReachability = nil
        }
    }
// MARK: -
// MARK: -
// MARK: -
// MARK: -
// MARK: - NSKeyValueObserving

    class func keyPathsForValuesAffectingValueForKey(key: String) -> Set<NSObject> {
        if (key == "reachable") || (key == "reachableViaWWAN") || (key == "reachableViaWiFi") {
            return Set<NSObject>(["networkReachabilityStatus"])
        }
        return super.keyPathsForValuesAffectingValueForKey(key)
    }

    var networkReachability = SCNetworkReachabilityRef()
    var networkReachabilityAssociation = AFNetworkReachabilityAssociation()
    var networkReachabilityStatus = AFNetworkReachabilityStatus()
    var networkReachabilityStatusBlock = AFNetworkReachabilityStatusBlock()
}
///----------------
/// @name Constants
///----------------
/**
 ## Network Reachability

 The following constants are provided by `AFNetworkReachabilityManager` as possible network reachability statuses.

 enum {
 AFNetworkReachabilityStatusUnknown,
 AFNetworkReachabilityStatusNotReachable,
 AFNetworkReachabilityStatusReachableViaWWAN,
 AFNetworkReachabilityStatusReachableViaWiFi,
 }

 `AFNetworkReachabilityStatusUnknown`
 The `baseURL` host reachability is not known.

 `AFNetworkReachabilityStatusNotReachable`
 The `baseURL` host cannot be reached.

 `AFNetworkReachabilityStatusReachableViaWWAN`
 The `baseURL` host can be reached via a cellular connection, such as EDGE or GPRS.

 `AFNetworkReachabilityStatusReachableViaWiFi`
 The `baseURL` host can be reached via a Wi-Fi connection.

 ### Keys for Notification UserInfo Dictionary

 Strings that are used as keys in a `userInfo` dictionary in a network reachability status change notification.

 `AFNetworkingReachabilityNotificationStatusItem`
 A key in the userInfo dictionary in a `AFNetworkingReachabilityDidChangeNotification` notification.
 The corresponding value is an `NSNumber` object representing the `AFNetworkReachabilityStatus` value for the current reachability status.
 */
///--------------------
/// @name Notifications
///--------------------
/**
 Posted when network reachability changes.
 This notification assigns no notification object. The `userInfo` dictionary contains an `NSNumber` object under the `AFNetworkingReachabilityNotificationStatusItem` key, representing the `AFNetworkReachabilityStatus` value for the current network reachability.

 @warning In order for network reachability to be monitored, include the `SystemConfiguration` framework in the active target's "Link Binary With Library" build phase, and add `#import <SystemConfiguration/SystemConfiguration.h>` to the header prefix of the project (`Prefix.pch`).
 */
let AFNetworkingReachabilityDidChangeNotification = ""

let AFNetworkingReachabilityNotificationStatusItem = ""

///--------------------
/// @name Functions
///--------------------
/**
 Returns a localized string representation of an `AFNetworkReachabilityStatus` value.
 */
var AFStringFromNetworkReachabilityStatus = ""

// AFNetworkReachabilityManager.m
// 
// Copyright (c) 2013-2014 AFNetworking (http://afnetworking.com)
//
// Permission is hereby granted, free of charge, to any person obtaining a copy
// of this software and associated documentation files (the "Software"), to deal
// in the Software without restriction, including without limitation the rights
// to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
// copies of the Software, and to permit persons to whom the Software is
// furnished to do so, subject to the following conditions:
// 
// The above copyright notice and this permission notice shall be included in
// all copies or substantial portions of the Software.
// 
// THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
// IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
// FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
// AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
// LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
// OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
// THE SOFTWARE.
import netinet
import netinet6
import arpa
import ifaddrs
import netdb
let AFNetworkingReachabilityDidChangeNotification = "com.alamofire.networking.reachability.change"

let AFNetworkingReachabilityNotificationStatusItem = "AFNetworkingReachabilityNotificationStatusItem"

typealias AFNetworkReachabilityStatusBlock = (status: AFNetworkReachabilityStatus) -> Void
enum AFNetworkReachabilityAssociation : Int {
    case AFNetworkReachabilityForAddress = 1
    case AFNetworkReachabilityForAddressPair = 2
    case AFNetworkReachabilityForName = 3
}

func AFStringFromNetworkReachabilityStatus(AFNetworkReachabilityStatus status) -> String {
    switch status {
        case .NotReachable:
            return NSLocalizedStringFromTable("Not Reachable", "AFNetworking")
        case .ReachableViaWWAN:
            return NSLocalizedStringFromTable("Reachable via WWAN", "AFNetworking")
        case .ReachableViaWiFi:
            return NSLocalizedStringFromTable("Reachable via WiFi", "AFNetworking")
        default:
            return NSLocalizedStringFromTable("Unknown", "AFNetworking")
    }

}

func .ForFlags(SCNetworkReachabilityFlags flags) -> AFNetworkReachabilityStatus {
    var isReachable = ((flags & kSCNetworkReachabilityFlagsReachable) != 0)
    var needsConnection = ((flags & kSCNetworkReachabilityFlagsConnectionRequired) != 0)
    var canConnectionAutomatically = ((flags & kSCNetworkReachabilityFlagsConnectionOnDemand) != 0) || ((flags & kSCNetworkReachabilityFlagsConnectionOnTraffic) != 0)
    var canConnectWithoutUserInteraction = (canConnectionAutomatically && (flags & kSCNetworkReachabilityFlagsInterventionRequired) == 0)
    var isNetworkReachable = (isReachable && (!needsConnection || canConnectWithoutUserInteraction))
    var status = .Unknown
    if isNetworkReachable == false {
        status = .NotReachable
    }
    else if (flags & kSCNetworkReachabilityFlagsIsWWAN) != 0 {
        status = .ReachableViaWWAN
    }
    else {
        status = .ReachableViaWiFi
    }

    return status
}

func AFNetworkReachabilityCallback(SCNetworkReachabilityRef __unused target, SCNetworkReachabilityFlags flags, Void) {
    var status = .ForFlags(flags)
    var block = (info as! AFNetworkReachabilityStatusBlock)
    if block {
        block(status)
    }
    dispatch_async(dispatch_get_main_queue(), {() -> Void in
        var notificationCenter = NSNotificationCenter.defaultCenter()
        notificationCenter.postNotificationName(AFNetworkingReachabilityDidChangeNotification, object: nil, userInfo: [AFNetworkingReachabilityNotificationStatusItem: (status)])
    })
}

let AFNetworkReachabilityRetainCallback

Block_copy(info)

func AFNetworkReachabilityReleaseCallback(info: Void) {
    if info {
        Block_release(info)
    }
}