// AFHTTPSessionManager.h
//
// Copyright (c) 2013-2014 AFNetworking (http://afnetworking.com)
//
// Permission is hereby granted, free of charge, to any person obtaining a copy
// of this software and associated documentation files (the "Software"), to deal
// in the Software without restriction, including without limitation the rights
// to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
// copies of the Software, and to permit persons to whom the Software is
// furnished to do so, subject to the following conditions:
//
// The above copyright notice and this permission notice shall be included in
// all copies or substantial portions of the Software.
//
// THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
// IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
// FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
// AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
// LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
// OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
// THE SOFTWARE.
import Foundation
import SystemConfiguration
import Availability
#if __IPHONE_OS_VERSION_MIN_REQUIRED
import MobileCoreServices
#else
import CoreServices
#endif
/**
 `AFHTTPSessionManager` is a subclass of `AFURLSessionManager` with convenience methods for making HTTP requests. When a `baseURL` is provided, requests made with the `GET` / `POST` / et al. convenience methods can be made with relative paths.
 
 ## Subclassing Notes
 
 Developers targeting iOS 7 or Mac OS X 10.9 or later that deal extensively with a web service are encouraged to subclass `AFHTTPSessionManager`, providing a class method that returns a shared singleton object on which authentication and other configuration can be shared across the application.
 
 For developers targeting iOS 6 or Mac OS X 10.8 or earlier, `AFHTTPRequestOperationManager` may be used to similar effect.

 ## Methods to Override

 To change the behavior of all data task operation construction, which is also used in the `GET` / `POST` / et al. convenience methods, override `dataTaskWithRequest:completionHandler:`.
 
 ## Serialization
 
 Requests created by an HTTP client will contain default headers and encode parameters according to the `requestSerializer` property, which is an object conforming to `<AFURLRequestSerialization>`.
 
 Responses received from the server are automatically validated and serialized by the `responseSerializers` property, which is an object conforming to `<AFURLResponseSerialization>`

 ## URL Construction Using Relative Paths

 For HTTP convenience methods, the request serializer constructs URLs from the path relative to the `-baseURL`, using `NSURL +URLWithString:relativeToURL:`, when provided. If `baseURL` is `nil`, `path` needs to resolve to a valid `NSURL` object using `NSURL +URLWithString:`.
 
 Below are a few examples of how `baseURL` and relative paths interact:

    NSURL *baseURL = [NSURL URLWithString:@"http://example.com/v1/"];
    [NSURL URLWithString:@"foo" relativeToURL:baseURL];                  // http://example.com/v1/foo
    [NSURL URLWithString:@"foo?bar=baz" relativeToURL:baseURL];          // http://example.com/v1/foo?bar=baz
    [NSURL URLWithString:@"/foo" relativeToURL:baseURL];                 // http://example.com/foo
    [NSURL URLWithString:@"foo/" relativeToURL:baseURL];                 // http://example.com/v1/foo
    [NSURL URLWithString:@"/foo/" relativeToURL:baseURL];                // http://example.com/foo/
    [NSURL URLWithString:@"http://example2.com/" relativeToURL:baseURL]; // http://example2.com/

 Also important to note is that a trailing slash will be added to any `baseURL` without one. This would otherwise cause unexpected behavior when constructing URLs using paths without a leading slash.
 */
#if (defined(__IPHONE_OS_VERSION_MAX_ALLOWED) && __IPHONE_OS_VERSION_MAX_ALLOWED >= 70000) || (defined(__MAC_OS_X_VERSION_MAX_ALLOWED) && __MAC_OS_X_VERSION_MAX_ALLOWED >= 1090)
class AFHTTPSessionManager: AFURLSessionManager, NSSecureCoding, NSCopying {
    /**
     The URL used to monitor reachability, and construct requests from relative paths in methods like `requestWithMethod:URLString:parameters:`, and the `GET` / `POST` / et al. convenience methods.
     */
    private(set) var baseURL: NSURL!
    /**
     Requests created with `requestWithMethod:URLString:parameters:` & `multipartFormRequestWithMethod:URLString:parameters:constructingBodyWithBlock:` are constructed with a set of default headers using a parameter serialization specified by this property. By default, this is set to an instance of `AFHTTPRequestSerializer`, which serializes query string parameters for `GET`, `HEAD`, and `DELETE` requests, or otherwise URL-form-encodes HTTP message bodies.
     
     @warning `requestSerializer` must not be `nil`.
     */
    weak var requestSerializer: AFURLRequestSerialization? {
        get {
            // TODO: add getter implementation
        }
        set(requestSerializer) {
            NSParameterAssert(requestSerializer)
            self.requestSerializer = requestSerializer
        }
    }
    /**
     Responses sent from the server in data tasks created with `dataTaskWithRequest:success:failure:` and run using the `GET` / `POST` / et al. convenience methods are automatically validated and serialized by the response serializer. By default, this property is set to an instance of `AFJSONResponseSerializer`.
    
     @warning `responseSerializer` must not be `nil`.
     */
    weak var responseSerializer: AFURLResponseSerialization? {
        get {
            // TODO: add getter implementation
        }
        set(responseSerializer) {
            NSParameterAssert(responseSerializer)
            super.responseSerializer = responseSerializer
        }
    }
    ///---------------------
    /// @name Initialization
    ///---------------------
    /**
     Creates and returns an `AFHTTPSessionManager` object.
     */

    class func manager() -> Self {
        return self.self(baseURL: nil)
    }
    /**
     Initializes an `AFHTTPSessionManager` object with the specified base URL.
     
     @param url The base URL for the HTTP client.
    
     @return The newly-initialized HTTP client
     */

    convenience override init(baseURL url: NSURL) {
        return self.init(baseURL: url, sessionConfiguration: nil)
    }
    /**
     Initializes an `AFHTTPSessionManager` object with the specified base URL.
    
     This is the designated initializer.
    
     @param url The base URL for the HTTP client.
     @param configuration The configuration used to create the managed session.
    
     @return The newly-initialized HTTP client
     */

    override init(baseURL url: NSURL, sessionConfiguration configuration: NSURLSessionConfiguration) {
        super.init(sessionConfiguration: configuration)
        if !self {
            return nil
        }
        // Ensure terminal slash for baseURL path, so that NSURL +URLWithString:relativeToURL: works as expected
        if url.path!.length > 0 && !url.absoluteString.hasSuffix("/") {
            url = url.URLByAppendingPathComponent("")
        }
        self.baseURL! = url
        self.requestSerializer = AFHTTPRequestSerializer.serializer()
        self.responseSerializer = AFJSONResponseSerializer.serializer()
    }
    ///---------------------------
    /// @name Making HTTP Requests
    ///---------------------------
    /**
     Creates and runs an `NSURLSessionDataTask` with a `GET` request.
    
     @param URLString The URL string used to create the request URL.
     @param parameters The parameters to be encoded according to the client request serializer.
     @param success A block object to be executed when the task finishes successfully. This block has no return value and takes two arguments: the data task, and the response object created by the client response serializer.
     @param failure A block object to be executed when the task finishes unsuccessfully, or that finishes successfully, but encountered an error while parsing the response data. This block has no return value and takes a two arguments: the data task and the error describing the network or parsing error that occurred.
    
     @see -dataTaskWithRequest:completionHandler:
     */

    func GET(URLString: String, parameters: AnyObject, success: (task: NSURLSessionDataTask, responseObject: AnyObject) -> Void, failure: (task: NSURLSessionDataTask, error: NSError) -> Void) -> NSURLSessionDataTask {
        var request = try! self.requestSerializer(method: "GET", URLString: NSURL(string: URLString, relativeToURL: self.baseURL!)!.absoluteString, parameters: parameters)
        var task = self.dataTaskWithRequest(request, completionHandler: {(response: NSURLResponse, responseObject: AnyObject, error: NSError) -> Void in
                if error {
                    if failure {
                        failure(task, error)
                    }
                }
                else {
                    if success {
                        success(task, responseObject)
                    }
                }
            })
        task.resume()
        return task
    }
    /**
     Creates and runs an `NSURLSessionDataTask` with a `HEAD` request.
    
     @param URLString The URL string used to create the request URL.
     @param parameters The parameters to be encoded according to the client request serializer.
     @param success A block object to be executed when the task finishes successfully. This block has no return value and takes a single arguments: the data task.
     @param failure A block object to be executed when the task finishes unsuccessfully, or that finishes successfully, but encountered an error while parsing the response data. This block has no return value and takes a two arguments: the data task and the error describing the network or parsing error that occurred.
    
     @see -dataTaskWithRequest:completionHandler:
     */

    func HEAD(URLString: String, parameters: AnyObject, success: (task: NSURLSessionDataTask) -> Void, failure: (task: NSURLSessionDataTask, error: NSError) -> Void) -> NSURLSessionDataTask {
        var request = try! self.requestSerializer(method: "HEAD", URLString: NSURL(string: URLString, relativeToURL: self.baseURL!)!.absoluteString, parameters: parameters)
        var task = self.dataTaskWithRequest(request, completionHandler: {(response: NSURLResponse, responseObject: id __unused, error: NSError) -> Void in
                if error {
                    if failure {
                        failure(task, error)
                    }
                }
                else {
                    if success {
                        success(task)
                    }
                }
            })
        task.resume()
        return task
    }
    /**
     Creates and runs an `NSURLSessionDataTask` with a `POST` request.
    
     @param URLString The URL string used to create the request URL.
     @param parameters The parameters to be encoded according to the client request serializer.
     @param success A block object to be executed when the task finishes successfully. This block has no return value and takes two arguments: the data task, and the response object created by the client response serializer.
     @param failure A block object to be executed when the task finishes unsuccessfully, or that finishes successfully, but encountered an error while parsing the response data. This block has no return value and takes a two arguments: the data task and the error describing the network or parsing error that occurred.
    
     @see -dataTaskWithRequest:completionHandler:
     */

    func POST(URLString: String, parameters: AnyObject, success: (task: NSURLSessionDataTask, responseObject: AnyObject) -> Void, failure: (task: NSURLSessionDataTask, error: NSError) -> Void) -> NSURLSessionDataTask {
        var request = try! self.requestSerializer(method: "POST", URLString: NSURL(string: URLString, relativeToURL: self.baseURL!)!.absoluteString, parameters: parameters)
        var task = self.dataTaskWithRequest(request, completionHandler: {(response: NSURLResponse, responseObject: AnyObject, error: NSError) -> Void in
                if error {
                    if failure {
                        failure(task, error)
                    }
                }
                else {
                    if success {
                        success(task, responseObject)
                    }
                }
            })
        task.resume()
        return task
    }
    /**
     Creates and runs an `NSURLSessionDataTask` with a multipart `POST` request.
    
     @param URLString The URL string used to create the request URL.
     @param parameters The parameters to be encoded according to the client request serializer.
     @param block A block that takes a single argument and appends data to the HTTP body. The block argument is an object adopting the `AFMultipartFormData` protocol.
     @param success A block object to be executed when the task finishes successfully. This block has no return value and takes two arguments: the data task, and the response object created by the client response serializer.
     @param failure A block object to be executed when the task finishes unsuccessfully, or that finishes successfully, but encountered an error while parsing the response data. This block has no return value and takes a two arguments: the data task and the error describing the network or parsing error that occurred.
    
     @see -dataTaskWithRequest:completionHandler:
     */

    func POST(URLString: String, parameters: AnyObject, constructingBodyWithBlock block: (formData: AFMultipartFormData) -> Void, success: (task: NSURLSessionDataTask, responseObject: AnyObject) -> Void, failure: (task: NSURLSessionDataTask, error: NSError) -> Void) -> NSURLSessionDataTask {
        var request = try! self.requestSerializer.multipartFormRequestWithMethod("POST", URLString: NSURL(string: URLString, relativeToURL: self.baseURL!)!.absoluteString, parameters: parameters, constructingBodyWithBlock: block)
        var task = self.uploadTaskWithStreamedRequest(request, progress: nil, completionHandler: {(response: NSURLResponse, responseObject: AnyObject, error: NSError) -> Void in
                if error {
                    if failure {
                        failure(task, error)
                    }
                }
                else {
                    if success {
                        success(task, responseObject)
                    }
                }
            })
        task.resume()
        return task
    }
    /**
     Creates and runs an `NSURLSessionDataTask` with a `PUT` request.
    
     @param URLString The URL string used to create the request URL.
     @param parameters The parameters to be encoded according to the client request serializer.
     @param success A block object to be executed when the task finishes successfully. This block has no return value and takes two arguments: the data task, and the response object created by the client response serializer.
     @param failure A block object to be executed when the task finishes unsuccessfully, or that finishes successfully, but encountered an error while parsing the response data. This block has no return value and takes a two arguments: the data task and the error describing the network or parsing error that occurred.
    
     @see -dataTaskWithRequest:completionHandler:
     */

    func PUT(URLString: String, parameters: AnyObject, success: (task: NSURLSessionDataTask, responseObject: AnyObject) -> Void, failure: (task: NSURLSessionDataTask, error: NSError) -> Void) -> NSURLSessionDataTask {
        var request = try! self.requestSerializer(method: "PUT", URLString: NSURL(string: URLString, relativeToURL: self.baseURL!)!.absoluteString, parameters: parameters)
        var task = self.dataTaskWithRequest(request, completionHandler: {(response: NSURLResponse, responseObject: AnyObject, error: NSError) -> Void in
                if error {
                    if failure {
                        failure(task, error)
                    }
                }
                else {
                    if success {
                        success(task, responseObject)
                    }
                }
            })
        task.resume()
        return task
    }
    /**
     Creates and runs an `NSURLSessionDataTask` with a `PATCH` request.
    
     @param URLString The URL string used to create the request URL.
     @param parameters The parameters to be encoded according to the client request serializer.
     @param success A block object to be executed when the task finishes successfully. This block has no return value and takes two arguments: the data task, and the response object created by the client response serializer.
     @param failure A block object to be executed when the task finishes unsuccessfully, or that finishes successfully, but encountered an error while parsing the response data. This block has no return value and takes a two arguments: the data task and the error describing the network or parsing error that occurred.
    
     @see -dataTaskWithRequest:completionHandler:
     */

    func PATCH(URLString: String, parameters: AnyObject, success: (task: NSURLSessionDataTask, responseObject: AnyObject) -> Void, failure: (task: NSURLSessionDataTask, error: NSError) -> Void) -> NSURLSessionDataTask {
        var request = try! self.requestSerializer(method: "PATCH", URLString: NSURL(string: URLString, relativeToURL: self.baseURL!)!.absoluteString, parameters: parameters)
        var task = self.dataTaskWithRequest(request, completionHandler: {(response: NSURLResponse, responseObject: AnyObject, error: NSError) -> Void in
                if error {
                    if failure {
                        failure(task, error)
                    }
                }
                else {
                    if success {
                        success(task, responseObject)
                    }
                }
            })
        task.resume()
        return task
    }
    /**
     Creates and runs an `NSURLSessionDataTask` with a `DELETE` request.
    
     @param URLString The URL string used to create the request URL.
     @param parameters The parameters to be encoded according to the client request serializer.
     @param success A block object to be executed when the task finishes successfully. This block has no return value and takes two arguments: the data task, and the response object created by the client response serializer.
     @param failure A block object to be executed when the task finishes unsuccessfully, or that finishes successfully, but encountered an error while parsing the response data. This block has no return value and takes a two arguments: the data task and the error describing the network or parsing error that occurred.
    
     @see -dataTaskWithRequest:completionHandler:
     */

    func DELETE(URLString: String, parameters: AnyObject, success: (task: NSURLSessionDataTask, responseObject: AnyObject) -> Void, failure: (task: NSURLSessionDataTask, error: NSError) -> Void) -> NSURLSessionDataTask {
        var request = try! self.requestSerializer(method: "DELETE", URLString: NSURL(string: URLString, relativeToURL: self.baseURL!)!.absoluteString, parameters: parameters)
        var task = self.dataTaskWithRequest(request, completionHandler: {(response: NSURLResponse, responseObject: AnyObject, error: NSError) -> Void in
                if error {
                    if failure {
                        failure(task, error)
                    }
                }
                else {
                    if success {
                        success(task, responseObject)
                    }
                }
            })
        task.resume()
        return task
    }


    convenience override init() {
        return self.init(baseURL: nil)
    }

    convenience override init(sessionConfiguration configuration: NSURLSessionConfiguration) {
        return self.init(baseURL: nil, sessionConfiguration: configuration)
    }
// MARK: -
#if _SYSTEMCONFIGURATION_H
#endif
// MARK: -
// MARK: - NSObject

    override func description() -> String {
        return "<\(NSStringFromClass(self.self)): \(self), baseURL: \(self.baseURL!.absoluteString), session: \(self.session), operationQueue: \(self.operationQueue!)>"
    }
// MARK: - NSSecureCoding

    class func supportsSecureCoding() -> Bool {
        return true
    }

    convenience required init?(coder decoder: NSCoder) {
        var baseURL = decoder.decodeObjectOfClass(NSURL.self, forKey: NSStringFromSelector(#selector(self.baseURL)))!
        var configuration = decoder.decodeObjectOfClass(NSURLSessionConfiguration.self, forKey: "sessionConfiguration")!
        if !configuration {
            var configurationIdentifier = decoder.decodeObjectOfClass(String.self, forKey: "identifier")!
            if configurationIdentifier != "" {
#if (defined(__IPHONE_OS_VERSION_MAX_ALLOWED) && __IPHONE_OS_VERSION_MAX_ALLOWED >= 80000) || (defined(__MAC_OS_X_VERSION_MAX_ALLOWED) && __MAC_OS_X_VERSION_MAX_ALLOWED >= 1100)
                configuration = NSURLSessionConfiguration.backgroundSessionConfigurationWithIdentifier(configurationIdentifier)
#else
                configuration = NSURLSessionConfiguration.backgroundSessionConfiguration(configurationIdentifier)
#endif
            }
        }
        self.init(baseURL: baseURL, sessionConfiguration: configuration)
        if !self {
            return nil
        }
        self.requestSerializer = decoder.decodeObjectOfClass(AFHTTPRequestSerializer.self, forKey: NSStringFromSelector(#selector(self.requestSerializer)))!
        self.responseSerializer = decoder.decodeObjectOfClass(AFHTTPResponseSerializer.self, forKey: NSStringFromSelector(#selector(self.responseSerializer)))!
    }

    override func encodeWithCoder(coder: NSCoder) {
        super.encodeWithCoder(coder)
        coder.encodeObject(self.baseURL!, forKey: NSStringFromSelector(#selector(self.baseURL)))
        if self.session.configuration is NSCoding {
            coder.encodeObject(self.session.configuration, forKey: "sessionConfiguration")
        }
        else {
            coder.encodeObject(self.session.configuration.identifier, forKey: "identifier")
        }
        coder.encodeObject(self.requestSerializer, forKey: NSStringFromSelector(#selector(self.requestSerializer)))
        coder.encodeObject(self.responseSerializer, forKey: NSStringFromSelector(#selector(self.responseSerializer)))
    }
// MARK: - NSCopying

    override func copyWithZone(zone: NSZone) -> AnyObject {
        var HTTPClient = self.self.allocWithZone(zone)(baseURL: self.baseURL!, sessionConfiguration: self.session.configuration)
        HTTPClient.requestSerializer = self.requestSerializer.copyWithZone(zone)
        HTTPClient.responseSerializer = self.responseSerializer.copyWithZone(zone)
        return HTTPClient
    }

    var baseURL: NSURL!
}
#endif
// AFHTTPSessionManager.m
//
// Copyright (c) 2013-2014 AFNetworking (http://afnetworking.com)
//
// Permission is hereby granted, free of charge, to any person obtaining a copy
// of this software and associated documentation files (the "Software"), to deal
// in the Software without restriction, including without limitation the rights
// to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
// copies of the Software, and to permit persons to whom the Software is
// furnished to do so, subject to the following conditions:
//
// The above copyright notice and this permission notice shall be included in
// all copies or substantial portions of the Software.
//
// THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
// IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
// FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
// AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
// LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
// OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
// THE SOFTWARE.
#if (defined(__IPHONE_OS_VERSION_MAX_ALLOWED) && __IPHONE_OS_VERSION_MAX_ALLOWED >= 70000) || (defined(__MAC_OS_X_VERSION_MAX_ALLOWED) && __MAC_OS_X_VERSION_MAX_ALLOWED >= 1090)
import Availability
import Security
#if _SYSTEMCONFIGURATION_H
import netinet
import netinet6
import arpa
import ifaddrs
import netdb
#endif
#if defined(__IPHONE_OS_VERSION_MIN_REQUIRED)
import UIKit
#endif