// AFHTTPRequestOperationManager.h
//
// Copyright (c) 2013-2014 AFNetworking (http://afnetworking.com)
//
// Permission is hereby granted, free of charge, to any person obtaining a copy
// of this software and associated documentation files (the "Software"), to deal
// in the Software without restriction, including without limitation the rights
// to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
// copies of the Software, and to permit persons to whom the Software is
// furnished to do so, subject to the following conditions:
//
// The above copyright notice and this permission notice shall be included in
// all copies or substantial portions of the Software.
//
// THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
// IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
// FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
// AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
// LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
// OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
// THE SOFTWARE.
import Foundation
import SystemConfiguration
import Availability
import MobileCoreServices
import Foundation
import Availability
import Security
import UIKit
/**
 `AFHTTPRequestOperationManager` encapsulates the common patterns of communicating with a web application over HTTP, including request creation, response serialization, network reachability monitoring, and security, as well as request operation management.

 ## Subclassing Notes
 
 Developers targeting iOS 7 or Mac OS X 10.9 or later that deal extensively with a web service are encouraged to subclass `AFHTTPSessionManager`, providing a class method that returns a shared singleton object on which authentication and other configuration can be shared across the application.
 
 For developers targeting iOS 6 or Mac OS X 10.8 or earlier, `AFHTTPRequestOperationManager` may be used to similar effect.
 
 ## Methods to Override

 To change the behavior of all request operation construction for an `AFHTTPRequestOperationManager` subclass, override `HTTPRequestOperationWithRequest:success:failure`.

 ## Serialization
 
 Requests created by an HTTP client will contain default headers and encode parameters according to the `requestSerializer` property, which is an object conforming to `<AFURLRequestSerialization>`. 
 
 Responses received from the server are automatically validated and serialized by the `responseSerializers` property, which is an object conforming to `<AFURLResponseSerialization>`

 ## URL Construction Using Relative Paths

 For HTTP convenience methods, the request serializer constructs URLs from the path relative to the `-baseURL`, using `NSURL +URLWithString:relativeToURL:`, when provided. If `baseURL` is `nil`, `path` needs to resolve to a valid `NSURL` object using `NSURL +URLWithString:`.

 Below are a few examples of how `baseURL` and relative paths interact:

    NSURL *baseURL = [NSURL URLWithString:@"http://example.com/v1/"];
    [NSURL URLWithString:@"foo" relativeToURL:baseURL];                  // http://example.com/v1/foo
    [NSURL URLWithString:@"foo?bar=baz" relativeToURL:baseURL];          // http://example.com/v1/foo?bar=baz
    [NSURL URLWithString:@"/foo" relativeToURL:baseURL];                 // http://example.com/foo
    [NSURL URLWithString:@"foo/" relativeToURL:baseURL];                 // http://example.com/v1/foo
    [NSURL URLWithString:@"/foo/" relativeToURL:baseURL];                // http://example.com/foo/
    [NSURL URLWithString:@"http://example2.com/" relativeToURL:baseURL]; // http://example2.com/

 Also important to note is that a trailing slash will be added to any `baseURL` without one. This would otherwise cause unexpected behavior when constructing URLs using paths without a leading slash.

 ## Network Reachability Monitoring

 Network reachability status and change monitoring is available through the `reachabilityManager` property. Applications may choose to monitor network reachability conditions in order to prevent or suspend any outbound requests. See `AFNetworkReachabilityManager` for more details.

 ## NSSecureCoding & NSCopying Caveats

 `AFHTTPRequestOperationManager` conforms to the `NSSecureCoding` and `NSCopying` protocols, allowing operations to be archived to disk, and copied in memory, respectively. There are a few minor caveats to keep in mind, however:

 - Archives and copies of HTTP clients will be initialized with an empty operation queue.
 - NSSecureCoding cannot serialize / deserialize block properties, so an archive of an HTTP client will not include any reachability callback block that may be set.
 */
class AFHTTPRequestOperationManager: NSObject, NSSecureCoding, NSCopying {
    /**
     The URL used to monitor reachability, and construct requests from relative paths in methods like `requestWithMethod:URLString:parameters:`, and the `GET` / `POST` / et al. convenience methods.
     */
    private(set) var baseURL: NSURL!
    /**
     Requests created with `requestWithMethod:URLString:parameters:` & `multipartFormRequestWithMethod:URLString:parameters:constructingBodyWithBlock:` are constructed with a set of default headers using a parameter serialization specified by this property. By default, this is set to an instance of `AFHTTPRequestSerializer`, which serializes query string parameters for `GET`, `HEAD`, and `DELETE` requests, or otherwise URL-form-encodes HTTP message bodies.
     
     @warning `requestSerializer` must not be `nil`.
     */
    weak var requestSerializer: AFURLRequestSerialization? {
        get {
            // TODO: add getter implementation
        }
        set(requestSerializer) {
            NSParameterAssert(requestSerializer)
            self.requestSerializer = requestSerializer
        }
    }
    /**
     Responses sent from the server in data tasks created with `dataTaskWithRequest:success:failure:` and run using the `GET` / `POST` / et al. convenience methods are automatically validated and serialized by the response serializer. By default, this property is set to a JSON serializer, which serializes data from responses with a `application/json` MIME type, and falls back to the raw data object. The serializer validates the status code to be in the `2XX` range, denoting success. If the response serializer generates an error in `-responseObjectForResponse:data:error:`, the `failure` callback of the session task or request operation will be executed; otherwise, the `success` callback will be executed.
    
     @warning `responseSerializer` must not be `nil`.
     */
    weak var responseSerializer: AFURLResponseSerialization? {
        get {
            // TODO: add getter implementation
        }
        set(responseSerializer) {
            NSParameterAssert(responseSerializer)
            self.responseSerializer = responseSerializer
        }
    }
    /**
     The operation queue on which request operations are scheduled and run.
     */
    var operationQueue: NSOperationQueue!
    ///-------------------------------
    /// @name Managing URL Credentials
    ///-------------------------------
    /**
     Whether request operations should consult the credential storage for authenticating the connection. `YES` by default.
    
     @see AFURLConnectionOperation -shouldUseCredentialStorage
     */
    var shouldUseCredentialStorage = false
    /**
     The credential used by request operations for authentication challenges.
    
     @see AFURLConnectionOperation -credential
     */
    var credential: NSURLCredential!
    ///-------------------------------
    /// @name Managing Security Policy
    ///-------------------------------
    /**
     The security policy used by created request operations to evaluate server trust for secure connections. `AFHTTPRequestOperationManager` uses the `defaultPolicy` unless otherwise specified.
     */
    var securityPolicy: AFSecurityPolicy!
    ///------------------------------------
    /// @name Managing Network Reachability
    ///------------------------------------
    /**
     The network reachability manager. `AFHTTPRequestOperationManager` uses the `sharedManager` by default.
     */
    var reachabilityManager: AFNetworkReachabilityManager!
    ///-------------------------------
    /// @name Managing Callback Queues
    ///-------------------------------
    /**
     The dispatch queue for the `completionBlock` of request operations. If `NULL` (default), the main queue is used.
     */
    var completionQueue = dispatch_queue_t()
    /**
     The dispatch group for the `completionBlock` of request operations. If `NULL` (default), a private dispatch group is used.
     */
    var completionGroup = dispatch_group_t()
    ///---------------------------------------------
    /// @name Creating and Initializing HTTP Clients
    ///---------------------------------------------
    /**
     Creates and returns an `AFHTTPRequestOperationManager` object.
     */

    class func manager() -> Self {
        return self.init(baseURL: nil)
    }
    /**
     Initializes an `AFHTTPRequestOperationManager` object with the specified base URL.
     
     This is the designated initializer.
     
     @param url The base URL for the HTTP client.
    
     @return The newly-initialized HTTP client
     */

    override init(baseURL url: NSURL) {
        super.init()
        if !self {
            return nil
        }
        // Ensure terminal slash for baseURL path, so that NSURL +URLWithString:relativeToURL: works as expected
        if url.path!.length > 0 && !url.absoluteString.hasSuffix("/") {
            url = url.URLByAppendingPathComponent("")
        }
        self.baseURL! = url
        self.requestSerializer = AFHTTPRequestSerializer.serializer()
        self.responseSerializer = AFJSONResponseSerializer.serializer()
        self.securityPolicy = AFSecurityPolicy.defaultPolicy()
        self.reachabilityManager = AFNetworkReachabilityManager.sharedManager()
        self.operationQueue! = NSOperationQueue()
        self.shouldUseCredentialStorage = true
    }
    ///---------------------------------------
    /// @name Managing HTTP Request Operations
    ///---------------------------------------
    /**
     Creates an `AFHTTPRequestOperation`, and sets the response serializers to that of the HTTP client.
    
     @param request The request object to be loaded asynchronously during execution of the operation.
     @param success A block object to be executed when the request operation finishes successfully. This block has no return value and takes two arguments: the created request operation and the object created from the response data of request.
     @param failure A block object to be executed when the request operation finishes unsuccessfully, or that finishes successfully, but encountered an error while parsing the response data. This block has no return value and takes two arguments:, the created request operation and the `NSError` object describing the network or parsing error that occurred.
     */

    func HTTPRequestOperationWithRequest(request: NSURLRequest, success: (operation: AFHTTPRequestOperation, responseObject: AnyObject) -> Void, failure: (operation: AFHTTPRequestOperation, error: NSError) -> Void) -> AFHTTPRequestOperation {
        var operation = AFHTTPRequestOperation(request: request)
        operation.responseSerializer = self.responseSerializer
        operation.shouldUseCredentialStorage = self.shouldUseCredentialStorage
        operation.credential = self.credential
        operation.securityPolicy = self.securityPolicy
        operation.setCompletionBlockWithSuccess(success, failure: failure)
        operation.completionQueue = self.completionQueue
        operation.completionGroup = self.completionGroup
        return operation
    }
    ///---------------------------
    /// @name Making HTTP Requests
    ///---------------------------
    /**
     Creates and runs an `AFHTTPRequestOperation` with a `GET` request.
    
     @param URLString The URL string used to create the request URL.
     @param parameters The parameters to be encoded according to the client request serializer.
     @param success A block object to be executed when the request operation finishes successfully. This block has no return value and takes two arguments: the request operation, and the response object created by the client response serializer.
     @param failure A block object to be executed when the request operation finishes unsuccessfully, or that finishes successfully, but encountered an error while parsing the response data. This block has no return value and takes a two arguments: the request operation and the error describing the network or parsing error that occurred.
    
     @see -HTTPRequestOperationWithRequest:success:failure:
     */

    func GET(URLString: String, parameters: AnyObject, success: (operation: AFHTTPRequestOperation, responseObject: AnyObject) -> Void, failure: (operation: AFHTTPRequestOperation, error: NSError) -> Void) -> AFHTTPRequestOperation {
        var request = try! self.requestSerializer(method: "GET", URLString: NSURL(string: URLString, relativeToURL: self.baseURL!)!.absoluteString, parameters: parameters)
        var operation = self.HTTPRequestOperationWithRequest(request, success: success, failure: failure)
        self.operationQueue!.addOperation(operation)
        return operation
    }
    /**
     Creates and runs an `AFHTTPRequestOperation` with a `HEAD` request.
    
     @param URLString The URL string used to create the request URL.
     @param parameters The parameters to be encoded according to the client request serializer.
     @param success A block object to be executed when the request operation finishes successfully. This block has no return value and takes a single arguments: the request operation.
     @param failure A block object to be executed when the request operation finishes unsuccessfully, or that finishes successfully, but encountered an error while parsing the response data. This block has no return value and takes a two arguments: the request operation and the error describing the network or parsing error that occurred.
    
     @see -HTTPRequestOperationWithRequest:success:failure:
     */

    func HEAD(URLString: String, parameters: AnyObject, success: (operation: AFHTTPRequestOperation) -> Void, failure: (operation: AFHTTPRequestOperation, error: NSError) -> Void) -> AFHTTPRequestOperation {
        var request = try! self.requestSerializer(method: "HEAD", URLString: NSURL(string: URLString, relativeToURL: self.baseURL!)!.absoluteString, parameters: parameters)
        var operation = self.HTTPRequestOperationWithRequest(request, success: {(requestOperation: AFHTTPRequestOperation, responseObject: id) -> Void in
                if success {
                    success(requestOperation)
                }
            }, failure: failure)
        self.operationQueue!.addOperation(operation)
        return operation
    }
    /**
     Creates and runs an `AFHTTPRequestOperation` with a `POST` request.
    
     @param URLString The URL string used to create the request URL.
     @param parameters The parameters to be encoded according to the client request serializer.
     @param success A block object to be executed when the request operation finishes successfully. This block has no return value and takes two arguments: the request operation, and the response object created by the client response serializer.
     @param failure A block object to be executed when the request operation finishes unsuccessfully, or that finishes successfully, but encountered an error while parsing the response data. This block has no return value and takes a two arguments: the request operation and the error describing the network or parsing error that occurred.
    
     @see -HTTPRequestOperationWithRequest:success:failure:
     */

    func POST(URLString: String, parameters: AnyObject, success: (operation: AFHTTPRequestOperation, responseObject: AnyObject) -> Void, failure: (operation: AFHTTPRequestOperation, error: NSError) -> Void) -> AFHTTPRequestOperation {
        var request = try! self.requestSerializer(method: "POST", URLString: NSURL(string: URLString, relativeToURL: self.baseURL!)!.absoluteString, parameters: parameters)
        var operation = self.HTTPRequestOperationWithRequest(request, success: success, failure: failure)
        self.operationQueue!.addOperation(operation)
        return operation
    }
    /**
     Creates and runs an `AFHTTPRequestOperation` with a multipart `POST` request.
    
     @param URLString The URL string used to create the request URL.
     @param parameters The parameters to be encoded according to the client request serializer.
     @param block A block that takes a single argument and appends data to the HTTP body. The block argument is an object adopting the `AFMultipartFormData` protocol.
     @param success A block object to be executed when the request operation finishes successfully. This block has no return value and takes two arguments: the request operation, and the response object created by the client response serializer.
     @param failure A block object to be executed when the request operation finishes unsuccessfully, or that finishes successfully, but encountered an error while parsing the response data. This block has no return value and takes a two arguments: the request operation and the error describing the network or parsing error that occurred.
    
     @see -HTTPRequestOperationWithRequest:success:failure:
     */

    func POST(URLString: String, parameters: AnyObject, constructingBodyWithBlock block: (formData: AFMultipartFormData) -> Void, success: (operation: AFHTTPRequestOperation, responseObject: AnyObject) -> Void, failure: (operation: AFHTTPRequestOperation, error: NSError) -> Void) -> AFHTTPRequestOperation {
        var request = try! self.requestSerializer.multipartFormRequestWithMethod("POST", URLString: NSURL(string: URLString, relativeToURL: self.baseURL!)!.absoluteString, parameters: parameters, constructingBodyWithBlock: block)
        var operation = self.HTTPRequestOperationWithRequest(request, success: success, failure: failure)
        self.operationQueue!.addOperation(operation)
        return operation
    }
    /**
     Creates and runs an `AFHTTPRequestOperation` with a `PUT` request.
    
     @param URLString The URL string used to create the request URL.
     @param parameters The parameters to be encoded according to the client request serializer.
     @param success A block object to be executed when the request operation finishes successfully. This block has no return value and takes two arguments: the request operation, and the response object created by the client response serializer.
     @param failure A block object to be executed when the request operation finishes unsuccessfully, or that finishes successfully, but encountered an error while parsing the response data. This block has no return value and takes a two arguments: the request operation and the error describing the network or parsing error that occurred.
    
     @see -HTTPRequestOperationWithRequest:success:failure:
     */

    func PUT(URLString: String, parameters: AnyObject, success: (operation: AFHTTPRequestOperation, responseObject: AnyObject) -> Void, failure: (operation: AFHTTPRequestOperation, error: NSError) -> Void) -> AFHTTPRequestOperation {
        var request = try! self.requestSerializer(method: "PUT", URLString: NSURL(string: URLString, relativeToURL: self.baseURL!)!.absoluteString, parameters: parameters)
        var operation = self.HTTPRequestOperationWithRequest(request, success: success, failure: failure)
        self.operationQueue!.addOperation(operation)
        return operation
    }
    /**
     Creates and runs an `AFHTTPRequestOperation` with a `PATCH` request.
    
     @param URLString The URL string used to create the request URL.
     @param parameters The parameters to be encoded according to the client request serializer.
     @param success A block object to be executed when the request operation finishes successfully. This block has no return value and takes two arguments: the request operation, and the response object created by the client response serializer.
     @param failure A block object to be executed when the request operation finishes unsuccessfully, or that finishes successfully, but encountered an error while parsing the response data. This block has no return value and takes a two arguments: the request operation and the error describing the network or parsing error that occurred.
    
     @see -HTTPRequestOperationWithRequest:success:failure:
     */

    func PATCH(URLString: String, parameters: AnyObject, success: (operation: AFHTTPRequestOperation, responseObject: AnyObject) -> Void, failure: (operation: AFHTTPRequestOperation, error: NSError) -> Void) -> AFHTTPRequestOperation {
        var request = try! self.requestSerializer(method: "PATCH", URLString: NSURL(string: URLString, relativeToURL: self.baseURL!)!.absoluteString, parameters: parameters)
        var operation = self.HTTPRequestOperationWithRequest(request, success: success, failure: failure)
        self.operationQueue!.addOperation(operation)
        return operation
    }
    /**
     Creates and runs an `AFHTTPRequestOperation` with a `DELETE` request.
    
     @param URLString The URL string used to create the request URL.
     @param parameters The parameters to be encoded according to the client request serializer.
     @param success A block object to be executed when the request operation finishes successfully. This block has no return value and takes two arguments: the request operation, and the response object created by the client response serializer.
     @param failure A block object to be executed when the request operation finishes unsuccessfully, or that finishes successfully, but encountered an error while parsing the response data. This block has no return value and takes a two arguments: the request operation and the error describing the network or parsing error that occurred.
    
     @see -HTTPRequestOperationWithRequest:success:failure:
     */

    func DELETE(URLString: String, parameters: AnyObject, success: (operation: AFHTTPRequestOperation, responseObject: AnyObject) -> Void, failure: (operation: AFHTTPRequestOperation, error: NSError) -> Void) -> AFHTTPRequestOperation {
        var request = try! self.requestSerializer(method: "DELETE", URLString: NSURL(string: URLString, relativeToURL: self.baseURL!)!.absoluteString, parameters: parameters)
        var operation = self.HTTPRequestOperationWithRequest(request, success: success, failure: failure)
        self.operationQueue!.addOperation(operation)
        return operation
    }


    convenience override init() {
        return self.init(baseURL: nil)
    }
// MARK: -
#if _SYSTEMCONFIGURATION_H
#endif
// MARK: -
// MARK: -
// MARK: - NSObject

    override func description() -> String {
        return "<\(NSStringFromClass(self.self)): \(self), baseURL: \(self.baseURL!.absoluteString), operationQueue: \(self.operationQueue!)>"
    }
// MARK: - NSSecureCoding

    class func supportsSecureCoding() -> Bool {
        return true
    }

//    convenience required init?(coder decoder: NSCoder) {
//        var baseURL = decoder.decodeObjectForKey(NSStringFromSelector(#selector(self.baseURL)))!
//        self.init(baseURL: baseURL)
//        if !self {
//            return nil
//        }
//        self.requestSerializer = decoder.decodeObjectOfClass(AFHTTPRequestSerializer.self, forKey: NSStringFromSelector(#selector(self.requestSerializer)))!
//        self.responseSerializer = decoder.decodeObjectOfClass(AFHTTPResponseSerializer.self, forKey: NSStringFromSelector(#selector(self.responseSerializer)))!
//    }
//
//    override func encodeWithCoder(coder: NSCoder) {
//        coder.encodeObject(self.baseURL!, forKey: NSStringFromSelector(#selector(self.baseURL)))
//        coder.encodeObject(self.requestSerializer, forKey: NSStringFromSelector(#selector(self.requestSerializer)))
//        coder.encodeObject(self.responseSerializer, forKey: NSStringFromSelector(#selector(self.responseSerializer)))
//    }
// MARK: - NSCopying

    override func copyWithZone(zone: NSZone) -> AnyObject {
        var HTTPClient = self.self.allocWithZone(zone)(baseURL: self.baseURL!)
        HTTPClient.requestSerializer = self.requestSerializer.copyWithZone(zone)
        HTTPClient.responseSerializer = self.responseSerializer.copyWithZone(zone)
        return HTTPClient
    }

    var baseURL: NSURL!
}