// AFHTTPRequestOperation.h
//
// Copyright (c) 2013-2014 AFNetworking (http://afnetworking.com)
//
// Permission is hereby granted, free of charge, to any person obtaining a copy
// of this software and associated documentation files (the "Software"), to deal
// in the Software without restriction, including without limitation the rights
// to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
// copies of the Software, and to permit persons to whom the Software is
// furnished to do so, subject to the following conditions:
//
// The above copyright notice and this permission notice shall be included in
// all copies or substantial portions of the Software.
//
// THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
// IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
// FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
// AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
// LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
// OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
// THE SOFTWARE.
import UIKit
import Foundation
/**
 `AFHTTPRequestOperation` is a subclass of `AFURLConnectionOperation` for requests using the HTTP or HTTPS protocols. It encapsulates the concept of acceptable status codes and content types, which determine the success or failure of a request.
 */
class AFHTTPRequestOperation: AFURLConnectionOperation {
    ///------------------------------------------------
    /// @name Getting HTTP URL Connection Information
    ///------------------------------------------------
    /**
     The last HTTP response received by the operation's connection.
     */
    private(set) var response: NSHTTPURLResponse!
    /**
     Responses sent from the server in data tasks created with `dataTaskWithRequest:success:failure:` and run using the `GET` / `POST` / et al. convenience methods are automatically validated and serialized by the response serializer. By default, this property is set to an AFHTTPResponse serializer, which uses the raw data as its response object. The serializer validates the status code to be in the `2XX` range, denoting success. If the response serializer generates an error in `-responseObjectForResponse:data:error:`, the `failure` callback of the session task or request operation will be executed; otherwise, the `success` callback will be executed.
    
     @warning `responseSerializer` must not be `nil`. Setting a response serializer will clear out any cached value 
     */
    weak var responseSerializer: AFURLResponseSerialization? {
        get {
            // TODO: add getter implementation
        }
        set(responseSerializer) {
            NSParameterAssert(responseSerializer)
            self.lock().lock()
            self.responseSerializer = responseSerializer
            self.responseObject = nil
            self.responseSerializationError = nil
            self.lock().unlock()
        }
    }
    /**
     An object constructed by the `responseSerializer` from the response and response data. Returns `nil` unless the operation `isFinished`, has a `response`, and has `responseData` with non-zero content length. If an error occurs during serialization, `nil` will be returned, and the `error` property will be populated with the serialization error.
     */
    var responseObject: AnyObject? {
        self.lock().lock()
            if !responseObject && self.isFinished() && !self.error {
                var error: NSError? = nil
                do {
                    self.responseObject = try self.responseSerializer.responseObjectForResponse(self.response, data: self.responseData)
                }
                catch let error {
                }
                if error != nil {
                    self.responseSerializationError = error
                }
            }
            self.lock().unlock()
            return responseObject
    }
    ///-----------------------------------------------------------
    /// @name Setting Completion Block Success / Failure Callbacks
    ///-----------------------------------------------------------
    /**
     Sets the `completionBlock` property with a block that executes either the specified success or failure block, depending on the state of the request on completion. If `error` returns a value, which can be caused by an unacceptable status code or content type, then `failure` is executed. Otherwise, `success` is executed.
    
     This method should be overridden in subclasses in order to specify the response object passed into the success block.
     
     @param success The block to be executed on the completion of a successful request. This block has no return value and takes two arguments: the receiver operation and the object constructed from the response data of the request.
     @param failure The block to be executed on the completion of an unsuccessful request. This block has no return value and takes two arguments: the receiver operation and the error that occurred during the request.
     */

    func setCompletionBlockWithSuccess(success: (operation: AFHTTPRequestOperation, responseObject: AnyObject) -> Void, failure: (operation: AFHTTPRequestOperation, error: NSError) -> Void) {
        // completionBlock is manually nilled out in AFURLConnectionOperation to break the retain cycle.
//#pragma clang diagnostic push
//#pragma clang diagnostic ignored "-Warc-retain-cycles"
//#pragma clang diagnostic ignored "-Wgnu"
        self.completionBlock() = {() -> Void in
            if self.completionGroup {
                dispatch_group_enter(self.completionGroup)
            }
            dispatch_async(http_request_operation_processing_queue(), {() -> Void in
                if self.error {
                    if failure {
                        dispatch_group_async(self.completionGroup ?? http_request_operation_completion_group(), self.completionQueue ?? dispatch_get_main_queue(), {() -> Void in
                            failure(self, self.error)
                        })
                    }
                }
                else {
                    var responseObject = self.responseObject
                    if self.error {
                        if failure {
                            dispatch_group_async(self.completionGroup ?? http_request_operation_completion_group(), self.completionQueue ?? dispatch_get_main_queue(), {() -> Void in
                                failure(self, self.error)
                            })
                        }
                    }
                    else {
                        if success {
                            dispatch_group_async(self.completionGroup ?? http_request_operation_completion_group(), self.completionQueue ?? dispatch_get_main_queue(), {() -> Void in
                                success(self, responseObject)
                            })
                        }
                    }
                }
                if self.completionGroup {
                    dispatch_group_leave(self.completionGroup)
                }
            })
        }
//#pragma clang diagnostic pop
    }


    override init(request urlRequest: NSURLRequest) {
        super.init(request: urlRequest)
        if !self {
            return nil
        }
        self.responseSerializer = AFHTTPResponseSerializer.serializer()
    }

    override func error() -> NSError? {
        if responseSerializationError {
            return responseSerializationError
        }
        else {
            return super.error
        }
    }
// MARK: - AFHTTPRequestOperation
// MARK: - AFURLRequestOperation

    override func pause() {
        super.pause()
        var offset = 0
        if self.outputStream.propertyForKey(NSStreamFileCurrentOffsetKey)! {
            offset = CUnsignedLongLong(Int(self.outputStream.propertyForKey(NSStreamFileCurrentOffsetKey)!))
        }
        else {
            offset = (self.outputStream.propertyForKey(NSStreamDataWrittenToMemoryStreamKey)! as! NSData).length
        }
        var mutableURLRequest = self.request
        if self.response.respondsToSelector(#selector(self.allHeaderFields)) && (self.response.allHeaderFields().valueForKey("ETag") as! String) {
            mutableURLRequest.setValue((self.response.allHeaderFields().valueForKey("ETag") as! String), forHTTPHeaderField: "If-Range")
        }
        mutableURLRequest.setValue("bytes=\(offset)-", forHTTPHeaderField: "Range")
        self.request = mutableURLRequest
    }
// MARK: - NSSecureCoding

    class func supportsSecureCoding() -> Bool {
        return true
    }

    required init?(coder decoder: NSCoder) {
        super.init(coder: decoder)
        if !self {
            return nil
        }
        self.responseSerializer = decoder.decodeObjectOfClass(AFHTTPResponseSerializer.self, forKey: NSStringFromSelector(#selector(self.responseSerializer)))!
    }

    override func encodeWithCoder(coder: NSCoder) {
        super.encodeWithCoder(coder)
        coder.encodeObject(self.responseSerializer, forKey: NSStringFromSelector(#selector(self.responseSerializer)))
    }
// MARK: - NSCopying

    override func copyWithZone(zone: NSZone) -> AnyObject {
        var operation = super.copyWithZone(zone)
        operation.responseSerializer = self.responseSerializer.copyWithZone(zone)
        operation.completionQueue = self.completionQueue
        operation.completionGroup = self.completionGroup
        return operation
    }

    var response: NSHTTPURLResponse!
    var responseObject: AnyObject? {
        self.lock().lock()
            if !responseObject && self.isFinished() && !self.error {
                var error: NSError? = nil
                do {
                    self.responseObject = try self.responseSerializer.responseObjectForResponse(self.response, data: self.responseData)
                }
                catch let error {
                }
                if error != nil {
                    self.responseSerializationError = error
                }
            }
            self.lock().unlock()
            return responseObject
    }
    var responseSerializationError: NSError!
    var lock: NSRecursiveLock!
}
// AFHTTPRequestOperation.m
//
// Copyright (c) 2013-2014 AFNetworking (http://afnetworking.com)
//
// Permission is hereby granted, free of charge, to any person obtaining a copy
// of this software and associated documentation files (the "Software"), to deal
// in the Software without restriction, including without limitation the rights
// to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
// copies of the Software, and to permit persons to whom the Software is
// furnished to do so, subject to the following conditions:
//
// The above copyright notice and this permission notice shall be included in
// all copies or substantial portions of the Software.
//
// THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
// IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
// FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
// AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
// LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
// OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
// THE SOFTWARE.
func http_request_operation_processing_queue() -> dispatch_queue_t {
    var af_http_request_operation_processing_queue: dispatch_queue_t
    var onceToken: dispatch_once_t
    dispatch_once(onceToken, {() -> Void in
        af_http_request_operation_processing_queue = dispatch_queue_create("com.alamofire.networking.http-request.processing", DISPATCH_QUEUE_CONCURRENT)
    })
    return af_http_request_operation_processing_queue
}

func http_request_operation_completion_group() -> dispatch_group_t {
    var af_http_request_operation_completion_group: dispatch_group_t
    var onceToken: dispatch_once_t
    dispatch_once(onceToken, {() -> Void in
        af_http_request_operation_completion_group = dispatch_group_create()
    })
    return af_http_request_operation_completion_group
}

// MARK: -