<?php /*

 Composr
 Copyright (c) ocProducts, 2004-2016

 See text/EN/licence.txt for full licencing information.


 NOTE TO PROGRAMMERS:
   Do not edit this file. If you need to make changes, save your changed file to the appropriate *_custom folder
   **** If you ignore this advice, then your website upgrades (e.g. for bug fixes) will likely kill your changes ****

*/

/**
 * @license    http://opensource.org/licenses/cpal_1.0 Common Public Attribution License
 * @copyright  ocProducts Ltd
 * @package    core_forum_drivers
 */

/**
 * phpBB: Encode hash
 *
 * @param  string $input To encode
 * @param  integer $count How many chars to encode
 * @param  string $itoa64 Lookup table used internally
 * @return string The encoded output
 *
 * @ignore
 */
function _hash_encode64($input, $count, &$itoa64)
{
    $output = '';
    $i = 0;

    do {
        $value = ord($input[$i]);
        $i++;
        $output .= $itoa64[$value & 0x3f];

        if ($i < $count) {
            $value |= ord($input[$i]) << 8;
        }

        $output .= $itoa64[($value >> 6) & 0x3f];

        if ($i >= $count) {
            $i++;
            break;
        }
        $i++;

        if ($i < $count) {
            $value |= ord($input[$i]) << 16;
        }

        $output .= $itoa64[($value >> 12) & 0x3f];

        if ($i >= $count) {
            $i++;
            break;
        }
        $i++;

        $output .= $itoa64[($value >> 18) & 0x3f];
    } while ($i < $count);

    return $output;
}

/**
 * phpBB: The crypt function/replacement
 * Get the rows for the top given number of posters on the forum.
 *
 * @param  string $password To encode
 * @param  string $setting Encode settings in special format
 * @param  string $itoa64 Lookup table used internally
 * @return ~string The encoded output (false: error)
 * @ignore
 */
function _hash_crypt_private($password, $setting, &$itoa64)
{
    $output = '*';

    if (substr($setting, 0, 3) == '$H$') {
        $count_log2 = strpos($itoa64, $setting[3]);

        if ($count_log2 < 7 || $count_log2 > 30) {
            return $output;
        }

        $count = 1 << $count_log2;
        $salt = substr($setting, 4, 8);

        if (strlen($salt) != 8) {
            return $output;
        }

        /**
         * We're kind of forced to use MD5 here since it's the only
         * cryptographic primitive available in all versions of PHP
         * currently in use.  To implement our own low-level crypto
         * in PHP would result in much worse performance and
         * consequently in lower iteration counts and hashes that are
         * quicker to crack (by non-PHP code).
         */
        $hash = pack('H*', md5($salt . $password));
        do {
            $hash = pack('H*', md5($hash . $password));
            --$count;
        } while ($count > 0);

        $output = substr($setting, 0, 12);
        $output .= _hash_encode64($hash, 16, $itoa64);
    }

    if ((substr($setting, 0, 3) == '$2y') || (substr($setting, 0, 3) == '$2a$')) {
        $hash = $setting;
        $salt = substr($hash, 0, 29);
        $output = crypt($password, $salt);
        if (strlen($output) < 60) {
            return false;
        }
    }

    return $output;
}

/**
 * Forum driver class.
 *
 * @package    core_forum_drivers
 */
class Forum_driver_phpbb3 extends Forum_driver_base
{
    /**
     * Check the connected DB is valid for this forum driver.
     *
     * @return boolean Whether it is valid
     */
    public function check_db()
    {
        $test = $this->connection->query('SELECT COUNT(*) FROM ' . $this->connection->get_table_prefix() . 'users', null, null, true);
        return !is_null($test);
    }

    /**
     * Get the rows for the top given number of posters on the forum.
     *
     * @param  integer $limit The limit to the number of top posters to fetch
     * @return array The rows for the given number of top posters in the forum
     */
    public function get_top_posters($limit)
    {
        return $this->connection->query('SELECT * FROM ' . $this->connection->get_table_prefix() . 'users WHERE user_id<>' . strval($this->get_guest_id()) . ' ORDER BY user_posts DESC', $limit);
    }

    /**
     * Attempt to to find the member's language from their forum profile. It converts between language-identifiers using a map (lang/map.ini).
     *
     * @param  MEMBER $member The member who's language needs to be fetched
     * @return ?LANGUAGE_NAME The member's language (null: unknown)
     */
    public function forum_get_lang($member)
    {
        return $this->get_member_row_field($member, 'user_lang');
    }

    /**
     * Find if the login cookie contains the login name instead of the member ID.
     *
     * @return boolean Whether the login cookie contains a login name or a member ID
     */
    public function is_cookie_login_name()
    {
        return false;
    }

    /**
     * Find if login cookie is md5-hashed.
     *
     * @return boolean Whether the login cookie is md5-hashed
     */
    public function is_hashed()
    {
        return false;
    }

    /**
     * Find the member ID of the forum guest member.
     *
     * @return MEMBER The member ID of the forum guest member
     */
    public function get_guest_id()
    {
        return 1;
    }

    /**
     * Get the forums' table prefix for the database.
     *
     * @return string The forum database table prefix
     */
    public function get_drivered_table_prefix()
    {
        global $SITE_INFO;
        return $SITE_INFO['phpbb_table_prefix'];
    }

    /**
     * Add the specified custom field to the forum (some forums implemented this using proper custom profile fields, others through adding a new field).
     *
     * @param  string $name The name of the new custom field
     * @param  integer $length The length of the new custom field
     * @return boolean Whether the custom field was created successfully
     */
    public function install_create_custom_field($name, $length)
    {
        $this->connection->query('ALTER TABLE ' . $this->connection->get_table_prefix() . 'users ADD cms_' . $name . ' TEXT', null, null, true);
        return true;
    }

    /**
     * Get an array of attributes to take in from the installer. Almost all forums require a table prefix, which the requirement there-of is defined through this function.
     * The attributes have 4 values in an array
     * - name, the name of the attribute for _config.php
     * - default, the default value (perhaps obtained through autodetection from forum config)
     * - description, a textual description of the attributes
     * - title, a textual title of the attribute
     *
     * @return array The attributes for the forum
     */
    public function install_specifics()
    {
        global $PROBED_FORUM_CONFIG;
        $a = array();
        $a['name'] = 'phpbb_table_prefix';
        $a['default'] = array_key_exists('sql_tbl_prefix', $PROBED_FORUM_CONFIG) ? $PROBED_FORUM_CONFIG['sql_tbl_prefix'] : 'phpbb_';
        $a['description'] = do_lang('MOST_DEFAULT');
        $a['title'] = 'phpBB ' . do_lang('TABLE_PREFIX');
        return array($a);
    }

    /**
     * Searches for forum auto-config at this path.
     *
     * @param  PATH $path The path in which to search
     * @return boolean Whether the forum auto-config could be found
     */
    public function install_test_load_from($path)
    {
        global $PROBED_FORUM_CONFIG;
        if (@file_exists($path . '/config.php')) {
            global $FILE_BASE;
            $cms_path = str_replace('\\', '/', $FILE_BASE);
            $path = str_replace('\\', '/', $path);
            $lcs = ''; // longest common substring
            for ($i = 0; $i < min(strlen($cms_path), strlen($path)); $i++) {
                if ($cms_path[$i] == $path[$i]) {
                    $lcs .= $path[$i];
                } else {
                    break;
                }
            }
            $str_diff_forum = substr($path, strlen($lcs));
            $str_diff_site = substr($cms_path, strlen($lcs));

            $base_url = get_base_url();
            if (substr($base_url, -strlen($str_diff_site)) == $str_diff_site) { // hopefully what didn't make it in the lcs is on the end of the base URL - allowing us to get a lock
                $base_url = substr($base_url, 0, strlen($base_url) - strlen($str_diff_site)) . $str_diff_forum;
            } else {
                $base_url = '';
            }

            $dbname = '';
            $dbuser = '';
            $dbpasswd = '';
            $table_prefix = '';
            @include($path . '/config.php');
            $PROBED_FORUM_CONFIG['sql_database'] = $dbname;
            $PROBED_FORUM_CONFIG['sql_user'] = $dbuser;
            $PROBED_FORUM_CONFIG['sql_pass'] = $dbpasswd;
            $PROBED_FORUM_CONFIG['board_url'] = $base_url;
            $PROBED_FORUM_CONFIG['sql_tbl_prefix'] = $table_prefix;
            $PROBED_FORUM_CONFIG['cookie_member_id'] = 'phpbb_u';
            $PROBED_FORUM_CONFIG['cookie_member_hash'] = 'phpbb_k';
            return true;
        }
        return false;
    }

    /**
     * Get an array of paths to search for config at.
     *
     * @return array The paths in which to search for the forum config
     */
    public function install_get_path_search_list()
    {
        return array(
            0 => 'forums',
            1 => 'forum',
            2 => 'boards',
            3 => 'board',
            4 => 'phpBB',
            5 => 'phpBB2',
            6 => 'upload',
            7 => 'uploads',
            8 => 'phpbb',
            9 => 'phpbb2',
            10 => '../forums',
            11 => '../forum',
            12 => '../boards',
            13 => '../board',
            14 => '../phpBB',
            15 => '../phpBB2',
            16 => '../upload',
            17 => '../uploads',
            18 => '../phpbb',
            19 => '../phpbb2');
    }

    /**
     * Get an emoticon chooser template.
     *
     * @param  string $field_name The ID of the form field the emoticon chooser adds to
     * @return Tempcode The emoticon chooser template
     */
    public function get_emoticon_chooser($field_name = 'post')
    {
        require_code('comcode_compiler');
        $emoticons = $this->connection->query_select('smilies', array('code'), null, 'GROUP BY code');
        $em = new Tempcode();
        foreach ($emoticons as $emo) {
            $code = $emo['code'];
            $em->attach(do_template('EMOTICON_CLICK_CODE', array('_GUID' => '6eef1777ed25b62f2c47964eebb655ac', 'FIELD_NAME' => $field_name, 'CODE' => $code, 'IMAGE' => apply_emoticons($code))));
        }

        return $em;
    }

    /**
     * Pin a topic.
     *
     * @param  AUTO_LINK $id The topic ID
     * @param  boolean $pin True: pin it, False: unpin it
     */
    public function pin_topic($id, $pin = true)
    {
        $this->connection->query_update('topics', array('topic_type' => $pin ? 1 : 0), array('topic_id' => $id), '', 1);
    }

    /**
     * Set a custom profile field's value, if the custom field exists. Only works on specially-named (titled) fields.
     *
     * @param  MEMBER $member The member ID
     * @param  string $field The field name (e.g. "firstname" for the CPF with a title of "cms_firstname")
     * @param  string $value The value
     */
    public function set_custom_field($member, $field, $value)
    {
        $this->connection->query_update('users', array('cms_' . $field => $value), array('user_id' => $member), '', null, null, false, true);
    }

    /**
     * Get custom profile fields values for all 'cms_' prefixed keys.
     *
     * @param  MEMBER $member The member ID
     * @return ?array A map of the custom profile fields, key_suffix=>value (null: no fields)
     */
    public function get_custom_fields($member)
    {
        $row = $this->get_member_row($member);
        $out = array();
        foreach ($row as $attribute => $value) {
            if (substr($attribute, 0, 4) == 'cms_') {
                $out[substr($attribute, 4)] = $value;
            }
        }
        return $out;
    }

    /**
     * Get a member row for the member of the given name.
     *
     * @param  SHORT_TEXT $name The member name
     * @return ?array The profile-row (null: not found)
     */
    public function get_mrow($name)
    {
        $rows = $this->connection->query_select('users', array('*'), array('username_clean' => strtolower($name)), '', 1);
        if (!array_key_exists(0, $rows)) {
            return null;
        }
        return $rows[0];
    }

    /**
     * From a member row, get the member's primary usergroup.
     *
     * @param  array $r The profile-row
     * @return GROUP The member's primary usergroup
     */
    public function mrow_group($r)
    {
        $id = $r['user_id'];

        // The first usergroup they're joined to
        $group = $this->connection->query_select_value_if_there('user_group u LEFT JOIN ' . $this->connection->get_table_prefix() . 'groups g ON u.group_id=g.group_id', 'g.group_id AS group_id', array('user_id' => $id, 'user_pending' => 0));
        return $group;
    }

    /**
     * From a member row, get the member's member ID.
     *
     * @param  array $r The profile-row
     * @return MEMBER The member ID
     */
    public function mrow_id($r)
    {
        return $r['user_id'];
    }

    /**
     * From a member row, get the member's last visit date.
     *
     * @param  array $r The profile-row
     * @return TIME The last visit date
     */
    public function mrow_lastvisit($r)
    {
        return $r['user_lastvisit'];
    }

    /**
     * From a member row, get the member's name.
     *
     * @param  array $r The profile-row
     * @return string The member name
     */
    public function mrow_username($r)
    {
        return $r['username'];
    }

    /**
     * From a member row, get the member's e-mail address.
     *
     * @param  array $r The profile-row
     * @return SHORT_TEXT The member e-mail address
     */
    public function mrow_email($r)
    {
        return $r['user_email'];
    }

    /**
     * Get a URL to the specified member's home (control panel).
     *
     * @param  MEMBER $id The member ID
     * @return URLPATH The URL to the members home
     */
    public function member_home_url($id)
    {
        return get_forum_base_url() . '/memberlist.php?mode=editprofile';
    }

    /**
     * Get the photo thumbnail URL for the specified member ID.
     *
     * @param  MEMBER $member The member ID
     * @return URLPATH The URL (blank: none)
     */
    public function get_member_photo_url($member)
    {
        return '';
    }

    /**
     * Get the avatar URL for the specified member ID.
     *
     * @param  MEMBER $member The member ID
     * @return URLPATH The URL (blank: none)
     */
    public function get_member_avatar_url($member)
    {
        $options = $this->connection->query('SELECT * FROM ' . $this->connection->get_table_prefix() . 'config WHERE ' . db_string_equal_to('config_name', 'avatar_path') . ' OR ' . db_string_equal_to('config_name', 'avatar_gallery_path'));
        $avatar_path = $options[0]['config_value'];
        $avatar_gallery_path = $options[1]['config_value'];

        $type = $this->get_member_row_field($member, 'user_avatar_type');
        $filename = $this->get_member_row_field($member, 'user_avatar');

        switch ($type) {
            case '1': // Upload
                return get_forum_base_url() . '/' . $avatar_path . '/' . $filename;
            case '2': // Remote
                return $filename;
            case '3': // Gallery
                return get_forum_base_url() . '/' . $avatar_gallery_path . '/' . $filename;
        }
        return '';
    }

    /**
     * Get a URL to the specified member's profile.
     *
     * @param  MEMBER $id The member ID
     * @return URLPATH The URL to the member profile
     */
    protected function _member_profile_url($id)
    {
        return get_forum_base_url() . '/memberlist.php?mode=viewprofile&u=' . strval($id);
    }

    /**
     * Get a URL to the registration page (for people to create member accounts).
     *
     * @return URLPATH The URL to the registration page
     */
    protected function _join_url()
    {
        return get_forum_base_url() . '/ucp.php?mode=register';
    }

    /**
     * Get a URL to the members-online page.
     *
     * @return URLPATH The URL to the members-online page
     */
    protected function _users_online_url()
    {
        return get_forum_base_url() . '/viewonline.php';
    }

    /**
     * Get a URL to send a private/personal message to the given member.
     *
     * @param  MEMBER $id The member ID
     * @return URLPATH The URL to the private/personal message page
     */
    protected function _member_pm_url($id)
    {
        return get_forum_base_url() . '/ucp.php?i=pm&mode=compose&u=' . strval($id);
    }

    /**
     * Get a URL to the specified forum.
     *
     * @param  integer $id The forum ID
     * @return URLPATH The URL to the specified forum
     */
    protected function _forum_url($id)
    {
        return get_forum_base_url() . '/viewforum.php?f=' . strval($id);
    }

    /**
     * Get the forum ID from a forum name.
     *
     * @param  SHORT_TEXT $forum_name The forum name
     * @return ?integer The forum ID (null: not found)
     */
    public function forum_id_from_name($forum_name)
    {
        return is_numeric($forum_name) ? intval($forum_name) : $this->connection->query_select_value_if_there('forums', 'forum_id', array('forum_name' => $forum_name));
    }

    /**
     * Convert an IP address into phpBB hexadecimal string format.
     *
     * @param  IP $ip The normal IP address
     * @return string The phpBB IP address
     */
    protected function _phpbb_ip($ip)
    {
        $ip_apart = explode('.', $ip);
        $_ip = dechex(intval($ip_apart[0])) . dechex(intval($ip_apart[1])) . dechex(intval($ip_apart[2])) . dechex(intval($ip_apart[3]));
        return $_ip;
    }

    /**
     * Convert an IP address from phpBB hexadecimal string format.
     *
     * @param  string $ip The phpBB IP address
     * @return IP The normal IP address
     */
    protected function _un_phpbb_ip($ip)
    {
        $_ip = strval(hexdec($ip[0] . $ip[1])) . '.' . strval(hexdec($ip[2] . $ip[3])) . '.' . strval(hexdec($ip[4] . $ip[5])) . '.' . strval(hexdec($ip[6] . $ip[7]));
        return $_ip;
    }

    /**
     * Makes a post in the specified forum, in the specified topic according to the given specifications. If the topic doesn't exist, it is created along with a spacer-post.
     * Spacer posts exist in order to allow staff to delete the first true post in a topic. Without spacers, this would not be possible with most forum systems. They also serve to provide meta information on the topic that cannot be encoded in the title (such as a link to the content being commented upon).
     *
     * @param  SHORT_TEXT $forum_name The forum name
     * @param  SHORT_TEXT $topic_identifier The topic identifier (usually <content-type>_<content-id>)
     * @param  MEMBER $member The member ID
     * @param  LONG_TEXT $post_title The post title
     * @param  LONG_TEXT $_post The post content in Comcode format
     * @param  string $content_title The topic title; must be same as content title if this is for a comment topic
     * @param  string $topic_identifier_encapsulation_prefix This is put together with the topic identifier to make a more-human-readable topic title or topic description (hopefully the latter and a $content_title title, but only if the forum supports descriptions)
     * @param  ?URLPATH $content_url URL to the content (null: do not make spacer post)
     * @param  ?TIME $time The post time (null: use current time)
     * @param  ?IP $ip The post IP address (null: use current members IP address)
     * @param  ?BINARY $validated Whether the post is validated (null: unknown, find whether it needs to be marked unvalidated initially). This only works with the Conversr driver.
     * @param  ?BINARY $topic_validated Whether the topic is validated (null: unknown, find whether it needs to be marked unvalidated initially). This only works with the Conversr driver.
     * @param  boolean $skip_post_checks Whether to skip post checks
     * @param  SHORT_TEXT $poster_name_if_guest The name of the poster
     * @param  ?AUTO_LINK $parent_id ID of post being replied to (null: N/A)
     * @param  boolean $staff_only Whether the reply is only visible to staff
     * @return array Topic ID (may be null), and whether a hidden post has been made
     */
    public function make_post_forum_topic($forum_name, $topic_identifier, $member, $post_title, $_post, $content_title, $topic_identifier_encapsulation_prefix, $content_url = null, $time = null, $ip = null, $validated = null, $topic_validated = 1, $skip_post_checks = false, $poster_name_if_guest = '', $parent_id = null, $staff_only = false)
    {
        $__post = comcode_to_tempcode($_post);
        $post = $__post->evaluate();

        if (is_null($time)) {
            $time = time();
        }
        if (is_null($ip)) {
            $ip = get_ip_address();
        }
        $forum_id = $this->forum_id_from_name($forum_name);
        if (is_null($forum_id)) {
            warn_exit(do_lang_tempcode('MISSING_FORUM', escape_html($forum_name)));
        }
        $test = $this->connection->query_select('forums', array('*'), null, '', 1);
        $fm = array_key_exists('hide_forum_in_cat', $test[0]);
        $topic_id = $this->find_topic_id_for_topic_identifier($forum_name, $topic_identifier);
        $ip_address = $this->_phpbb_ip($ip);
        $local_ip_address = $this->_phpbb_ip('127.0.0.1');
        $is_new = is_null($topic_id);
        if ($is_new) {
            $map = array('forum_id' => $forum_id, 'topic_title' => $content_title . ', ' . $topic_identifier_encapsulation_prefix . ': #' . $topic_identifier, 'topic_poster' => $member, 'topic_time' => $time, 'topic_views' => 0, 'topic_replies' => 0, 'topic_replies_real' => 0, 'topic_status' => 0, 'topic_type' => 0, 'topic_first_post_id' => 0, 'topic_last_post_id' => 0, 'topic_moved_id' => 0);
            if ($fm) {
                $map = array_merge($map, array('answer_status' => 0, 'topic_attachment' => 0, 'topic_icon' => 0, 'rating_rank_id' => 0, 'title_compl_infos' => null, 'topic_priority' => 0));
            }
            $topic_id = $this->connection->query_insert('topics', $map, true);
            $home_link = hyperlink($content_url, $content_title, false, true);
            $map = array('topic_id' => $topic_id, 'forum_id' => $forum_id, 'poster_id' => -1, 'post_text' => do_lang('SPACER_POST', $home_link->evaluate(), '', '', get_site_default_lang()), 'post_time' => $time, 'poster_ip' => $local_ip_address, 'post_username' => $this->get_username($member), 'enable_bbcode' => 1, 'enable_smilies' => 1, 'enable_sig' => 1, 'post_edit_time' => 0, 'post_edit_count' => 0);
            if ($fm) {
                $map = array_merge($map, array('post_attachment' => 0, 'post_edit_user' => null, 'post_icon' => 0, 'post_bluecard' => null, 'rating_rank_id' => 0, 'user_avatar' => null, 'user_avatar_type' => 0, 'urgent_post' => 0));
            }
            $post_id = $this->connection->query_insert('posts', $map, true);
            $this->connection->query_update('topics', array('topic_first_post_id' => $post_id), array('topic_id' => $topic_id), '', 1);
            $this->connection->query('UPDATE ' . $this->connection->get_table_prefix() . 'forums SET forum_topics_real=(forum_topics_real+1),forum_topics=(forum_topics+1),forum_posts=(forum_posts+1) WHERE forum_id=' . strval($forum_id), 1);
        }

        $GLOBALS['LAST_TOPIC_ID'] = $topic_id;
        $GLOBALS['LAST_TOPIC_IS_NEW'] = $is_new;

        if ($post == '') {
            return array($topic_id, false);
        }

        $map = array('topic_id' => $topic_id, 'forum_id' => $forum_id, 'poster_id' => $member, 'post_text' => $post, 'post_time' => $time, 'poster_ip' => $ip_address, 'post_username' => $this->get_username($member), 'enable_bbcode' => 1, 'enable_smilies' => 1, 'enable_sig' => 1, 'post_edit_time' => 0, 'post_edit_count' => 0);
        if ($fm) {
            $map = array_merge($map, array('post_attachment' => 0, 'post_edit_user' => null, 'post_icon' => 0, 'post_bluecard' => null, 'rating_rank_id' => 0, 'user_avatar' => null, 'user_avatar_type' => 0, 'urgent_post' => 0));
        }
        $post_id = $this->connection->query_insert('posts', $map, true);
        $this->connection->query('UPDATE ' . $this->connection->get_table_prefix() . 'forums SET forum_posts=(forum_posts+1), forum_last_post_id=' . strval($post_id) . ' WHERE forum_id=' . strval($forum_id), 1);
        $this->connection->query('UPDATE ' . $this->connection->get_table_prefix() . 'topics SET topic_replies=(topic_replies+1), topic_replies_real=(topic_replies_real+1), topic_last_post_id=' . strval($post_id) . ' WHERE topic_id=' . strval($topic_id), 1);

        return array($topic_id, false);
    }

    /**
     * Get an array of maps for the topic in the given forum.
     *
     * @param  integer $topic_id The topic ID
     * @param  integer $count The comment count will be returned here by reference
     * @param  integer $max Maximum comments to returned
     * @param  integer $start Comment to start at
     * @param  boolean $mark_read Whether to mark the topic read (ignored for this forum driver)
     * @param  boolean $reverse Whether to show in reverse
     * @return mixed The array of maps (Each map is: title, message, member, date) (-1 for no such forum, -2 for no such topic)
     */
    public function get_forum_topic_posts($topic_id, &$count, $max = 100, $start = 0, $mark_read = true, $reverse = false)
    {
        if (is_null($topic_id)) {
            return (-2);
        }
        $order = $reverse ? 'post_time DESC' : 'post_time';
        $rows = $this->connection->query('SELECT * FROM ' . $this->connection->get_table_prefix() . 'posts p WHERE topic_id=' . strval($topic_id) . ' AND post_text NOT LIKE \'' . db_encode_like(substr(do_lang('SPACER_POST', '', '', '', get_site_default_lang()), 0, 20) . '%') . '\' ORDER BY ' . $order, $max, $start);
        $count = $this->connection->query_value_if_there('SELECT COUNT(*) FROM ' . $this->connection->get_table_prefix() . 'posts p WHERE topic_id=' . strval($topic_id) . ' AND post_text NOT LIKE \'' . db_encode_like(substr(do_lang('SPACER_POST', '', '', '', get_site_default_lang()), 0, 20) . '%') . '\'');
        $out = array();
        foreach ($rows as $myrow) {
            $temp = array();
            $temp['title'] = $myrow['post_subject'];
            if (is_null($temp['title'])) {
                $temp['title'] = '';
            }
            global $LAX_COMCODE;
            $temp2 = $LAX_COMCODE;
            $LAX_COMCODE = true;
            $temp['message'] = comcode_to_tempcode($this->_cleanup_post($myrow['bbcode_uid'], $myrow['post_text']), $myrow['poster_id']);
            $LAX_COMCODE = $temp2;
            $temp['member'] = $myrow['poster_id'];
            $temp['date'] = $myrow['post_time'];

            $out[] = $temp;
        }

        return $out;
    }

    /**
     * Get a URL to the specified topic ID. Most forums don't require the second parameter, but some do, so it is required in the interface.
     *
     * @param  integer $id The topic ID
     * @param  string $forum The forum ID
     * @return URLPATH The URL to the topic
     */
    public function topic_url($id, $forum)
    {
        return get_forum_base_url() . '/viewtopic.php?t=' . strval($id);
    }

    /**
     * Get a URL to the specified post ID.
     *
     * @param  integer $id The post ID
     * @param  string $forum The forum ID
     * @return URLPATH The URL to the post
     */
    public function post_url($id, $forum)
    {
        $topic_id = $this->connection->query_select_value_if_there('posts', 'tid', array('pid' => $id));
        if (is_null($topic_id)) {
            return '?';
        }
        $url = get_forum_base_url() . '/viewtopic.php?t=' . strval($topic_id) . '#' . strval($id);
        return $url;
    }

    /**
     * Get the topic ID from a topic identifier in the specified forum. It is used by comment topics, which means that the unique-topic-name assumption holds valid.
     *
     * @param  string $forum The forum name / ID
     * @param  SHORT_TEXT $topic_identifier The topic identifier
     * @return ?integer The topic ID (null: not found)
     */
    public function find_topic_id_for_topic_identifier($forum, $topic_identifier)
    {
        if (is_integer($forum)) {
            $forum_id = $forum;
        } else {
            $forum_id = $this->forum_id_from_name($forum);
        }
        if (is_null($forum_id)) {
            return null;
        }
        return $this->connection->query_value_if_there('SELECT topic_id FROM ' . $this->connection->get_table_prefix() . 'topics WHERE forum_id=' . strval($forum_id) . ' AND (' . db_string_equal_to('topic_title', $topic_identifier) . ' OR topic_title LIKE \'%: #' . db_encode_like($topic_identifier) . '\')');
    }

    /**
     * Get an array of topics in the given forum. Each topic is an array with the following attributes:
     * - id, the topic ID
     * - title, the topic title
     * - lastusername, the username of the last poster
     * - lasttime, the timestamp of the last reply
     * - closed, a Boolean for whether the topic is currently closed or not
     * - firsttitle, the title of the first post
     * - firstpost, the first post (only set if $show_first_posts was true)
     *
     * @param  mixed $name The forum name or an array of forum IDs
     * @param  integer $limit The limit
     * @param  integer $start The start position
     * @param  integer $max_rows The total rows (not a parameter: returns by reference)
     * @param  SHORT_TEXT $filter_topic_title The topic title filter
     * @param  boolean $show_first_posts Whether to show the first posts
     * @param  string $date_key The date key to sort by
     * @set    lasttime firsttime
     * @param  boolean $hot Whether to limit to hot topics
     * @param  SHORT_TEXT $filter_topic_description The topic description filter
     * @return ?array The array of topics (null: error)
     */
    public function show_forum_topics($name, $limit, $start, &$max_rows, $filter_topic_title = '', $show_first_posts = false, $date_key = 'lasttime', $hot = false, $filter_topic_description = '')
    {
        if (is_integer($name)) {
            $id_list = 'forum_id=' . strval($name);
        } elseif (!is_array($name)) {
            $id = $this->forum_id_from_name($name);
            if (is_null($id)) {
                return null;
            }
            $id_list = 'forum_id=' . strval($id);
        } else {
            $id_list = '';
            foreach (array_keys($name) as $id) {
                if ($id_list != '') {
                    $id_list .= ' OR ';
                }
                $id_list .= 'forum_id=' . strval($id);
            }
            if ($id_list == '') {
                return null;
            }
        }

        $topic_filter = ($filter_topic_title != '') ? 'AND topic_title LIKE \'' . db_encode_like($filter_topic_title) . '\'' : '';
        $topic_filter .= ' ORDER BY ' . (($date_key == 'lasttime') ? 'topic_last_post_id' : 'topic_time') . ' DESC';
        $rows = $this->connection->query('SELECT * FROM ' . $this->connection->get_table_prefix() . 'topics WHERE (' . $id_list . ') ' . $topic_filter, $limit, $start);
        $max_rows = $this->connection->query_value_if_there('SELECT COUNT(*) FROM ' . $this->connection->get_table_prefix() . 'topics WHERE (' . $id_list . ') ' . $topic_filter);
        $i = 0;
        $firsttime = array();
        $username = array();
        $memberid = array();
        $datetimes = array();
        $rs = array();
        while (array_key_exists($i, $rows)) {
            $r = $rows[$i];

            $id = $r['topic_id'];
            $firsttime[$id] = $r['topic_time'];

            $post_rows = $this->connection->query('SELECT * FROM ' . $this->connection->get_table_prefix() . 'posts p WHERE topic_id=' . strval($id) . ' AND post_text NOT LIKE \'' . db_encode_like(substr(do_lang('SPACER_POST', '', '', '', get_site_default_lang()), 0, 20) . '%') . '\' ORDER BY post_time DESC', 1);
            if (!array_key_exists(0, $post_rows)) {
                $i++;
                continue;
            }
            $r2 = $post_rows[0];

            $username[$id] = $this->get_username($r2['poster_id']);
            $memberid[$id] = $r2['poster_id'];
            $datetimes[$id] = $r2['post_time'];
            $rs[$id] = $r;

            $i++;
        }
        if ($i > 0) {
            arsort($datetimes);
            $i = 0;
            $out = array();
            if (count($datetimes) > 0) {
                foreach ($datetimes as $id => $datetime) {
                    $r = $rs[$id];

                    $out[$i] = array();
                    $out[$i]['id'] = $id;
                    $out[$i]['num'] = $r['topic_replies_real'] + 1;
                    $out[$i]['title'] = $r['topic_title'];
                    $out[$i]['description'] = $r['topic_title'];
                    $out[$i]['firsttime'] = $r['topic_time'];
                    $out[$i]['firstusername'] = $this->get_username($r['topic_poster']);
                    $out[$i]['lastusername'] = $username[$id];
                    $out[$i]['firstmemberid'] = $r['topic_poster'];
                    $out[$i]['lastmemberid'] = $memberid[$id];
                    $out[$i]['lasttime'] = $datetime;
                    $out[$i]['closed'] = ($r['topic_status'] == 1);

                    $fp_rows = $this->connection->query('SELECT post_subject,post_text,bbcode_uid,poster_id FROM ' . $this->connection->get_table_prefix() . 'posts p WHERE post_text NOT LIKE \'' . db_encode_like(substr(do_lang('SPACER_POST', '', '', '', get_site_default_lang()), 0, 20) . '%') . '\' AND post_time=' . strval($firsttime[$id]) . ' AND topic_id=' . strval($id), 1);
                    if (!array_key_exists(0, $fp_rows)) {
                        unset($out[$i]);
                        continue;
                    }
                    $out[$i]['firsttitle'] = $fp_rows[0]['post_subject'];
                    if ($show_first_posts) {
                        global $LAX_COMCODE;
                        $temp = $LAX_COMCODE;
                        $LAX_COMCODE = true;
                        $out[$i]['firstpost'] = comcode_to_tempcode($this->_cleanup_post($fp_rows[0]['bbcode_uid'], $fp_rows[0]['post_text']), $fp_rows[0]['poster_id']);
                        $LAX_COMCODE = $temp;
                    }

                    $i++;
                    if ($i == $limit) {
                        break;
                    }
                }
            }

            return $out;
        }
        return null;
    }

    /**
     * Cleanup a post to match Comcode.
     *
     * @param  string $uid Bbcode embedded UID (which we strip)
     * @param  string $text The post
     * @return string Cleaned post
     */
    protected function _cleanup_post($uid, $text)
    {
        if ($uid != '') {
            $text = str_replace(':' . $uid, '', $text);
        }
        $text = preg_replace('#<!-- s([^\s]*) --><img src="\{SMILIES_PATH\}/[^"]*" alt="([^\s]*)" title="[^"]*" /><!-- s([^\s]*) -->#U', '${1}', $text);
        $text = preg_replace('#\[(/?\w+):[^\]]*\]#', '[${1}]', $text);
        return html_entity_decode($text, ENT_QUOTES);
    }

    /**
     * Get an array of members who are in at least one of the given array of usergroups.
     *
     * @param  array $groups The array of usergroups
     * @param  ?integer $max Return up to this many entries for primary members and this many entries for secondary members (null: no limit, only use no limit if querying very restricted usergroups!)
     * @param  integer $start Return primary members after this offset and secondary members after this offset
     * @return ?array The array of members (null: no members)
     */
    public function member_group_query($groups, $max = null, $start = 0)
    {
        $_groups = '';
        foreach ($groups as $group) {
            if ($_groups != '') {
                $_groups .= ' OR ';
            }
            $_groups .= 'g.group_id=' . strval($group);
        }
        if ($_groups == '') {
            return array();
        }
        return $this->connection->query('SELECT * FROM ' . $this->connection->get_table_prefix() . 'user_group g LEFT JOIN ' . $this->connection->get_table_prefix() . 'users u ON u.user_id=g.user_id WHERE (' . $_groups . ') AND user_pending=0 ORDER BY g.group_id ASC', $max, $start, false, true);
    }

    /**
     * This is the opposite of the get_next_member function.
     *
     * @param  MEMBER $member The member ID to decrement
     * @return ?MEMBER The previous member ID (null: no previous member)
     */
    public function get_previous_member($member)
    {
        $tempid = $this->connection->query_value_if_there('SELECT user_id FROM ' . $this->connection->get_table_prefix() . 'users WHERE user_id<' . strval($member) . ' AND user_id>0 ORDER BY user_id DESC');
        return $tempid;
    }

    /**
     * Get the member ID of the next member after the given one, or null.
     * It cannot be assumed there are no gaps in member IDs, as members may be deleted.
     *
     * @param  MEMBER $member The member ID to increment
     * @return ?MEMBER The next member ID (null: no next member)
     */
    public function get_next_member($member)
    {
        $tempid = $this->connection->query_value_if_there('SELECT user_id FROM ' . $this->connection->get_table_prefix() . 'users WHERE user_id>' . strval($member) . ' ORDER BY user_id');
        return $tempid;
    }

    /**
     * Try to find a member with the given IP address
     *
     * @param  IP $ip The IP address
     * @return array The distinct rows found
     */
    public function probe_ip($ip)
    {
        $_ip = $this->_phpbb_ip($ip);
        return $this->connection->query_select('posts', array('DISTINCT poster_id AS id'), array('poster_ip' => $_ip));
    }

    /**
     * Get the name relating to the specified member ID.
     * If this returns null, then the member has been deleted. Always take potential null output into account.
     *
     * @param  MEMBER $member The member ID
     * @return ?SHORT_TEXT The member name (null: member deleted)
     */
    protected function _get_username($member)
    {
        if ($member == $this->get_guest_id()) {
            return do_lang('GUEST');
        }
        return $this->get_member_row_field($member, 'username');
    }

    /**
     * Get the e-mail address for the specified member ID.
     *
     * @param  MEMBER $member The member ID
     * @return SHORT_TEXT The e-mail address
     */
    protected function _get_member_email_address($member)
    {
        return $this->get_member_row_field($member, 'user_email');
    }

    /**
     * Find if this member may have e-mails sent to them
     *
     * @param  MEMBER $member The member ID
     * @return boolean Whether the member may have e-mails sent to them
     */
    public function get_member_email_allowed($member)
    {
        $v = $this->get_member_row_field($member, 'user_viewemail');
        if ($v == 1) {
            return true;
        }
        return false;
    }

    /**
     * Get the timestamp of a member's join date.
     *
     * @param  MEMBER $member The member ID
     * @return TIME The timestamp
     */
    public function get_member_join_timestamp($member)
    {
        return $this->get_member_row_field($member, 'user_regdate');
    }

    /**
     * Find all members with a name matching the given SQL LIKE string.
     *
     * @param  string $pattern The pattern
     * @param  ?integer $limit Maximum number to return (limits to the most recent active) (null: no limit)
     * @return ?array The array of matched members (null: none found)
     */
    public function get_matching_members($pattern, $limit = null)
    {
        $rows = $this->connection->query('SELECT * FROM ' . $this->connection->get_table_prefix() . 'users WHERE username LIKE \'' . db_encode_like(strtolower($pattern)) . '\' AND user_id<>' . strval($this->get_guest_id()) . ' ORDER BY user_lastvisit DESC', $limit);
        sort_maps_by($rows, 'username');
        return $rows;
    }

    /**
     * Get the given member's post count.
     *
     * @param  MEMBER $member The member ID
     * @return integer The post count
     */
    public function get_post_count($member)
    {
        return $this->get_member_row_field($member, 'user_posts');
    }

    /**
     * Get the given member's topic count.
     *
     * @param  MEMBER $member The member ID
     * @return integer The topic count
     */
    public function get_topic_count($member)
    {
        return $this->connection->query_select_value('topics', 'COUNT(*)', array('topic_poster' => $member));
    }

    /**
     * Find out if the given member ID is banned.
     *
     * @param  MEMBER $member The member ID
     * @return boolean Whether the member is banned
     */
    public function is_banned($member)
    {
        $banned = $this->connection->query_select_value_if_there('banlist', 'ban_userid', array('ban_userid' => $member));
        if (!is_null($banned)) {
            return true;
        }
        return false;
    }

    /**
     * Find the base URL to the emoticons.
     *
     * @return URLPATH The base URL
     */
    public function get_emo_dir()
    {
        return get_forum_base_url() . '/images/smilies/';
    }

    /**
     * Get a map between emoticon codes and templates representing the HTML-image-code for this emoticon. The emoticons presented of course depend on the forum involved.
     *
     * @return array The map
     */
    public function find_emoticons()
    {
        if (!is_null($this->EMOTICON_CACHE)) {
            return $this->EMOTICON_CACHE;
        }
        $rows = $this->connection->query_select('smilies', array('*'), null, '', null, null, true);
        if (is_null($rows)) {
            $rows = array();
        }
        $this->EMOTICON_CACHE = array();
        foreach ($rows as $myrow) {
            $src = $myrow['smiley_url'];
            if (url_is_local($src)) {
                $src = $this->get_emo_dir() . $src;
            }
            $this->EMOTICON_CACHE[$myrow['code']] = array('EMOTICON_IMG_CODE_DIR', $src, $myrow['code']);
        }
        uksort($this->EMOTICON_CACHE, '_strlen_sort');
        $this->EMOTICON_CACHE = array_reverse($this->EMOTICON_CACHE);
        return $this->EMOTICON_CACHE;
    }

    /**
     * Find a list of all forum skins (aka themes).
     *
     * @return array The list of skins
     */
    public function get_skin_list()
    {
        $table = 'styles';
        $codename = 'style_name';

        $rows = $this->connection->query_select($table, array($codename));
        return collapse_1d_complexity($codename, $rows);
    }

    /**
     * Try to find the theme that the logged-in/guest member is using, and map it to a Composr theme.
     * The themes/map.ini file functions to provide this mapping between forum themes, and Composr themes, and has a slightly different meaning for different forum drivers. For example, some drivers map the forum themes theme directory to the Composr theme name, while others made the humanly readeable name.
     *
     * @param  boolean $skip_member_specific Whether to avoid member-specific lookup
     * @return ID_TEXT The theme
     */
    public function _get_theme($skip_member_specific = false)
    {
        $def = '';

        // Load in remapper
        require_code('files');
        $map = file_exists(get_file_base() . '/themes/map.ini') ? better_parse_ini_file(get_file_base() . '/themes/map.ini') : array();

        // Work out
        if (!$skip_member_specific) {
            $member = get_member();
            if ($member > 0) {
                $skin = $this->get_member_row_field($member, 'user_style');
            } else {
                $skin = 0;
            }
            if ($skin > 0) { // User has a custom theme
                $phpbb = $this->connection->query_select_value_if_there('styles', 'style_name', array('style_id' => $skin));
                if (!is_null($phpbb)) {
                    $def = array_key_exists($phpbb, $map) ? $map[$phpbb] : $phpbb;
                }
            }
        }

        // Look for a skin according to our site name (we bother with this instead of 'default' because Composr itself likes to never choose a theme when forum-theme integration is on: all forum [via map] or all Composr seems cleaner, although it is complex)
        if ((!(strlen($def) > 0)) || (!file_exists(get_custom_file_base() . '/themes/' . $def))) {
            $phpbb = $this->connection->query_select_value_if_there('styles', 'style_name', array('style_name' => get_site_name()));
            if (!is_null($phpbb)) {
                $def = array_key_exists($phpbb, $map) ? $map[$phpbb] : $phpbb;
            }
        }

        // Default then!
        if ((!(strlen($def) > 0)) || (!file_exists(get_custom_file_base() . '/themes/' . $def))) {
            $def = array_key_exists('default', $map) ? $map['default'] : 'default';
        }

        return $def;
    }

    /**
     * Find if the specified member ID is marked as staff or not.
     *
     * @param  MEMBER $member The member ID
     * @return boolean Whether the member is staff
     */
    protected function _is_staff($member)
    {
        return (array_intersect($this->get_members_groups($member), $this->get_moderator_groups()) != array());
    }

    /**
     * Find if the specified member ID is marked as a super admin or not.
     *
     * @param  MEMBER $member The member ID
     * @return boolean Whether the member is a super admin
     */
    protected function _is_super_admin($member)
    {
        return (array_intersect($this->get_members_groups($member), $this->get_super_admin_groups()) != array());
    }

    /**
     * If we can't get a list of admins via a usergroup query, we have to disable the staff filter - else the staff filtering can cause disaster at the point of being turned on (because it can't automatically sync).
     *
     * @return boolean Whether to disable the staff filter
     */
    protected function _disable_staff_filter()
    {
        return true;
    }

    /**
     * Get the number of members currently online on the forums.
     *
     * @return integer The number of members
     */
    public function get_num_users_forums()
    {
        return $this->connection->query_value_if_there('SELECT COUNT(*) FROM ' . $this->connection->get_table_prefix() . 'sessions WHERE session_time>' . strval(time() - 60 * intval(get_option('users_online_time'))));
    }

    /**
     * Get the number of members registered on the forum.
     *
     * @return integer The number of members
     */
    public function get_members()
    {
        return $this->connection->query_select_value('users', 'COUNT(*)') - 1;
    }

    /**
     * Get the total topics ever made on the forum.
     *
     * @return integer The number of topics
     */
    public function get_topics()
    {
        static $num_topics = null;
        if (is_null($num_topics)) {
            $num_topics = $this->connection->query_select_value('topics', 'COUNT(*)');
        }
        return $num_topics;
    }

    /**
     * Get the total posts ever made on the forum.
     *
     * @return integer The number of posts
     */
    public function get_num_forum_posts()
    {
        static $num_posts = null;
        if (is_null($num_posts)) {
            $num_posts = $this->connection->query_select_value('posts', 'COUNT(*)');
        }
        return $num_posts;
    }

    /**
     * Get the number of new forum posts.
     *
     * @return integer The number of posts
     */
    protected function _get_num_new_forum_posts()
    {
        return $this->connection->query_value_if_there('SELECT COUNT(*) FROM ' . $this->connection->get_table_prefix() . 'posts WHERE post_time>' . strval(time() - 60 * 60 * 24));
    }

    /**
     * Get a member ID from the given member's username.
     *
     * @param  SHORT_TEXT $name The member name
     * @return MEMBER The member ID
     */
    public function get_member_from_username($name)
    {
        return $this->connection->query_select_value_if_there('users', 'user_id', array('username_clean' => strtolower($name)));
    }

    /**
     * Get the IDs of the admin usergroups.
     *
     * @return array The admin usergroup IDs
     */
    protected function _get_super_admin_groups()
    {
        $admin_group = $this->connection->query_select_value_if_there('groups', 'group_id', array('group_name' => 'ADMINISTRATORS'), 'ORDER BY group_id DESC');
        if (is_null($admin_group)) {
            return array();
        }
        return array($admin_group);
    }

    /**
     * Get the IDs of the moderator usergroups.
     * It should not be assumed that a member only has one usergroup - this depends upon the forum the driver works for. It also does not take the staff site filter into account.
     *
     * @return array The moderator usergroup IDs
     */
    protected function _get_moderator_groups()
    {
        $moderator_group = $this->connection->query_value_if_there('SELECT group_id FROM ' . $this->connection->get_table_prefix() . 'groups WHERE group_name=\'GLOBAL_MODERATORS\'');
        if (is_null($moderator_group)) {
            return array();
        }
        return array($moderator_group);
    }

    /**
     * Get the forum usergroup list.
     *
     * @return array The usergroup list
     */
    protected function _get_usergroup_list()
    {
        $results = $this->connection->query('SELECT group_id,group_name FROM ' . $this->connection->get_table_prefix() . 'groups');
        $results2 = collapse_2d_complexity('group_id', 'group_name', $results);
        return $results2;
    }

    /**
     * Get the forum usergroup relating to the specified member ID.
     *
     * @param  MEMBER $member The member ID
     * @return array The array of forum usergroups
     */
    protected function _get_members_groups($member)
    {
        //if ($member == $this->get_guest_id()) return array(1); May not hold true

        $groups = collapse_1d_complexity('group_id', $this->connection->query_select('user_group', array('group_id'), array('user_pending' => 0, 'user_id' => $member)));
        $groups[] = $this->get_member_row_field($member, 'group_id');
        if (count($groups) <= 1) {
            $all_groups = $this->get_usergroup_list();
            if ((count($groups) == 0) || (!array_key_exists($groups[0], $all_groups))) {
                $groups[] = 1;
            }
        }
        return $groups;
    }

    /**
     * The hashing algorithm of this forum driver. NOT used for cookie logins for this forum driver (cookies store a generated session ID).
     *
     * @param  string $data The data to hash (the password in actuality)
     * @param  string $key The string converted member-ID in actuality, although this function is more general. For cookie logins, 'ys'
     * @param  boolean $just_first Whether to just get the old style hash
     * @return string The hashed data
     */
    public function forum_md5($data, $key, $just_first = false)
    {
        $itoa64 = './0123456789ABCDEFGHIJKLMNOPQRSTUVWXYZabcdefghijklmnopqrstuvwxyz';
        $hash = $GLOBALS['FORUM_DB']->query_select_value_if_there('users', 'user_password', array('username_clean' => strtolower($key)));
        if (is_null($hash)) {
            return '';
        }

        $test = _hash_crypt_private($data, $hash, $itoa64);
        if ($test === false) {
            return '';
        }
        return $test;
    }

    /**
     * Create a member login cookie.
     *
     * @param  MEMBER $id The member ID
     * @param  ?SHORT_TEXT $name The username (null: lookup)
     * @param  string $password The password
     */
    public function forum_create_cookie($id, $name, $password)
    {
        $member_cookie_name = get_member_cookie();
        $colon_pos = strpos($member_cookie_name, ':');
        if ($colon_pos !== false) {
            $base = substr($member_cookie_name, 0, $colon_pos);
            $real_member_cookie = substr($member_cookie_name, $colon_pos + 1);
            $real_pass_cookie = substr(get_pass_cookie(), $colon_pos + 1);
            $real_session_cookie = 'sid';
        } else {
            $real_member_cookie = $member_cookie_name;
            $real_pass_cookie = get_pass_cookie();
            $real_session_cookie = preg_replace('#\_u$#', '_sid', $real_member_cookie);
        }

        require_code('crypt');
        $hash = substr(get_rand_password(), 0, 17);
        $this->connection->query_insert('sessions_keys', array('key_id' => md5($hash), 'user_id' => $id, 'last_ip' => ip2long(get_ip_address()), 'last_login' => time()));

        require_code('crypt');
        $session_id = get_rand_password();
        $this->connection->query_insert('sessions', array(
            'session_id' => $session_id,
            'session_user_id' => $id,
            'session_forum_id' => 0,
            'session_last_visit' => time(),
            'session_start' => time(),
            'session_time' => time(),
            'session_ip' => get_ip_address(),
            'session_browser' => get_browser_string(),
            'session_forwarded_for' => '',
            'session_page' => '',
            'session_viewonline' => 1,
            'session_autologin' => 1,
            'session_admin' => $this->_is_super_admin($id),
        ));

        $cookie = serialize(array($real_member_cookie => strval($id), $real_pass_cookie => $hash, $real_session_cookie => $session_id));

        if ($colon_pos !== false) {
            cms_setcookie($base, $cookie);
            $_COOKIE[$base] = $cookie;
        } else {
            cms_setcookie($real_member_cookie, strval($id));
            cms_setcookie($real_pass_cookie, $hash);
            cms_setcookie($real_session_cookie, $session_id);
            $_COOKIE[$real_member_cookie] = strval($id);
            $_COOKIE[$real_pass_cookie] = $hash;
            $_COOKIE[$real_session_cookie] = $session_id;
        }
    }

    /**
     * Find if the given member ID and password is valid. If username is null, then the member ID is used instead.
     * All authorisation, cookies, and form-logins, are passed through this function.
     * Some forums do cookie logins differently, so a Boolean is passed in to indicate whether it is a cookie login.
     *
     * @param  ?SHORT_TEXT $username The member username (null: don't use this in the authentication - but look it up using the ID if needed)
     * @param  MEMBER $userid The member ID
     * @param  SHORT_TEXT $password_hashed The md5-hashed password
     * @param  string $password_raw The raw password
     * @param  boolean $cookie_login Whether this is a cookie login
     * @return array A map of 'id' and 'error'. If 'id' is null, an error occurred and 'error' is set
     */
    public function forum_authorise_login($username, $userid, $password_hashed, $password_raw, $cookie_login = false)
    {
        $out = array();
        $out['id'] = null;

        if (is_null($userid)) {
            $rows = $this->connection->query_select('users', array('*'), array('username_clean' => strtolower($username)), '', 1);
            if (array_key_exists(0, $rows)) {
                $this->MEMBER_ROWS_CACHED[$rows[0]['user_id']] = $rows[0];
            }
        } else {
            $rows = array();
            $rows[0] = $this->get_member_row($userid);
        }

        if (!array_key_exists(0, $rows) || $rows[0] === null) { // All hands to lifeboats
            $out['error'] = (do_lang_tempcode('_MEMBER_NO_EXIST', $username));
            return $out;
        }
        $row = $rows[0];
        if ($this->is_banned($row['user_id'])) { // All hands to the guns
            $out['error'] = (do_lang_tempcode('YOU_ARE_BANNED'));
            return $out;
        }
        if ($cookie_login) {
            $lookup = $this->connection->query_select_value_if_there('sessions_keys', 'user_id', array('key_id' => md5($password_raw)));
            if ($row['user_id'] !== $lookup) {
                $out['error'] = (do_lang_tempcode('MEMBER_BAD_PASSWORD'));
                return $out;
            }
        } else {
            if ($row['user_password'] != $password_hashed) {
                $out['error'] = (do_lang_tempcode('MEMBER_BAD_PASSWORD'));
                return $out;
            }
        }

        $pos = strpos(get_member_cookie(), '_data:userid');
        require_code('users_active_actions');
        if ($pos !== false) {
            cms_eatcookie(substr(get_member_cookie(), 0, $pos) . '_sid');
        }

        $out['id'] = $row['user_id'];
        return $out;
    }

    /**
     * Get a first known IP address of the given member.
     *
     * @param  MEMBER $member The member ID
     * @return IP The IP address
     */
    public function get_member_ip($member)
    {
        $ip = $this->connection->query_select_value_if_there('posts', 'poster_ip', array('poster_id' => $member));
        if (!is_null($ip)) {
            return $this->_un_phpbb_ip($ip);
        }
        return '';
    }

    /**
     * Gets a whole member row from the database.
     *
     * @param  MEMBER $member The member ID
     * @return ?array The member row (null: no such member)
     */
    public function get_member_row($member)
    {
        if (array_key_exists($member, $this->MEMBER_ROWS_CACHED)) {
            return $this->MEMBER_ROWS_CACHED[$member];
        }

        $rows = $this->connection->query_select('users', array('*'), array('user_id' => $member), '', 1);
        if (!array_key_exists(0, $rows)) {
            return null;
        }
        $this->MEMBER_ROWS_CACHED[$member] = $rows[0];
        return $this->MEMBER_ROWS_CACHED[$member];
    }

    /**
     * Gets a named field of a member row from the database.
     *
     * @param  MEMBER $member The member ID
     * @param  string $field The field identifier
     * @return mixed The field
     */
    public function get_member_row_field($member, $field)
    {
        $row = $this->get_member_row($member);
        return is_null($row) ? null : $row[$field];
    }
}
